/*
** File: pci9xxx.h  
** Project: ADB3 core driver
** Purpose: Defintions for PLX Technology PCI9080 and PCI9656 chips.
**
** (C) Copyright Alpha Data 2009-2010
*/

#ifndef _ADB3_CORE_PCI9XXX_H
#define _ADB3_CORE_PCI9XXX_H

#include <df.h>

/*
** PCI9056 / PCI9080 / PCI9656 definitions
*/

#define PLX_CNTRL_GPO		          (1U << 16)
#define PLX_CNTRL_GPI		          (1U << 17)
#define PLX_CNTRL_EESK            (1U << 24) /* Value to drive on EEPROM serial clock pin EESK */
#define PLX_CNTRL_EECS            (1U << 25) /* Value to drive on EEPROM chip enable pin EECS (1 => asserted) */
#define PLX_CNTRL_EEDI            (1U << 26) /* Value to drive on EEPROM data pin EEDI */
#define PLX_CNTRL_EEDO            (1U << 27) /* Level on EEPROM data input pin EEDO */
#define PLX_CNTRL_EEPROM_PRESENT  (1U << 28) /* 1 => EEPROM present, 0 => EEPROM not present */
#define PLX_CNTRL_RST		          (1U << 30)
#define PLX_CNTRL_PCI9656_EEDI_OD (1U << 31) /* Active high output disable for EEPROM data pin EEDI/EEDO (PCI9656 only) */

#define PLX_INTCSR_PCI_IE	  (1U << 8)
#define PLX_INTCSR_DB_IE	  (1U << 9)
#define PLX_INTCSR_LOC_IE	  (1U << 11)
#define PLX_INTCSR_LOC_INT	(1U << 15)
#define PLX_INTCSR_DMA0_IE	(1U << 18)
#define PLX_INTCSR_DMA1_IE	(1U << 19)
#define PLX_INTCSR_DMA0_INT	(1U << 21)
#define PLX_INTCSR_DMA1_INT	(1U << 22)

#define PLX_DMACSR_ENABLE	(1U << 0)
#define PLX_DMACSR_START	(1U << 1)
#define PLX_DMACSR_ABORT	(1U << 2)
#define PLX_DMACSR_CLRINT	(1U << 3)
#define PLX_DMACSR_DONE		(1U << 4)

/*
**  PCI9056 / PCI9080 / PCI9656 chained DMA descriptor definitions
**
**  PCI9xxx Chaining DMA descriptor format is similar to the
**  internal registers of the device.
**
**  Format of the "link" member / DPR register is:
**
**  Bit   value   function
**
**    0     0     Descriptors are in local memory
**          1     Descriptors are in PCI memory
**
**    1     0     Fetch next descriptor
**          1     This is the last in the chain
**
**    2     0     No interrupt
**          1     Interrupt on completion of chain
**
**    3     0     Direction is from PCI to local
**          1     Direction is from local to PCI
**
** 31:4           Base address of the next descriptor quad aligned
**/

#define PCI9xxxDESCF_FETCH_LOCAL  (0U << 0)
#define PCI9xxxDESCF_FETCH_PCI    (1U << 0)
#define PCI9xxxDESCF_ENDCHAIN     (1U << 1)
#define PCI9xxxDESCF_INTERRUPT    (1U << 2)
#define PCI9xxxDESCF_PCI_TO_LOCAL (0U << 3)
#define PCI9xxxDESCF_LOCAL_TO_PCI (1U << 3)

#pragma pack(1)
typedef struct _Pci9xxxDmaDescriptor {
	uint32_t pci;
	uint32_t local;
	uint32_t count;
	uint32_t link;
} Pci9xxxDmaDescriptor;
#pragma pack()

/* This structure is one set of DMA engine registers */
#pragma pack(1)
typedef struct _Pci9xxxDmaRegisters {
	uint32_t MODE;
	uint32_t PADR;
	uint32_t LADR;
	uint32_t SIZE;
	uint32_t DPR;
} Pci9xxxDmaRegisters;
#pragma pack()

/*
** PCI9080-specific
*/

#define PCI9080_NUM_DMA_CHANNEL (2)

#pragma pack(1)
typedef struct _Pci9080Registers {
  uint32_t			LAS0RR;         					/*  0x00 */
  uint32_t			LAS0BA;         					/*  0x04 */
  uint32_t			MARBR;          					/*  0x08 */
  uint32_t			BIGEND;         					/*  0x0C */
  uint32_t			EROMRR;         					/*  0x10 */
  uint32_t			EROMBA;         					/*  0x14 */
  uint32_t			LBRD0;          					/*  0x18 */
  uint32_t			DMRR;           					/*  0x1C */
  uint32_t			DMLBAM;         					/*  0x20 */
  uint32_t			DMLBAI;         					/*  0x24 */
  uint32_t			DMPBAM;         					/*  0x28 */
  uint32_t			DMCFGA;         					/*  0x2C */
  uint32_t			OPLFIS;         					/*  0x30 */
  uint32_t			OPLFIM;         					/*  0x34 */
  uint32_t			reserved38[2];  					/*  0x38 */
  uint32_t			MAILBOX[8];     					/*  0x40 */
  uint32_t			P2LDBELL;       					/*  0x60 */
  uint32_t			L2PDBELL;       					/*  0x64 */
  uint32_t			INTCSR;         					/*  0x68 */
  uint32_t			CNTRL;          					/*  0x6C */
  uint32_t			PCIHDR;         					/*  0x70 */
  uint16_t			PCIHREV;        					/*  0x74 */
  uint16_t			reserved76;							  /*  0x76 */
  uint32_t			MBOX0;							      /*  0x78  always present */
  uint32_t			MBOX1;							      /*  0x7C  always present */
  Pci9xxxDmaRegisters DMA[PCI9080_NUM_DMA_CHANNEL]; /*  0x80 */
  union {
    uint8_t			asByte[PCI9080_NUM_DMA_CHANNEL];
    uint16_t    asWord;
  } DMACSR;                               /*  0xA8 */
  uint8_t				reservedAA[2];						/*  0xAA */
  uint32_t			DMAARB;         					/*  0xAC */
  uint32_t			DMATHR;         					/*  0xB0 */
  uint32_t			reservedB4[3];  					/*  0xB4 */
  uint32_t			MQCR;           					/*  0xC0 */
  uint32_t			QBAR;           					/*  0xC4 */
  uint32_t			IFHPR;          					/*  0xC8 */
  uint32_t			IFTPR;          					/*  0xCC */
  uint32_t			IPHPR;          					/*  0xD0 */
  uint32_t			IPTPR;          					/*  0xD4 */
  uint32_t			OFHPR;          					/*  0xD8 */
  uint32_t			OFTPR;          					/*  0xDC */
  uint32_t			OPHPR;          					/*  0xE0 */
  uint32_t			OPTPR;          					/*  0xE4 */
  uint32_t			QSR;            					/*  0xE8 */
  uint32_t			reservedEC;     					/*  0xEC */
  uint32_t			LAS1RR;         					/*  0xF0 */
  uint32_t			LAS1BA;         					/*  0xF4 */
  uint32_t			LBRD1;          					/*  0xF8 */
  uint32_t			reservedFC;	    					/*  0xFC */
} Pci9080Registers;
#pragma pack()

/*
** PCI9x56-specific
*/

#define PCI9656_NUM_DMA_CHANNEL (2)

#define PCI9656_DMAMODE_WIDTHMASK        (0x3U << 0)
#define PCI9656_DMAMODE_8BIT             (0x0U << 0)
#define PCI9656_DMAMODE_16BIT            (0x1U << 0)
#define PCI9656_DMAMODE_32BIT            (0x2U << 0)
#define PCI9656_DMAMODE_USE_READY        (0x1U << 6)
#define PCI9656_DMAMODE_USE_BTERM        (0x1U << 7)
#define PCI9656_DMAMODE_ENABLE_BURST     (0x1U << 8)
#define PCI9656_DMAMODE_CHAINED          (0x1U << 9)
#define PCI9656_DMAMODE_DONE_INTERRUPT   (0x1U << 10)
#define PCI9656_DMAMODE_FIXED_LOCAL      (0x1U << 11)
#define PCI9656_DMAMODE_DEMAND           (0x1U << 12)
#define PCI9656_DMAMODE_WRITE_INVALIDATE (0x1U << 13)
#define PCI9656_DMAMODE_INTERRUPT_PCI    (0x1U << 17)

#pragma pack(1)
typedef struct _Pci9656Registers {
  uint32_t			LAS0RR;								/*  00 */
  uint32_t			LAS0BA;								/*  04 */
  uint32_t			MARBR;								/*  08 */
  uint8_t				BIGEND;								/*  0C */
  uint8_t				LMISC1;								/*  0D */
  uint8_t				PROT_AREA;					  /*  0E */
  uint8_t				LMISC2;								/*  0F */
  uint32_t			EROMRR;								/*  10 */
  uint32_t			EROMBA;								/*  14 */
  uint32_t			LBRD0;								/*  18 */
  uint32_t			DMRR;								  /*  1C */
  uint32_t			DMLBAM;								/*  20 */
  uint32_t			DMLBAI;								/*  24 */
  uint32_t			DMPBAM;								/*  28 */
  uint32_t			DMCFGA;								/*  2C */
  uint32_t			OPQIS;								/*  30 */
  uint32_t			OPQIM;								/*  34 */
  uint8_t				reserved1[8];				  /*  38 */
  union {
    struct {
      uint32_t	MBOX0;								/*  40 */
      uint32_t	MBOX1;								/*  44 */
      uint32_t	MBOX2;								/*  48 */
      uint32_t	MBOX3;								/*  4C */
      uint32_t	MBOX4;								/*  50 */
      uint32_t	MBOX5;								/*  54 */
      uint32_t	MBOX6;								/*  58 */
      uint32_t	MBOX7;								/*  5C */
    } NoI2O;
    struct {
      uint32_t	IQP;								  /*  40 */
      uint32_t	OQP;								  /*  44 */
      uint32_t	MBOX2;								/*  48 */
      uint32_t	MBOX3;								/*  4C */
      uint32_t	MBOX4;								/*  50 */
      uint32_t	MBOX5;								/*  54 */
      uint32_t	MBOX6;								/*  58 */
      uint32_t	MBOX7;								/*  5C */
    } WithI2O;
  } MBOXA;
  uint32_t			P2LDBELL;							/*  60 */
  uint32_t			LD2PBELL;							/*  64 */
  uint32_t			INTCSR;								/*  68 */
  uint32_t			CNTRL;								/*  6C */
  uint32_t			PCIHIDR;							/*  70 */
  uint8_t				PCIHREV;							/*  74 */
  uint8_t				reserved2[3];				  /*  75 */
  union {
    struct {
      uint32_t  reserved3[2];
    } NoI2O;
    struct {
      uint32_t	MBOX0;								/*  78 */
      uint32_t	MBOX1;								/*  7C */
    } WithI2O;
  } MBOXB;
  Pci9xxxDmaRegisters DMA[PCI9656_NUM_DMA_CHANNEL];		/*  80 */
  uint16_t      DMACSR;               /*  A8 */
  uint8_t				reserved4[2];					/*  AA */
  uint32_t			DMAARB;								/*  AC */
  uint32_t			DMATHR;								/*  B0 */
  uint32_t			DMADAC0;							/*  B4 */
  uint32_t			DMADAC1;							/*  B8 */
  uint8_t				reserved5[4];					/*  BC */
  uint32_t			MQCR;								  /*  C0 */
  uint32_t			QBAR;								  /*  C4 */
  uint32_t			IFHPR;								/*  C8 */
  uint32_t			IFTPR;								/*  CC */
  uint32_t			IPHPR;								/*  D0 */
  uint32_t			IPTPR;								/*  D4 */
  uint32_t			OFHPR;								/*  D8 */
  uint32_t			OFTPR;								/*  DC */
  uint32_t			OPHPR;								/*  E0 */
  uint32_t			OPTPR;								/*  E4 */
  uint32_t			QSR;								  /*  E8 */
  uint32_t			reserved6;					  /*  EC */
  uint32_t			LAS1RR;								/*  F0 */
  uint32_t			LAS1BA;								/*  F4 */
  uint32_t			LBRD1;								/*	F8 */
  uint32_t			DMDAC;								/*  FC */
  uint32_t			PCIARB;								/* 100 */
  uint32_t			PABTADR;							/* 104 */
} Pci9656Registers;
#pragma pack()

#endif
