/*
** File: model_admxrc7v1.c  
** Project: ADB3 core driver
** Purpose: Code specific to ADM-XRC-7V1 & ADM-VPX3-7V2, including function to bootstrap device context.
**
** (C) Copyright Alpha Data 2012
**
** NOTES
**
** 1. Organization of Flash device U?? (all offsets in bytes):
**
**  0x0000000 - 0x07FFFFF  Alternate ADB3 PCI-E to OCP Bridge bitstream
**  0x0800000 - 0x0FFFFFF  Default ADB3 PCI-E to OCP Bridge bitstream
**  0x1000000 - 0x10FFFFF  Alpha Data VPD region
**  0x1100000 - 0x11FFFFF  Customer VPD region
**  0x1200000 - 0x28FFFFF  Target FPGA bitstream
**  0x2900000 - 0x3FFFFFF  Failsafe target FPGA bitstream
**
** 2. Frequency synthesizer in PCI-E Bridge is programmable clock index 0.
*/

#include <df.h>

#include "adb3.h"
#include "adb3_common.h"
#include "admxrc7v1.h"
#include "admxrc6tx_common.h"
#include "avr_common.h"
#include "avr_clock_common.h"
#include "avr_flash.h"
#include "avr_fmc.h"
#include "avr_sensor.h"
#include "coreclock.h"
#include "coreif.h"
#include "coredma.h"
#include "device.h"
#include "model_boot.h"
#include "flash.h"
#include "flash_cfi.h"
#include "i2c_common.h"
#include "v6clksynth.h"
#include "v6clksynth_common.h"

/* TO DO - error values */
static CoreSensorInfo g_avrVoltageInfoAdmxrc7v1[] = {
  { "5V/12V XMC VPWR rail",        0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC1 */
  { "12V XMC power rail",          0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC2 */
  { "5V XMC power rail",           0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC3 */
  { "3.3V XMC power rail",         0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC4 */
  { "2.5V power rail",             0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC5 */
  { "2.0V target VCCAux IO",       0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC6 */
  { "1.8V power rail",             0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC7 */
  { "1.8V target VCCAux",          0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_B1 */
  { "1.8V target MGT VCCAux",      0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_B2 */
  { "1.5V DDR3 SDRAM power rail",  0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_B3 */
  { "XRM variable VI/O rail",      0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_B4 */
  { "1.0V power rail",             0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_B5 */
  { "1.2V target MGT AVTT",        0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_B6 */
  { "1.0V target MGT AVCC",        0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }  /* AVR ADC_B7 */
};

/* TO DO - error values */
static CoreSensorInfo g_avrVoltageInfoAdmvpx37v2[] = {
  { "12V VPX power rail",          0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC1 */
  { "5V VPX power rail",           0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC2 */
  { "3.3V VPX power rail",         0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC3 */
  { "2.5V power rail",             0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC4 */
  { "2.0V target VCCAux IO",       0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC5 */
  { "1.8V power rail",             0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC6 */
  { "1.8V target VCCAux",          0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC7 */
  { "1.8V target MGT VCCAux",      0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_B1 */
  { "1.5V DDR3 SDRAM power rail",  0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_B2 */
  { "FMC VADJ power rail",         0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_B3 */
  { "1.0V power rail",             0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_B4 */
  { "1.2V target MGT AVCC",        0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_B5 */
  { "1.0V target MGT AVCC",        0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC_B6 */
  { "1.2V bridge MGT AVCC",        0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }  /* AVR ADC_B7 */
};

/* TO DO - error values */
static CoreSensorInfo g_avrTempInfo[] = {
  { "uC internal temperature",     0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* Microcontroller internal temp */
  { "Board temp. (central)"      , 0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* Temperature sensor near middle of board */
  { "Bridge FPGA int temp. diode", 0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* Temperature sensor diode in bridge FPGA */
  { "Target FPGA int temp. diode", 0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }  /* Temperature sensor diode in target FPGA */
};

/* TO DO - error values */
static CoreSensorInfo g_avrMiscInfo[] = {
  { "Total powered on time",       0, { 0, 16384 }, CoreSensorDataDouble, CoreSensorUnitS,    0 }, /* Number of seconds spent powered on */
  { "Event count",                 0, { 0,     0 }, CoreSensorDataUInt32, CoreSensorUnitNone, 0 }  /* Number of elapsed events */
};

/* These correspond to 'g_avrVoltageInfo' above, to convert the raw ADC values into Volt */
static AvrVoltageScaling g_avrVoltageScaling[] = {
  { 10U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC1 */
  { 12U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC2 */
  { 14U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC3 */
  { 16U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC4 */
  { 18U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC5 */
  { 20U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC6 */
  { 22U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC7 */
  { 26U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_B1 */
  { 28U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_B2 */
  { 30U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_B3 */
  { 32U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_B4 */
  { 34U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_B5 */
  { 36U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }, /* AVR ADC_B6 */
  { 38U, 0xFFFFU, FALSE, 0x04000  /* factor = 1/4096 */ }  /* AVR ADC_B7 */
};

/* These correspond to 'g_avrTempInfo' above, to convert the raw ADC values into deg. C */
static AvrTempScaling g_avrTempScaling[] = {
  { 40U, 0xFFFFU, FALSE, 0x10000, 0xFEEED99A, /* factor = 1/16, offset = -273.15 */ },
  { 42U, 0xFFFFU, FALSE, 0x10000, 0xFEEED99A, /* factor = 1/16, offset = -273.15 */ },
  { 44U, 0xFFFFU, FALSE, 0x10000, 0xFEEED99A, /* factor = 1/16, offset = -273.15 */ },
  { 46U, 0xFFFFU, FALSE, 0x10000, 0xFEEED99A, /* factor = 1/16, offset = -273.15 */ }
};

static AvrMiscScaling g_avrMiscScaling = {
  { 0U },
  { 4U, FALSE }
};

static void
initSensorInfo(
  Adb3CoreDeviceContext* pDevCtx)
{
  unsigned int i = 0;

  if (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice == PCI_SUBDEVICE_ADB3_ADMVPX37V2) {
    dfCopyKK(pDevCtx->info.sensor + i, g_avrVoltageInfoAdmvpx37v2, sizeof(g_avrVoltageInfoAdmvpx37v2));
    i += (unsigned int)DF_ARRAY_LENGTH(g_avrVoltageInfoAdmvpx37v2);
  } else {
    dfCopyKK(pDevCtx->info.sensor + i, g_avrVoltageInfoAdmxrc7v1, sizeof(g_avrVoltageInfoAdmxrc7v1));
    i += (unsigned int)DF_ARRAY_LENGTH(g_avrVoltageInfoAdmxrc7v1);
  }
  dfCopyKK(pDevCtx->info.sensor + i, g_avrTempInfo, sizeof(g_avrTempInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_avrTempInfo);
  i += admxrc6txInitSensorInfoXsmon(pDevCtx, i, TRUE);
  dfCopyKK(pDevCtx->info.sensor + i, g_avrMiscInfo, sizeof(g_avrMiscInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_avrMiscInfo);
}

static boolean_t
createFakeVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  VpdAdmxrc7v1Rev0* pVpd = &pDevCtx->info.vpd.pData->admxrc7v1.rev0;
  int bank;

  DF_ZERO_OBJECT(pVpd);
  pVpd->version = 0;
  pVpd->length = sizeof(VpdAdmxrc7v1Rev0);
  pVpd->serial = 100U;
  pVpd->modifications = 0U;
  pVpd->si5338Ref = 25000000;
  for (bank = 0; bank < DF_ARRAY_LENGTH(pVpd->sdram); bank++) {
    pVpd->sdram[bank].order = 27; /* 512 MiB per bank */
    pVpd->sdram[bank].speed = 8000;
    pVpd->sdram[bank].link = 0;
  }
  pVpd->fpga.device = 258; /* 7VX485T */
  dfAssert(sizeof(char) == sizeof(uint8_t));
  dfStrCpy((char*)pVpd->fpga.speedGrade, DF_ARRAY_LENGTH(pVpd->fpga.speedGrade), "1");
  dfStrCpy((char*)pVpd->fpga.stepping, DF_ARRAY_LENGTH(pVpd->fpga.stepping), "IES");
  pVpd->fpga.tempGrade = ADMXRC7V1_VPD_TEMPGRADE_COMMERCIAL;
  pVpd->pcbRev = 0x10U;
  pVpd->cpldRev = 0x10U;

  pVpd->checksum = 0U;

  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE;
}

static boolean_t
readVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  CoreVpdStatus status;
  VpdAdmxrc7v1Rev0* pVpd = &pDevCtx->info.vpd.pData->admxrc7v1.rev0;
  VpdAdmxrc7v1Rev0 vpdRaw;
  uint16_t vpdVersion = 0xffffU;
  uint16_t vpdLength = 0U;
  int i;

  status = admxrc6txReadAvrVpdData(pDevCtx, 0, sizeof(vpdVersion), &vpdVersion);
  if (CoreVpdSuccess != status) {
  	dfDebugPrint(0, ("*** readVpd: failed to read VPD version field\n"));
    return FALSE;
  }
  vpdVersion = dfLe16ToCpu(vpdVersion);
  status = admxrc6txReadAvrVpdData(pDevCtx, 2, sizeof(vpdLength), &vpdLength);
  if (CoreVpdSuccess != status) {
  	dfDebugPrint(0, ("*** readVpd: failed to read VPD length field\n"));
    return FALSE;
  }
  vpdLength = dfLe16ToCpu(vpdLength);
  switch (vpdVersion) {
  case 0:
    /* Ok */
    break;

  case 0xffffU:
  	dfDebugPrint(0, ("*** readVpd: invalid VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    return FALSE;

  default:
  	dfDebugPrint(0, ("+++ readVpd: unrecognised VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    break;
  }

  admxrc6txVerifyAvrVpdChecksum(pDevCtx, vpdLength);

  status = admxrc6txReadAvrVpdData(pDevCtx, 0, sizeof(vpdRaw), &vpdRaw);
  if (CoreVpdSuccess != status) {
  	dfDebugPrint(0, ("*** readVpd: failed to block read VPD\n"));
    return FALSE;
  }
  DF_ZERO_OBJECT(pVpd);
  pVpd->version = vpdVersion;
  pVpd->length = vpdLength;
  pVpd->serial = dfLe32ToCpu(vpdRaw.serial);
  pVpd->modifications = dfLe32ToCpu(vpdRaw.modifications);
  pVpd->si5338Ref = dfLe32ToCpu(vpdRaw.si5338Ref);
  for (i = 0; i < DF_ARRAY_LENGTH(vpdRaw.sdram); i++) {
    pVpd->sdram[i].order = dfLe8ToCpu(vpdRaw.sdram[i].order);
    pVpd->sdram[i].speed = dfLe16ToCpu(vpdRaw.sdram[i].speed);
    pVpd->sdram[i].link = dfLe16ToCpu(vpdRaw.sdram[i].link);
  }
  pVpd->fpga.device = dfLe16ToCpu(vpdRaw.fpga.device);
  dfAssert(sizeof(char) == sizeof(uint8_t));
  dfStrCpy((char*)pVpd->fpga.speedGrade, DF_ARRAY_LENGTH(pVpd->fpga.speedGrade), (const char*)vpdRaw.fpga.speedGrade);
  dfStrCpy((char*)pVpd->fpga.stepping, DF_ARRAY_LENGTH(pVpd->fpga.stepping), (const char*)vpdRaw.fpga.stepping);
  pVpd->fpga.tempGrade = dfLe8ToCpu(vpdRaw.fpga.tempGrade);
  pVpd->pcbRev = dfLe8ToCpu(vpdRaw.pcbRev);
  pVpd->checksum = dfLe8ToCpu(vpdRaw.checksum);

  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE;
}

static void
enableAvrRxNotEmptyInt(
  AvrDeviceContext* pAvrCtx)
{
	Adb3CoreDeviceContext* pDevCtx = DF_CONTAINER_OF(pAvrCtx, Adb3CoreDeviceContext, model.admxrc7v1.avr.context);
  DfInterruptObject* pInterruptObject = &pDevCtx->interrupt.object;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  DfSpinLockFlags f;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  f = dfInterruptSpinLockGet(pInterruptObject);
  pDevCtx->model.admxrc7v1.shadow.intCtl |= ADMXRC7V1_V_INTCTL_ADVRXNE;
  dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admxrc7v1.shadow.intCtl | ADMXRC7V1_V_INTCTL_LEVELBITS);
  dfPciMemRead32(hModel, &pModelRegs->intCtl);
  dfInterruptSpinLockPut(pInterruptObject, f);
}

static void
enableAvrTxEmptyInt(
  AvrDeviceContext* pAvrCtx)
{
	Adb3CoreDeviceContext* pDevCtx = DF_CONTAINER_OF(pAvrCtx, Adb3CoreDeviceContext, model.admxrc7v1.avr.context);
  DfInterruptObject* pInterruptObject = &pDevCtx->interrupt.object;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  DfSpinLockFlags f;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  f = dfInterruptSpinLockGet(pInterruptObject);
  pDevCtx->model.admxrc7v1.shadow.intCtl |= ADMXRC7V1_V_INTCTL_ADVTXE;
  dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admxrc7v1.shadow.intCtl | ADMXRC7V1_V_INTCTL_LEVELBITS);
  dfPciMemRead32(hModel, &pModelRegs->intCtl);
  dfInterruptSpinLockPut(pInterruptObject, f);
}

/* Returns TRUE if alert interrupt is active for specified target FPGA */
static boolean_t
alertPoll(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int targetIndex)
{
  DfInterruptObject* pInterruptObject;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t intStat;
  DfSpinLockFlags f;

  if (targetIndex > 0) {
    return FALSE;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  intStat = dfPciMemRead32(hModel, &pModelRegs->intStat);
  if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
    /* Alert interrupt still active */
    dfDebugPrint(9, ("alertPoll: alert interrupt still active\n"));
    return TRUE;
  } else {
    /* Alert interrupt no longer active - reenable alert interrupt */
    dfDebugPrint(9, ("alertPoll: alert interrupt no longer active\n"));
    pInterruptObject = &pDevCtx->interrupt.object;
    f = dfInterruptSpinLockGet(pInterruptObject);
    pDevCtx->model.admxrc7v1.shadow.intCtl |= ADMXRC6Tx_V_INTCTL_SYSENABLE;
    dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admxrc7v1.shadow.intCtl | ADMXRC7V1_V_INTCTL_LEVELBITS);
    dfPciMemRead32(hModel, &pModelRegs->intCtl);
    dfInterruptSpinLockPut(pInterruptObject, f);
    return FALSE;
  }
}

static void
onV6ClkSynthProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  V6ClkSynthDeviceContext* pV6ClkSynthCtx,
  V6ClkSynthProgramRequest* pRequest,
  V6ClkSynthProgramStatus status)
{
  onClockProgramDone(pDevCtx, 0, v6ClkSynthMapProgramStatus(status));
}

static void
onAvrClockProgramDone(
  AvrClockContext* pAvrClockCtx,
  AvrClockRequest* pRequest,
  AvrClockStatus status,
  unsigned int clockIndex,
  void* pContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pContext;

  switch (clockIndex) {
  case 0:
    clockIndex = 2;
    break;

  case 1:
    clockIndex = 1;
    break;

  case 2:
  case 3:
    clockIndex++;
    break;

  default:
    dfAssert(FALSE);
    return;
  }
  onClockProgramDone(pDevCtx, clockIndex, avrClockMapStatus(status));
}

static ClockProgramStatus
clockProgram(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  const CoreClockWord* pClockWord)
{
  if (clockIndex >= pDevCtx->info.bootstrap.numClockGen) {
    return ClockProgramInvalidIndex;
  }

  switch (clockIndex) {
  case 0:
    return v6ClkSynthMapProgramStatus(v6ClkSynthProgram(pDevCtx, &pDevCtx->model.admxrc7v1.v6ClkSynth.context, &pDevCtx->model.admxrc7v1.v6ClkSynth.programming, pClockWord, onV6ClkSynthProgramDone));

  case 1:
    clockIndex = 1;
    break;

  case 2:
    clockIndex = 0;
    break;

  case 3:
  case 4:
    clockIndex--;
    break;

  default:
    return ClockProgramInvalidIndex;
  }

  return avrClockMapStatus(avrClockSetFrequency(&pDevCtx->model.admxrc7v1.avrClock.context, &pDevCtx->model.admxrc7v1.avrClock.request[clockIndex], clockIndex, pClockWord, onAvrClockProgramDone, (void*)pDevCtx));
}

static ClockWordStatus
clockWord(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  uint32_t flags,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  if (clockIndex >= pDevCtx->info.bootstrap.numClockGen) {
    return ClockWordInvalidIndex;
  }

  switch (clockIndex) {
  case 0:
    return v6ClkSynthMapClockWordStatus(v6ClkSynthClockWord(flags, pDevCtx->model.admxrc7v1.v6ClkSynth.refClkFrequency, frequencyReq, frequencyMin, frequencyMax, pClockWord));

  case 1:
  case 2:
  case 3:
  case 4:
    /* Only frequency field of clock word is used */
    pClockWord->frequency = frequencyReq;
    return ClockWordSuccess;

  default:
    return ClockWordInvalidIndex;
  }
}

static void
admvpx37v2FlashRemap(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t logicalAddress,
  uint64_t length,
  uint64_t* pPhysicalAddress,
  uint64_t* pLengthOut)
{
  /*
  ** If 128 MiB Flash device is fitted, remap Flash addresses. This is because in order for
  ** the firmware to be able to use either a 64 MiB or 128 MiB Flash device (appropriate to
  ** the FPGA device fitted), the target bitstreams are stored noncontiguously in the Flash
  ** device, in terms of physical address. The driver maps logical addresses to physical
  ** addresses in order to make the target bitstream images logically contiguous.
  **
  ** Logical address           Usage                                Physical address
  ** -------------------------------------------------------------------------------------
  ** 0x00000000 - 0x11FFFFF => Bridge bitstreams / reserved      => 0x00000000 - 0x11FFFFF
  ** 0x12000000 - 0x28FFFFF => Target FPGA bitstream lo 23 MiB   => 0x12000000 - 0x28FFFFF
  ** 0x29000000 - 0x48FFFFF => Target FPGA bitstream hi 32 MiB   => 0x40000000 - 0x5FFFFFF (A)
  ** 0x49000000 - 0x5FFFFFF => Failsafe FPGA bitstream lo 23 MiB => 0x29000000 - 0x3FFFFFF (B)
  ** 0x60000000 - 0x7FFFFFF => Failsafe FPGA bitstream hi 32 MiB => 0x60000000 - 0x7FFFFFF
  **
  ** Thus, only the two regions marked (A) and (B) are remapped.
  */
  static const struct {
    uint64_t logicalStart;
    uint64_t physicalStart;
    uint64_t length;
  } regions[] = {
    { 0x29000000U, 0x4000000U, 0x2000000U }, /* (A) */
    { 0x49000000U, 0x2900000U, 0x1700000U }  /* (B) */
  };
  uint64_t offset, regionLength, physicalAddress = logicalAddress, newLength = length;
  unsigned int i;

  if (bankIndex != 0) {
    goto done;
  }

  if (pDevCtx->info.flash[0].totalSize <= 64*1024*1024) {
    /* Remapping not necessary for 64 MiB Flash device */
    goto done;
  }

  for (i = 0; i < DF_ARRAY_LENGTH(regions); i++) {
    if (logicalAddress >= regions[i].logicalStart) {
      regionLength = regions[i].length;
      offset = logicalAddress - regions[i].logicalStart;
      if (offset < regionLength) {
        physicalAddress = regions[i].physicalStart + offset;
        if (offset + length > regionLength) {
          newLength = regions[i].length - offset;
        }
      }
    }
  }

done:
  *pPhysicalAddress = physicalAddress;
  *pLengthOut = newLength;
}

static void
flash(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  boolean_t bWrite,
  uint64_t location,
  uint32_t* pValue)
{
  boolean_t bIsAdmvpx37v2 = (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice == PCI_SUBDEVICE_ADB3_ADMVPX37V2) ? TRUE : FALSE;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t flctl, flpage;

  dfDebugPrint(9, ("flash: bankIndex=%u bWrite=%lu location=0x%08lx_%08lx *pValue=0x%lx\n",
    bankIndex, (unsigned long)bWrite, dfSplitUint64(location), (unsigned long)*pValue));

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  if (location >= 0x10000000U) {
    return;
  }

  hModel = pDevCtx->hardware.hModel;

  if (bIsAdmvpx37v2) {
    flpage = (uint32_t)location & 0x7fe0000U;
  } else {
    flpage = (uint32_t)location & 0x3fe0000U;
  }
  dfPciMemWrite32(hModel, &pModelRegs->flash.addr, flpage);
  (void)dfPciMemRead32(hModel, &pModelRegs->flash.addr);

  if (bWrite) {
    /* Select mode 0, set fl_a, Drive fl_a, fl_d, fl_ce_n, fl_oe_n, fl_we_n, assert fl_ce_n */
    flctl = ((uint32_t)location & 0x1fffeU) | (0x7U << 24) | (0x1U << 28);
    dfPciMemWrite32(hModel, &pModelRegs->flash.ctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flash.ctl);

    /* Set fl_d */
    dfPciMemWrite32(hModel, &pModelRegs->flash.data, (uint16_t)*pValue);
    (void)dfPciMemRead32(hModel, &pModelRegs->flash.data);

    /* Assert fl_we_n */
    flctl |= (0x1U << 30);
    dfPciMemWrite32(hModel, &pModelRegs->flash.ctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flash.ctl);

    /* Deassert fl_we_n */
    flctl &= ~(0x1U << 30);
    dfPciMemWrite32(hModel, &pModelRegs->flash.ctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flash.ctl);

    /* Deassert fl_ce_n */
    flctl &= ~(0x1U << 28);
    dfPciMemWrite32(hModel, &pModelRegs->flash.ctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flash.ctl);

    /* Stop driving fl_a, fl_d, fl_ce_n, fl_oe_n, fl_we_n */
    flctl &= ~(0x7U << 24);
    dfPciMemWrite32(hModel, &pModelRegs->flash.ctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flash.ctl);
	} else {
    /* Select mode 0, set fl_a, Drive fl_a, fl_ce_n, fl_oe_n, fl_we_n, assert fl_ce_n */
    flctl = ((uint32_t)location & 0x1fffeU) | (0x3U << 25) | (0x1U << 28);
    dfPciMemWrite32(hModel, &pModelRegs->flash.ctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flash.ctl);

    /* Assert fl_oe_n */
    flctl |= (0x1U << 29);
    dfPciMemWrite32(hModel, &pModelRegs->flash.ctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flash.ctl);

    /* Read fl_d */
    *pValue = (uint16_t)dfPciMemRead32(hModel, &pModelRegs->flash.data);

    /* Deassert fl_oe_n */
    flctl &= ~(0x1U << 29);
    dfPciMemWrite32(hModel, &pModelRegs->flash.ctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flash.ctl);

    /* Deassert fl_ce_n */
    flctl &= ~(0x1U << 28);
    dfPciMemWrite32(hModel, &pModelRegs->flash.ctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flash.ctl);

    /* Stop driving fl_a, fl_ce_n, fl_oe_n, fl_we_n */
    flctl &= ~(0x3U << 25);
    dfPciMemWrite32(hModel, &pModelRegs->flash.ctl, flctl);
    (void)dfPciMemRead32(hModel, &pModelRegs->flash.ctl);

    dfDebugPrint(9, ("flash: *pValue=0x%lx\n", (unsigned long)*pValue));
	}
}

static FpgaControlStatus
fpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  boolean_t* pbValue)
{
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t val32;
  boolean_t bOk;

  dfDebugPrint(8, ("fpgaControl: index=%lu opCode=%lu pbValue=%p\n",
    (unsigned long)index, (unsigned long)opCode, pbValue));
  
  if (0 != index) {
    return FpgaControlInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  switch (opCode) {
  case CoreFpgaControlOpGetDone:
    bOk = TRUE;
    val32 = dfPciMemRead32(hModel, &pModelRegs->fpgaCtl);
    if (!(val32 & 0x4U)) {
      dfDebugPrint(9, ("fpgaControl: fpgaStat=0x%lx\n", (unsigned long)val32));
      bOk = FALSE;
    }
    *pbValue = bOk;
    dfDebugPrint(9, ("fpgaControl: opCode=GetDone val32=0x%lx *pbValue=%s\n", (unsigned long)val32, bOk ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetLinkStatus:
    bOk = TRUE;
    /* Check that MPTL is up */
    val32 = dfPciMemRead32(hModel, &pModelRegs->mptlStat);
    if ((val32 & 0x7U) != 0x7U) {
      dfDebugPrint(9, ("fpgaControl: mptlStat=0x%lx\n", (unsigned long)val32));
      bOk = FALSE;
    }
    *pbValue = bOk;
    dfDebugPrint(9, ("fpgaControl: opCode=GetLinkStat val32=0x%lx *pbValue=%s\n", (unsigned long)val32, bOk ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpGetInit:
    val32 = dfPciMemRead32(hModel, &pModelRegs->fpgaCtl);
    *pbValue = (val32 & 0x2U) ? TRUE : FALSE;
    dfDebugPrint(9, ("fpgaControl: opCode=GetInit val32=0x%lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    break;

  case CoreFpgaControlOpSetProg:
    val32 = pDevCtx->model.admxrc7v1.shadow.fpgaCtl;
    if (*pbValue) {
      val32 |= 0x1U;
    } else {
      val32 &= ~0x1U;
    }
    dfDebugPrint(9, ("fpgaControl: opCode=SetProg val32=0x%lx *pbValue=%s\n", (unsigned long)val32, *pbValue ? "TRUE" : "FALSE"));
    dfPciMemWrite32(hModel, &pModelRegs->fpgaCtl, val32);
    dfPciMemRead32(hModel, &pModelRegs->fpgaCtl);
    break;

  default:
    return FpgaControlInvalidOp;
  }

  return FpgaControlSuccess;
}

static FpgaSelectMapStatus
fpgaSelectMap(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  boolean_t bWrite,
  size_t length,
  void* pBuffer)
{
  const size_t mask32 = 0x3U;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t* pBuffer8;
  uint32_t* pBuffer32;
  size_t length32, align32;
  uint8_t val8;
  uint32_t val32;

  dfDebugPrint(8, ("fpgaSelectMap: index=%lu bWrite=%s length=" DF_FMT_U64 "(0x%08lx_%08lx) pBuffer=%p\n",
    (unsigned long)index, bWrite ? "TRUE" : "FALSE", (unsigned long long)length, dfSplitUint64((uint64_t)length), pBuffer));
  
  if (0 != index) {
    return FpgaSelectMapInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;
  pBuffer8 = (uint8_t*)pBuffer;

  if (bWrite) {
    /* Align pointer to 32-bit boundary using 8-bit transfers */
    align32 = (~(size_t)(uintptr_t)pBuffer8 + 1) & mask32;
    if (align32 > length) {
      align32 = length;
    }
    length -= align32;
    while (align32--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, &pModelRegs->selectMap[0], val8);
    }

    /* Do bulk of data using 32-bit transfers */
    length32 = length & ~mask32;
    pBuffer32 = (uint32_t*)pBuffer8;
    length -= length32;
    while (length32) {
      val32 = dfLe32ToCpu(*pBuffer32++);
      dfPciMemWrite32(hModel, (uint32_t*)&pModelRegs->selectMap[0], val32);
      length32 -= 4;
    }

    /* Do remainder using 8-bit transfers */
    pBuffer8 = (uint8_t*)pBuffer32;
    while (length--) {
      val8 = *pBuffer8++;
      dfPciMemWrite8(hModel, &pModelRegs->selectMap[0], val8);
    }

    dfPciMemRead32(hModel, &pModelRegs->fpgaCtl); /* Ensure above writes have propagated to destination */
  } else {
    while (length--) {
      val8 = dfPciMemRead8(hModel, &pModelRegs->selectMap[0]);
      *pBuffer8++ = val8;
    }
  }

  return FpgaSelectMapSuccess;
}

static boolean_t
initHardware(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bInit,
  unsigned int phase)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  unsigned int i;
  uint32_t fOutV6ClkSynth;
  uint32_t fOutAvrClock[ADMXRC7V1_NUM_AVR_CLOCK];
  uint8_t xrmStatus;
  boolean_t bIsFmc;

  dfDebugPrint(2, ("initHardware: bInit=%s\n", bInit ? "TRUE" : "FALSE"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  if (bInit) {
    switch (phase) {
    case 0:
      if (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice == PCI_SUBDEVICE_ADB3_ADMVPX37V2) {
        pDevCtx->info.model = CoreModelAdmvpx37v2;
      } else {
        pDevCtx->info.model = CoreModelAdmxrc7v1;
      }

      /* The set of Bridge interrupts that we are interested in */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnableMaster = ADB3IRQ_DMA_ALL | ADB3IRQ_MODEL;

      /* Ensure Bridge interrupts are disabled */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnable = 0;
      dfPciMemWrite32(hBridge, &pRegs->irqEnable, pDevCtx->hardware.shadow.bridge.adb3.irqEnable);
      dfPciMemRead32(hBridge, &pRegs->irqEnable);

      /* Enable model-specific interrupts that we are interested in */
      pDevCtx->model.admxrc7v1.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE | ADMXRC6Tx_V_INTCTL_SYSENABLE;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admxrc7v1.shadow.intCtl | ADMXRC7V1_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);

      /* Initialize DMA enables */
      dfPciMemWrite32(hBridge, &pRegs->dmaEnable, ADB3DMAEN_LOCAL_MASTER | ADB3DMAEN_DMA_ALL);

      if (pDevCtx->pDevObj->hardwareId.variant.pci.revision < 0x04U) {
        /* Reinitialize PCI-E packet control for rev 0x03 or lower; allow up to 3 PCIe packets in flight at a time */
        dfPciMemWrite32(hBridge, &pRegs->pcieControl, 0x02100100U);           
      }

      /* Figure out number of DMA engines present, and directions they support */
      pDevCtx->info.bootstrap.numDmaChannel = adb3CountDmaEngines(pDevCtx);

      /* Figure out if we need MSI bug workaround */
      pDevCtx->model.admxrc7v1.bNeedMsiWorkaround = !pDevCtx->interrupt.bPolledMode && adb3NeedMsiWorkaround(pDevCtx);

      /* Initialize DMA engine registers */
      adb3InitDmaEngines(pDevCtx, pDevCtx->info.bootstrap.numDmaChannel);

      /* Initialize FPGACTL register for 8-bit SelectMap port */
      pDevCtx->model.admxrc7v1.shadow.fpgaCtl = 0;
      dfPciMemWrite32(hModel, &pModelRegs->fpgaCtl, pDevCtx->model.admxrc7v1.shadow.fpgaCtl);
		  dfPciMemRead32(hModel, &pModelRegs->fpgaCtl);

      /* Identify Flash device */
      admxrc6txFlashResetBank(pDevCtx, &pDevCtx->hardware.model.pAdmxrc6tx->flash);
      if (flashCfiIdentify(pDevCtx, 0, 0)) {
        pDevCtx->info.flash[0].bPresent = TRUE;
        pDevCtx->info.flash[0].targetArea.start = 0x1200000U;
        pDevCtx->info.flash[0].targetArea.length = pDevCtx->info.flash[0].totalSize - pDevCtx->info.flash[0].targetArea.start;
      } else {
        dfDebugPrint(0, ("*** initHardware: failed to identify Flash\n"));
      }

      /* Read in VPD */
      if (!readVpd(pDevCtx)) {
        createFakeVpd(pDevCtx);
      }

      /* Identify any XRM module fitted */
      xrmStatus = dfPciMemRead8(hModel, pModelRegs->model3.avr.sensor + ADMXRC7V1_SENSOR_OFFSET_XRMSTATUS);
      bIsFmc = (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice == PCI_SUBDEVICE_ADB3_ADMVPX37V2) ? TRUE : FALSE;
      avrFmcIdentify(pDevCtx, bIsFmc, xrmStatus);

      /* Initialize V6 clock synthesizer stuff */
      pDevCtx->model.admxrc7v1.v6ClkSynth.refClkFrequency = 200000000U;
      if (!v6ClkSynthInit(pDevCtx, &pDevCtx->model.admxrc7v1.v6ClkSynth.context, pDevCtx->model.admxrc7v1.v6ClkSynth.refClkFrequency, &fOutV6ClkSynth)) {
        dfDebugPrint(0, ("*** initHardware: failed to initialize V6 clock synthesizer\n"));
      } else {
        dfDebugPrint(2, ("initHardware: V6 clock synthesiser initial frequency is %lu Hz\n", (unsigned long)fOutV6ClkSynth));
        pDevCtx->clockGenerator[0].currentFrequency = fOutV6ClkSynth;
      }

      /* Initialize information needed for production test */
      pDevCtx->info.prodTest.serialNumber = pDevCtx->info.vpd.pData->admxrc7v1.rev0.serial;
      pDevCtx->info.prodTest.cpldRevision = pDevCtx->info.vpd.pData->admxrc7v1.rev0.cpldRev;
      pDevCtx->info.prodTest.pcbRevision = pDevCtx->info.vpd.pData->admxrc7v1.rev0.pcbRev;
      pDevCtx->info.prodTest.bridgeDate = dfPciMemRead32(hModel, &pModelRegs->buildDate);
      pDevCtx->info.prodTest.bridgeTime = dfPciMemRead32(hModel, &pModelRegs->buildTime);

      /* Find base addresses and sizes of local bus windows */
      i = 0;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      i++;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[3].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[3].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.pBridgedPrefetch;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[3].size;
      i++;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = 0;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.model.pGeneric;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      i++;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = 0;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[0].size;
      i++;

      /* Initialize sensor information structures */
      initSensorInfo(pDevCtx);
      break;

    case 1:
      /* Initialize the AVR uC interface */
      if (avrInit(
        &pDevCtx->model.admxrc7v1.avr.context,
        &pDevCtx->interrupt.object,
        &pDevCtx->hardware.model.pAdmxrc6tx->model2.avr.curr.ctlStat,
        pDevCtx->hardware.hModel,
        ADMXRC7V1_AVR_FLASH_SIZE,
        ADMXRC7V1_AVR_FLASH_PAGE_SIZE,
        enableAvrRxNotEmptyInt,
        enableAvrTxEmptyInt))
      {
        pDevCtx->model.admxrc7v1.bAvrInitialized = TRUE;
      }

      /* Initialize the AVR clock programming context */
      if (avrClockInit(
        &pDevCtx->model.admxrc7v1.avrClock.context,
        &pDevCtx->model.admxrc7v1.avr.context,
        DF_ARRAY_LENGTH(fOutAvrClock),
        fOutAvrClock))
      {
        pDevCtx->model.admxrc7v1.bAvrClockInitialized = TRUE;
        dfDebugPrint(2, ("initHardware: AVR clock initial frequencies are { %lu, %lu, %lu, %lu } Hz\n",
          (unsigned long)fOutAvrClock[0], (unsigned long)fOutAvrClock[1], (unsigned long)fOutAvrClock[2], (unsigned long)fOutAvrClock[3]));
        pDevCtx->clockGenerator[1].currentFrequency = fOutAvrClock[1];
        pDevCtx->clockGenerator[2].currentFrequency = fOutAvrClock[0];
        pDevCtx->clockGenerator[3].currentFrequency = fOutAvrClock[2];
        pDevCtx->clockGenerator[4].currentFrequency = fOutAvrClock[3];
      }
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    switch (phase) {
    case 0:
      /* Deinitialize V6 clock synthesizer stuff */
      v6ClkSynthUninit(pDevCtx, &pDevCtx->model.admxrc7v1.v6ClkSynth.context);

      if (pDevCtx->hardware.bCanTouch) {
        /* Deinitialize DMA enables */
        dfPciMemWrite32(hBridge, &pRegs->dmaEnable, 0);
      }
      break;

    case 1:
      /* Deinitialize the AVR clock programming context */
      if (pDevCtx->model.admxrc7v1.bAvrClockInitialized) {
        avrClockUninit(&pDevCtx->model.admxrc7v1.avrClock.context);
        pDevCtx->model.admxrc7v1.bAvrClockInitialized = FALSE;
      }

      /* Deinitialize the AVR interface */
      if (pDevCtx->model.admxrc7v1.bAvrInitialized) {
        avrUninit(&pDevCtx->model.admxrc7v1.avr.context);
        pDevCtx->model.admxrc7v1.bAvrInitialized = FALSE;
      }
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  }

  return TRUE;
}

static boolean_t
isr(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t irqStatus, dmaIrqStatus;
  uint32_t intStat, intCtl;
  unsigned int channel, n;
  DfSpinLockFlags f;

  dfDebugPrint(8, ("isr: entered\n"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  /* Sample the currently active interrupts. */
  irqStatus = dfPciMemRead32(hBridge, &pRegs->irqStatus);

  f = dfIsrSpinLockGet(pInterruptObj);
  /* Mask the currently active interrupts with the interrupts we are interested in right now. */
  irqStatus &= pDevCtx->hardware.shadow.bridge.adb3.irqEnable;
  /* Clear the interrupts that we just saw. */
  dfPciMemWrite32(hBridge, &pRegs->irqStatus, irqStatus);

  if (irqStatus & ADB3IRQ_MODEL) {
    dfDebugPrint(9, ("isr: model-specific interrupt\n"));
    intCtl = pDevCtx->model.admxrc7v1.shadow.intCtl;
    intStat = dfPciMemRead32(hModel, &pModelRegs->intStat);
    intStat &= intCtl;
    if (intStat & ADMXRC7V1_V_INTSTAT_LEVELBITS) {
      if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
        /* Disable hardware monitor interrupt; gets reenabled by hardware monitor polling. */
        intCtl &= ~ADMXRC6Tx_V_INTCTL_SYSENABLE;
      }
      if (intStat & ADMXRC7V1_V_INTSTAT_ADVRXNE) {
        /* Disable RX FIFO not empty interrupt. */
        intCtl &= ~ADMXRC7V1_V_INTSTAT_ADVRXNE;
      }
      if (intStat & ADMXRC7V1_V_INTSTAT_ADVTXE) {
        /* Disable TX FIFO empty interrupt. */
        intCtl &= ~ADMXRC7V1_V_INTSTAT_ADVTXE;
      }
      /* Write new model-specific interrupt enables. */
      pDevCtx->model.admxrc7v1.shadow.intCtl = intCtl;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, intCtl | ADMXRC7V1_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);
    }
    dfIsrSpinLockPut(pInterruptObj, f);

    /* Clear the model-specific interrupts that we saw. */
    dfPciMemWrite32(hModel, &pModelRegs->intStat, intStat);

    /* Take action for the model-specific interrupts that we saw. */
    if (intStat & ADMXRC6Tx_V_INTSTAT_FP) {
      dfDebugPrint(9, ("isr: target FPGA 0 interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.fpga[0].dpc, (void*)(uintptr_t)0);
    }
    if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
      dfDebugPrint(9, ("isr: hardware monitor interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.hwMon[0].dpc, (void*)(uintptr_t)0);
    }
    if (intStat & ADMXRC7V1_V_INTSTAT_ADVRXNE) {
      dfDebugPrint(9, ("isr: AVR RX FIFO not empty interrupt\n"));
      dfDpcSchedule(&pDevCtx->model.admxrc7v1.avr.context.transport.rx.dpc, NULL);
    }
    if (intStat & ADMXRC7V1_V_INTSTAT_ADVTXE) {
      dfDebugPrint(9, ("isr: AVR TX FIFO empty interrupt\n"));
      dfDpcSchedule(&pDevCtx->model.admxrc7v1.avr.context.transport.tx.dpc, NULL);
    }
  } else {
    dfIsrSpinLockPut(pInterruptObj, f);
  }

  n = pDevCtx->info.bootstrap.numDmaChannel;
  for (channel = 0; channel < n; channel++) {
    if (irqStatus & (ADB3IRQ_DMA0 << channel)) {
      dmaIrqStatus = dfPciMemRead32(hBridge, &pRegs->dmaEngine[channel].irq.status);
      dfDebugPrint(9, ("isr: DMA%lu interrupt, DMA%lu.irqStatus=0x%08lx\n",
        (unsigned long)channel, (unsigned long)channel, (unsigned long)dmaIrqStatus));
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[channel].count++;
#endif
      dfPciMemWrite32(hBridge, &pRegs->dmaEngine[channel].irq.ack, dmaIrqStatus);
      dfDpcSchedule(&pDevCtx->interrupt.dma[channel].dpc, (void*)(uintptr_t)channel);
    }
  }

  if (pDevCtx->interrupt.bUsingMsi && pDevCtx->model.admxrc7v1.bNeedMsiWorkaround) {
    /* MSI problem workaround - disable all interrupts then reenable */
    adb3MsiWorkaround(pDevCtx);
  }

  return irqStatus ? TRUE : FALSE;
}

static boolean_t
mapBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated,
	boolean_t bUnmap)
{
	dfDebugPrint(1, ("mapBusResources: bUnmap=%s\n", bUnmap ? "TRUE" : "FALSE"));

	if (bUnmap) {
		if (NULL != pDevCtx->hardware.bridge.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.bridge.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 generic registers\n"));
			}
      pDevCtx->hardware.bridge.pGeneric = NULL;
		}
		if (NULL != pDevCtx->hardware.model.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.model.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 model-specific registers\n"));
			}
      pDevCtx->hardware.model.pGeneric = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridged) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridged)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 bridged space\n"));
			}
      pDevCtx->hardware.pBridged = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridgedPrefetch) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridgedPrefetch)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 prefetchable bridged space\n"));
			}
      pDevCtx->hardware.pBridgedPrefetch = NULL;
		}
		dfDebugPrint(2, ("mapBusResources: unmapped\n"));
	} else {
		pDevCtx->hardware.bridge.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 0, 0, pTranslated->variant.pci.memBar[0].size, &pDevCtx->hardware.hBridge);
		if (NULL == pDevCtx->hardware.bridge.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 generic registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 generic registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[0].base), pDevCtx->hardware.bridge.pGeneric));
		}
		pDevCtx->hardware.model.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 1, 0, pTranslated->variant.pci.memBar[1].size, &pDevCtx->hardware.hModel);
		if (NULL == pDevCtx->hardware.model.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 model-specific registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 model-specific registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[1].base), pDevCtx->hardware.model.pGeneric));
		}
    pDevCtx->hardware.pBridged = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 2, 0, pTranslated->variant.pci.memBar[2].size, &pDevCtx->hardware.hBridged);
		if (NULL == pDevCtx->hardware.pBridged) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 bridged space\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 bridged space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[2].base), pDevCtx->hardware.pBridged));
		}
    pDevCtx->hardware.pBridgedPrefetch = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 3, 0, pTranslated->variant.pci.memBar[3].size, &pDevCtx->hardware.hBridgedPrefetch);
    if (NULL == pDevCtx->hardware.pBridgedPrefetch) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 prefetchable bridged space\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped ADB3 prefetchable bridged space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[3].base), pDevCtx->hardware.pBridgedPrefetch));
    }
		dfDebugPrint(2, ("mapBusResources: mapped OK\n"));
	}

	return TRUE;
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  V6ClkSynthProgramStatus v6ClkSynthStatus;
  AvrClockStatus avrClkStatus;

  dfAssert(newState == 0 || newState == 3);

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
      /* Save V6 clock synthesizer state */
      v6ClkSynthSaveState(pDevCtx, &pDevCtx->model.admxrc7v1.v6ClkSynth.context);

      /* Save AVR clock programming state */
      avrClockSaveState(&pDevCtx->model.admxrc7v1.avrClock.context);
    } else { /* Powering up (from D3) */
      /* Reenable model-specific interrupts that we are interested in */
      pDevCtx->model.admxrc7v1.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE | ADMXRC6Tx_V_INTCTL_SYSENABLE;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admxrc7v1.shadow.intCtl | ADMXRC7V1_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);

      /* Reinitialize DMA enables */
      dfPciMemWrite32(hBridge, &pRegs->dmaEnable, ADB3DMAEN_LOCAL_MASTER | ADB3DMAEN_DMA_ALL);

      if (pDevCtx->pDevObj->hardwareId.variant.pci.revision < 0x04U) {
        /* Reinitialize PCI-E packet control for rev 0x03 or lower; allow up to 3 PCIe packets in flight at a time */
        dfPciMemWrite32(hBridge, &pRegs->pcieControl, 0x02100100U);           
      }

      /* Reinitialize DMA engine registers */
      adb3InitDmaEngines(pDevCtx, pDevCtx->info.bootstrap.numDmaChannel);

      /* Restore FPGACTL register for 8-bit SelectMap port */
      dfPciMemWrite32(hModel, &pModelRegs->fpgaCtl, pDevCtx->model.admxrc7v1.shadow.fpgaCtl);
		  dfPciMemRead32(hModel, &pModelRegs->fpgaCtl);

      /* Reset Flash device */
      admxrc6txFlashResetBank(pDevCtx, &pDevCtx->hardware.model.pAdmxrc6tx->flash);

      /* Restore AVR clock programming state */
      avrClkStatus = avrClockRestoreState(&pDevCtx->model.admxrc7v1.avrClock.context);
      if (AvrClockStatusSuccess != avrClkStatus) {
        dfDebugPrint(0, ("*** powerChange: failed to restore AVR clock programming state, status=%u\n", avrClkStatus));
      }

      /* Restore V6 clock synthesizer state */
      v6ClkSynthStatus = v6ClkSynthRestoreState(pDevCtx, &pDevCtx->model.admxrc7v1.v6ClkSynth.context);
      if (V6ClkSynthProgramStatusSuccess != v6ClkSynthStatus) {
        dfDebugPrint(0, ("*** powerChange: failed to restore V6 clock synthesizer state, status=%u\n", v6ClkSynthStatus));
      }
    }
  }
}

static CoreSensorStatus
readSensor(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreSensorValue* pReading)
{
  CoreSensorStatus status = CoreSensorInvalidIndex;

  if (index >= pDevCtx->info.bootstrap.numSensor) {
    return CoreSensorInvalidIndex;
  }

  status = avrSensorVoltageRead(pDevCtx, index, g_avrVoltageScaling, DF_ARRAY_LENGTH(g_avrVoltageScaling), pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_avrVoltageScaling);
  }

  status = avrSensorTempRead(pDevCtx, index, g_avrTempScaling, DF_ARRAY_LENGTH(g_avrTempScaling), pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_avrTempInfo);
  }

  status = admxrc6txReadSensorXsmon(pDevCtx, index, pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= ADMXRC6TX_NUM_XSMON_SENSOR;
  }

  status = avrSensorMiscRead(pDevCtx, index, &g_avrMiscScaling, pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_avrMiscInfo);
  }

  return status;
}

static CoreVpdStatus
mapAvrFlashStatus(
  AvrFlashStatus status)
{
  switch (status) {
  case AvrFlashSuccess:
    return CoreVpdSuccess;

  case AvrFlashTimeout:
    return CoreVpdTimeout;

  case AvrFlashNotServiceMode:
    return CoreVpdNotPresent;

  case AvrFlashInvalidRegion:
    return CoreVpdInvalidRegion;

  case AvrFlashGeneralFailure:
    return CoreVpdGeneralFailure;

  default:
    if (status >= AvrFlashHardwareFailure && status < AvrFlashHardwareFailure + 0x100U) {
      return CoreVpdGeneralFailure;
    }
    return CoreVpdGeneralFailure;
  }
}

static CoreVpdStatus
readWriteVpd(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWrite,
  size_t address,
  size_t length,
  void* pData)
{
  AvrFlashStatus status;

  if (CHECK_BOUNDS_OFFSET(address, length, pDevCtx->info.bootstrap.vpdOffset, pDevCtx->info.bootstrap.vpdLength)) {
    return CoreVpdInvalidRegion;
  }
  if (bWrite) {
    /* Casts are safe because of above bounds check */
    status = avrFlashWrite(pDevCtx, &pDevCtx->model.admxrc7v1.avr.context, (uint32_t)address, (uint32_t)length, pData);
  } else {
    /* Casts are safe because of above bounds check */
    status = avrFlashRead(pDevCtx, &pDevCtx->model.admxrc7v1.avr.context, (uint32_t)address, (uint32_t)length, pData);
  }
  return mapAvrFlashStatus(status);
}

static boolean_t
validateBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated)
{
	dfDebugPrint(1, ("validateBusResources: entered\n"));

	if (pTranslated->variant.pci.numMemBar < 4) {
		dfDebugPrint(0, ("*** validateBusResources: less than 4 memory BARs\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[0].size < 0x1000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR0 smaller than 0x1000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[1].size < sizeof(ModelRegsAdmxrc6tx)) {
		dfDebugPrint(0, ("*** validateBusResources: BAR1 smaller than 0x%lu bytes\n", (unsigned long)sizeof(ModelRegsAdmxrc6tx)));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[2].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR2 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
  if (pTranslated->variant.pci.memBar[3].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR3 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.numIrq != 1) {
		dfDebugPrint(0, ("*** validateBusResources: not exactly 1 interrupt\n"));
		return FALSE;
	}

	dfDebugPrint(2, ("validateBusResources: OK\n"));

	return TRUE;
}

static void
bootstrap(
	Adb3CoreDeviceContext* pDevCtx)
{
  boolean_t bExposeAllClocks = FALSE;
  boolean_t bIsAdmvpx37v2 = (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice == PCI_SUBDEVICE_ADB3_ADMVPX37V2) ? TRUE : FALSE;

  dfAssert(DF_OFFSET_OF(VpdAdmxrc7v1Rev0, checksum) == sizeof(VpdAdmxrc7v1Rev0) - 1);

  pDevCtx->methods.pAlertPoll = alertPoll;
  pDevCtx->methods.pClearErrors = adb3ClearErrors;
  pDevCtx->methods.pClockProgram = clockProgram;
  pDevCtx->methods.pClockWord = clockWord;
  pDevCtx->methods.pDmaList = adb3DmaList;
  pDevCtx->methods.pDmaTransfer = adb3DmaTransfer;
  pDevCtx->methods.pEnableInterrupts = adb3EnableInterrupts;
  pDevCtx->methods.pFlash = flash;
  if (bIsAdmvpx37v2) {
    pDevCtx->methods.pFlashRemap = admvpx37v2FlashRemap;
  }
  pDevCtx->methods.pFpgaControl = fpgaControl;
  pDevCtx->methods.pFpgaSelectMap = fpgaSelectMap;
  pDevCtx->methods.pGetDeviceStatus = adb3GetDeviceStatus;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIsr = isr;
  pDevCtx->methods.pMapBusResources = mapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pReadSensor = readSensor;
  pDevCtx->methods.pReadWriteVpd = readWriteVpd;
  pDevCtx->methods.pValidateBusResources = validateBusResources;
  pDevCtx->methods.pWindowConfig = adb3WindowConfig;

  pDevCtx->info.bootstrap.numClockGen = 1U;
  dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "Si5338ExposeAllClocks", &bExposeAllClocks);
  pDevCtx->info.bootstrap.numClockGen += bExposeAllClocks ? ADMXRC7V1_NUM_AVR_CLOCK : ADMXRC7V1_NUM_AVR_USER_CLOCK;
  pDevCtx->info.bootstrap.numDmaChannel = ADMXRC7V1_NUM_DMA_CHANNEL;
  pDevCtx->info.bootstrap.numIoModule = 1;
  pDevCtx->info.bootstrap.numSensor = bIsAdmvpx37v2 ? DF_ARRAY_LENGTH(g_avrVoltageInfoAdmvpx37v2) : DF_ARRAY_LENGTH(g_avrVoltageInfoAdmxrc7v1);
  pDevCtx->info.bootstrap.numSensor += (unsigned int)(DF_ARRAY_LENGTH(g_avrTempInfo) + DF_ARRAY_LENGTH(g_avrMiscInfo) + ADMXRC6TX_NUM_XSMON_SENSOR);
  pDevCtx->info.bootstrap.numTargetFpga = 1;
  pDevCtx->info.bootstrap.numWindow = 4;
  pDevCtx->info.bootstrap.numFlashBank = 1;
  pDevCtx->info.bootstrap.bVpdInFlash = FALSE;
  pDevCtx->info.bootstrap.vpdLength = ADMXRC7V1_AVR_FLASH_SIZE;
  pDevCtx->info.bootstrap.vpdOffset = 0;
  pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdmxrc7v1Rev0);
  pDevCtx->info.bootstrap.flash[0].deviceWidth = 2;
  pDevCtx->info.bootstrap.flash[0].width = 2;
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "PciAddress64Bit", &pDevCtx->info.bootstrap.b64BitDmaAddress);
  pDevCtx->info.bootstrap.dmaDescriptorSize = 32;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x1000000U;
  pDevCtx->info.bootstrap.bWorkAround4kCrossing = (pDevCtx->pDevObj->hardwareId.variant.pci.revision <= 0x01U) ? TRUE : FALSE;
  pDevCtx->info.bootstrap.bI2cInterruptDriven = FALSE;
}

void
adb3CoreBootstrapAdmxrc7v1(
	Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdmxrc7v1: entered\n"));
  bootstrap(pDevCtx);
}

void
adb3CoreBootstrapAdmvpx37v2(
	Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdmvpx37v2: entered\n"));
  bootstrap(pDevCtx);
}
