/*
** File: model_admxrc6tadv8.c  
** Project: ADB3 core driver
** Purpose: ADM-XRC-6T-ADV8 code, including function to bootstrap device
**          context.
**
** (C) Copyright Alpha Data 2011
**
** Notes for REVISION 2 board:
**
** 1. Organization of Flash device U37 (all offsets in bytes):
**
**  0x0000000 - 0x1FFFFFF  1st image (strap 00), failsafe ADB3 PCI-E to OCP
**                         Bridge bitstream
**  0x2000000 - 0x3FFFFFF  2nd image (strap 01), default ADB3 PCI-E to OCP
**                         Bridge bitstream
**  0x4000000 - 0x5FFFFFF  3rd image (strap 10), currently unused
**  0x6000000 - 0x7FFFFFF  4th image (strap 11), currently unused
**
** If a 64 MiB Flash device is fitted, the 3rd and 4th images are not
** available.
** If a 128 MiB Flash device is fitted, the 3rd and 4th images are available.
**
** 'strap' is a 2-bit binary value representing resistor straps on the board.
** The normal build of the ADM-XRC-6T-ADV8 uses a strap of 01 so that the 2nd
** image is the default bitstream loaded, and the 1st image is the fallback
** bitstream that is loaded if the FPGA cannot be configured (i.e. DONE does
** not go high) from the 2nd image.
*/

#include <df.h>

#include "adb3.h"
#include "adb3_common.h"
#include "admxrc6tadv8.h"
#include "admxrc6tx_common.h"
#include "avr_common.h"
#include "avr_flash.h"
#include "avr_sensor.h"
#include "coredma.h"
#include "coreif.h"
#include "device.h"
#include "model_boot.h"
#include "flash.h"
#include "flash_cfi.h"
#include "i2c_common.h"

/* TO DO - error values */
static CoreSensorInfo g_avrVoltageInfo[] = {
  { "3.3V supply rail",            0, { 0,  1311 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC0 */
  { "5V/12V XMC VPWR rail",        0, { 0,  1966 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC1 */
  { "5V supply rail",              0, { 0,  2359 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC2 */
  { "1V supply rail",              0, { 0,  3277 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC3 */
  { "1.5V supply rail",            0, { 0,  4325 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC4 */
  { "1.8V supply rail",            0, { 0,  6554 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC5 */
  { "2.5V supply rail",            0, { 0, 15729 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC6 */
  { "VDD for 156.25 MHz osc.",     0, { 0,  3277 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* AVR ADC7 */
  /* Following sensor is NOT present in board rev. 2 */
  { "VDD for 150 MHz osc.",        0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }  /* AVR ADC8 */
};

/* TO DO - error values */
static CoreSensorInfo g_avrTempInfo[] = {
  { "uC internal temperature",     0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* Microcontroller internal temp */
  { "FPGA int temp. diode",        0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* Temperature sensor diode in FPGA */
  { "Board temp. @ optical xcvr.", 0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }  /* Temperature sensor in TMP412 chip near optical transceivers */
};

/* TO DO - error values */
static CoreSensorInfo g_avrMiscInfo[] = {
  { "Total 3.3VAux on time",       0, { 0, 16384 }, CoreSensorDataDouble, CoreSensorUnitS,    0 }, /* Number of seconds spent with 3.3VAux powered on */
  { "Event count",                 0, { 0,     0 }, CoreSensorDataUInt32, CoreSensorUnitNone, 0 }  /* Number of elapsed events */
};

/* These correspond to 'g_avrVoltageInfo' above, to convert the raw ADC values into Volt */
static AvrVoltageScaling g_avrVoltageScaling[] = {
  {  8U, 0x0FFFU, FALSE, 0x17AAB  /* factor = 0.00144450f */ },
  { 10U, 0x0FFFU, FALSE, 0x40000  /* factor = 0.00390625f */ },
  { 12U, 0x0FFFU, FALSE, 0x17AAB  /* factor = 0.00144450f */ },
  { 14U, 0x0FFFU, FALSE, 0x06000  /* factor = 0.00036621f */ },
  { 16U, 0x0FFFU, FALSE, 0x17AAB  /* factor = 0.00144450f */ },
  { 18U, 0x0FFFU, FALSE, 0x17AAB  /* factor = 0.00144450f */ },
  { 20U, 0x0FFFU, FALSE, 0x17AAB  /* factor = 0.00144450f */ },
  { 22U, 0x0FFFU, FALSE, 0x17AAB  /* factor = 0.00144450f */ },
  { 24U, 0x0FFFU, FALSE, 0x17AAB  /* factor = 0.00144450f */ }
};

/* These correspond to 'g_avrTempInfo' above, to convert the raw ADC values into deg. C */
static AvrTempScaling g_avrTempScaling[] = {
  { 26U, 0xFFFFU, FALSE, 0x10000, 0xFEEED99A, /* factor = 1/16, offset = -273.15 */ },
  { 30U, 0xFFF0U,  TRUE,  0x1000,          0, /* factor = 1/256, offset = 0 */ },
  { 28U, 0xFFF0U,  TRUE,  0x1000,          0, /* factor = 1/256, offset = 0 */ }
};

static AvrMiscScaling g_avrMiscScaling = {
  { 0U },
  { 5U, TRUE }
};

static void
initSensorInfo(
  Adb3CoreDeviceContext* pDevCtx)
{
  unsigned int i = 0, n = pDevCtx->model.admxrc6tadv8.avr.numVoltageSensor;
  boolean_t bCompatSensor1Name = FALSE;

  dfCopyKK(pDevCtx->info.sensor + i, g_avrVoltageInfo, n * sizeof(CoreSensorInfo));
  dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "Admxrc6tadv8CompatSensor1Name", &bCompatSensor1Name);
  if (bCompatSensor1Name) {
    /* Old sensor name might be required for compatibility with some software */
    dfStrCpy(pDevCtx->info.sensor[i + 1].description, DF_ARRAY_LENGTH(pDevCtx->info.sensor[i + 1].description), "12V supply rail");
  }
  i += n;
  dfCopyKK(pDevCtx->info.sensor + i, g_avrTempInfo, sizeof(g_avrTempInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_avrTempInfo);
  i += admxrc6txInitSensorInfoXsmon(pDevCtx, i, FALSE);
  dfCopyKK(pDevCtx->info.sensor + i, g_avrMiscInfo, sizeof(g_avrMiscInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_avrMiscInfo);
}

static boolean_t
readVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  CoreVpdStatus status;
  VpdAdmxrc6tadv8Rev0* pVpd = &pDevCtx->info.vpd.pData->admxrc6tadv8.rev0;
  VpdAdmxrc6tadv8Rev0 vpdRaw;
  uint16_t vpdVersion = 0xffffU;
  uint16_t vpdLength = 0U;
  int i;

  status = admxrc6txReadAvrVpdData(pDevCtx, 0, sizeof(vpdVersion), &vpdVersion);
  if (CoreVpdSuccess != status) {
  	dfDebugPrint(0, ("*** readVpd: failed to read VPD version field\n"));
    return FALSE;
  }
  vpdVersion = dfLe16ToCpu(vpdVersion);
  status = admxrc6txReadAvrVpdData(pDevCtx, 2, sizeof(vpdLength), &vpdLength);
  if (CoreVpdSuccess != status) {
  	dfDebugPrint(0, ("*** readVpd: failed to read VPD length field\n"));
    return FALSE;
  }
  vpdLength = dfLe16ToCpu(vpdLength);
  switch (vpdVersion) {
  case 0:
    /* Ok */
    break;

  case 0xffffU:
  	dfDebugPrint(0, ("*** readVpd: invalid VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    return FALSE;

  default:
  	dfDebugPrint(0, ("+++ readVpd: unrecognised VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    break;
  }

  admxrc6txVerifyAvrVpdChecksum(pDevCtx, vpdLength);

  status = admxrc6txReadAvrVpdData(pDevCtx, 0, sizeof(vpdRaw), &vpdRaw);
  if (CoreVpdSuccess != status) {
  	dfDebugPrint(0, ("*** readVpd: failed to block read VPD\n"));
    return FALSE;
  }
  DF_ZERO_OBJECT(pVpd);
  pVpd->version = vpdVersion;
  pVpd->length = vpdLength;
  pVpd->serial = dfLe32ToCpu(vpdRaw.serial);
  pVpd->modifications = dfLe32ToCpu(vpdRaw.modifications);
  pVpd->idelayRef = dfLe32ToCpu(vpdRaw.idelayRef);
  pVpd->mgt1Ref = dfLe32ToCpu(vpdRaw.mgt1Ref);
  pVpd->mgt2Ref = dfLe32ToCpu(vpdRaw.mgt2Ref);
  for (i = 0; i < 4; i++) {
    pVpd->sdram[i].order = dfLe8ToCpu(vpdRaw.sdram[i].order);
    pVpd->sdram[i].speed = dfLe16ToCpu(vpdRaw.sdram[i].speed);
    pVpd->sdram[i].link = dfLe16ToCpu(vpdRaw.sdram[i].link);
  }
  pVpd->adv8NotFitted = dfLe32ToCpu(vpdRaw.adv8NotFitted);
  pVpd->fpga.device = dfLe16ToCpu(vpdRaw.fpga.device);
  dfAssert(sizeof(char) == sizeof(uint8_t));
  dfStrCpy((char*)pVpd->fpga.speedGrade, DF_ARRAY_LENGTH(pVpd->fpga.speedGrade), (const char*)vpdRaw.fpga.speedGrade);
  dfStrCpy((char*)pVpd->fpga.stepping, DF_ARRAY_LENGTH(pVpd->fpga.stepping), (const char*)vpdRaw.fpga.stepping);
  pVpd->fpga.tempGrade = dfLe8ToCpu(vpdRaw.fpga.tempGrade);
  pVpd->pcbRev = dfLe8ToCpu(vpdRaw.pcbRev);
  pVpd->checksum = dfLe8ToCpu(vpdRaw.checksum);

  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE;
}

static void
enableAvrRxNotEmptyInt(
  AvrDeviceContext* pAvrCtx)
{
	Adb3CoreDeviceContext* pDevCtx = DF_CONTAINER_OF(pAvrCtx, Adb3CoreDeviceContext, model.admxrc6tadv8.avr.context);
  DfInterruptObject* pInterruptObject = &pDevCtx->interrupt.object;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  DfSpinLockFlags f;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  f = dfInterruptSpinLockGet(pInterruptObject);
  pDevCtx->model.admxrc6tadv8.shadow.intCtl |= ADMXRC6TADV8_V_INTCTL_ADVRXNE;
  dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admxrc6tadv8.shadow.intCtl | ADMXRC6TADV8_V_INTCTL_LEVELBITS);
  dfPciMemRead32(hModel, &pModelRegs->intCtl);
  dfInterruptSpinLockPut(pInterruptObject, f);
}

static void
enableAvrTxEmptyInt(
  AvrDeviceContext* pAvrCtx)
{
	Adb3CoreDeviceContext* pDevCtx = DF_CONTAINER_OF(pAvrCtx, Adb3CoreDeviceContext, model.admxrc6tadv8.avr.context);
  DfInterruptObject* pInterruptObject = &pDevCtx->interrupt.object;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  DfSpinLockFlags f;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  f = dfInterruptSpinLockGet(pInterruptObject);
  pDevCtx->model.admxrc6tadv8.shadow.intCtl |= ADMXRC6TADV8_V_INTCTL_ADVTXE;
  dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admxrc6tadv8.shadow.intCtl | ADMXRC6TADV8_V_INTCTL_LEVELBITS);
  dfPciMemRead32(hModel, &pModelRegs->intCtl);
  dfInterruptSpinLockPut(pInterruptObject, f);
}

/* Returns TRUE if alert interrupt is active for specified target FPGA */
static boolean_t
alertPoll(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int targetIndex)
{
  DfInterruptObject* pInterruptObject;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t intStat;
  DfSpinLockFlags f;

  if (targetIndex > 0) {
    return FALSE;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  intStat = dfPciMemRead32(hModel, &pModelRegs->intStat);
  if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
    /* Alert interrupt still active */
    return TRUE;
  } else {
    /* Alert interrupt no longer active - reenable alert interrupt */
    pInterruptObject = &pDevCtx->interrupt.object;
    f = dfInterruptSpinLockGet(pInterruptObject);
    pDevCtx->model.admxrc6tadv8.shadow.intCtl |= ADMXRC6Tx_V_INTCTL_SYSENABLE;
    dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admxrc6tadv8.shadow.intCtl | ADMXRC6TADV8_V_INTCTL_LEVELBITS);
    dfPciMemRead32(hModel, &pModelRegs->intCtl);
    dfInterruptSpinLockPut(pInterruptObject, f);
    return FALSE;
  }
}

static ClockProgramStatus
clockProgram(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  const CoreClockWord* pClockWord)
{
  return ClockProgramInvalidIndex;
}

static ClockWordStatus
clockWord(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  uint32_t flags,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  return ClockWordInvalidIndex;
}

static FpgaControlStatus
fpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  boolean_t* pbValue)
{
  dfDebugPrint(8, ("fpgaControl: index=%lu opCode=%lu pbValue=%p\n",
    (unsigned long)index, (unsigned long)opCode, pbValue));
  
  if (0 != index) {
    return FpgaControlInvalidIndex;
  }

  /* ADM-XRC-6T-ADV8 cannot be fully reconfigured */
  return FpgaControlNotSupported;
}

static FpgaSelectMapStatus
fpgaSelectMap(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  boolean_t bWrite,
  size_t length,
  void* pBuffer)
{
  dfDebugPrint(8, ("fpgaSelectMap: index=%lu bWrite=%s length=" DF_FMT_U64 "(0x%08lx_%08lx) pBuffer=%p\n",
    (unsigned long)index, bWrite ? "TRUE" : "FALSE", (unsigned long long)length, dfSplitUint64((uint64_t)length), pBuffer));
  
  if (0 != index) {
    return FpgaSelectMapInvalidIndex;
  }
  return FpgaSelectMapNotSupported;
}

static boolean_t
initHardware(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bInit,
  unsigned int phase)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  unsigned int i;

  dfDebugPrint(2, ("initHardware: bInit=%s phase=%u\n", bInit ? "TRUE" : "FALSE", phase));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  if (bInit) {
    switch (phase) {
    case 0:
      pDevCtx->info.model = CoreModelAdmxrc6tadv8;

      /* The set of Bridge interrupts that we are interested in */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnableMaster = ADB3IRQ_DMA_ALL | ADB3IRQ_MODEL;

      /* Ensure Bridge interrupts are disabled */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnable = 0;
      dfPciMemWrite32(hBridge, &pRegs->irqEnable, pDevCtx->hardware.shadow.bridge.adb3.irqEnable);
      dfPciMemRead32(hBridge, &pRegs->irqEnable);

      /* Enable model-specific interrupts that we are interested in */
      pDevCtx->model.admxrc6tadv8.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE | ADMXRC6Tx_V_INTCTL_SYSENABLE;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admxrc6tadv8.shadow.intCtl | ADMXRC6TADV8_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);

      /* Initialize DMA enables */
      dfPciMemWrite32(hBridge, &pRegs->dmaEnable, ADB3DMAEN_LOCAL_MASTER | ADB3DMAEN_DMA_ALL);

      /* Figure out number of DMA engines present, and directions they support */
      pDevCtx->info.bootstrap.numDmaChannel = adb3CountDmaEngines(pDevCtx);

      /* Figure out if we need MSI bug workaround */
      pDevCtx->model.admxrc6tadv8.bNeedMsiWorkaround = !pDevCtx->interrupt.bPolledMode && adb3NeedMsiWorkaround(pDevCtx);

      /* Initialize DMA engine registers */
      adb3InitDmaEngines(pDevCtx, pDevCtx->info.bootstrap.numDmaChannel);

      /* Identify Flash device */
      admxrc6txFlashResetBank(pDevCtx, &pDevCtx->hardware.model.pAdmxrc6tx->flash);
      if (flashCfiIdentify(pDevCtx, 0, 0)) {
        pDevCtx->info.flash[0].bPresent = TRUE;
        pDevCtx->info.flash[0].targetArea.start = 0U;
        pDevCtx->info.flash[0].targetArea.length = pDevCtx->info.flash[0].totalSize - pDevCtx->info.flash[0].targetArea.start;
      } else {
        dfDebugPrint(0, ("*** initHardware: failed to identify Flash\n"));
      }

      /* Initialize I2C stuff */
      i2cInit(&pDevCtx->model.admxrc6tadv8.i2c.context, NULL, admxrc6txI2cStart, admxrc6txI2cPoll, pDevCtx);

      /* Get the VPD */
      readVpd(pDevCtx);
      pDevCtx->model.admxrc6tadv8.avr.numVoltageSensor = DF_ARRAY_LENGTH(g_avrVoltageInfo);
      /* Correct the number of sensors for board rev. 2, if necessary */
      if (pDevCtx->info.vpd.pData->admxrc6tadv8.rev0.pcbRev >= 0x20U) {
        /* Last sensor is not present on board rev. 2 */
        pDevCtx->model.admxrc6tadv8.avr.numVoltageSensor--;
        pDevCtx->info.bootstrap.numSensor--;
      }

      pDevCtx->info.prodTest.serialNumber = pDevCtx->info.vpd.pData->admxrc6tadv8.rev0.serial;
      pDevCtx->info.prodTest.cpldRevision = 0; /* TO DO - get microcontroller revision */
      pDevCtx->info.prodTest.pcbRevision = pDevCtx->info.vpd.pData->admxrc6tadv8.rev0.pcbRev;
      pDevCtx->info.prodTest.bridgeDate = dfPciMemRead32(hModel, &pModelRegs->buildDate);
      pDevCtx->info.prodTest.bridgeTime = dfPciMemRead32(hModel, &pModelRegs->buildTime);

      /* Find base addresses and sizes of local bus windows */
      i = 0;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      i++;
      if (NULL != pDevCtx->hardware.pBridgedPrefetch) {
        pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[3].base;
        pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[3].size;
        pDevCtx->info.window[i].localBase = 0;
        pDevCtx->info.window[i].localSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
        pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[3].base;
        pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
        pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.pBridgedPrefetch;
        pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[3].size;
        i++;
      }
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = 0;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.model.pGeneric;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      i++;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = 0;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[0].size;
      i++;

      /* Initialize sensor information structures */
      initSensorInfo(pDevCtx);
      break;

    case 1:
      /* Initialise the AVR uC interface */
      if (avrInit(
        &pDevCtx->model.admxrc6tadv8.avr.context,
        &pDevCtx->interrupt.object,
        &pDevCtx->hardware.model.pAdmxrc6tx->model2.avr.curr.ctlStat,
        pDevCtx->hardware.hModel,
        ADMXRC6TADV8_AVR_FLASH_SIZE,
        ADMXRC6TADV8_AVR_FLASH_PAGE_SIZE,
        enableAvrRxNotEmptyInt,
        enableAvrTxEmptyInt))
      {
        pDevCtx->model.admxrc6tadv8.bAvrInitialized = TRUE;
      }
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    switch (phase) {
    case 0:
      /* Uninitialize I2C stuff */
      i2cUninit(&pDevCtx->model.admxrc6tadv8.i2c.context);

      if (pDevCtx->hardware.bCanTouch) {
        /* Deinitialize DMA enables */
        dfPciMemWrite32(hBridge, &pRegs->dmaEnable, 0);
      }
      break;

    case 1:
      /* Deinitialize the AVR interface */
      if (pDevCtx->model.admxrc6tadv8.bAvrInitialized) {
        avrUninit(&pDevCtx->model.admxrc6tadv8.avr.context);
        pDevCtx->model.admxrc6tadv8.bAvrInitialized = FALSE;
      }
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  }

  return TRUE;
}

static boolean_t
isr(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t irqStatus, dmaIrqStatus;
  uint32_t intStat, intCtl;
  unsigned int channel, n;
  DfSpinLockFlags f;

  dfDebugPrint(8, ("isr: entered\n"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  /* Sample the currently active interrupts. */
  irqStatus = dfPciMemRead32(hBridge, &pRegs->irqStatus);

  f = dfIsrSpinLockGet(pInterruptObj);
  /* Mask the currently active interrupts with the interrupts we are interested in right now. */
  irqStatus &= pDevCtx->hardware.shadow.bridge.adb3.irqEnable;
  /* Clear the interrupts that we just saw. */
  dfPciMemWrite32(hBridge, &pRegs->irqStatus, irqStatus);

  if (irqStatus & ADB3IRQ_MODEL) {
    dfDebugPrint(9, ("isr: model-specific interrupt\n"));
    intCtl = pDevCtx->model.admxrc6tadv8.shadow.intCtl;
    intStat = dfPciMemRead32(hModel, &pModelRegs->intStat);
    intStat &= intCtl;
    if (intStat & ADMXRC6TADV8_V_INTSTAT_LEVELBITS) {
      if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
        /* Disable hardware monitor interrupt; gets reenabled by hardware monitor polling. */
        intCtl &= ~ADMXRC6Tx_V_INTCTL_SYSENABLE;
      }
      if (intStat & ADMXRC6TADV8_V_INTSTAT_ADVRXNE) {
        /* Disable RX FIFO not empty interrupt. */
        intCtl &= ~ADMXRC6TADV8_V_INTSTAT_ADVRXNE;
      }
      if (intStat & ADMXRC6TADV8_V_INTSTAT_ADVTXE) {
        /* Disable TX FIFO empty interrupt. */
        intCtl &= ~ADMXRC6TADV8_V_INTSTAT_ADVTXE;
      }
      /* Write new model-specific interrupt enables. */
      pDevCtx->model.admxrc6tadv8.shadow.intCtl = intCtl;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, intCtl | ADMXRC6TADV8_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);
    }
    dfIsrSpinLockPut(pInterruptObj, f);

    /* Clear the model-specific interrupts that we saw. */
    dfPciMemWrite32(hModel, &pModelRegs->intStat, intStat);

    /* Take action for the model-specific interrupts that we saw. */
    if (intStat & ADMXRC6Tx_V_INTSTAT_FP) {
      dfDebugPrint(9, ("isr: target FPGA 0 interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.fpga[0].dpc, (void*)(uintptr_t)0);
    }
    if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
      dfDebugPrint(9, ("isr: hardware monitor interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.hwMon[0].dpc, (void*)(uintptr_t)0);
    }
    if (intStat & ADMXRC6TADV8_V_INTSTAT_ADVRXNE) {
      dfDebugPrint(9, ("isr: AVR RX FIFO not empty interrupt\n"));
      dfDpcSchedule(&pDevCtx->model.admxrc6tadv8.avr.context.transport.rx.dpc, NULL);
    }
    if (intStat & ADMXRC6TADV8_V_INTSTAT_ADVTXE) {
      dfDebugPrint(9, ("isr: AVR TX FIFO empty interrupt\n"));
      dfDpcSchedule(&pDevCtx->model.admxrc6tadv8.avr.context.transport.tx.dpc, NULL);
    }
  } else {
    dfIsrSpinLockPut(pInterruptObj, f);
  }

  n = pDevCtx->info.bootstrap.numDmaChannel;
  for (channel = 0; channel < n; channel++) {
    if (irqStatus & (ADB3IRQ_DMA0 << channel)) {
      dmaIrqStatus = dfPciMemRead32(hBridge, &pRegs->dmaEngine[channel].irq.status);
      dfDebugPrint(9, ("isr: DMA%lu interrupt, DMA%lu.irqStatus=0x%08lx\n",
        (unsigned long)channel, (unsigned long)channel, (unsigned long)dmaIrqStatus));
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[channel].count++;
#endif
      dfPciMemWrite32(hBridge, &pRegs->dmaEngine[channel].irq.ack, dmaIrqStatus);
      dfDpcSchedule(&pDevCtx->interrupt.dma[channel].dpc, (void*)(uintptr_t)channel);
    }
  }

  if (pDevCtx->interrupt.bUsingMsi && pDevCtx->model.admxrc6tadv8.bNeedMsiWorkaround) {
    /* MSI problem workaround - disable all interrupts then reenable */
    adb3MsiWorkaround(pDevCtx);
  }

  return irqStatus ? TRUE : FALSE;
}

static boolean_t
mapBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated,
	boolean_t bUnmap)
{
	dfDebugPrint(1, ("mapBusResources: bUnmap=%s\n", bUnmap ? "TRUE" : "FALSE"));

	if (bUnmap) {
		if (NULL != pDevCtx->hardware.bridge.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.bridge.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 generic registers\n"));
			}
      pDevCtx->hardware.bridge.pGeneric = NULL;
		}
		if (NULL != pDevCtx->hardware.model.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.model.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 model-specific registers\n"));
			}
      pDevCtx->hardware.model.pGeneric = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridged) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridged)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 bridged space\n"));
			}
      pDevCtx->hardware.pBridged = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridgedPrefetch) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridgedPrefetch)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 prefetchable bridged space\n"));
			}
      pDevCtx->hardware.pBridgedPrefetch = NULL;
      pDevCtx->info.bootstrap.numWindow--;
		}
		dfDebugPrint(2, ("mapBusResources: unmapped\n"));
	} else {
		pDevCtx->hardware.bridge.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 0, 0, pTranslated->variant.pci.memBar[0].size, &pDevCtx->hardware.hBridge);
		if (NULL == pDevCtx->hardware.bridge.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 generic registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 generic registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[0].base), pDevCtx->hardware.bridge.pGeneric));
		}
		pDevCtx->hardware.model.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 1, 0, pTranslated->variant.pci.memBar[1].size, &pDevCtx->hardware.hModel);
		if (NULL == pDevCtx->hardware.model.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 model-specific registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 model-specific registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[1].base), pDevCtx->hardware.model.pGeneric));
		}
    pDevCtx->hardware.pBridged = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 2, 0, pTranslated->variant.pci.memBar[2].size, &pDevCtx->hardware.hBridged);
		if (NULL == pDevCtx->hardware.pBridged) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 bridged space\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 bridged space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[2].base), pDevCtx->hardware.pBridged));
		}
    if (pTranslated->variant.pci.numMemBar >= 4) {
      pDevCtx->hardware.pBridgedPrefetch = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 3, 0, pTranslated->variant.pci.memBar[3].size, &pDevCtx->hardware.hBridgedPrefetch);
      if (NULL == pDevCtx->hardware.pBridgedPrefetch) {
        dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 prefetchable bridged space\n"));
        return FALSE;
      } else {
        dfDebugPrint(2, ("mapBusResources: mapped ADB3 prefetchable bridged space @ 0x%08lx_%08lx to %p\n",
          dfSplitUint64(pTranslated->variant.pci.memBar[3].base), pDevCtx->hardware.pBridgedPrefetch));
        pDevCtx->info.bootstrap.numWindow++;
      }
    }
		dfDebugPrint(2, ("mapBusResources: mapped OK\n"));
	}

	return TRUE;
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;

  dfAssert(newState == 0 || newState == 3);

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
      /* Nothing to do yet */
    } else { /* Powering up (from D3) */
      /* Reenable model-specific interrupts that we are interested in */
      pDevCtx->model.admxrc6tadv8.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE | ADMXRC6Tx_V_INTCTL_SYSENABLE;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admxrc6tadv8.shadow.intCtl | ADMXRC6TADV8_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);

      /* Reinitialize DMA enables */
      dfPciMemWrite32(hBridge, &pRegs->dmaEnable, ADB3DMAEN_LOCAL_MASTER | ADB3DMAEN_DMA_ALL);

      /* Reinitialize DMA engine registers */
      adb3InitDmaEngines(pDevCtx, pDevCtx->info.bootstrap.numDmaChannel);

      /* Reset Flash device */
      admxrc6txFlashResetBank(pDevCtx, &pDevCtx->hardware.model.pAdmxrc6tx->flash);
    }
  }
}

static CoreSensorStatus
readSensor(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreSensorValue* pReading)
{
  CoreSensorStatus status = CoreSensorInvalidIndex;

  if (index >= pDevCtx->info.bootstrap.numSensor) {
    return CoreSensorInvalidIndex;
  }

  status = avrSensorVoltageRead(pDevCtx, index, g_avrVoltageScaling, pDevCtx->model.admxrc6tadv8.avr.numVoltageSensor, pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= pDevCtx->model.admxrc6tadv8.avr.numVoltageSensor;
  }

  status = avrSensorTempRead(pDevCtx, index, g_avrTempScaling, DF_ARRAY_LENGTH(g_avrTempScaling), pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_avrTempInfo);
  }

  status = admxrc6txReadSensorXsmon(pDevCtx, index, pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= ADMXRC6TX_NUM_XSMON_SENSOR;
  }

  status = avrSensorMiscRead(pDevCtx, index, &g_avrMiscScaling, pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_avrMiscInfo);
  }

  return status;
}

static CoreVpdStatus
mapAvrFlashStatus(
  AvrFlashStatus status)
{
  switch (status) {
  case AvrFlashSuccess:
    return CoreVpdSuccess;

  case AvrFlashTimeout:
    return CoreVpdTimeout;

  case AvrFlashNotServiceMode:
    return CoreVpdNotPresent;

  case AvrFlashInvalidRegion:
    return CoreVpdInvalidRegion;

  case AvrFlashGeneralFailure:
    return CoreVpdGeneralFailure;

  default:
    if (status >= AvrFlashHardwareFailure && status < AvrFlashHardwareFailure + 0x100U) {
      return CoreVpdGeneralFailure;
    }
    return CoreVpdGeneralFailure;
  }
}

static CoreVpdStatus
readWriteVpd(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWrite,
  size_t address,
  size_t length,
  void* pData)
{
  AvrFlashStatus status;

  if (CHECK_BOUNDS_OFFSET(address, length, pDevCtx->info.bootstrap.vpdOffset, pDevCtx->info.bootstrap.vpdLength)) {
    return CoreVpdInvalidRegion;
  }
  if (bWrite) {
    /* Casts are safe because of above bounds check */
    status = avrFlashWrite(pDevCtx, &pDevCtx->model.admxrc6tadv8.avr.context, (uint32_t)address, (uint32_t)length, pData);
  } else {
    /* Casts are safe because of above bounds check */
    status = avrFlashRead(pDevCtx, &pDevCtx->model.admxrc6tadv8.avr.context, (uint32_t)address, (uint32_t)length, pData);
  }
  return mapAvrFlashStatus(status);
}

static boolean_t
validateBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated)
{
	dfDebugPrint(1, ("validateBusResources: entered\n"));

	if (pTranslated->variant.pci.numMemBar < 3) {
		dfDebugPrint(0, ("*** validateBusResources: less than 3 memory BARs\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[0].size < 0x1000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR0 smaller than 0x1000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[1].size < sizeof(ModelRegsAdmxrc6tx)) {
		dfDebugPrint(0, ("*** validateBusResources: BAR1 smaller than 0x%lu bytes\n", (unsigned long)sizeof(ModelRegsAdmxrc6tx)));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[2].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR2 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
  if (pTranslated->variant.pci.numMemBar >= 4 && pTranslated->variant.pci.memBar[3].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR3 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.numIrq != 1) {
		dfDebugPrint(0, ("*** validateBusResources: not exactly 1 interrupt\n"));
		return FALSE;
	}

	dfDebugPrint(2, ("validateBusResources: OK\n"));

	return TRUE;
}

void
adb3CoreBootstrapAdmxrc6tadv8(
	Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdmxrc6tadv8: entered\n"));

  dfAssert(DF_OFFSET_OF(VpdAdmxrc6tadv8Rev0, checksum) == sizeof(VpdAdmxrc6tadv8Rev0) - 1);

  pDevCtx->methods.pAlertPoll = alertPoll;
  pDevCtx->methods.pClearErrors = adb3ClearErrors;
  pDevCtx->methods.pClockProgram = clockProgram;
  pDevCtx->methods.pClockWord = clockWord;
  pDevCtx->methods.pDmaList = adb3DmaList;
  pDevCtx->methods.pDmaTransfer = adb3DmaTransfer;
  pDevCtx->methods.pEnableInterrupts = adb3EnableInterrupts;
  pDevCtx->methods.pFlash = admxrc6txFlashSingleBank;
  pDevCtx->methods.pFpgaControl = fpgaControl;
  pDevCtx->methods.pFpgaSelectMap = fpgaSelectMap;
  pDevCtx->methods.pGetDeviceStatus = adb3GetDeviceStatus;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIsr = isr;
  pDevCtx->methods.pMapBusResources = mapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pReadSensor = readSensor;
  pDevCtx->methods.pReadWriteVpd = readWriteVpd;
  pDevCtx->methods.pSyncVpd = NULL;
  pDevCtx->methods.pValidateBusResources = validateBusResources;
  pDevCtx->methods.pWindowConfig = adb3WindowConfig;

  pDevCtx->info.bootstrap.numClockGen = 0;
  pDevCtx->info.bootstrap.numDmaChannel = ADMXRC6TADV8_NUM_DMA_CHANNEL;
  pDevCtx->info.bootstrap.numIoModule = 0;
  pDevCtx->info.bootstrap.numSensor = ADMXRC6TX_NUM_XSMON_SENSOR + DF_ARRAY_LENGTH(g_avrVoltageInfo) + DF_ARRAY_LENGTH(g_avrTempInfo) + DF_ARRAY_LENGTH(g_avrMiscInfo);
  pDevCtx->info.bootstrap.numTargetFpga = 1;
  pDevCtx->info.bootstrap.numWindow = 3; /* May be incremented in mapBusResources method */
  pDevCtx->info.bootstrap.numFlashBank = 1;
  pDevCtx->info.bootstrap.bVpdInFlash = FALSE;
  pDevCtx->info.bootstrap.vpdLength = ADMXRC6TADV8_AVR_FLASH_SIZE;
  pDevCtx->info.bootstrap.vpdOffset = 0;
  pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdmxrc6tadv8Rev0);
  pDevCtx->info.bootstrap.flash[0].deviceWidth = 2;
  pDevCtx->info.bootstrap.flash[0].width = 2;
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "PciAddress64Bit", &pDevCtx->info.bootstrap.b64BitDmaAddress);
  pDevCtx->info.bootstrap.dmaDescriptorSize = 32;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x1000000U;
  pDevCtx->info.bootstrap.bWorkAround4kCrossing = FALSE;
  pDevCtx->info.bootstrap.bI2cInterruptDriven = FALSE;
}
