/*
** File: model_admxpl.c  
** Project: ADB3 core driver
** Purpose: ADM-XPL-specific code, including function to bootstrap device context.
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>

#include "adb1_common.h"
#include "admxpl_common.h"
#include "admxpl.h"
#include "coreclock.h"
#include "coredma.h"
#include "device.h"
#include "flash.h"
#include "flash_cfi.h"
#include "ics307_common.h"
#include "model_boot.h"

static boolean_t
readVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  VpdAdmxplRev0* pVpd = &pDevCtx->info.vpd.pData->admxpl.rev0;
  VpdAdmxplRev0 vpdRaw;
  uint16_t vpdVersion = 0xffffU;

  flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdVersion), &vpdVersion);
  vpdVersion = dfLe16ToCpu(vpdVersion);
  switch (vpdVersion) {
  case 0:
    /* Ok */
    break;

  case 0xffffU:
  	dfDebugPrint(0, ("*** readVpd: invalid VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    return FALSE;

  default:
  	dfDebugPrint(0, ("+++ readVpd: unrecognised VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    break;
  }

  flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdRaw), &vpdRaw);
  DF_ZERO_OBJECT(pVpd);
  pVpd->version = vpdVersion;
  pVpd->boardType = dfLe16ToCpu(vpdRaw.boardType);
  pVpd->modifications = dfLe32ToCpu(vpdRaw.modifications);
  pVpd->refClockFreq = dfLe32ToCpu(vpdRaw.refClockFreq);
  pVpd->ssram.order = dfLe8ToCpu(vpdRaw.ssram.order);
  pVpd->ssram.width = dfLe8ToCpu(vpdRaw.ssram.width);
  pVpd->ssram.typeMask = dfLe16ToCpu(vpdRaw.ssram.typeMask);
  pVpd->sdram.order = dfLe8ToCpu(vpdRaw.sdram.order);
  pVpd->sdram.width = dfLe8ToCpu(vpdRaw.sdram.width);
  pVpd->sdram.typeMask = dfLe16ToCpu(vpdRaw.sdram.typeMask);
  pVpd->cardId = dfLe32ToCpu(vpdRaw.cardId);
  pVpd->serialNumber = dfLe32ToCpu(vpdRaw.serialNumber);
  pVpd->fpgaType = dfLe16ToCpu(vpdRaw.fpgaType);
  pVpd->pcbRev = dfLe8ToCpu(vpdRaw.pcbRev);
  pVpd->logicRev = (uint8_t)(0x10 + pDevCtx->pDevObj->hardwareId.variant.pci.revision);

  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE;
}

static void
onIcs307ProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  Ics307DeviceContext* pIcs307SynthCtx,
  Ics307ProgramRequest* pRequest,
  Ics307ProgramStatus status)
{
  onClockProgramDone(pDevCtx, 0, ics307MapProgramStatus(status));
}

static ClockProgramStatus
clockProgram(
	Adb3CoreDeviceContext*	pDevCtx,
  unsigned int clockIndex,
	const CoreClockWord* pClockWord)
{
  switch (clockIndex) {
  case 0:
    return ics307MapProgramStatus(ics307Program(pDevCtx, &pDevCtx->model.admxpl.ics307.context, &pDevCtx->model.admxpl.ics307.programming, pClockWord, onIcs307ProgramDone));

  default:
    return ClockProgramInvalidIndex;
  }
}

static ClockWordStatus
clockWord(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  uint32_t flags,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  VpdAdmxplRev0* pVpd = &pDevCtx->info.vpd.pData->admxpl.rev0;

  if (clockIndex >= pDevCtx->info.bootstrap.numClockGen) {
    return ClockWordInvalidIndex;
  }

  switch (clockIndex) {
  case 0:
    return ics307MapClockWordStatus(clockWordIcs307(flags, pVpd->refClockFreq, frequencyReq, frequencyMin, frequencyMax, TRUE, pClockWord));

  default:
    return ClockWordInvalidIndex;
	}
}

static FpgaControlStatus
fpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  boolean_t* pbValue)
{
  return admxplFpgaControl(pDevCtx, index, opCode, &pDevCtx->model.admxpl.shadow.fpctl, pbValue);
}

static boolean_t
initHardware(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bInit,
  unsigned int phase)
{
  Adb1BridgeRegisters* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxpl* pModelRegs;
  DfMemoryHandle hModel;
  VpdAdmxplRev0* pVpd = &pDevCtx->info.vpd.pData->admxpl.rev0;
  uint32_t ls0b, ls0r, ls1b, ls1r;
  uint32_t fOut;

  dfDebugPrint(1, ("initHardware: bInit=%s\n", bInit ? "TRUE" : "FALSE"));

  pRegs = pDevCtx->hardware.bridge.pAdb1;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxpl;
  hModel = pDevCtx->hardware.hModel;

  if (bInit) {
    switch (phase) {
    case 0:
      pDevCtx->info.model = CoreModelAdmxpl;

      /* Ensure that all interrupts are disabled in Bridge */
      pDevCtx->hardware.shadow.bridge.adb1.pictlMaster = ADB1_PICTL_ENMODEL | ADB1_PICTL_ENDMA_ALL | ADB1_PICTL_ENINTA;
      dfPciMemWrite32(hBridge, &pRegs->pictl, 0);
      (void)dfPciMemRead32(hBridge, &pRegs->pictl);

      /* Enable target FPGA interrupt */
      pDevCtx->model.admxpl.shadow.ictl = ADMXPL_ICTL_ENFPGA;
      dfPciMemWrite32(hModel, &pModelRegs->ictl, pDevCtx->model.admxpl.shadow.ictl);
      (void)dfPciMemRead32(hModel, &pModelRegs->ictl);

      /* Enable DMA engines */
      adb1EnableDmaEngines(pDevCtx, TRUE);

      /* Establish FPGA control register shadow */
      pDevCtx->model.admxpl.shadow.fpctl = 0;
      dfPciMemWrite32(hModel, &pModelRegs->fpctl, pDevCtx->model.admxpl.shadow.fpctl);

      /* Reset Flash and identify it */
      admxplFlashReset(pDevCtx);
      if (flashCfiIdentify(pDevCtx, 0, 0)) {
        pDevCtx->info.flash[0].bPresent = TRUE;
        pDevCtx->info.flash[0].targetArea.start = ADMXPL_FLASH_TARGET_AREA_START;
        pDevCtx->info.flash[0].targetArea.length = pDevCtx->info.flash[0].totalSize - ADMXPL_FLASH_TARGET_AREA_START;
        readVpd(pDevCtx);
      } else {
        dfDebugPrint(0, ("*** initHardware: failed to identify Flash\n"));
      }

      /* Initialize ICS307 stuff */
      if (!ics307Init(
        pDevCtx,
        &pDevCtx->model.admxpl.ics307.context,
        pVpd->refClockFreq,
        Ics307InterfaceTypeXpl,
        TRUE,
        &pDevCtx->hardware.model.pAdmxpl->clock,
        pDevCtx->hardware.hModel,
        &fOut))
      {
        dfDebugPrint(0, ("*** initHardware: failed to initialize ICS307 clock synthesizer\n"));
      } else {
        pDevCtx->clockGenerator[0].currentFrequency = fOut;
      }

      /* Find base addresses and sizes of local bus windows */
      pDevCtx->info.window[0].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      ls0b = dfPciMemRead32(hBridge, &pRegs->ls0b);
      ls0r = dfPciMemRead32(hBridge, &pRegs->ls0r);
      ls0r &= 0xfffffff0U;
      ls0b &= ls0r;
      pDevCtx->info.window[0].localBase = ls0b;
      pDevCtx->info.window[0].localSize = ~ls0r + 1;
      pDevCtx->info.window[0].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[0].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[0].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[1].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      ls1b = dfPciMemRead32(hBridge, &pRegs->ls1b);
      ls1r = dfPciMemRead32(hBridge, &pRegs->ls1r);
      ls1r &= 0xfffffff0U;
      ls1b &= ls1r;
      pDevCtx->info.window[1].localBase = ls1b;
      pDevCtx->info.window[1].localSize = ~ls1r + 1;
      pDevCtx->info.window[1].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[1].pKernelBase = pDevCtx->hardware.model.pAdmxpl;
      pDevCtx->info.window[1].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[2].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].localBase = 0;
      pDevCtx->info.window[2].localSize = 0;
      pDevCtx->info.window[2].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[2].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[0].size;
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    switch (phase) {
    case 0:
      /* Deinitialize ICS307 stuff */
      ics307Uninit(pDevCtx, &pDevCtx->model.admxpl.ics307.context);

      if (pDevCtx->hardware.bCanTouch) {
        /* Disable DMA engines */
        adb1EnableDmaEngines(pDevCtx, FALSE);
      }
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  }

  dfDebugPrint(1, ("initHardware: done\n"));
  return TRUE;
}

static boolean_t
isr(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;

  return admxplHandleInterrupt(pDevCtx, pDevCtx->hardware.shadow.bridge.adb1.pictl, pDevCtx->model.admxpl.shadow.ictl);
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  Ics307ProgramStatus ics307Status;
  ModelRegsAdmxpl* pModelRegs;
  DfMemoryHandle hModel;

  dfAssert(newState == 0 || newState == 3);

  pModelRegs = pDevCtx->hardware.model.pAdmxpl;
  hModel = pDevCtx->hardware.hModel;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
      /* Save ICS307 state */
      ics307SaveState(pDevCtx, &pDevCtx->model.admxpl.ics307.context);
    } else { /* Powering up (from D3) */
      /* Reset Flash device */
      admxplFlashReset(pDevCtx);

      /* Reenable target FPGA interrupt */
      dfPciMemWrite32(hModel, &pModelRegs->ictl, pDevCtx->model.admxpl.shadow.ictl);
      (void)dfPciMemRead32(hModel, &pModelRegs->ictl);

      /* Restore ICS307 state */
      ics307Status = ics307RestoreState(pDevCtx, &pDevCtx->model.admxpl.ics307.context);
      if (Ics307ProgramStatusSuccess != ics307Status) {
        dfDebugPrint(0, ("*** powerChange: failed to restore ICS307 state, status=%u\n", ics307Status));
      }
    }
  }
}

void
adb3CoreBootstrapAdmxpl(
  Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdmxpl entered\n"));

  pDevCtx->methods.pClockProgram = clockProgram;
  pDevCtx->methods.pClockWord = clockWord;
  pDevCtx->methods.pDmaList = adb1DmaList;
  pDevCtx->methods.pDmaTransfer = adb1DmaTransfer;
  pDevCtx->methods.pFlash = admxplFlash;
  pDevCtx->methods.pEnableInterrupts = adb1EnableInterrupts;
  pDevCtx->methods.pFpgaControl = fpgaControl;
  pDevCtx->methods.pFpgaSelectMap = admxplFpgaSelectMap;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIsr = isr;
  pDevCtx->methods.pMapBusResources = admxplMapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pReadWriteVpd = admxplReadWriteVpd;
  pDevCtx->methods.pSyncVpd = admxplSyncVpd;
  pDevCtx->methods.pValidateBusResources = admxplValidateBusResources;
  pDevCtx->methods.pWindowConfig = adb1WindowConfig;

  pDevCtx->info.bootstrap.numClockGen = 1;
  pDevCtx->info.bootstrap.numDmaChannel = ADMXPL_NUM_DMA_CHANNEL;
  pDevCtx->info.bootstrap.numIoModule = 1U;
  pDevCtx->info.bootstrap.numSensor = 0;
  pDevCtx->info.bootstrap.numTargetFpga = 1;
  pDevCtx->info.bootstrap.numWindow = 3;
  pDevCtx->info.bootstrap.numFlashBank = 1;
  pDevCtx->info.bootstrap.bVpdInFlash = TRUE;
  pDevCtx->info.bootstrap.vpdLength = ADMXPL_VPD_SIZE_IN_FLASH;
  pDevCtx->info.bootstrap.vpdOffset = ADMXPL_VPD_OFFSET_IN_FLASH;
  pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdmxplRev0);
  pDevCtx->info.bootstrap.flash[0].deviceWidth = 2;
  pDevCtx->info.bootstrap.flash[0].width = 1;
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  pDevCtx->info.bootstrap.dmaDescriptorSize = 32;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x800000U;
}
