/*
** File: model_admpcieku3.c  
** Project: ADB3 core driver
** Purpose: Code specific to ADM-PCIE-KU3, including function to bootstrap device context.
**
** (C) Copyright Alpha Data 2015
**
** SI5338 output to clock index mapping:
**
** SI5338#   Output#   Signal        Clock index
** ============================================
**    0         0      REFCLK156M25_0    0
**    0         1      REFCLK250M        2
**    0         2      REFCLK156M25_1    1
**    0         3      REFCLK133M        7
**    1         0      REFCLK400M_0      3
**    1         1      REFCLK200M        6
**    1         2      REFCLK400M_1      4
**    1         3      REFCLK150M        5
*/

#include <df.h>

#include "adb3.h"
#include "adb3_common.h"
#include "admpcieku3.h"
#include "admxrc6tx_common.h"
#include "coreclock.h"
#include "coreif.h"
#include "coredma.h"
#include "device.h"
#include "ddr3_spd_common.h"
#include "model_boot.h"
#include "flash.h"
#include "flash_cfi.h"
#include "i2c_common.h"
#include "icap_common.h"
#include "ucd90120buf.h"
#include "si5338.h"
#include "si5338_common.h"

/* Mapping from voltage sensor index to 1-based rail number */
static unsigned int g_voltageMapping[] = {
  1, 2, 10, 7, 6, 8, 9, 5, 4, 3
};

/* Mapping from current sensor index to 1-based rail number */
static unsigned int g_currentMapping[] = {
  1, 2
};

/* Mapping from temp. sensor index to 1-based rail number */
static unsigned int g_tempMapping[] = {
  11
};

/* TO DO - error values */
static CoreSensorInfo g_voltageInfo[] = {
  { "12V rail (edge conn.)",     0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* UCD90120 MON1, rail 1 */
  { "3.3V rail (edge conn.)",    0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* UCD90120 MON3, rail 2 */
  { "3.3V power (internal)",     0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* UCD90120 MON12, rail 10 */
  { "1.8V power",                0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* UCD90120 MON10, rail 7 */
  { "1.8V MGT VCCAux",           0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* UCD90120 MON11, rail 6 */
  { "1.5V DIMM 0 power",         0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* UCD90120 MON8, rail 8 */
  { "1.5V DIMM 1 power",         0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* UCD90120 MON9, rail 9 */
  { "1.2V MGT AVTT",             0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* UCD90120 MON7, rail 5 */
  { "1.0V MGT AVCC",             0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* UCD90120 MON6, rail 4 */
  { "0.95V power",               0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }  /* UCD90120 MON5, rail 3 */
};

/* TO DO - error values */
static CoreSensorInfo g_currentInfo[] = {
  { "12V current (edge conn.)",  0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitA,    0 }, /* UCD90120 MON2, rail 1 */
  { "3.3V current (edge conn.)", 0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitA,    0 }  /* UCD90120 MON4, rail 2 */
};

/* TO DO - error values */
static CoreSensorInfo g_tempInfo[] = {
  { "Target FPGA int temp. diode", 0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }  /* UCD90120 MON13, rail 11 */
};

static void
initSensorInfo(
  Adb3CoreDeviceContext* pDevCtx)
{
  unsigned int i = 0;

  dfCopyKK(pDevCtx->info.sensor + i, g_voltageInfo, sizeof(g_voltageInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_voltageInfo);
  dfCopyKK(pDevCtx->info.sensor + i, g_currentInfo, sizeof(g_currentInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_currentInfo);
  dfCopyKK(pDevCtx->info.sensor + i, g_tempInfo, sizeof(g_tempInfo));
  i += (unsigned int)DF_ARRAY_LENGTH(g_tempInfo);
}

static CoreVpdStatus
readShadowedVpd(
  Adb3CoreDeviceContext* pDevCtx,
  size_t address,
  size_t length,
  void* pData)
{
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t* pDst;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  pDst = (uint8_t*)pData;
  while (length) {
    *pDst++ = dfPciMemRead8(hModel, pModelRegs->xrmBuf + address);
    length--;
    address++;
  }

  return CoreVpdSuccess;
}

static boolean_t
verifyVpdChecksum(
	Adb3CoreDeviceContext* pDevCtx,
  uint16_t vpdLength)
{
  CoreVpdStatus status;
  uint8_t buff[16];
  uint8_t sum = 0, checksum;
  uint16_t offset = 0;
  uint16_t remaining = (uint16_t)(vpdLength - 1U);
  uint16_t i, chunk;

  if (vpdLength < 5 || vpdLength > ADMPCIEKU3_VPD_ROM_SIZE) {
  	dfDebugPrint(0, ("*** verifyVpdChecksum: invalid VPD length %lu(0x%lx)\n",
      (unsigned long)vpdLength, (unsigned long)vpdLength));
    pDevCtx->info.vpd.bChecksumError = TRUE;
    return FALSE;
  }

  /* Get checksum from VPD shadow memory; always last byte of structure */
  status = readShadowedVpd(pDevCtx, remaining, sizeof(checksum), &checksum);
  if (CoreVpdSuccess != status) {
  	dfDebugPrint(0, ("*** verifyVpdChecksum: failed to read checksum from VPD area\n"));
    pDevCtx->info.vpd.bChecksumError = TRUE;
    return FALSE;
  }

  /* Compute checksum over all but last byte of structure */
  while (remaining) {
    chunk = (uint16_t)((remaining > sizeof(buff)) ? sizeof(buff) : remaining);
    status = readShadowedVpd(pDevCtx, offset, chunk, buff);
    if (CoreVpdSuccess != status) {
  	  dfDebugPrint(0, ("*** verifyVpdChecksum: failed to data chunk from VPD area for checksumming\n"));
      pDevCtx->info.vpd.bChecksumError = TRUE;
      return FALSE;
    }

    for (i = 0; i < chunk; i++) {
      sum = (uint8_t)(sum + buff[i]);
    }
    offset = (uint16_t)(offset + chunk);
    remaining = (uint16_t)(remaining - chunk);
  }
  sum = (uint8_t)-sum;

  if (sum == checksum) {
    return TRUE;
  } else {
  	dfDebugPrint(0, ("*** verifyVpdChecksum: checksum invalid, computed 0x%02lx found 0x%02lx\n",
      (unsigned long)sum, (unsigned long)checksum));
    pDevCtx->info.vpd.bChecksumError = TRUE;
    return FALSE;
  }
}

static boolean_t
initVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  VpdAdmpcieku3* pVpd = &pDevCtx->info.vpd.pData->admpcieku3;
  unsigned int i;

  DF_ZERO_OBJECT(pVpd);
  pVpd->prom.rev0.version = 0;
  pVpd->prom.rev0.length = sizeof(VpdAdmpcieku3Rev0);
  pVpd->prom.rev0.partNumber = 1284U;
  pVpd->prom.rev0.serial = 99U;
  if (pDevCtx->model.admpcieku3.bIs2ndEndpoint) {
    pVpd->prom.rev0.serial |= 0x80000000U;
  }
  pVpd->prom.rev0.pcbRev = 0x0U;
  pVpd->prom.rev0.si5338Ref = 100000000U;
  pVpd->prom.rev0.dimmRef[1] = pVpd->prom.rev0.dimmRef[0] = 400000000U;
  pVpd->prom.rev0.sfpMgtRef = 156250000U;
  pVpd->prom.rev0.sataMgtRef = 150000000U;
  pVpd->prom.rev0.cclkRef = 80000000U;
  pVpd->prom.rev0.fpga.device = 323U; /* KU060 */
  dfAssert(sizeof(char) == sizeof(uint8_t));
  dfStrCpy((char*)pVpd->prom.rev0.fpga.speedGrade, DF_ARRAY_LENGTH(pVpd->prom.rev0.fpga.speedGrade), "1");
  dfStrCpy((char*)pVpd->prom.rev0.fpga.stepping, DF_ARRAY_LENGTH(pVpd->prom.rev0.fpga.stepping), "");
  pVpd->prom.rev0.fpga.tempGrade = ADMPCIEKU3_VPD_TEMPGRADE_COMMERCIAL;
  pVpd->prom.rev0.checksum = 0U;
  for (i = 0; i < DF_ARRAY_LENGTH(pVpd->dimm); i++) {
    pVpd->dimm[i].physicalSize = 0U;
    pVpd->dimm[i].freqMax = 0U;
    pVpd->dimm[i].bHasEcc = FALSE;
  }

  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE;
}

static boolean_t
readVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  CoreVpdStatus status;
  VpdAdmpcieku3* pVpd = &pDevCtx->info.vpd.pData->admpcieku3;
  VpdAdmpcieku3Rev0 vpdRaw;
  uint16_t vpdVersion = 0xffffU;
  uint16_t vpdLength = 0U;

  status = readShadowedVpd(pDevCtx, 0U, sizeof(vpdVersion), &vpdVersion);
  if (CoreVpdSuccess != status) {
  	dfDebugPrint(0, ("*** readVpd: failed to read VPD version field\n"));
    return FALSE;
  }
  vpdVersion = dfLe16ToCpu(vpdVersion);
  status = readShadowedVpd(pDevCtx, 2U, sizeof(vpdLength), &vpdLength);
  if (CoreVpdSuccess != status) {
  	dfDebugPrint(0, ("*** readVpd: failed to read VPD length field\n"));
    return FALSE;
  }
  vpdLength = dfLe16ToCpu(vpdLength);
  switch (vpdVersion) {
  case 0:
    /* Ok */
    break;

  case 0xffffU:
  	dfDebugPrint(0, ("*** readVpd: invalid VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    return FALSE;

  default:
  	dfDebugPrint(0, ("+++ readVpd: unrecognised VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    break;
  }
  verifyVpdChecksum(pDevCtx, vpdLength);

  status = readShadowedVpd(pDevCtx, 0U, sizeof(vpdRaw), &vpdRaw);
  if (CoreVpdSuccess != status) {
  	dfDebugPrint(0, ("*** readVpd: failed to block read VPD\n"));
    return FALSE;
  }
  DF_ZERO_OBJECT(pVpd);
  pVpd->prom.rev0.version = vpdVersion;
  pVpd->prom.rev0.length = vpdLength;
  pVpd->prom.rev0.partNumber = dfLe32ToCpu(vpdRaw.partNumber);
  pVpd->prom.rev0.serial = dfLe32ToCpu(vpdRaw.serial);
  if (pDevCtx->model.admpcieku3.bIs2ndEndpoint) {
    pVpd->prom.rev0.serial |= 0x80000000U;
  }
  pVpd->prom.rev0.pcbRev = dfLe8ToCpu(vpdRaw.pcbRev);
  pVpd->prom.rev0.si5338Ref = dfLe32ToCpu(vpdRaw.si5338Ref);
  pVpd->prom.rev0.dimmRef[0] = dfLe32ToCpu(vpdRaw.dimmRef[0]);
  pVpd->prom.rev0.dimmRef[1] = dfLe32ToCpu(vpdRaw.dimmRef[1]);
  pVpd->prom.rev0.sfpMgtRef = dfLe32ToCpu(vpdRaw.sfpMgtRef);
  pVpd->prom.rev0.sataMgtRef = dfLe32ToCpu(vpdRaw.sataMgtRef);
  pVpd->prom.rev0.cclkRef = dfLe32ToCpu(vpdRaw.cclkRef);
  dfAssert(sizeof(char) == sizeof(uint8_t));
  dfStrCpy((char*)pVpd->prom.rev0.fpga.stepping, DF_ARRAY_LENGTH(pVpd->prom.rev0.fpga.stepping), (const char*)vpdRaw.fpga.stepping);
  dfStrCpy((char*)pVpd->prom.rev0.fpga.speedGrade, DF_ARRAY_LENGTH(pVpd->prom.rev0.fpga.speedGrade), (const char*)vpdRaw.fpga.speedGrade);
  pVpd->prom.rev0.fpga.device = dfLe16ToCpu(vpdRaw.fpga.device);
  pVpd->prom.rev0.fpga.tempGrade = dfLe8ToCpu(vpdRaw.fpga.tempGrade);
  pVpd->prom.rev0.checksum = dfLe8ToCpu(vpdRaw.checksum);

  return TRUE; /* Success */
}

static boolean_t
readDimmVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  VpdAdmpcieku3* pVpd = &pDevCtx->info.vpd.pData->admpcieku3;
  unsigned int i;

  /* Figure out which DIMM slots are populated, and the overall geometry of any modules fitted */
  for (i = 0; i < DF_ARRAY_LENGTH(pVpd->dimm); i++) {
    Ddr3DimmSummary summary;

    /* Assume DIMM slot is unpopulated, will correct below if necessary. */
    pVpd->dimm[i].physicalSize = 0U;
    pVpd->dimm[i].bHasEcc = FALSE;
    pVpd->dimm[i].freqMax = 0U;
    if (admxrc6txGetDdr3DimmSpd(&pDevCtx->model.admpcieku3.i2c.context, ADMPCIEKU3_DIMM_SPD_I2C_BUS, (uint8_t)ADMPCIEKU3_DIMM_SPD_I2C_SLOT(i), &summary, NULL)) {
      dfDebugPrint(1, ("readVpd: DIMM %u summary: bRomCorrupt=%s physicalSize=0x%08lx_%08lx dataWidth=%u eccWidth=%u freqMax=%lu\n",
        i, summary.bRomCorrupt ? "TRUE" : "FALSE", dfSplitUint64(summary.physicalSize), summary.dataWidth, summary.eccWidth, (unsigned long)summary.freqMax));
      if (summary.bRomCorrupt) {
        dfDebugPrint(0, ("*** readVpd: DIMM %u: corruption detected in SPD ROM\n", i, summary.dataWidth));
      }
      if (summary.dataWidth != 64) {
        dfDebugPrint(0, ("*** readVpd: DIMM %u: unexpected data width: expecting 64, got %u\n", i, summary.dataWidth));
      }
      if (summary.eccWidth != 8U && summary.eccWidth != 0U) {
        dfDebugPrint(0, ("*** readVpd: DIMM %u: unexpected ECC width: expecting 0 or 8, got %u\n", i, summary.eccWidth));
      }
      if (!summary.bRomCorrupt && summary.dataWidth == 64U) {
        pVpd->dimm[i].physicalSize = summary.physicalSize;
        pVpd->dimm[i].bHasEcc = (summary.eccWidth == 8) ? TRUE : FALSE;
        pVpd->dimm[i].freqMax = summary.freqMax;
      }
    } else {
      /* DIMM slot is unpopulated or SPD structure is invalid. */
      dfDebugPrint(1, ("readVpd: DIMM %u not fitted or SPD ROM is invalid.\n",
        i, dfSplitUint64(summary.physicalSize)));
    }
  }

  return TRUE;
}

/* Returns TRUE if alert interrupt is active for specified target FPGA */
static boolean_t
alertPoll(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int targetIndex)
{
  DfInterruptObject* pInterruptObject;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t intStat;
  DfSpinLockFlags f;

  if (targetIndex > 0) {
    return FALSE;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  intStat = dfPciMemRead32(hModel, &pModelRegs->intStat);
  if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
    /* Alert interrupt still active */
    return TRUE;
  } else {
    /* Alert interrupt no longer active - reenable alert interrupt */
    pInterruptObject = &pDevCtx->interrupt.object;
    f = dfInterruptSpinLockGet(pInterruptObject);
    pDevCtx->model.admpcieku3.shadow.intCtl |= ADMXRC6Tx_V_INTCTL_SYSENABLE;
    dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admpcieku3.shadow.intCtl | ADMPCIEKU3_V_INTCTL_LEVELBITS);
    dfPciMemRead32(hModel, &pModelRegs->intCtl);
    dfInterruptSpinLockPut(pInterruptObject, f);
    return FALSE;
  }
}

static void
onSi5338ProgramDone(
  Si5338DeviceContext* pSi5338Ctx,
  Si5338ProgramRequest* pRequest,
  unsigned int msIndex,
  Si5338ProgramStatus status,
  void* pContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pContext;
  unsigned int clockIndex = 0;

  if (pSi5338Ctx == &pDevCtx->model.admpcieku3.si5338[0].context) {
    switch (msIndex) {
    case 0: /* REFCLK156M25_0 */
      clockIndex = 0;
      break;

    case 1: /* REFCLK250M */
      clockIndex = 2;
      break;

    case 2: /* REFCLK156M25_1 */
      clockIndex = 1;
      break;

    case 3: /* REFCLK133M */
      clockIndex = 7;
      break;

    default:
      dfAssert(FALSE);
      return;
    }
  } else if (pSi5338Ctx == &pDevCtx->model.admpcieku3.si5338[1].context) {
    switch (msIndex) {
    case 0: /* REFCLK400M_0 */
      clockIndex = 3;
      break;

    case 1: /* REFCLK200M */
      clockIndex = 6;
      break;

    case 2: /* REFCLK400M_1 */
      clockIndex = 4;
      break;

    case 3: /* REFCLK150M */
      clockIndex = 5;
      break;

    default:
      dfAssert(FALSE);
      return;
    }
  } else {
    dfAssert(FALSE);
    return;
  }

  onClockProgramDone(pDevCtx, clockIndex, si5338MapProgramStatus(status));
}

static ClockProgramStatus
clockProgram(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  const CoreClockWord* pClockWord)
{
  unsigned int si5338Index, si5338MsIndex;

  if (clockIndex >= pDevCtx->info.bootstrap.numClockGen) {
    return ClockProgramInvalidIndex;
  }

  switch (clockIndex) {
  case 0: /* REFCLK156M25_0 */
    si5338Index = 0U;
    si5338MsIndex = 0U;
    break;

  case 1: /* REFCLK156M25_1 */
    si5338Index = 0U;
    si5338MsIndex = 2U;
    break;

  case 2: /* REFCLK250M */
    si5338Index = 0U;
    si5338MsIndex = 1U;
    break;

  case 3: /* REFCLK400M_0 */
    si5338Index = 1U;
    si5338MsIndex = 0U;
    break;

  case 4: /* REFCLK400M_1 */
    si5338Index = 1U;
    si5338MsIndex = 2U;
    break;

  case 5: /* REFCLK150M */
    si5338Index = 1U;
    si5338MsIndex = 3U;
    break;

  case 6: /* REFCLK200M */
    si5338Index = 1U;
    si5338MsIndex = 1U;
    break;

  case 7: /* REFCLK133M */
    si5338Index = 0U;
    si5338MsIndex = 3U;
    break;

  default:
    return ClockProgramInvalidIndex;
  }

  return si5338MapProgramStatus(
    si5338ClockProgram(
      &pDevCtx->model.admpcieku3.si5338[si5338Index].context,
      &pDevCtx->model.admpcieku3.si5338[si5338Index].programming[si5338MsIndex],
      si5338MsIndex,
      pClockWord,
      onSi5338ProgramDone,
      pDevCtx));
}

static ClockWordStatus
clockWord(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  uint32_t flags,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  unsigned int si5338Index;

  if (clockIndex >= pDevCtx->info.bootstrap.numClockGen) {
    return ClockWordInvalidIndex;
  }

  switch (clockIndex) {
  case 0: /* REFCLK156M25_0 */
  case 1: /* REFCLK156M25_1 */
  case 2: /* REFCLK250M */
  case 7: /* REFCLK133M */
    si5338Index = 0;
    break;

  case 3: /* REFCLK400M_0 */
  case 4: /* REFCLK400M_1 */
  case 5: /* REFCLK150M */
  case 6: /* REFCLK200M */
    si5338Index = 1;
    break;

  default:
    return ClockWordInvalidIndex;
  }

  return si5338MapClockWordStatus(
    si5338ClockWord(
      pDevCtx->model.admpcieku3.si5338[si5338Index].context.vco.frequency,
      frequencyReq,
      frequencyMin,
      frequencyMax,
      pDevCtx->model.admpcieku3.si5338[si5338Index].context.bIsSi5338a,
      pClockWord));
}

static FpgaControlStatus
fpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  boolean_t* pbValue)
{
  dfDebugPrint(8, ("fpgaControl: index=%lu opCode=%lu pbValue=%p\n",
    (unsigned long)index, (unsigned long)opCode, pbValue));
  
  if (0 != index) {
    return FpgaControlInvalidIndex;
  }

  /* ADM-PCIE-KU3 cannot be fully reconfigured */
  return FpgaControlNotSupported;
}

static FpgaSelectMapStatus
fpgaSelectMap(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  boolean_t bWrite,
  size_t length,
  void* pBuffer)
{
  dfDebugPrint(8, ("fpgaSelectMap: index=%lu bWrite=%s length=" DF_FMT_U64 "(0x%08lx_%08lx) pBuffer=%p\n",
    (unsigned long)index, bWrite ? "TRUE" : "FALSE", (unsigned long long)length, dfSplitUint64((uint64_t)length), pBuffer));
  
  if (0 != index) {
    return FpgaSelectMapInvalidIndex;
  }
  return FpgaSelectMapNotSupported;
}

static void
dmaEngineInit(
  Adb3CoreDeviceContext* pDevCtx)
{
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;

  /* Initialize DMA enables */
  dfPciMemWrite32(hBridge, &pRegs->dmaEnable, ADB3DMAEN_LOCAL_MASTER | ADB3DMAEN_DMA_ALL);

  /* Initialize DMA engine registers */
  adb3InitDmaEngines(pDevCtx, pDevCtx->info.bootstrap.numDmaChannel);

  if (pDevCtx->pDevObj->hardwareId.variant.pci.revision >= 0x03U) {
    uint32_t val32 = dfPciMemRead32(hBridge, &pRegs->pcieControl) & 0xF0FFFFFFU;
    /* Initialize PCI-E packet control for rev 0x03 or higher: allow up to 8 PCIe read packets in flight */
    dfPciMemWrite32(hBridge, &pRegs->pcieControl, val32 | 0x07000000U);           
  }
}

static boolean_t
initHardware(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bInit,
  unsigned int phase)
{
  boolean_t bIs2ndEndpoint = (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice == PCI_SUBDEVICE_ADB3_ADMPCIEKU3_2ND) ? TRUE : FALSE;
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  I2cStatus i2cStatus;
  unsigned int i;
  uint8_t val8;

  dfDebugPrint(2, ("initHardware: bInit=%s\n", bInit ? "TRUE" : "FALSE"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  if (bInit) {
    switch (phase) {
    case 0:
      pDevCtx->model.admpcieku3.bIs2ndEndpoint = bIs2ndEndpoint;
      pDevCtx->info.model = bIs2ndEndpoint ? CoreModelAdmpcieku32nd : CoreModelAdmpcieku3;

      /* The set of Bridge interrupts that we are interested in */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnableMaster = ADB3IRQ_DMA_ALL | ADB3IRQ_MODEL;

      /* Ensure Bridge interrupts are disabled */
      pDevCtx->hardware.shadow.bridge.adb3.irqEnable = 0;
      dfPciMemWrite32(hBridge, &pRegs->irqEnable, pDevCtx->hardware.shadow.bridge.adb3.irqEnable);
      dfPciMemRead32(hBridge, &pRegs->irqEnable);

      /* Enable model-specific interrupts that we are interested in */
      if (bIs2ndEndpoint) {
        pDevCtx->model.admpcieku3.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE;
      } else {
        pDevCtx->model.admpcieku3.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE | ADMXRC6Tx_V_INTCTL_SYSENABLE;
      }
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admpcieku3.shadow.intCtl | ADMPCIEKU3_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);

      /* Figure out number of DMA engines present, and directions they support */
      pDevCtx->info.bootstrap.numDmaChannel = adb3CountDmaEngines(pDevCtx);

      /* Initialize DMA engine stuff */
      dmaEngineInit(pDevCtx);

      /* Initialize ICAP interface */
      pDevCtx->model.admpcieku3.icap.bPresent = adb3HasIcapInterface(pDevCtx);
      if (pDevCtx->model.admpcieku3.icap.bPresent) {
        icapInit(&pDevCtx->model.admpcieku3.icap.context, hModel, &pModelRegs->icap, ADMPCIEKU3_IPROG_DELAYCNT_FREQ_HZ);
      } else {
        /* Ensure that IPROG methods are not called */
        pDevCtx->methods.pIprogControl = NULL;
        pDevCtx->methods.pIprogStatus = NULL;
      }

      /* Initialize I2C stuff */
      if (!bIs2ndEndpoint) {
        i2cInit(&pDevCtx->model.admpcieku3.i2c.context, NULL, admxrc6txI2cStart, admxrc6txI2cPoll, pDevCtx);
      }

      /* Start off with fake VPD, in case subsequent VPD reading fails. */
      initVpd(pDevCtx);
      /* Read VPD from PROM. */
      readVpd(pDevCtx);
      if (!bIs2ndEndpoint) {
        /* Read memory bank VPD from SODIMM slots. */
        readDimmVpd(pDevCtx);
      }

      if (!bIs2ndEndpoint) {
        /* Look for SI5338 1 on I2C bus 1; if it's there, board must be rev. 2 */
        i2cStatus = i2cReadSync(&pDevCtx->model.admpcieku3.i2c.context, ADMPCIEKU3_SI5338_I2C_BUS, ADMPCIEKU3_SI5338_1_I2C_SLOT, 0U, &val8);
        if (I2cStatusSuccess == i2cStatus) {
          uint32_t initFreq[4];

          /* Initialize SI5338 1 */
          si5338Init(&pDevCtx->model.admpcieku3.i2c.context, &pDevCtx->model.admpcieku3.si5338[0].context, ADMPCIEKU3_SI5338_I2C_BUS, ADMPCIEKU3_SI5338_1_I2C_SLOT);
          /* Find the current SI5338 output frequencies */
          if (!si5338Interrogate(&pDevCtx->model.admpcieku3.si5338[0].context, pDevCtx->info.vpd.pData->admpcieku3.prom.rev0.si5338Ref, initFreq)) {
            dfDebugPrint(0, ("*** initHardware: failed to interrogate SI5338 1\n"));
          } else {
            pDevCtx->clockGenerator[0].currentFrequency = initFreq[0];
            pDevCtx->clockGenerator[2].currentFrequency = initFreq[1];
            pDevCtx->clockGenerator[1].currentFrequency = initFreq[2];
            pDevCtx->clockGenerator[7].currentFrequency = initFreq[3];
          }
          if (dfDebugLevel(2)) {
            dfDebugPrint(0, ("initHardware: SI5338 1 register dump:\n"));
            si5338DumpRegisters(&pDevCtx->model.admpcieku3.si5338[0].context);
          }

          /* Initialize SI5338 2 */
          si5338Init(&pDevCtx->model.admpcieku3.i2c.context, &pDevCtx->model.admpcieku3.si5338[1].context, ADMPCIEKU3_SI5338_I2C_BUS, ADMPCIEKU3_SI5338_2_I2C_SLOT);
          /* Find the current SI5338 output frequencies */
          if (!si5338Interrogate(&pDevCtx->model.admpcieku3.si5338[1].context, pDevCtx->info.vpd.pData->admpcieku3.prom.rev0.si5338Ref, initFreq)) {
            dfDebugPrint(0, ("*** initHardware: failed to interrogate SI5338 2\n"));
          } else {
            pDevCtx->clockGenerator[3].currentFrequency = initFreq[0];
            pDevCtx->clockGenerator[6].currentFrequency = initFreq[1];
            pDevCtx->clockGenerator[4].currentFrequency = initFreq[2];
            pDevCtx->clockGenerator[5].currentFrequency = initFreq[3];
          }
          if (dfDebugLevel(2)) {
            dfDebugPrint(0, ("initHardware: SI5338 2 register dump:\n"));
            si5338DumpRegisters(&pDevCtx->model.admpcieku3.si5338[1].context);
          }
        }
      }

      if (!bIs2ndEndpoint) {
        /* Identify Flash device */
        admxrc6txFlashResetBank(pDevCtx, &pDevCtx->hardware.model.pAdmxrc6tx->flash);
        if (flashCfiIdentify(pDevCtx, 0, 0)) {
          pDevCtx->info.flash[0].bPresent = TRUE;
          pDevCtx->info.flash[0].targetArea.start = 0x0U;
          pDevCtx->info.flash[0].targetArea.length = pDevCtx->info.flash[0].totalSize - pDevCtx->info.flash[0].targetArea.start;
        } else {
          dfDebugPrint(0, ("*** initHardware: failed to identify Flash\n"));
        }
      }

      /* Initialize information needed for production test */
      pDevCtx->info.prodTest.serialNumber = pDevCtx->info.vpd.pData->admpcieku3.prom.rev0.serial;
      pDevCtx->info.prodTest.cpldRevision = 0U;
      pDevCtx->info.prodTest.pcbRevision = pDevCtx->info.vpd.pData->admpcieku3.prom.rev0.pcbRev;
      pDevCtx->info.prodTest.bridgeDate = dfPciMemRead32(hModel, &pModelRegs->buildDate);
      pDevCtx->info.prodTest.bridgeTime = dfPciMemRead32(hModel, &pModelRegs->buildTime);

      /* Find base addresses and sizes of local bus windows */
      i = 0;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      i++;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[3].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[3].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[3].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.pBridgedPrefetch;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[3].size;
      i++;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = 0;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.model.pGeneric;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      i++;
      pDevCtx->info.window[i].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[i].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[i].localBase = 0;
      pDevCtx->info.window[i].localSize = 0;
      pDevCtx->info.window[i].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[i].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[i].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[i].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[0].size;
      i++;

      /* Initialize sensor information structures */
      initSensorInfo(pDevCtx);
      break;

    case 1:
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    switch (phase) {
    case 0:
      if (!bIs2ndEndpoint) {
        /* Uninitialize I2C stuff */
        i2cUninit(&pDevCtx->model.admpcieku3.i2c.context);
      }

      if (pDevCtx->model.admpcieku3.icap.bPresent) {
        /* Uninitialize ICAP interface */
        icapUninit(&pDevCtx->model.admpcieku3.icap.context, pDevCtx->hardware.bCanTouch);
      }

      if (pDevCtx->hardware.bCanTouch) {
        /* Deinitialize DMA enables */
        dfPciMemWrite32(hBridge, &pRegs->dmaEnable, 0);
      }
      break;

    case 1:
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  }

  return TRUE;
}

static IprogResult
iprogCtl(
	Adb3CoreDeviceContext* pDevCtx,
  uint32_t targetIndex,
  IprogControlOp op,
  uint64_t address,
  uint32_t delayMs)
{
  if (targetIndex != 0U) {
    return IprogResultBadFpgaIndex; 
  }

  switch (op) {
  case IprogControlOpScheduleFromNow:
    if (address > 0xFFFFFFFFU) {
      return IprogResultBadAddress; 
    }
    return iprogScheduleFromNow(&pDevCtx->model.admpcieku3.icap.context, (uint32_t)address, delayMs);
  
  case IprogControlOpScheduleOnStop:
    if (address > 0xFFFFFFFFU) {
      return IprogResultBadAddress; 
    }
    return iprogScheduleOnStop(&pDevCtx->model.admpcieku3.icap.context, (uint32_t)address, delayMs);

  case IprogControlOpAbort:
    return iprogAbort(&pDevCtx->model.admpcieku3.icap.context);

  default:
    return IprogResultUnknownError;
  }
}

static IprogResult
iprogStat(
	Adb3CoreDeviceContext* pDevCtx,
  uint32_t targetIndex,
  IprogScheduleState* pState,
  uint64_t* pAddress,
  uint32_t* pDelayMs)
{
  uint32_t cfgaddrValue;

  if (targetIndex != 0) {
    return IprogResultBadFpgaIndex; 
  }

  iprogStatus(&pDevCtx->model.admpcieku3.icap.context, pState, &cfgaddrValue, pDelayMs);
  *pAddress = cfgaddrValue;

  return IprogResultOk;
}

static boolean_t
isr(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t irqStatus, dmaIrqStatus;
  uint32_t intStat, intCtl;
  unsigned int channel, n;
  DfSpinLockFlags f;

  dfDebugPrint(8, ("isr: entered\n"));

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  /* Sample the currently active interrupts. */
  irqStatus = dfPciMemRead32(hBridge, &pRegs->irqStatus);

  f = dfIsrSpinLockGet(pInterruptObj);
  /* Mask the currently active interrupts with the interrupts we are interested in right now. */
  irqStatus &= pDevCtx->hardware.shadow.bridge.adb3.irqEnable;
  /* Clear the interrupts that we just saw. */
  dfPciMemWrite32(hBridge, &pRegs->irqStatus, irqStatus);

  if (irqStatus & ADB3IRQ_MODEL) {
    dfDebugPrint(9, ("isr: model-specific interrupt\n"));
    intCtl = pDevCtx->model.admpcieku3.shadow.intCtl;
    intStat = dfPciMemRead32(hModel, &pModelRegs->intStat);
    intStat &= intCtl;
    if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
      /* Disable hardware monitor interrupt; gets reenabled by hardware monitor polling. */
      intCtl &= ~ADMXRC6Tx_V_INTCTL_SYSENABLE;
      /* Write new model-specific interrupt enables. */
      pDevCtx->model.admpcieku3.shadow.intCtl = intCtl;
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, intCtl | ADMPCIEKU3_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);
    }
    dfIsrSpinLockPut(pInterruptObj, f);

    /* Clear the model-specific interrupts that we saw */
    dfPciMemWrite32(hModel, &pModelRegs->intStat, intStat);

    /* Take action for the model-specific interrupts that we saw. */
    if (intStat & ADMXRC6Tx_V_INTSTAT_FP) {
      dfDebugPrint(9, ("isr: target FPGA 0 interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.fpga[0].dpc, (void*)(uintptr_t)0);
    }
    if (intStat & ADMXRC6Tx_V_INTSTAT_SYS) {
      dfDebugPrint(9, ("isr: hardware monitor interrupt\n"));
      dfDpcSchedule(&pDevCtx->interrupt.hwMon[0].dpc, (void*)(uintptr_t)0);
    }
  } else {
    dfIsrSpinLockPut(pInterruptObj, f);
  }

  n = pDevCtx->info.bootstrap.numDmaChannel;
  for (channel = 0; channel < n; channel++) {
    if (irqStatus & (ADB3IRQ_DMA0 << channel)) {
      dmaIrqStatus = dfPciMemRead32(hBridge, &pRegs->dmaEngine[channel].irq.status);
      dfDebugPrint(9, ("isr: DMA%lu interrupt, DMA%lu.irqStatus=0x%08lx\n",
        (unsigned long)channel, (unsigned long)channel, (unsigned long)dmaIrqStatus));
#ifdef ADB3_DBG_IRQ_COUNT
      pDevCtx->interrupt.dma[channel].count++;
#endif
      dfPciMemWrite32(hBridge, &pRegs->dmaEngine[channel].irq.ack, dmaIrqStatus);
      dfDpcSchedule(&pDevCtx->interrupt.dma[channel].dpc, (void*)(uintptr_t)channel);
    }
  }

  return irqStatus ? TRUE : FALSE;
}

static boolean_t
mapBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated,
	boolean_t bUnmap)
{
	dfDebugPrint(1, ("mapBusResources: bUnmap=%s\n", bUnmap ? "TRUE" : "FALSE"));

	if (bUnmap) {
		if (NULL != pDevCtx->hardware.bridge.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.bridge.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 generic registers\n"));
			}
      pDevCtx->hardware.bridge.pGeneric = NULL;
		}
		if (NULL != pDevCtx->hardware.model.pGeneric) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.model.pGeneric)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 model-specific registers\n"));
			}
      pDevCtx->hardware.model.pGeneric = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridged) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridged)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 bridged space\n"));
			}
      pDevCtx->hardware.pBridged = NULL;
		}
    if (NULL != pDevCtx->hardware.pBridgedPrefetch) {
			if (!dfIoSpaceUnmap(pDevCtx->pDevObj, pDevCtx->hardware.pBridgedPrefetch)) {
				dfDebugPrint(0, ("*** mapBusResources: failed to unmap ADB3 prefetchable bridged space\n"));
			}
      pDevCtx->hardware.pBridgedPrefetch = NULL;
		}
		dfDebugPrint(2, ("mapBusResources: unmapped\n"));
	} else {
		pDevCtx->hardware.bridge.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 0, 0, pTranslated->variant.pci.memBar[0].size, &pDevCtx->hardware.hBridge);
		if (NULL == pDevCtx->hardware.bridge.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 generic registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 generic registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[0].base), pDevCtx->hardware.bridge.pGeneric));
		}
		pDevCtx->hardware.model.pGeneric = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 1, 0, pTranslated->variant.pci.memBar[1].size, &pDevCtx->hardware.hModel);
		if (NULL == pDevCtx->hardware.model.pGeneric) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 model-specific registers\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 model-specific registers @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[1].base), pDevCtx->hardware.model.pGeneric));
		}
    pDevCtx->hardware.pBridged = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 2, 0, pTranslated->variant.pci.memBar[2].size, &pDevCtx->hardware.hBridged);
		if (NULL == pDevCtx->hardware.pBridged) {
			dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 bridged space\n"));
			return FALSE;
		} else {
			dfDebugPrint(2, ("mapBusResources: mapped ADB3 bridged space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[2].base), pDevCtx->hardware.pBridged));
		}
    pDevCtx->hardware.pBridgedPrefetch = dfIoSpaceMap(pDevCtx->pDevObj, pTranslated, 3, 0, pTranslated->variant.pci.memBar[3].size, &pDevCtx->hardware.hBridgedPrefetch);
    if (NULL == pDevCtx->hardware.pBridgedPrefetch) {
      dfDebugPrint(0, ("*** mapBusResources: failed to map ADB3 prefetchable bridged space\n"));
      return FALSE;
    } else {
      dfDebugPrint(2, ("mapBusResources: mapped ADB3 prefetchable bridged space @ 0x%08lx_%08lx to %p\n",
        dfSplitUint64(pTranslated->variant.pci.memBar[3].base), pDevCtx->hardware.pBridgedPrefetch));
    }
		dfDebugPrint(2, ("mapBusResources: mapped OK\n"));
	}

	return TRUE;
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  boolean_t bIs2ndEndpoint = pDevCtx->model.admpcieku3.bIs2ndEndpoint;
  Adb3GenericRegs* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  Si5338ProgramStatus si5338Status;
  unsigned int index;

  dfAssert(newState == 0 || newState == 3);

  pRegs = pDevCtx->hardware.bridge.pAdb3;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
      if (pDevCtx->model.admpcieku3.icap.bPresent) {
        /* Save ICAP interface state */
        icapSaveState(&pDevCtx->model.admpcieku3.icap.context);
      }

      if (!bIs2ndEndpoint) {
        /* Save SI5338 clock generator state */
        for (index = 0; index < DF_ARRAY_LENGTH(pDevCtx->model.admpcieku3.si5338); index++) {
          si5338SaveState(&pDevCtx->model.admpcieku3.si5338[index].context);
        }
      }
    } else { /* Powering up (from D3) */
      /* Reenable model-specific interrupts that we are interested in */
      if (bIs2ndEndpoint) {
        pDevCtx->model.admpcieku3.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE;
      } else {
        pDevCtx->model.admpcieku3.shadow.intCtl = ADMXRC6Tx_V_INTCTL_FPENABLE | ADMXRC6Tx_V_INTCTL_SYSENABLE;
      }
      dfPciMemWrite32(hModel, &pModelRegs->intCtl, pDevCtx->model.admpcieku3.shadow.intCtl | ADMPCIEKU3_V_INTCTL_LEVELBITS);
      dfPciMemRead32(hModel, &pModelRegs->intCtl);

      /* Reinitialize DMA engine stuff */
      dmaEngineInit(pDevCtx);

      if (!bIs2ndEndpoint) {
        /* Reset Flash device */
        admxrc6txFlashResetBank(pDevCtx, &pDevCtx->hardware.model.pAdmxrc6tx->flash);
      }

      if (!bIs2ndEndpoint) {
        /* Restore SI5338 clock generator state */
        for (index = 0; index < DF_ARRAY_LENGTH(pDevCtx->model.admpcieku3.si5338); index++) {
          si5338Status = si5338RestoreState(&pDevCtx->model.admpcieku3.si5338[index].context);
          if (Si5338ProgramStatusSuccess != si5338Status) {
            dfDebugPrint(0, ("*** powerChange: failed to restore SI5338 %u state, status=%u\n", index, si5338Status));
          }
        }
      }

      if (pDevCtx->model.admpcieku3.icap.bPresent) {
        /* Restore ICAP interface state */
        icapRestoreState(&pDevCtx->model.admpcieku3.icap.context);
      }
    }
  }
}

static CoreSensorStatus
readSensor(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreSensorValue* pReading)
{
  CoreSensorStatus status = CoreSensorInvalidIndex;
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;

  if (index >= pDevCtx->info.bootstrap.numSensor) {
    return CoreSensorInvalidIndex;
  }

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  if (index < DF_ARRAY_LENGTH(g_voltageInfo)) {
    ucd90120BufReadSensor(pModelRegs->model3.ucd90120.sensor, hModel, g_voltageMapping[index] - 1U, UCD90120SENSORTYPE_VOLTAGE, pReading);
    return CoreSensorSuccess;
  }
  index -= (unsigned int)DF_ARRAY_LENGTH(g_voltageInfo);

  if (index < DF_ARRAY_LENGTH(g_currentInfo)) {
    ucd90120BufReadSensor(pModelRegs->model3.ucd90120.sensor, hModel, g_currentMapping[index] - 1U, UCD90120SENSORTYPE_CURRENT, pReading);
    return CoreSensorSuccess;
  }
  index -= (unsigned int)DF_ARRAY_LENGTH(g_currentInfo);

  if (index < DF_ARRAY_LENGTH(g_tempInfo)) {
    int64_t tmp, tmp1, tmp2;

    ucd90120BufReadSensor(pModelRegs->model3.ucd90120.sensor, hModel, g_tempMapping[index] - 1U, UCD90120SENSORTYPE_VOLTAGE, pReading);
    
    /*
    ** A voltage reading from LTC2997 chip connected to the temperature
    ** diode of the target FPGA is used for sensing temperature, and it
    ** is scaled in the UCD90120 such that:
    **
    **   tempC = voltage * 10
    */

    /* Implement 10x as (2x + 8x) */
    tmp = (int64_t)pReading->doubleValue.intPart << 32;
    tmp |= (int64_t)pReading->doubleValue.fracPart;
    tmp1 = tmp << 1;
    tmp2 = tmp << 3;
    tmp = tmp1 + tmp2;

    /* Now put back into int/frac format */
    pReading->doubleValue.intPart = (int32_t)(tmp >> 32);
    pReading->doubleValue.fracPart = (uint32_t)tmp;
    return CoreSensorSuccess;
  }
  index -= (unsigned int)DF_ARRAY_LENGTH(g_tempInfo);

  return status;
}

static CoreVpdStatus
readWriteVpd(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bWrite,
  size_t address,
  size_t length,
  void* pData)
{
  I2cStatus i2cStatus = I2cStatusSuccess;
  uint8_t* p = (uint8_t*)pData;
  uint8_t val8;

  dfAssertPassiveContext();

  if (CHECK_BOUNDS_OFFSET(address, length, pDevCtx->info.bootstrap.vpdOffset, pDevCtx->info.bootstrap.vpdLength)) {
    return CoreVpdInvalidRegion;
  }

  if (bWrite) {
    while (length) {
      i2cStatus = i2cWriteSync(&pDevCtx->model.admpcieku3.i2c.context, (uint8_t)ADMPCIEKU3_VPD_ROM_I2C_BUS, (uint8_t)ADMPCIEKU3_VPD_ROM_I2C_SLOT, (uint8_t)address, *p++);
      if (I2cStatusSuccess != i2cStatus) {
        dfDebugPrint(0, ("*** readWriteVpd: bWrite=%s i2cStatus=%u\n", bWrite ? "true" : "false", i2cStatus));
        goto done;
      }
      length--;
      address++;
    }
  } else {
    while (length) {
      i2cStatus = i2cReadSync(&pDevCtx->model.admpcieku3.i2c.context, (uint8_t)ADMPCIEKU3_VPD_ROM_I2C_BUS, (uint8_t)ADMPCIEKU3_VPD_ROM_I2C_SLOT, (uint8_t)address, &val8);
      if (I2cStatusSuccess != i2cStatus) {
        dfDebugPrint(0, ("*** readWriteVpd: bWrite=%s i2cStatus=%u\n", bWrite ? "true" : "false", i2cStatus));
        goto done;
      }
      *p++ = val8;
      length--;
      address++;
    }
  }
done:
  switch (i2cStatus) {
  case I2cStatusSuccess:
    return CoreFlashSuccess;

  case I2cStatusTimeout:
    return CoreVpdTimeout;

  case I2cStatusInvalidBus:
  case I2cStatusInvalidDevice:
  case I2cStatusInvalidAddress:
    return CoreVpdInvalidParameter;

  case I2cStatusHardwareError:
  case I2cStatusGeneralFailure:
  default:
    return CoreVpdGeneralFailure;
  }
}

static boolean_t
validateBusResources(
	Adb3CoreDeviceContext* pDevCtx,
	DfBusResources* pRaw,
	DfBusResources* pTranslated)
{
	dfDebugPrint(1, ("validateBusResources: entered\n"));

	if (pTranslated->variant.pci.numMemBar < 4) {
		dfDebugPrint(0, ("*** validateBusResources: less than 4 memory BARs\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[0].size < 0x1000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR0 smaller than 0x1000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[1].size < sizeof(ModelRegsAdmxrc6tx)) {
		dfDebugPrint(0, ("*** validateBusResources: BAR1 smaller than 0x%lu bytes\n", (unsigned long)sizeof(ModelRegsAdmxrc6tx)));
		return FALSE;
	}
	if (pTranslated->variant.pci.memBar[2].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR2 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
  if (pTranslated->variant.pci.memBar[3].size < 0x100000U) {
		dfDebugPrint(0, ("*** validateBusResources: BAR3 smaller than 0x100000 bytes\n"));
		return FALSE;
	}
	if (pTranslated->variant.pci.numIrq != 1) {
		dfDebugPrint(0, ("*** validateBusResources: not exactly 1 interrupt\n"));
		return FALSE;
	}

	dfDebugPrint(2, ("validateBusResources: OK\n"));

	return TRUE;
}

static void
bootstrap(
	Adb3CoreDeviceContext* pDevCtx)
{
  boolean_t bExposeAllClocks = FALSE;
  boolean_t bIs2ndEndpoint = (pDevCtx->pDevObj->hardwareId.variant.pci.subDevice == PCI_SUBDEVICE_ADB3_ADMPCIEKU3_2ND) ? TRUE : FALSE;

  dfAssert(DF_OFFSET_OF(VpdAdmpcieku3Rev0, checksum) == sizeof(VpdAdmpcieku3Rev0) - 1);

  pDevCtx->methods.pAlertPoll = alertPoll;
  pDevCtx->methods.pClearErrors = adb3ClearErrors;
  pDevCtx->methods.pClockProgram = clockProgram;
  pDevCtx->methods.pClockWord = clockWord;
  pDevCtx->methods.pDmaList = adb3DmaList;
  pDevCtx->methods.pDmaTransfer = adb3DmaTransfer;
  pDevCtx->methods.pEnableInterrupts = adb3EnableInterrupts;
  pDevCtx->methods.pFlash = admxrc6txFlashSingleBank;
  pDevCtx->methods.pFpgaControl = fpgaControl;
  pDevCtx->methods.pFpgaSelectMap = fpgaSelectMap;
  pDevCtx->methods.pGetDeviceStatus = adb3GetDeviceStatus;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIprogControl = iprogCtl; /* Assume that we have ICAP interface; will nullify later if necessary */
  pDevCtx->methods.pIprogStatus = iprogStat; /* Assume that we have ICAP interface; will nullify later if necessary */
  pDevCtx->methods.pIsr = isr;
  pDevCtx->methods.pMapBusResources = mapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pReadSensor = readSensor;
  pDevCtx->methods.pReadWriteVpd = readWriteVpd;
  pDevCtx->methods.pValidateBusResources = validateBusResources;

  pDevCtx->info.bootstrap.numClockGen = 0;
  if (!bIs2ndEndpoint) {
    pDevCtx->info.bootstrap.numClockGen += 6U;
    dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "Si5338ExposeAllClocks", &bExposeAllClocks);
    if (bExposeAllClocks) {
      /* Expose REFCLK200M & REFCLK133M as clock generators 6 & 7 if Si5338ExposeAllClocks driver param is true. */
      pDevCtx->info.bootstrap.numClockGen += 2U;
    }
  }
  pDevCtx->info.bootstrap.numDmaChannel = ADMPCIEKU3_NUM_DMA_CHANNEL;
  pDevCtx->info.bootstrap.numIoModule = 0U;
  pDevCtx->info.bootstrap.numSensor = 0U;
  if (!bIs2ndEndpoint) {
    pDevCtx->info.bootstrap.numSensor +=
      (unsigned int)DF_ARRAY_LENGTH(g_voltageInfo) +
      (unsigned int)DF_ARRAY_LENGTH(g_currentInfo) +
      (unsigned int)DF_ARRAY_LENGTH(g_tempInfo);
  }
  pDevCtx->info.bootstrap.numTargetFpga = 1;
  pDevCtx->info.bootstrap.numWindow = 4;
  pDevCtx->info.bootstrap.numFlashBank = bIs2ndEndpoint ? 0 : 1;
  pDevCtx->info.bootstrap.bVpdInFlash = FALSE;
  pDevCtx->info.bootstrap.vpdLength = bIs2ndEndpoint ? 0 : ADMPCIEKU3_VPD_ROM_SIZE;
  pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdmpcieku3);
  pDevCtx->info.bootstrap.vpdOffset = 0;
  pDevCtx->info.bootstrap.flash[0].deviceWidth = 2;
  pDevCtx->info.bootstrap.flash[0].width = 2;
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "PciAddress64Bit", &pDevCtx->info.bootstrap.b64BitDmaAddress);
  pDevCtx->info.bootstrap.dmaDescriptorSize = 32;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x1000000U;
  pDevCtx->info.bootstrap.bWorkAround4kCrossing = FALSE;
  pDevCtx->info.bootstrap.bI2cInterruptDriven = FALSE;
}

void
adb3CoreBootstrapAdmpcieku3(
	Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdmpcieku3: entered\n"));
  bootstrap(pDevCtx);
}
