/*
** File: ics307.c  
** Project: ADB3 core driver
** Purpose: Functions for programming the ICS307 clock generator.
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>

#include "ics307_common.h"

static const uint32_t g_fRefMin = 5000000U;   /* Minimum allowed reference clock frequency in Hz, from datasheet */
static const uint32_t g_fRefMax = 270000000U; /* Maximum allowed reference clock frequency in Hz, from datasheet */

/* Use Industrial temperature range values */
static const uint32_t g_fVcoMin = 60000000U;  /* Minimum allowed VCO frequency in Hz, from datasheet */
static const uint32_t g_fVcoMax = 360000000U; /* Maximum allowed VCO frequency in Hz, from datasheet */

static const uint32_t g_fOutMin = 6000000U;   /* Minimum possible output frequency */
static const uint32_t g_fOutMax = 200000000U; /* Maximum possible output frequency */

static const uint32_t g_ttlVal = 0x1U;
static const uint32_t g_fVal = 0x3U;
static const uint32_t g_cVal = 0x0U;

/* OD values */
static const uint32_t g_odVal[8] = {
  10, 2, 8, 4, 5, 7, 3, 6
};

Ics307ClockWordStatus
clockWordIcs307(
  uint32_t flags,
  uint32_t refFrequency,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  boolean_t bUsingClk2,
  CoreClockWord* pClockWord)
{
  uint32_t maxRDWd;
  uint32_t odIdx, RDWd, VDWd;
  uint32_t bestFrequency = 0, bestError, bestRDW = 0, bestVDW = 0, bestOdIdx = 0;
  uint32_t fVco, f, error;
  
  dfDebugPrint(3, ("clockWordIcs307: refFrequency=%lu frequencyReq=%lu\n", (unsigned long)refFrequency, (unsigned long)frequencyReq));

  if (refFrequency < g_fRefMin || refFrequency > g_fRefMax) {
    dfDebugPrint(0, ("*** clockWordIcs307: refFrequency=%lu, out of range; must be 5..27 MHz\n", (unsigned long)refFrequency));
    return Ics307ClockWordStatusGeneralFailure;
  }

  if (bUsingClk2) {
    /* We're using CLK2 as the as the output clock, which is. So we have to multiply the desired output frequency by 2. */
    if (frequencyReq < g_fOutMin || frequencyReq * 2 > g_fOutMax) {
      return Ics307ClockWordStatusOutOfRange;
    }
    frequencyReq = frequencyReq << 1;
  } else {
    if (frequencyReq < g_fOutMin || frequencyReq > g_fOutMax) {
      return Ics307ClockWordStatusOutOfRange;
    }
  }

  /* Get the maximum allowed value for reference divider word. */
  maxRDWd = refFrequency / 200000;
  if (maxRDWd > 129) {
    maxRDWd = 129;
  }
  
  /* Brute force guessing of best OD, RDW and VDW combination. */
  bestError = 0xFFFFFFFFU;
  for (odIdx = 0; odIdx < 8; odIdx++) {
    for (RDWd = 3; RDWd <= maxRDWd; RDWd++) {
      for (VDWd = 12; VDWd <= 519; VDWd++) {
        fVco = (uint32_t)refFrequency * 2 / RDWd * VDWd;
        if (fVco < g_fVcoMin || fVco > g_fVcoMax) {
          continue;
        }
        f = fVco / g_odVal[odIdx];
        if ((flags & CoreClockMinimumFrequency) && f < frequencyMin) {
          continue;
        }
        if ((flags & CoreClockMaximumFrequency) && f > frequencyMax) {
          continue;
        }
        error = (f > frequencyReq) ? f - (uint32_t)frequencyReq : (uint32_t)frequencyReq - f;
        if (error < bestError) {
          bestFrequency = f;
          bestError = error;
          bestRDW = RDWd - 2;
          bestVDW = VDWd - 8;
          bestOdIdx = odIdx;
        }
      }
    }
  }
  
  if (bestFrequency == 0) {
    /* Failed to find a suitable value */
    return Ics307ClockWordStatusOutOfRange;
  }

  pClockWord->d0 = 0;
  pClockWord->d0 |= (bestRDW & 0x7fU)  << 0;
  pClockWord->d0 |= (bestVDW & 0x1ffU) << 7;
  pClockWord->d0 |= (bestOdIdx & 0x7U) << 16;
  pClockWord->d0 |= (g_fVal & 0x3U) << 19;
  pClockWord->d0 |= (g_ttlVal & 0x1U) << 21;
  pClockWord->d0 |= (g_cVal & 0x3U) << 22;
  if (bUsingClk2) {
    pClockWord->frequency = bestFrequency >> 1;
  } else {
    pClockWord->frequency = bestFrequency;
  }

  if (pClockWord->frequency < 50001000) {
		/* SelectMap LCLK divide of 1 for 0-50MHz */
		pClockWord->d1 = 2;
	} else if (pClockWord->frequency < 75001000) {
		/* SelectMap LCLK divide of 1.5 for 50-75MHz */
		pClockWord->d1 = 1;
	} else {
		/* SelectMap LCLK divide of 2 for 75+MHz */
		pClockWord->d1 = 0;
	}

  dfDebugPrint(3, ("clockWordIcs307: bestFrequency=%lu clockWord=0x%08lx newClkDivide=%u\n",
    (unsigned long)bestFrequency, (unsigned long)pClockWord->d0, (unsigned int)pClockWord->d1));

  return Ics307ClockWordStatusSuccess;
}

boolean_t
ics307Init(
  Adb3CoreDeviceContext* pDevCtx,
  Ics307DeviceContext* pIcs307Ctx,
  uint32_t refFrequency,
  Ics307InterfaceType progIf,
  boolean_t bUsingClk2,
  uint32_t* pReg,
  DfMemoryHandle hReg,
  uint32_t* pFOut)
{
  uint32_t val32, clockWord;
  uint32_t r, v, s, f;
  uint32_t fVco, fOutClk1, fOut;
  uint8_t smapClkDiv;

  dfTimerInit(&pIcs307Ctx->timer, pDevCtx);
  pIcs307Ctx->pReg = pReg;
  pIcs307Ctx->hReg = hReg;
  pIcs307Ctx->save.bLastValid = FALSE;
  pIcs307Ctx->progIf = progIf;

  if (refFrequency < g_fRefMin || refFrequency > g_fRefMax) {
    dfDebugPrint(0, ("*** ics307Init: refFrequency=%lu, out of range\n", (unsigned long)refFrequency));
    return FALSE;
  }

  /* Figure out current output frequency of ICS307 */
  val32 = dfPciMemRead32(pIcs307Ctx->hReg, pIcs307Ctx->pReg);
  dfDebugPrint(2, ("ics307Init: CLOCK=0x%08lX\n", (unsigned long)val32));
  clockWord = (pIcs307Ctx->progIf == Ics307InterfaceTypeXpl) ? val32 >> 8 : val32;
  r = ((clockWord >> 0) & 0x7FU) + 2U;
  v = ((clockWord >> 7) & 0x1FFU) + 8U;
  s = (clockWord >> 16) & 0x7U;
  f = (clockWord >> 19) & 0x3U;

  fVco = (uint32_t)refFrequency * 2 / r * v;
  if (fVco < g_fVcoMin || fVco > g_fVcoMax) {
    dfDebugPrint(0, ("*** ics307Init: VCO frequency=%lu, out of range\n", (unsigned long)fVco));
  }
  fOutClk1 = fVco / g_odVal[s];

  if (bUsingClk2) {
    switch (f) {
    case 0:
      fOut = refFrequency;
      dfDebugPrint(0, ("*** ics307Init: outputting reference clock, fOut=%lu\n", (unsigned long)fOut));
      break;

    case 1:
      fOut = refFrequency >> 1;
      dfDebugPrint(0, ("*** ics307Init: outputting reference clock / 2, fOut=%lu\n", (unsigned long)fOut));
      break;

    case 2:
      fOut = 0;
      dfDebugPrint(0, ("*** ics307Init: output is OFF\n"));
      break;

    case 3:
    default:
      fOut = fOutClk1 >> 1;
      break;
    }
  } else {
    fOut = fOutClk1;
  }

  dfDebugPrint(1, ("ics307Init: fVco=%lu fOut=%lu output=%s\n",
    (unsigned long)fVco, (unsigned long)fOut, bUsingClk2 ? "CLK2" : "CLK1"));

  pIcs307Ctx->curr.frequency = fOut;

  if (pIcs307Ctx->progIf == Ics307InterfaceTypeXpl) {
    smapClkDiv = (uint8_t)((val32 >> 2) & 0x3U);
  } else {
    smapClkDiv = 0; /* Not used in ADM-XRC-4LX / -4SX programming interface */
  }
  pIcs307Ctx->curr.smapClkDiv = smapClkDiv;

  *pFOut = fOut;

  return TRUE;
}

void
ics307Uninit(
  Adb3CoreDeviceContext* pDevCtx,
  Ics307DeviceContext* pIcs307Ctx)
{
  dfTimerUninit(&pIcs307Ctx->timer);
}

static void
onCheckIcs307(
  DfTimer* pTimer,
  void* pTimerContext,
  void* pCallbackContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pTimerContext;
  Ics307ProgramRequest* pRequest = (Ics307ProgramRequest*)pCallbackContext;
  Ics307DeviceContext* pIcs307Ctx = pRequest->pIcs307Ctx;
  const CoreClockWord* pClockWord = &pRequest->clockWord;
  Ics307ProgramStatus status = Ics307ProgramStatusSuccess;
  uint32_t* pReg;
  DfMemoryHandle hReg;
  uint32_t val32, clockWord;
  uint8_t smapClkDiv = 0;
  uint64_t newFreq;
  boolean_t bSuccess;

  pReg = pIcs307Ctx->pReg;
  hReg = pIcs307Ctx->hReg;

  val32 = dfPciMemRead32(hReg, pReg);
  dfDebugPrint(5, ("onCheckIcs307: CLOCK=0x%08lX\n", (unsigned long)val32));
  if (pIcs307Ctx->progIf == Ics307InterfaceTypeXpl) {
    bSuccess = (val32 & 0x1U) == 0x1U ? TRUE : FALSE;
  } else {
    bSuccess = (val32 & 0x80000000U) == 0x80000000U ? TRUE : FALSE;
  }
  if (!bSuccess) {
    status = Ics307ProgramStatusHardwareError;
  }

  if (Ics307ProgramStatusSuccess == status) {
    newFreq = pClockWord->frequency;
    clockWord = pClockWord->d0;

    if (pIcs307Ctx->progIf == Ics307InterfaceTypeXpl) {
      smapClkDiv = (uint8_t)pClockWord->d1;
      if (newFreq <= pIcs307Ctx->curr.frequency) {
        /* Decreasing frequency means change SelectMap clock division factor after changing frequency */
        val32 = ((clockWord & 0xffffffU) << 8) | ((uint32_t)(smapClkDiv & 0x3U) << 2);
        dfDebugPrint(5, ("onCheckIcs307: writing CLOCK register with 0x%08lX\n", (unsigned long)val32));
        dfPciMemWrite32(hReg, pReg, val32);
        dfPciMemRead32(hReg, pReg);
      }
    }

    pIcs307Ctx->curr.frequency = (uint32_t)newFreq;
    pIcs307Ctx->curr.smapClkDiv = smapClkDiv;
    pIcs307Ctx->save.bLastValid = TRUE;
    pIcs307Ctx->save.last = pRequest->clockWord;
  }

  pRequest->pCallback(pDevCtx, pIcs307Ctx, pRequest, status);
}

Ics307ProgramStatus
ics307Program(
  Adb3CoreDeviceContext* pDevCtx,
  Ics307DeviceContext* pIcs307Ctx,
  Ics307ProgramRequest* pRequest,
  const CoreClockWord* pClockWord,
  Ics307ProgramCallback* pCallback)
{
  uint32_t* pReg;
  DfMemoryHandle hReg;
  uint32_t val32, clockWord;
  uint8_t smapClkDiv;
  uint64_t newFreq;

  pRequest->pIcs307Ctx = pIcs307Ctx;
  pRequest->pCallback = pCallback;
  pRequest->clockWord = *pClockWord;

  newFreq = pClockWord->frequency;
  clockWord = pClockWord->d0;

  pReg = pIcs307Ctx->pReg;
  hReg = pIcs307Ctx->hReg;

  if (pIcs307Ctx->progIf == Ics307InterfaceTypeXpl) {
    if (newFreq > pIcs307Ctx->curr.frequency) {
      /* Increasing frequency means change SelectMap clock division factor at the same time as changing frequency */
      smapClkDiv = (uint8_t)pClockWord->d1;
    } else {
      /* Decreasing frequency means change SelectMap clock division factor after changing frequency */
      smapClkDiv = pIcs307Ctx->curr.smapClkDiv;
    }
    val32 = 0x1U | ((clockWord & 0xffffffU) << 8) | ((uint32_t)(smapClkDiv & 0x3U) << 2);
  } else {
    val32 = 0x80000000U | clockWord;
  }

  dfDebugPrint(5, ("ics307Program: writing CLOCK register with 0x%08lX\n", (unsigned long)val32));
  dfPciMemWrite32(hReg, pReg, val32);
  dfPciMemRead32(hReg, pReg);

  dfTimerSchedule(&pIcs307Ctx->timer, dfTimeGet() + dfMicrosecondsToTime(50000), onCheckIcs307, pRequest);
  return Ics307ProgramStatusSuccess;
}

static void
onProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  Ics307DeviceContext* pV6ClkSynthCtx,
  Ics307ProgramRequest* pRequest,
  Ics307ProgramStatus status)
{
  Ics307ProgramRequestSync* pRequestSync = (Ics307ProgramRequestSync*)pRequest;

  pRequestSync->status = status;
  dfEventSignal(&pRequestSync->ev);
}

Ics307ProgramStatus
ics307ProgramSync(
  Adb3CoreDeviceContext* pDevCtx,
  Ics307DeviceContext* pIcs307Ctx,
  const CoreClockWord* pClockWord)
{
  Ics307ProgramRequestSync req;
  Ics307ProgramStatus status;

  dfEventInit(&req.ev);
  status = ics307Program(pDevCtx, pIcs307Ctx, &req.request, pClockWord, onProgramDone);
  if (Ics307ProgramStatusSuccess == status) {
    dfEventWait(&req.ev);
    status = req.status;
  }
  dfEventUninit(&req.ev);
  return status;
}

void
ics307SaveState(
  Adb3CoreDeviceContext* pDevCtx,
  Ics307DeviceContext* pIcs307Ctx)
{
  /* Nothing to do yet */
}

Ics307ProgramStatus
ics307RestoreState(
  Adb3CoreDeviceContext* pDevCtx,
  Ics307DeviceContext* pIcs307Ctx)
{
  Ics307ProgramStatus status = Ics307ProgramStatusSuccess;
 
  /* If the ICS307 had been programmed before power-down, reprogram it with the same thing */
  if (pIcs307Ctx->save.bLastValid) {
    status = ics307ProgramSync(pDevCtx, pIcs307Ctx, &pIcs307Ctx->save.last);
  }

  return status;
}

ClockProgramStatus
ics307MapProgramStatus(
  Ics307ProgramStatus status)
{
  switch (status) {
  case Ics307ProgramStatusSuccess:
    return ClockProgramSuccess;

  case Ics307ProgramStatusHardwareError:
    return ClockProgramHardwareError;

  case Ics307ProgramStatusGeneralFailure:
  default:
    return ClockProgramGeneralFailure;
  }
}

ClockWordStatus
ics307MapClockWordStatus(
  Ics307ClockWordStatus status)
{
  switch (status) {
  case Ics307ClockWordStatusSuccess:
    return ClockWordSuccess;

  case Ics307ClockWordStatusOutOfRange:
    return ClockWordOutOfRange;

  case Ics307ClockWordStatusGeneralFailure:
  default:
    return ClockWordGeneralFailure;
  }
}
