/*
** File: icd2061.c  
** Project: ADB3 core driver
** Purpose: Functions for programming the ICD2061 clock generator.
**
** (C) Copyright Alpha Data 2008-2011
*/

#include <df.h>

#include "device.h"
#include "icd2061_common.h"

typedef struct _Icd2061ProgramRequestSync {
  Icd2061ProgramRequest request;
  DfEvent ev;
  Icd2061ProgramStatus status;
} Icd2061ProgramRequestSync;

static void
onCconSerializerAcquired(
  SerializerQueue* pQueue,
  SerializerRequest* pRequest,
  void* pContextIgnored);

Icd2061ClockWordStatus
icd2061ClockWord(
  unsigned int clockIndex,
  uint32_t flags,
	uint32_t refFrequency,
	uint64_t frequencyReq,
	uint64_t frequencyMin,
	uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  static const struct {
    uint32_t min, max;
  } mclkRange[16] = {
    { 49990000,  51000000 },
    { 50990000,  53200000 },
    { 53190000,  58500000 },
    { 58490000,  60700000 },
    { 60690000,  64400000 },
    { 60390000,  66800000 },
    { 66790000,  73500000 },
    { 73490000,  75600000 },
    { 75590000,  80900000 },
    { 80890000,  83200000 },
    { 83190000,  91500000 },
    { 91490000, 100000000 },
    { 99990000, 120100000 },
    { 99990000, 120100000 },
    { 99990000, 120100000 },
    { 99990000, 120100000 }
  };
  const uint32_t pMax = 131; /* Max allowable P, chosen to reduce jitter */
  uint32_t clockWord, i, d, p, q;
  uint32_t bestP = 0, bestQ = 0, bestD = 0; /* Init. to avoid compiler whine */
  uint32_t fVCO = 0;
  uint32_t error, bestError;
  uint32_t frequency, bestFrequency = 0; /* Init. to avoid compiler whine */
  uint32_t qMin, qMax; /* Min / max allowable Q */
  boolean_t bFound = FALSE;

  if (clockIndex >= 2) {
    return Icd2061ClockWordStatusInvalidIndex;
  }

  if (frequencyReq < 400000 || frequencyReq > 100000000 || clockIndex >= 2) {
    return Icd2061ClockWordStatusOutOfRange;
  }

  /* Brute force P and Q guessing */
  bestError = 1000000000U;
  qMax = (refFrequency / 200000) - 1;
  qMin = (refFrequency / 5000000) + 1;
  for (d = 0; d < 8; d++) {
    for (q = qMin; q <= qMax; q++) {
      for (p = 4; p < pMax; p++) {
        fVCO = (uint32_t)refFrequency * 2 / q * p;
        if (fVCO < 50000000 || fVCO > 120000000) {
          continue;
        }
        /*
        ** The associativity of multiplication and divide operators is
        ** left to right. IF THIS FORMULA IS MODIFIED, ENSURE THAT IT
        ** CANNOT OVERFLOW A uint32_t.
        */
        frequency = fVCO / (1 << d);
        if ((flags & CoreClockMinimumFrequency) && frequency < frequencyMin) {
          continue;
        }
        if ((flags & CoreClockMaximumFrequency) && frequency > frequencyMax) {
          continue;
        }
        if (frequency > (uint32_t)frequencyReq) {
          error = frequency - (uint32_t)frequencyReq;
        } else {
          error = (uint32_t)frequencyReq - frequency;
        }
        if (error < bestError) {
          bestQ = q;
          bestP = p;
          bestD = d;
          bestError = error;
          bestFrequency = frequency;
          bFound = TRUE;
        }
      }
    }
  }

  if (!bFound) {
    return Icd2061ClockWordStatusGeneralFailure;
  }

  /* Find the index value for presetting the VCO range */
  for (i = 0; i < 16; i++) {
    if ((fVCO > mclkRange[i].min) && (fVCO < mclkRange[i].max)) {
      break;
    }
  }

  /* Assemble the completed word apart from the register selection */
  clockWord = (i << 17) | ((bestP - 3) << 10) | (bestD << 7) | (bestQ - 2);
	
  /*
  ** Add the ICD2061 register index
  **
  ** index = 0 means MCLK register
  ** index = 1 means VCLK1 register
  */
  switch (clockIndex) {
  case 0:
    clockWord |= (3 << 21);  /* ICD2061's MCLK */
    break;

  case 1:	
    clockWord |= (0 << 21);  /* ICD2061's VCLK1 */
    break;

  default:
    dfAssert(FALSE);
    break;
  }

  pClockWord->frequency = bestFrequency;
  pClockWord->d0 = clockWord;

  return Icd2061ClockWordStatusSuccess;
}

extern boolean_t
icd2061Init(
  Icd2061DeviceContext* pIcd2061Ctx,
  uint8_t* pReg,
  DfMemoryHandle hReg)
{
  serializerQueueInit(&pIcd2061Ctx->cconSerializer);
  dfTimerInit(&pIcd2061Ctx->timer, pIcd2061Ctx);
  pIcd2061Ctx->save[0].bLastValid = FALSE;
  pIcd2061Ctx->save[1].bLastValid = FALSE;
  pIcd2061Ctx->pReg = pReg;
  pIcd2061Ctx->hReg = hReg;
  return TRUE;
}

extern void
icd2061Uninit(
  Icd2061DeviceContext* pIcd2061Ctx)
{
  dfTimerUninit(&pIcd2061Ctx->timer);
}

static void
onCheckIcd2061(
  DfTimer* pTimer,
  void* pTimerContext,
  void* pCallbackContext)
{
  Icd2061ProgramRequest* pProgRequest = (Icd2061ProgramRequest*)pCallbackContext;
  Icd2061DeviceContext* pIcd2061Ctx = (Icd2061DeviceContext*)pTimerContext;
  Icd2061ProgramStatus status = Icd2061ProgramStatusSuccess;
  DfMemoryHandle hCcon;
  uint8_t* pCcon;
  uint8_t val8;
  unsigned int attempts = pProgRequest->attempts;
  unsigned int index = pProgRequest->index;
  hCcon = pIcd2061Ctx->hReg;
  pCcon = pIcd2061Ctx->pReg;

  val8 = (uint8_t)(dfPciMemRead8(hCcon, pCcon) & 0x10U);
  if (val8) {
    /* Failed */
    if (attempts < 10) {
      pProgRequest->attempts = attempts + 1;
      onCconSerializerAcquired(&pIcd2061Ctx->cconSerializer, &pProgRequest->serRequest, NULL);
      return;
    }
  }
  serializerPut(&pIcd2061Ctx->cconSerializer);
  if (val8) {
    status = Icd2061ProgramStatusHardwareError;
  } else {
    pIcd2061Ctx->save[index].bLastValid = TRUE;
    pIcd2061Ctx->save[index].last = pProgRequest->clockWord;
  }
  pProgRequest->pCallback(pIcd2061Ctx, pProgRequest, index, status, pProgRequest->pContext);
}

static void
onCconSerializerAcquired(
  SerializerQueue* pQueue,
  SerializerRequest* pRequest,
  void* pContextIgnored)
{
  Icd2061ProgramRequest* pProgRequest = DF_CONTAINER_OF(pRequest, Icd2061ProgramRequest, serRequest);
  Icd2061DeviceContext* pIcd2061Ctx = pProgRequest->pIcd2061Ctx;
  DfMemoryHandle hCcon;
  uint8_t* pCcon;
  unsigned int i;
	uint32_t clockWord = pProgRequest->clockWord.d0;
  uint8_t val8;

  hCcon = pIcd2061Ctx->hReg;
  pCcon = pIcd2061Ctx->pReg;

  dfPciMemWrite8(hCcon, pCcon, ICD2061_CLK | ICD2061_DATA);
  (void)dfPciMemRead8(hCcon, pCcon);
  
  /*
  ** Send unlock sequence to ICD2061; at least five 0->1 CLK transitions with DATA
  **  held at 1, and than a 0->1 CLK transition with DATA at 0.
  */
  for (i = 0; i < 6; i++) {
    dfPciMemWrite8(hCcon, pCcon, ICD2061_DATA);
    (void)dfPciMemRead8(hCcon, pCcon);
    dfPciMemWrite8(hCcon, pCcon, ICD2061_CLK | ICD2061_DATA);
    (void)dfPciMemRead8(hCcon, pCcon);
  }
  dfPciMemWrite8(hCcon, pCcon, ICD2061_DATA);
  (void)dfPciMemRead8(hCcon, pCcon);
  dfPciMemWrite8(hCcon, pCcon, 0);
  (void)dfPciMemRead8(hCcon, pCcon);
  dfPciMemWrite8(hCcon, pCcon, ICD2061_CLK);
  (void)dfPciMemRead8(hCcon, pCcon);
  dfPciMemWrite8(hCcon, pCcon, 0);
  (void)dfPciMemRead8(hCcon, pCcon);
  dfPciMemWrite8(hCcon, pCcon, ICD2061_CLK);
  (void)dfPciMemRead8(hCcon, pCcon);
  
  /* Send the data bits to ICD2061 (uses a modified Manchester encoding) */
  for (i = 0; i < 24; i++) {
    if (clockWord & 0x1U) {
      val8 = 0;
    } else {
      val8 = 2;
    }
    clockWord >>= 1;
    
    dfPciMemWrite8(hCcon, pCcon, (uint8_t)(val8 | ICD2061_CLK));
    (void)dfPciMemRead8(hCcon, pCcon);
    dfPciMemWrite8(hCcon, pCcon, val8);
    (void)dfPciMemRead8(hCcon, pCcon);
    dfPciMemWrite8(hCcon, pCcon, (uint8_t)(val8 ^ ICD2061_DATA));
    (void)dfPciMemRead8(hCcon, pCcon);
    dfPciMemWrite8(hCcon, pCcon, (uint8_t)((val8 ^ ICD2061_DATA) | ICD2061_CLK));
    (void)dfPciMemRead8(hCcon, pCcon);
  }
  
  /*
  ** Send stop bit to ICD2061; this is a 1->0 and 0->1 transition of CLK while DATA
  ** is 1.
  */
  dfPciMemWrite8(hCcon, pCcon, ICD2061_CLK | ICD2061_DATA);
  (void)dfPciMemRead8(hCcon, pCcon);
  dfPciMemWrite8(hCcon, pCcon, ICD2061_DATA);
  (void)dfPciMemRead8(hCcon, pCcon);
  dfPciMemWrite8(hCcon, pCcon, ICD2061_CLK | ICD2061_DATA);
  (void)dfPciMemRead8(hCcon, pCcon);
  dfPciMemWrite8(hCcon, pCcon, 0);
  (void)dfPciMemRead8(hCcon, pCcon);

  dfTimerSchedule(&pIcd2061Ctx->timer, dfTimeGet() + dfMicrosecondsToTime(50000), onCheckIcd2061, pProgRequest);
}

extern Icd2061ProgramStatus
icd2061Program(
  Icd2061DeviceContext* pIcd2061Ctx,
  Icd2061ProgramRequest* pRequest,
  unsigned int index,
  const CoreClockWord* pClockWord,
  Icd2061ProgramCallback* pCallback,
  void* pContext)
{
  pRequest->pIcd2061Ctx = pIcd2061Ctx;
  pRequest->clockWord = *pClockWord;
  pRequest->pCallback = pCallback;
  pRequest->pContext = pContext;
  pRequest->index = index;
  pRequest->attempts = 0;
  serializerGet(&pIcd2061Ctx->cconSerializer, &pRequest->serRequest, onCconSerializerAcquired, NULL);
  return Icd2061ProgramStatusSuccess;
}

static void
onProgramDone(
  Icd2061DeviceContext* pV5ClkSynthCtx,
  Icd2061ProgramRequest* pRequest,
  unsigned int index,
  Icd2061ProgramStatus status,
  void* pContextIgnored)
{
  Icd2061ProgramRequestSync* pRequestSync = (Icd2061ProgramRequestSync*)pRequest;

  pRequestSync->status = status;
  dfEventSignal(&pRequestSync->ev);
}

extern Icd2061ProgramStatus
icd2061ProgramSync(
  Icd2061DeviceContext* pIcd2061Ctx,
  unsigned int index,
  const CoreClockWord* pClockWord)
{
  Icd2061ProgramRequestSync req;
  Icd2061ProgramStatus status;

  dfEventInit(&req.ev);
  status = icd2061Program(pIcd2061Ctx, &req.request, index, pClockWord, onProgramDone, NULL);
  if (Icd2061ProgramStatusSuccess == status) {
    dfEventWait(&req.ev);
    status = req.status;
  }
  dfEventUninit(&req.ev);
  return status;
}

extern void
icd2061SaveState(
  Icd2061DeviceContext* pIcd2061Ctx)
{
  /* Nothing to do yet */
}

extern Icd2061ProgramStatus
icd2061RestoreState(
  Icd2061DeviceContext* pIcd2061Ctx)
{
  Icd2061ProgramStatus status = Icd2061ProgramStatusSuccess;
  unsigned int index;

  for (index = 0; index < 2; index++) {
    if (pIcd2061Ctx->save[index].bLastValid) {
      status = icd2061ProgramSync(pIcd2061Ctx, index, &pIcd2061Ctx->save[index].last);
      if (Icd2061ProgramStatusSuccess != status) {
        return status;
      }
    }
  }

  return status;
}

ClockProgramStatus
icd2061MapProgramStatus(
  Icd2061ProgramStatus status)
{
  switch (status) {
  case Icd2061ProgramStatusSuccess:
    return ClockProgramSuccess;

  case Icd2061ProgramStatusHardwareError:
    return ClockProgramHardwareError;

  case Icd2061ProgramStatusGeneralFailure:
  default:
    return ClockProgramGeneralFailure;
  }
}

ClockWordStatus
icd2061MapClockWordStatus(
  Icd2061ClockWordStatus status)
{
  switch (status) {
  case Icd2061ClockWordStatusSuccess:
    return ClockWordSuccess;

  case Icd2061ClockWordStatusInvalidIndex:
    return ClockWordInvalidIndex;

  case Icd2061ClockWordStatusOutOfRange:
    return ClockWordOutOfRange;

  case Icd2061ClockWordStatusGeneralFailure:
  default:
    return ClockWordGeneralFailure;
  }
}
