/*
** File: flash_legacy.c  
** Project: ADB3 core driver
** Purpose: Implements functions for dealing with legacy (non-CFI) Flash devices.
**
** (C) Copyright Alpha Data 2013
*/

#include "device.h"
#include "flash_legacy.h"

#define DEBUGLEVEL_ERROR  (0) /* Level of debug messages relating to driver logic errors */
#define DEBUGLEVEL_NORMAL (3) /* Level of debug messages relating to normal operation */

#define MAKE_ID(V, D) (((uint32_t)(D)) | ((uint32_t)(V) << 16))

#define VENDOR_SST_ATMEL    (0xBFU)
#define DEVICE_SST_39VF016Q (0xD9U)

#define VENDOR_MICRON         (0x89U)
#define DEVICE_MICRON_28F008B (0x99U)

/* Check for SST 39VF016Q device (8-bit wide data) */
static boolean_t
identifySst39vf016q(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t* pTotalSize,
  LegacyFlashInformation* pLegacyInfo)
{
  const uint64_t baseAddr = 0x0U;
  uint32_t val32, vendor, device;
  boolean_t bFound = FALSE;

  /* Enter "Read ID" mode. */
  val32 = 0xAAU;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, baseAddr + 0x5555U, &val32);
  val32 = 0x55U;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, baseAddr + 0xAAAAU, &val32);
  val32 = 0x90U;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, baseAddr + 0x5555U, &val32);
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, baseAddr, &vendor);
  if (vendor == VENDOR_SST_ATMEL) {
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, baseAddr + 0x1U, &device);

    val32 = 0xF0U;
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, baseAddr, &val32);

    switch (device) {
    case DEVICE_SST_39VF016Q:
      /* Identified SST 39VF016Q, 16Mbit (2MByte) */
      *pTotalSize = 2*1024*1024;
      pLegacyInfo->vendorId = (uint16_t)vendor;
      pLegacyInfo->deviceId = (uint16_t)device;
      pLegacyInfo->numSpecialBlock = 0;
      pLegacyInfo->blockSize = 64*1024;
      pLegacyInfo->largestBlockSize = pLegacyInfo->blockSize;
      bFound = TRUE;
      break;

    default:
      break;
    }
  }
  
  /* Go to read array mode */
  val32 = 0xFFU;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, baseAddr, &val32);
  val32 = 0xF0U;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, baseAddr, &val32);

  return bFound;
}

/* Check for Micron 28F008B device (8-bit wide data) */
static boolean_t
identifyMicron28f008b(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t* pTotalSize,
  LegacyFlashInformation* pLegacyInfo)
{
  const uint64_t baseAddr = 0x0U;
  uint32_t val32, vendor, device;
  boolean_t bFound = FALSE;

  /* Enter "Read ID" mode. */
  val32 = 0x90U;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, baseAddr, &val32);
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, baseAddr, &vendor);
  if (vendor == VENDOR_MICRON) {
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, baseAddr + 0x1U, &device);

    val32 = 0xFFU;
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, baseAddr, &val32);

    switch (device) {
    case DEVICE_MICRON_28F008B:
      /* Identified Micron 28F008B, 8Mbit (1MByte) */
      *pTotalSize = 1024*1024;
      pLegacyInfo->vendorId = (uint16_t)vendor;
      pLegacyInfo->deviceId = (uint16_t)device;
      pLegacyInfo->blockSize = 128*1024;
      pLegacyInfo->largestBlockSize = pLegacyInfo->blockSize;
      pLegacyInfo->numSpecialBlock = 4;
      pLegacyInfo->specialBlock[0].address = (uint32_t)baseAddr;
      pLegacyInfo->specialBlock[0].size = 16*1024;
      pLegacyInfo->specialBlock[1].address = pLegacyInfo->specialBlock[0].address + pLegacyInfo->specialBlock[0].size;
      pLegacyInfo->specialBlock[1].size = 8*1024;
      pLegacyInfo->specialBlock[2].address = pLegacyInfo->specialBlock[1].address + pLegacyInfo->specialBlock[1].size;
      pLegacyInfo->specialBlock[2].size = 8*1024;
      pLegacyInfo->specialBlock[3].address = pLegacyInfo->specialBlock[2].address + pLegacyInfo->specialBlock[2].size;
      pLegacyInfo->specialBlock[3].size = 96*1024;
      bFound = TRUE;
      break;

    default:
      break;
    }
  }
  
  /* Go to read array mode */
  val32 = 0xFFU;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, baseAddr, &val32);
  val32 = 0xF0U;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, baseAddr, &val32);

  return bFound;
}

CoreFlashStatus
eraseBlockMicron28F008B(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t blockAddress)
{
  CoreFlashStatus status = CoreFlashSuccess;
  uint32_t val32;
  DfTime expire;
  DfTime delayIncrement = dfMicrosecondsToTime(1000); /* 1 ms */
  boolean_t bTimeout = FALSE;

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("eraseBlockMicron28F008B: bankIndex=%u blockAddress=0x%08lx_%08lx\n",
    bankIndex, dfSplitUint64(blockAddress)));

  /* Send ERASE-BLOCK command */
  val32 = 0x20U;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);
  val32 = 0xD0U;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);

  /* Wait for completion */
  expire = dfTimeGet() + dfMicrosecondsToTime(14000000); /* 14 sec */
  do {
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, blockAddress, &val32);
    if (val32 & 0x80U) {
      break;
    }
    if (dfTimeGreater(dfTimeGet(), expire)) {
      bTimeout = TRUE;
    } else {
      dfDelayThreadFor(delayIncrement);
    }
  } while (!bTimeout);
  if (bTimeout) {
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** eraseBlockMicron28F008B: Timed out waiting for ERASE-BLOCK\n"));
    status = CoreFlashTimeout;
  } else {
    /* Status-check */
    if (val32 & 0x1cU) {
      dfDebugPrint(DEBUGLEVEL_ERROR, ("*** eraseBlockMicron28F008B: Error detected after ERASE-BLOCK: SR[7:0] = 0x%02lx\n", (unsigned long)val32));
      status = CoreFlashEraseFailed;
    }
  }

  /* If an error occurred, attempt to return to normal operation */
  if (CoreFlashSuccess != status) {
    /* Send CLEAR-STATUS command */
    val32 = 0x50U;
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);
  }

  /* Send READ-ARRAY command */
  val32 = 0xFFU;
  pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);

  return status;
}

CoreFlashStatus
writeBlockMicron28F008B(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t blockAddress,
  uint64_t length,
  const void* pData)
{
  CoreFlashStatus status = CoreFlashSuccess;
  uint64_t i, limit;
  uint32_t val32;
  uint8_t* p8 = (uint8_t*)pData;
  unsigned int timeout;

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("writeBlockMicron28F008B: bankIndex=%u blockAddress=0x%08lx_%08lx, length=0x%08lx_%08lx\n",
    bankIndex, dfSplitUint64(blockAddress), dfSplitUint64(length)));

  limit = blockAddress + length;
  for (i = blockAddress; i < limit; i++) {
    /* Send WRITE-BYTE command */
    val32 = 0x40U;
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);
    val32 = *p8++;
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, i, &val32);
    
    /* Wait for completion (10 us) */
    timeout = 5;
    while (timeout) {
      dfDelayMicroseconds(2);
      /* Use data polling method */
      pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, i, &val32);
      if (val32 & 0x80U) {
        break;
      }
      timeout--;
    }
    if (0 == timeout) {
      dfDebugPrint(DEBUGLEVEL_ERROR, ("*** writeBlockMicron28F008B: Timed out waiting for WRITE-BYTE\n"));
      status = CoreFlashTimeout;
    } else {
      /* Status check */
      if (val32 & 0x0cU) {
        dfDebugPrint(DEBUGLEVEL_ERROR, ("*** writeBlockMicron28F008B: Error detected after WRITE-BYTE: SR[7:0] = 0x%02lx\n", (unsigned long)val32));
        status = CoreFlashWriteFailed;
      }
    }
    
    /* Return to READ-ARRAY mode */
    val32 = 0xffU;
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);

    if (CoreFlashSuccess != status) {
      break;
    }
  }

  return status;
}

CoreFlashStatus
eraseBlockSst39vf016q(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t blockAddress)
{
  CoreFlashStatus status = CoreFlashSuccess;
  uint32_t val32;
  DfTime expire;
  DfTime delayIncrement = dfMicrosecondsToTime(1000); /* 1 ms */
  boolean_t bTimeout = FALSE;

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("eraseBlockSst39vf016q: bankIndex=%u blockAddress=0x%08lx_%08lx\n",
    bankIndex, dfSplitUint64(blockAddress)));

  /* Send ERASE-BLOCK command */
	val32 = 0xAAU;
	pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, 0x5555, &val32);
	val32 = 0x55U;
	pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, 0x2AAA, &val32);
	val32 = 0x80U;
	pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, 0x5555, &val32);
	val32 = 0xAAU;
	pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, 0x5555, &val32);
	val32 = 0x55U;
	pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, 0x2AAA, &val32);
	val32 = 0x50U;
	pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, blockAddress, &val32);

  /* Wait for completion */
  expire = dfTimeGet() + dfMicrosecondsToTime(30000); /* 30 ms */
  do {
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, blockAddress, &val32);
		if (val32 & 0x80U) {
			break;
		}
    if (dfTimeGreater(dfTimeGet(), expire)) {
      bTimeout = TRUE;
    } else {
      dfDelayThreadFor(delayIncrement);
    }
  } while (!bTimeout);
  if (bTimeout) {
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** eraseBlockSst39vf016q: Timed out waiting for ERASE-BLOCK\n"));
    status = CoreFlashTimeout;
  }

  return status;
}

CoreFlashStatus
writeBlockSst39vf016q(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t blockAddress,
  uint64_t length,
  const void* pData)
{
  CoreFlashStatus status = CoreFlashSuccess;
  uint64_t i, limit;
  uint32_t val32;
  uint8_t* p8 = (uint8_t*)pData;
  uint8_t byte;
  unsigned int timeout;

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("writeBlockSst39vf016q: bankIndex=%u blockAddress=0x%08lx_%08lx, length=0x%08lx_%08lx\n",
    bankIndex, dfSplitUint64(blockAddress), dfSplitUint64(length)));

  limit = blockAddress + length;
  for (i = blockAddress; i < limit; i++) {
    /* Send WRITE-BYTE command */
    val32 = 0xAAU;
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, 0x5555, &val32);
    val32 = 0x55U;
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, 0x2AAA, &val32);
    val32 = 0xA0U;
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, 0x5555, &val32);
    val32 = byte = *p8++;;
    pDevCtx->methods.pFlash(pDevCtx, bankIndex, TRUE, i, &val32);

    /* Wait for completion (50 us) */
    timeout = 10;
    while (timeout) {
      dfDelayMicroseconds(5);
      /* Use data polling method */
      pDevCtx->methods.pFlash(pDevCtx, bankIndex, FALSE, i, &val32);
      if ((val32 & 0x80U) == (byte & 0x80U)) {
        break;
      }
      timeout--;
    }
    if (0 == timeout) {
      dfDebugPrint(DEBUGLEVEL_ERROR, ("*** writeBlockSst39vf016q: Timed out waiting for WRITE-BYTE\n"));
      status = CoreFlashTimeout;
    }
    
    if (CoreFlashSuccess != status) {
      break;
    }
  }

  return status;
}

/*
** -----------------------------------------------------------------
** Exported routines
** -----------------------------------------------------------------
*/

extern void
flashLegacyCleanup(
  LegacyFlashInformation* pLegacyInfo)
{
  dfFree(pLegacyInfo);
}

CoreFlashStatus
flashLegacyEraseBlock(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t blockAddress)
{
  LegacyFlashInformation* pLegacyInfo;
  uint32_t id;

  if (bankIndex >= pDevCtx->info.bootstrap.numFlashBank) {
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** flashLegacyIdentify: invalid bank index %u\n", bankIndex));
    return CoreFlashInvalidIndex;
  }

  dfAssert(pDevCtx->info.flash[bankIndex].bLegacyFlash);

  pLegacyInfo = pDevCtx->info.flash[bankIndex].detail.pLegacy;
  id = MAKE_ID(pLegacyInfo->vendorId, pLegacyInfo->deviceId);
  switch (id) {
  case MAKE_ID(VENDOR_SST_ATMEL, DEVICE_SST_39VF016Q):
    return eraseBlockSst39vf016q(pDevCtx, bankIndex, blockAddress);

  case MAKE_ID(VENDOR_MICRON, DEVICE_MICRON_28F008B):
    return eraseBlockMicron28F008B(pDevCtx, bankIndex, blockAddress);

  default:
    /* Don't understand this device */
    dfAssert(FALSE);
    return CoreFlashGeneralFailure;
  }
}

CoreFlashStatus
flashLegacyWriteBlock(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  uint64_t blockAddress,
  uint64_t length,
  const void* pData)
{
  LegacyFlashInformation* pLegacyInfo;
  uint32_t id;

  if (bankIndex >= pDevCtx->info.bootstrap.numFlashBank) {
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** flashLegacyIdentify: invalid bank index %u\n", bankIndex));
    return CoreFlashInvalidIndex;
  }

  dfAssert(pDevCtx->info.flash[bankIndex].bLegacyFlash);

  pLegacyInfo = pDevCtx->info.flash[bankIndex].detail.pLegacy;
  id = MAKE_ID(pLegacyInfo->vendorId, pLegacyInfo->deviceId);
  switch (id) {
  case MAKE_ID(VENDOR_SST_ATMEL, DEVICE_SST_39VF016Q):
    return writeBlockSst39vf016q(pDevCtx, bankIndex, blockAddress, length, pData);

  case MAKE_ID(VENDOR_MICRON, DEVICE_MICRON_28F008B):
    return writeBlockMicron28F008B(pDevCtx, bankIndex, blockAddress, length, pData);

  default:
    /* Don't understand this device */
    dfAssert(FALSE);
    return CoreFlashGeneralFailure;
  }
}

boolean_t
flashLegacyIdentify(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex)
{
  boolean_t bFound = FALSE;
  uint64_t totalSize = 0;
  LegacyFlashInformation legacyInfo;

  dfDebugPrint(DEBUGLEVEL_NORMAL, ("flashLegacyIdentify: entered, pDevCtx=%p bankIndex=%lu\n",
    (void*)pDevCtx, (unsigned long)bankIndex));

  if (bankIndex >= pDevCtx->info.bootstrap.numFlashBank) {
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** flashLegacyIdentify: invalid bank index %u\n", bankIndex));
    return FALSE;
  }

  dfZeroMemory(&legacyInfo, sizeof(legacyInfo));
  if (!bFound) {
    bFound = identifyMicron28f008b(pDevCtx, bankIndex, &totalSize, &legacyInfo);
  }
  if (!bFound) {
    bFound = identifySst39vf016q(pDevCtx, bankIndex, &totalSize, &legacyInfo);
  }

  if (bFound) {
    pDevCtx->info.flash[bankIndex].detail.pLegacy = dfMalloc(sizeof(LegacyFlashInformation));
    if (NULL == pDevCtx->info.flash[bankIndex].detail.pLegacy) {
      dfDebugPrint(DEBUGLEVEL_ERROR, ("*** flashLegacyIdentify: failed to allocate legacy flash information struct for bank %u\n", bankIndex));
      return FALSE;
    }
    *pDevCtx->info.flash[bankIndex].detail.pLegacy = legacyInfo;
    pDevCtx->info.flash[bankIndex].bLegacyFlash = TRUE;
    pDevCtx->info.flash[bankIndex].totalSize = totalSize;
  } else {
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** flashLegacyIdentify: failed to identify Flash device for bank %u\n", bankIndex));
  }

  return bFound;
}
