/*
** File: corevpd.c  
** Project: ADB3 core driver
** Purpose: Implements functions for VPD read and write in Core Interface.
**
** (C) Copyright Alpha Data 2009-2010
*/

#include <df.h>

#include "corenotify.h"

static Adb3CoreNotificationList*
getList(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int notifyType)
{
  Adb3CoreNotificationList* pList;
  unsigned int index;

  if (notifyType >= CORE_NOTIFY_FPGA_INTERRUPT(0) && notifyType < CORE_NOTIFY_FPGA_INTERRUPT(pDevCtx->info.bootstrap.numTargetFpga)) {
    index = notifyType - CORE_NOTIFY_FPGA_INTERRUPT(0);
    pList = &pDevCtx->notification.fpgaInterrupt[index];
  } else if (notifyType >= CORE_NOTIFY_FPGA_ALERT(0) && notifyType < CORE_NOTIFY_FPGA_ALERT(pDevCtx->info.bootstrap.numTargetFpga)) {
    index = notifyType - CORE_NOTIFY_FPGA_ALERT(0);
    pList = &pDevCtx->notification.fpgaAlert[index];
  } else if (notifyType == CORE_NOTIFY_SENSOR_POLL) {
    pList = &pDevCtx->notification.sensorPoll;
  } else {
    /* Invalid notification type */
    return NULL;
  }
  return pList;
}

void
coreNotifyListeners(
  Adb3CoreDeviceContext* pDevCtx,
  Adb3CoreNotificationList* pList)
{
  CoreNotification* pNotification;
  DfListNode* pNode;
  DfListNode* pNext;
  DfSpinLockFlags f;
  DfList removed;

  dfListInit(&removed);

  /* Traverse the notification list, invoking the callback for each node */
  f = dfSpinLockGet(&pList->lock);
  pNode = dfListGetHead(&pList->header);
  while (NULL != pNode) {
    pNotification = DF_CONTAINER_OF(pNode, CoreNotification, system.node);
    /* Increment ref. count to make sure the node can't be removed until we're finished with it */
    pNotification->system.refCount++;
    dfSpinLockPut(&pList->lock, f);

    /* We don't invoke the callback while holding any locks, to avoid deadlocks */
    pNotification->system.pCallback(pNotification, pNotification->system.notifyType, pNotification->system.pCallbackContext);

    f = dfSpinLockGet(&pList->lock);
    pNext = dfListGetNext(&pList->header, pNode);
    pNotification->system.refCount--; /* Decrement ref. count of node, so it can be removed */
    if (0 == pNotification->system.refCount) {
      /* Ref. count has returned to 0, so remove this node and add it to the list of nodes we have to signal */
      dfListRemove(pNode);
      dfListAddToTail(&removed, pNode);
    }
    pNode = pNext;
  }
  dfSpinLockPut(&pList->lock, f);

  /* Signal the nodes we removed */
  pNode = dfListGetHead(&removed);
  while (NULL != pNode) {
    pNotification = DF_CONTAINER_OF(pNode, CoreNotification, system.node);
    dfEventSignal(pNotification->system.pRemoveEvent);
    pNode = dfListGetNext(&removed, pNode);
  }
}

boolean_t
coreNotificationReg(
  void* pInterfaceContext,
  unsigned int notifyType,
  CoreNotification* pNotification,
  CoreNotificationCallback* pCallback,
  void* pCallbackContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  Adb3CoreNotificationList* pList;
  DfSpinLockFlags f;

  dfAssertPassiveContext();

  pNotification->system.notifyType = notifyType;
  pNotification->system.pCallback = pCallback;
  pNotification->system.pCallbackContext = pCallbackContext;
  pNotification->system.refCount = 1;

  pList = getList(pDevCtx, notifyType);
  if (NULL == pList) {
    /* Invalid notification type */
    return FALSE;
  }

  /* Add this node to the linked list */
  f = dfSpinLockGet(&pList->lock);
  dfListAddToTail(&pList->header, &pNotification->system.node);
  dfSpinLockPut(&pList->lock, f);

  return TRUE;
}

boolean_t
coreNotificationUnreg(
  void* pInterfaceContext,
  CoreNotification* pNotification)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  Adb3CoreNotificationList* pList;
  unsigned int notifyType;
  DfSpinLockFlags f;
  DfEvent wait;
  boolean_t bMustWait = FALSE; /* Will set TRUE if necessary */

  dfAssertPassiveContext();

  notifyType = pNotification->system.notifyType;
  pList = getList(pDevCtx, notifyType);
  if (NULL == pList) {
    /* Invalid notification type */
    return FALSE;
  }

  /* Remove this node from the linked list */
  dfEventInit(&wait);
  f = dfSpinLockGet(&pList->lock);
  pNotification->system.refCount--;
  if (0 == pNotification->system.refCount) {
    dfListRemove(&pNotification->system.node);
  } else {
    pNotification->system.pRemoveEvent = &wait;
    bMustWait = TRUE;
  }
  dfSpinLockPut(&pList->lock, f);
  if (bMustWait) {
    dfEventWait(&wait);
  }
  dfEventUninit(&wait);

  return TRUE;
}
