/*
** File: avr_clock.c  
** Project: ADB3 core driver
** Purpose: Routines for programming clock generators via AVR uC.
**
** (C) Copyright Alpha Data 2012
**
** NOTES
**
** 1. Clock word layout is:
**
**   d0[31:0]   (unused)
**   d1[31:0]   (unused)
**   d2[31:0]   (unused)
**   frequency  Frequency, in Hz
*/

#include "device.h"
#include "avr_common.h"
#include "avr_clock.h"
#include "avr_clock_common.h"

static AvrClockStatus
mapAvrTransStatus(
  AvrTransStatus transStatus)
{
  switch (transStatus) {
  case AvrTransStatusSuccess:
    return AvrClockStatusSuccess;

  case AvrTransStatusTimeout:
    return AvrClockStatusTimeout;

  default:
    return AvrClockStatusGeneralFailure;
  }
}

boolean_t
avrClockInit(
  AvrClockContext* pAvrClockCtx,
  AvrDeviceContext* pAvrDevCtx,
  unsigned int numClock,
  uint32_t* pFInit)
{
  AvrClockStatus status;
  boolean_t bOk = TRUE; /* Assume successful */
  uint32_t f;
  unsigned int i;

  if (numClock > AVR_CLOCK_MAX_CLOCKS) {
    dfDebugPrint(0, ("*** avrClockInit: Too many clocks, numClock=%u\n", numClock));
    return FALSE;
  }
  pAvrClockCtx->pAvrDevCtx = pAvrDevCtx;
  pAvrClockCtx->numClock = numClock;
  for (i = 0; i < numClock; i++) {
    pAvrClockCtx->clocks[i].index = i;
    status = avrClockGetFrequencySync(pAvrClockCtx, i, &f);
    if (AvrClockStatusSuccess != status) {
      dfDebugPrint(0, ("*** avrClockInit: Failed to get initial frequency for clock index %u, status=%u\n", i, status));
      pFInit[i] = 0;
      bOk = FALSE;
    } else {
      dfDebugPrint(2, ("avrClockInit: Initial frequency of clock index %u is %lu Hz\n", i, (unsigned long)f));
      pFInit[i] = f;
    }
    pAvrClockCtx->clocks[i].save.bLastValid = FALSE;
  }

  return bOk;
}

void
avrClockUninit(
  AvrClockContext* pAvrClockCtx)
{
  /* Nothing to do yet */
}

static void
onAvrGetFrequencyDone(
  AvrTransRequest* pTransactionIgnored,
  AvrTransStatus avrStatus,
  void* pContext)
{
  AvrClockRequest* pRequest = (AvrClockRequest*)pContext;
  unsigned int index = pRequest->index;
  AvrClockContext* pAvrClockCtx = pRequest->pAvrClockCtx;
  AvrClockGenContext* pClockGen = &pAvrClockCtx->clocks[index];
  AvrClockStatus avrClkStatus = AvrClockStatusSuccess;
  uint32_t frequency = 0;
  uint8_t reply;

  if (AvrTransStatusSuccess == avrStatus) {
    reply = pClockGen->transaction.getFrequency.reply[3];
    if (0 == reply) {
      /* GF command to AVR was successful */
      frequency = ((uint32_t)pClockGen->transaction.getFrequency.reply[4] << 0)
                | ((uint32_t)pClockGen->transaction.getFrequency.reply[5] << 8)
                | ((uint32_t)pClockGen->transaction.getFrequency.reply[6] << 16)
                | ((uint32_t)pClockGen->transaction.getFrequency.reply[7] << 24);
    } else {
      /* GF command to AVR indicated error */
      switch (reply) {
      case 0x1U:
        avrClkStatus = AvrClockStatusInvalidIndex;
        break;

      default:
        avrClkStatus = AvrClockStatusGeneralFailure;
        break;
      }
    }
  } else {
    /* AVR transaction failed */
    avrClkStatus = mapAvrTransStatus(avrStatus);
  }
  pRequest->callback.pGetFrequency(pAvrClockCtx, pRequest, avrClkStatus, index, frequency, pRequest->pContext);
}

AvrClockStatus
avrClockGetFrequency(
  AvrClockContext* pAvrClockCtx,
  AvrClockRequest* pRequest,
  unsigned int index,
  AvrClockGetFrequencyCallback* pCallback,
  void* pContext)
{
  AvrClockGenContext* pClockGen;

  if (index >= pAvrClockCtx->numClock) {
    return AvrClockStatusInvalidIndex;
  }

  pClockGen = &pAvrClockCtx->clocks[index];
  pRequest->pAvrClockCtx = pAvrClockCtx;
  pRequest->callback.pGetFrequency = pCallback;
  pRequest->pContext = pContext;
  pRequest->index = index;
  pClockGen->transaction.getFrequency.command[0] = 'G';
  pClockGen->transaction.getFrequency.command[1] = 'F';
  pClockGen->transaction.getFrequency.command[2] = '?';
  pClockGen->transaction.getFrequency.command[3] = (uint8_t)index;
  avrTransaction(
    pAvrClockCtx->pAvrDevCtx,
    &pClockGen->transaction.getFrequency.request,
    pClockGen->transaction.getFrequency.command,
    DF_ARRAY_LENGTH(pClockGen->transaction.getFrequency.command),
    pClockGen->transaction.getFrequency.reply,
    DF_ARRAY_LENGTH(pClockGen->transaction.getFrequency.reply),
    1000000U,
    onAvrGetFrequencyDone,
    pRequest);

  return AvrClockStatusSuccess;
}

static void
onAvrSetFrequencyDone(
  AvrTransRequest* pTransactionIgnored,
  AvrTransStatus avrStatus,
  void* pContext)
{
  AvrClockRequest* pRequest = (AvrClockRequest*)pContext;
  unsigned int index = pRequest->index;
  AvrClockContext* pAvrClockCtx = pRequest->pAvrClockCtx;
  AvrClockGenContext* pClockGen = &pAvrClockCtx->clocks[index];
  AvrClockStatus avrClkStatus = AvrClockStatusSuccess;
  uint8_t reply;

  if (AvrTransStatusSuccess == avrStatus) {
    reply = pClockGen->transaction.setFrequency.reply[3];
    if (0 == reply) {
      /* SF command to AVR was successful */
    } else {
      /* SF command to AVR indicated error */
      switch (reply) {
      case 0x1U:
        /* Invalid clock index */
        avrClkStatus = AvrClockStatusInvalidIndex;
        break;

      case 0x11U:
        /* Invalid requested frequency */
        avrClkStatus = AvrClockStatusOutOfRange;
        break;

      default:
        avrClkStatus = AvrClockStatusGeneralFailure;
        break;
      }
    }
  } else {
    /* AVR transaction failed */
    avrClkStatus = mapAvrTransStatus(avrStatus);
  }
  pRequest->callback.pSetFrequency(pAvrClockCtx, pRequest, avrClkStatus, index, pRequest->pContext);
}

AvrClockStatus
avrClockSetFrequency(
  AvrClockContext* pAvrClockCtx,
  AvrClockRequest* pRequest,
  unsigned int index,
  const CoreClockWord* pClockWord,
  AvrClockSetFrequencyCallback* pCallback,
  void* pContext)
{
  AvrClockGenContext* pClockGen;

  if (index >= pAvrClockCtx->numClock) {
    return AvrClockStatusInvalidIndex;
  }
  if (pClockWord->frequency >= 0xFFFFFFFFU) {
    return AvrClockStatusOutOfRange;
  }

  pClockGen = &pAvrClockCtx->clocks[index];
  pRequest->pAvrClockCtx = pAvrClockCtx;
  pRequest->callback.pSetFrequency = pCallback;
  pRequest->pContext = pContext;
  pRequest->index = index;
  pClockGen->transaction.setFrequency.command[0] = 'S';
  pClockGen->transaction.setFrequency.command[1] = 'F';
  pClockGen->transaction.setFrequency.command[2] = '?';
  pClockGen->transaction.setFrequency.command[3] = (uint8_t)index;
  pClockGen->transaction.setFrequency.command[4] = (uint8_t)(pClockWord->frequency >> 0);
  pClockGen->transaction.setFrequency.command[5] = (uint8_t)(pClockWord->frequency >> 8);
  pClockGen->transaction.setFrequency.command[6] = (uint8_t)(pClockWord->frequency >> 16);
  pClockGen->transaction.setFrequency.command[7] = (uint8_t)(pClockWord->frequency >> 24);
  avrTransaction(
    pAvrClockCtx->pAvrDevCtx,
    &pClockGen->transaction.setFrequency.request,
    pClockGen->transaction.setFrequency.command,
    DF_ARRAY_LENGTH(pClockGen->transaction.setFrequency.command),
    pClockGen->transaction.setFrequency.reply,
    DF_ARRAY_LENGTH(pClockGen->transaction.setFrequency.reply),
    1000000U,
    onAvrSetFrequencyDone,
    pRequest);
    
  return AvrClockStatusSuccess;
}

static void
getFrequencySyncCallback(
  AvrClockContext* pAvrClockCtx,
  AvrClockRequest* pRequest,
  AvrClockStatus status,
  unsigned int index,
  uint32_t frequency,
  void* pContextIgnored)
{
  AvrClockRequestSync* pRequestSync = DF_CONTAINER_OF(pRequest, AvrClockRequestSync, request);

  pRequestSync->status = status;
  pRequestSync->frequency = frequency;
  dfEventSignal(&pRequestSync->ev);
}

AvrClockStatus
avrClockGetFrequencySync(
  AvrClockContext* pAvrClockCtx,
  unsigned int index,
  uint32_t* pFrequency)
{
  AvrClockStatus status;
  AvrClockRequestSync req;

  dfEventInit(&req.ev);
  status = avrClockGetFrequency(pAvrClockCtx, &req.request, index, getFrequencySyncCallback, NULL);
  if (AvrClockStatusSuccess == status) {
    dfEventWait(&req.ev);
    status = req.status;
  }
  *pFrequency = req.frequency;
  dfEventUninit(&req.ev);
  return status;
}

static void
setFrequencySyncCallback(
  AvrClockContext* pAvrClockCtx,
  AvrClockRequest* pRequest,
  AvrClockStatus status,
  unsigned int index,
  void* pContextIgnored)
{
  AvrClockRequestSync* pRequestSync = DF_CONTAINER_OF(pRequest, AvrClockRequestSync, request);

  pRequestSync->status = status;
  dfEventSignal(&pRequestSync->ev);
}

AvrClockStatus
avrClockSetFrequencySync(
  AvrClockContext* pAvrClockCtx,
  unsigned int index,
  const CoreClockWord* pClockWord)
{
  AvrClockStatus status;
  AvrClockRequestSync req;

  dfEventInit(&req.ev);
  status = avrClockSetFrequency(pAvrClockCtx, &req.request, index, pClockWord, setFrequencySyncCallback, NULL);
  if (AvrClockStatusSuccess == status) {
    dfEventWait(&req.ev);
    status = req.status;
  }
  dfEventUninit(&req.ev);
  return status;
}

void
avrClockSaveState(
  AvrClockContext* pAvrClockCtx)
{
  /* Nothing to do yet */
}

AvrClockStatus
avrClockRestoreState(
  AvrClockContext* pAvrClockCtx)
{
  AvrClockStatus status = AvrClockStatusSuccess;
  unsigned int i;
  CoreClockWord clockWord;

  for (i = 0; i < pAvrClockCtx->numClock; i++) {
    /* If the clock synthesizer had been programmed before power-down, reprogram it with the same thing */
    if (pAvrClockCtx->clocks[i].save.bLastValid) {
      clockWord.frequency = pAvrClockCtx->clocks[i].save.lastFrequency;
      status = avrClockSetFrequencySync(pAvrClockCtx, i, &clockWord);
      if (AvrClockStatusSuccess != status) {
        dfDebugPrint(0, ("*** avrClockInit: Failed to restore frequency (%lu Hz) for clock index %u, status=%u\n",
          (unsigned long)clockWord.frequency, i, status));
        break;
      }
    }
  }

  return status;
}

ClockProgramStatus
avrClockMapStatus(
  AvrClockStatus status)
{
  switch (status) {
  case AvrClockStatusSuccess:
    return ClockProgramSuccess;

  case AvrClockStatusTimeout:
  case AvrClockStatusServiceMode:
    return ClockProgramHardwareError;

  case AvrClockStatusOutOfRange:
    return ClockProgramOutOfRange;

  case AvrClockStatusInvalidIndex:
    return ClockProgramInvalidIndex;

  case AvrClockStatusGeneralFailure:
  default:
    return ClockProgramGeneralFailure;
  }
}
