/*
** File: admxrc2_mmap.c  
** Project: ADMXRC2 API module library
** Purpose: Implements Linux-specific functions for mapping memory in
**          ADMXRC2 API module library.
**
** (C) Copyright Alpha Data 2013
*/

#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <poll.h>
#include <unistd.h>
#include <pthread.h>

#include "admxrc2_platform.h"
#include <ioctl_dflinux.h>

#define TRACE 0

#define ARRAY_LENGTH(x) (sizeof(x) / sizeof((x)[0]))

struct _MappedRegion;
typedef struct _MappedRegion MappedRegion;

struct _MappedRegion {
  MappedRegion* pPrev;
  MappedRegion* pNext;
  struct {
    ADMXRC2_HANDLE hDevice;
    unsigned int regionTag;
    uint64_t offset;
    uint64_t length;
  } parameters;
  struct {
    void* pAlignedVirtual;
    void* pBaseVirtual;
    size_t alignedLength;
  } mapped;
};

static struct _MappedList {
  MappedRegion* pHead;
  MappedRegion* pTail;
} g_mappedList;
static pthread_mutex_t g_mappedListLock = PTHREAD_MUTEX_INITIALIZER;
static int g_bInitialized = FALSE;

static void
initialize(
  void)
{
  pthread_mutex_lock(&g_mappedListLock);
  if (!g_bInitialized) {
#if TRACE
    fprintf(stderr, "initialize: initializing globals\n");
#endif
    g_mappedList.pHead = NULL;
    g_mappedList.pTail = NULL;
    g_bInitialized = TRUE;
  }
  pthread_mutex_unlock(&g_mappedListLock);
}

static void
addRegion(
  MappedRegion* pMappedRegion)
{
  pthread_mutex_lock(&g_mappedListLock);
  pMappedRegion->pPrev = g_mappedList.pTail;
  pMappedRegion->pNext = NULL;
  if (NULL == g_mappedList.pHead) {
    g_mappedList.pHead = pMappedRegion;
    g_mappedList.pTail = pMappedRegion;
  } else {
    g_mappedList.pTail->pNext = pMappedRegion;
    g_mappedList.pTail = pMappedRegion;
  }
  pthread_mutex_unlock(&g_mappedListLock);
}

/* Must be called while holding the semaphore */
static void
removeRegion(
  MappedRegion* pMappedRegion)
{
  if (NULL == pMappedRegion->pPrev) {
    g_mappedList.pHead = pMappedRegion->pNext;
  } else {
    pMappedRegion->pPrev->pNext = pMappedRegion->pNext;
  }
  if (NULL == pMappedRegion->pNext) {
    g_mappedList.pTail = pMappedRegion->pPrev;
  } else {
    pMappedRegion->pNext->pPrev = pMappedRegion->pPrev;
  }
  pMappedRegion->pPrev = NULL;
  pMappedRegion->pNext = NULL;
}

static MappedRegion*
findAndRemoveRegion(
  ADMXRC2_HANDLE hDevice,
  const void* pBaseVirtual,
  bool_t* pbFoundHandle)
{
  MappedRegion* p;
  bool_t bFoundHandle = FALSE;

  pthread_mutex_lock(&g_mappedListLock);
  p = g_mappedList.pHead;
  while (NULL != p) {
    if (p->parameters.hDevice == hDevice) {
      bFoundHandle = TRUE;
      if (p->mapped.pBaseVirtual == pBaseVirtual) {
        break;
      }
    }
    p = p->pNext;
  }
  if (NULL != p) {
#if TRACE
    fprintf(stderr, "findAndRemoveRegion: removing region, hDevice=%d pBaseVirtual=%p pRegion=%p\n", hDevice, pBaseVirtual, p);
#endif
    removeRegion(p);
  } else {
#if TRACE
    fprintf(stderr, "findAndRemoveRegion: failed to find region, hDevice=%d pBaseVirtual=%p pRegion=%p\n", hDevice, pBaseVirtual, p);
#endif
  }
  pthread_mutex_unlock(&g_mappedListLock);

  *pbFoundHandle = bFoundHandle;
  return p;
}

/* Must be called while holding mutex */
static MappedRegion*
findRegion(
  ADMXRC2_HANDLE hDevice,
  unsigned int regionTag,
  unsigned int ordinal,
  bool_t* pbFoundHandle)
{
  MappedRegion* p;
  bool_t bFoundHandle = FALSE;

  p = g_mappedList.pHead;
  while (NULL != p) {
    if (p->parameters.hDevice == hDevice) {
      bFoundHandle = TRUE;
      if (p->parameters.regionTag == regionTag) {
        if (ordinal == 0) {
          break;
        }
        ordinal--;
      }
    }
    p = p->pNext;
  }

  *pbFoundHandle = bFoundHandle;
  return p;
}

ADMXRC2_STATUS
admxrc2MapRegion(
  ADMXRC2_HANDLE hAdmxrc2,
  unsigned int regionTag,
  uint64_t offset,
  uint64_t length,
  void** ppVirtualBase)
{
  ADMXRC2_STATUS status = ADMXRC2_SUCCESS;
  DfLinuxIoctlGetMmapAddress io;
  uint8_t* p;
  int err, ioCode;
  long pageSize;
  uint64_t address;
  off_t pageMask, pageStart;
  size_t pageLength, pageOffset;
  MappedRegion* pMappedRegion = NULL;

  initialize();

  assert(sizeof(off_t) == sizeof(unsigned long));

  /* Sanity checks */
  if (0 == length || offset + length < offset) {
    return ADMXRC2_INVALID_REGION;
  }

  pMappedRegion = (MappedRegion*)malloc(sizeof(MappedRegion));
  if (NULL == pMappedRegion) {
    return ADMXRC2_NO_MEMORY;
  }

  ioCode = _IOC(_IOC_READ | _IOC_WRITE, 150, DFLINUX_IOCTLCODE_GETMMAPADDRESS, 0);
  io.in.regionTag = regionTag;
  err = ioctl(hAdmxrc2, ioCode, &io);
  if (0 != err) {
    if (err < 0) {
      switch (errno) {
      case EBADF:
        status = ADMXRC2_INVALID_HANDLE;
        break;

      default:
        fprintf(stderr, "admxrc2MapRegion: ioctl() failed, errno=%d (%s)\n", errno, strerror(errno));
        status = ADMXRC2_UNKNOWN_ERROR;
        break;
      }
    } else {
      switch ((DfLinuxMmapStatus)err) {
      case DfLinuxMmapInvalidRegion:
        status = ADMXRC2_INVALID_REGION;
        break;

      case DfLinuxMmapInvalidTag:
        status = ADMXRC2_INVALID_INDEX;
        break;

      case DfLinuxMmapNoMemory:
        status = ADMXRC2_NO_MEMORY;
        break;

      case DfLinuxMmapRegionTooLarge:
        status = ADMXRC2_REGION_TOO_LARGE;
        break;

      case DfLinuxMmapAccessDenied:
        status = ADMXRC2_ACCESS_DENIED;
        break;

      case DfLinuxMmapGeneralFailure:
      default:
        status = ADMXRC2_UNKNOWN_ERROR;
        break;
      }
    }
  }

  if (ADMXRC2_SUCCESS != status) {
    goto out;
  }

  pageSize = sysconf(_SC_PAGESIZE);
  if (pageSize == (unsigned long)-1) {
    fprintf(stderr, "admxrc2MapRegion: get page size from sysconf() failed, errno=%d (%s)\n", errno, strerror(errno));
    status = ADMXRC2_UNKNOWN_ERROR;
    goto out;
  }
  pageMask = (off_t)(pageSize - 1U);
  address = io.out.address + offset;
  if (length + offset > io.out.length) {
    status = ADMXRC2_INVALID_REGION;
    goto out;
  }
  pageStart = (off_t)address & ~pageMask;
  pageOffset = (size_t)address & (size_t)pageMask;
  pageLength = (size_t)((size_t)length + pageOffset + (size_t)pageMask) & ~(size_t)pageMask;
#if TRACE
  fprintf(stderr, "admxrc2MapRegion: calling mmap, hDevice=%d length=%p address=0x%llx\n",
    hAdmxrc2, (void*)pageLength, (unsigned long long)pageStart);
#endif
  p = (uint8_t*)mmap(NULL, pageLength, PROT_READ | PROT_WRITE, MAP_SHARED, hAdmxrc2, pageStart);
  if (MAP_FAILED == p) {
    switch (errno) {
    default:
      fprintf(stderr, "admxrc2MapRegion: mmap() failed, errno=%d (%s)\n", errno, strerror(errno));
      status = ADMXRC2_UNKNOWN_ERROR;
      goto out;
    }
  }
  pMappedRegion->parameters.hDevice = hAdmxrc2;
  pMappedRegion->parameters.regionTag = regionTag;
  pMappedRegion->parameters.offset = offset;
  pMappedRegion->parameters.length = length;
  pMappedRegion->mapped.alignedLength = pageLength;
  pMappedRegion->mapped.pAlignedVirtual  = p;
  p += pageOffset;
  pMappedRegion->mapped.pBaseVirtual = p;
#if TRACE
  fprintf(stderr, "admxrc2MapRegion: adding region, hDevice=%d pRegion%p\n", hAdmxrc2, pMappedRegion);
#endif
  addRegion(pMappedRegion);

out:
  if (ADMXRC2_SUCCESS == status) {
    *ppVirtualBase = (void*)p;
  } else {
    if (NULL != pMappedRegion) {
      free(pMappedRegion);
    }
  }
  return status;
}

ADMXRC2_STATUS
admxrc2UnmapRegion(
  ADMXRC2_HANDLE hAdmxrc2,
  void* pVirtualBase)
{
  ADMXRC2_STATUS status = ADMXRC2_SUCCESS;
  MappedRegion* p;
  bool_t bFoundHandle = FALSE;

  initialize();

  p = findAndRemoveRegion(hAdmxrc2, pVirtualBase, &bFoundHandle);
  if (NULL == p) {
    return bFoundHandle ? ADMXRC2_INVALID_REGION : ADMXRC2_INVALID_HANDLE;
  }

#if TRACE
  fprintf(stderr, "admxrc2UnmapRegion: calling munmap, aligned=%p length=%p\n",
    p->pAlignedVirtual, (void*)p->alignedLength);
#endif
  if (0 != munmap(p->mapped.pAlignedVirtual, p->mapped.alignedLength)) {
    switch (errno) {
    case EINVAL:
      status = ADMXRC2_INVALID_REGION;
      break;

    default:
      fprintf(stderr, "admxrc2UnmapRegion: munmap() failed, base=%p length=%p errno=%d (%s)\n",
        p->mapped.pAlignedVirtual, (void*)p->mapped.alignedLength, errno, strerror(errno));
      status = ADMXRC2_UNKNOWN_ERROR;
      break;
    }
  }
  free(p);
  
  return status;
}

void
admxrc2UnmapAllRegions(
  ADMXRC2_HANDLE hAdmxrc2)
{
  MappedRegion* p, * pNext;

  initialize();

  pthread_mutex_lock(&g_mappedListLock);
  p = g_mappedList.pHead;
  while (NULL != p) {
    pNext = p->pNext;
    if (p->parameters.hDevice == hAdmxrc2) {
#if TRACE
      fprintf(stderr, "admxrc2UnmapAllRegions: removing region, hDevice=%d pRegion=%p\n", hAdmxrc2, p);
#endif
      removeRegion(p);
#if TRACE
      fprintf(stderr, "admxrc2UnmapAllRegions: calling munmap, aligned=%p length=%p\n",
        p->pAlignedVirtual, (void*)p->alignedLength);
#endif
      if (0 != munmap(p->mapped.pAlignedVirtual, p->mapped.alignedLength)) {
        fprintf(stderr, "admxrc2UnmapAllRegions: munmap() failed, base=%p length=%p errno=%d (%s)\n",
          p->mapped.pAlignedVirtual, (void*)p->mapped.alignedLength, errno, strerror(errno));
      }
      free(p);
    }
    p = pNext;
  }
  pthread_mutex_unlock(&g_mappedListLock);
}

ADMXRC2_STATUS
admxrc2FindRegion(
  ADMXRC2_HANDLE hAdmxrc2,
  unsigned int regionTag,
  unsigned int ordinal,
  void** ppVirtualBase,
  uint64_t* pOffset,
  uint64_t* pLength)
{
  MappedRegion* pMappedRegion;
  bool_t bFoundHandle = FALSE;

  initialize();

  pthread_mutex_lock(&g_mappedListLock);
  pMappedRegion = findRegion(hAdmxrc2, regionTag, ordinal, &bFoundHandle);
  if (NULL == pMappedRegion) {
    pthread_mutex_unlock(&g_mappedListLock);
    return bFoundHandle ? ADMXRC2_INVALID_REGION : ADMXRC2_INVALID_HANDLE;
  }
  *ppVirtualBase = pMappedRegion->mapped.pBaseVirtual;
  *pOffset = pMappedRegion->parameters.offset;
  *pLength = pMappedRegion->parameters.length;
  pthread_mutex_unlock(&g_mappedListLock);
  
  return ADMXRC2_SUCCESS;
}
