/*******************************************************************************
 *	PupeMemTest.c	Simple Pupe Memtest program
 *			T.Barnaby,	BEAM Ltd,	2007-05-18
 *******************************************************************************
 */
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <getopt.h>
#include <sys/time.h>
#include <admxrc2.h>

#ifdef linux
#include <unistd.h>
#endif

// FPGA Control Registers
const int	IMEM_REG	= 0x0000 >> 2;	// Block RAM Index
const int	LOCKED		= 0x0008 >> 2;	// DCM Locked Status
const int	ADDR_REG	= 0x0010 >> 2;	// PAGE Register for SDRAM access
const int	MEM_REG		= 0x0018 >> 2;	// SDRAM Bank Select Register and BlockRam enable
const int	IER		= 0x0020 >> 2;	// Interrupt Enable Register
const int	ISR		= 0x0028 >> 2;	// Interrupt Status Register
const int	MEM_RAS		= 0x0030 >> 2;	// SDRAM Read Address Scaling
const int	MEM_RAO		= 0x0038 >> 2;	// SDRAM Read Address Offset
const int	MEM_GNT0	= 0x0400 >> 2;	// Memory Grant Registers
const int	MEM_GNT1	= 0x0408 >> 2;	// Memory Grant Registers
const int	MEM_GNT2	= 0x0410 >> 2;	// Memory Grant Registers
const int	MEM_GNT3	= 0x0408 >> 2;	// Memory Grant Registers

const int	FIRMWARE	= 0x0800 >> 2;	// Firmware ID code "CN" + version numbers
const int	ADC		= 0x0808 >> 2;	// ADC Control Register
const int	TIMING_IO	= 0x0810 >> 2;	// Timing I/O Register
const int	TESTCTRL	= 0x0818 >> 2;	// Test Data Control Register
const int	TESTLEN		= 0x0820 >> 2;	// Test Data Pattern Length

// Per Pick-Up registers
const int	PU0_REG_BASE	= 0x0880 >> 2;	// PickUp 0 Register Base
const int	PU1_REG_BASE	= 0x0900 >> 2;	// PickUp 1 Register Base
const int	PU2_REG_BASE	= 0x0980 >> 2;	// PickUp 2 Register Base

const int	CONTROL		= 0x0000 >> 2;	// PU General Control and Status register
const int	CYCLE		= 0x0008 >> 2;	// Cycle number
const int	TIME		= 0x0010 >> 2;	// Time in ms from start of cycle
const int	TIME_TBLADDR	= 0x0018 >> 2;	// Last write address in timing table
const int	PLL_FREQUENCY	= 0x0020 >> 2;	// PLL Reference orbit frequency
const int	PLL_FREQDELAY	= 0x0028 >> 2;	// PLL frequency load delay
const int	PLL_PHASEDELAY	= 0x0030 >> 2;	// PLL phase delay
const int	PLL_GAIN	= 0x0038 >> 2;	// PLL gain
const int	DDS_FREQ_MIN	= 0x0040 >> 2;	// PLL DDS minimum frequency
const int	DDS_FREQ_MAX	= 0x0048 >> 2;	// PLL DDS maximum frequency
const int	DIAG_CTRL	= 0x0050 >> 2;	// Diagnostics Control/Status
const int	DIAG_TRIGGER	= 0x0058 >> 2;	// Diagnostics Trigger
const int	TEST		= 0x0060 >> 2;	// Timing Test

// Memory access window
const int	FPGA_MEM	= 0x200000 >> 2;	// The FPGA memory

// SDRAM Data Banks
const int	SDRAM_DATA_PU0		= 0x08;	// PickUp 0 data
const int	SDRAM_DATA_PU1		= 0x09;	// PickUp 0 data
const int	SDRAM_DATA_PU2		= 0x0A;	// PickUp 0 data
const int	SDRAM_DATA_TEST		= 0x0B;	// Test input data
const int	SDRAM_BLOCKRAM		= 0x00;	// Test input data

const int	PU0_BRAM_BASE		= 0;	// PickUp 0 Block Ram ID base
const int	PU1_BRAM_BASE		= 8;	// PickUp 0 Block Ram ID base
const int	PU2_BRAM_BASE		= 16;	// PickUp 0 Block Ram ID base

const int	BRAM_CYCLE_TIMING	= 0;	// Cycle Timing Table
const int	BRAM_CYCLE_INFO		= 1;	// Cycle Information Table
const int	BRAM_PHASE_TABLE	= 2;	// Timing Phase Table
const int	BRAM_SWITCH_TABLE	= 3;	// Timing Switch Table
const int	BRAM_DIAG_TABLE		= 4;	// Diagnostics table
const int	BRAM_BUNCH_MEAN0	= 5;	// Bunch Mean Table #0
const int	BRAM_BUNCH_MEAN1	= 6;	// Bunch Mean Table #1

// Control register bits
const int	CONTROL_INIT				= 0x01;	// Reset
const int	CONTROL_LOOP_CONTROL			= 0x02;	// 
const int	CONTROL_DDS_FREQUENCY_LIMIT_ENABLE	= 0x04;	// 

// Interrupt bits
const int	INT_PU0_CYCLE_START	(1<<0);		// 0 PU #0 CYCLE_START
const int	INT_PU0_CYCLE_STOP	(1<<1);		// 1 PU #0 CYCLE_STOP
const int	INT_PU0_ERROR		(1<<2);		// 2 PU #0 ERROR
const int	INT_PU0_DIAG		(1<<3);		// 3 PU #0 DIAGNOSTIC INFO CAPTURED
const int	INT_PU0_WRITE_FIFO	(1<<4);		// 4 PU #0 SDRAM Write FIFO Half Full
const int	INT_PU0_USER0		(1<<5);		// 5 PU #0 User Interrupts (Switch Table bits 5)
const int	INT_PU0_USER1		(1<<6);		// 6 PU #0 User Interrupts (Switch Table bits 6)
const int	INT_PU0_USER2		(1<<7);		// 7 PU #0 User Interrupts (Switch Table bits 7)

const int	INT_PU1_CYCLE_START	(1<<8);		// 8 PU #1 CYCLE_START
const int	INT_PU1_CYCLE_STOP	(1<<9);		// 9 PU #1 CYCLE_STOP
const int	INT_PU1_ERROR		(1<<10);	// 10 PU #1 ERROR
const int	INT_PU1_DIAG		(1<<11);	// 11 PU #1 DIAGNOSTIC INFO CAPTURED
const int	INT_PU1_WRITE_FIFO	(1<<12);	// 12 PU #1 SDRAM Write FIFO Half Full
const int	INT_PU1_USER0		(1<<13);	// 13 PU #1 User Interrupts (Switch Table bits 5)
const int	INT_PU1_USER1		(1<<14);	// 14 PU #1 User Interrupts (Switch Table bits 6)
const int	INT_PU1_USER2		(1<<15);	// 15 PU #1 User Interrupts (Switch Table bits 7)

const int	INT_PU2_CYCLE_START	(1<<16);	// 16 PU #2 CYCLE_START
const int	INT_PU2_CYCLE_STOP	(1<<17);	// 17 PU #2 CYCLE_STOP
const int	INT_PU2_ERROR		(1<<18);	// 18 PU #2 ERROR
const int	INT_PU2_DIAG		(1<<19);	// 19 PU #2 DIAGNOSTIC INFO CAPTURED
const int	INT_PU2_WRITE_FIFO	(1<<20);	// 20 PU #2 SDRAM Write FIFO Half Full
const int	INT_PU2_USER0		(1<<21);	// 21 PU #2 User Interrupts (Switch Table bits 5)
const int	INT_PU2_USER1		(1<<22);	// 22 PU #2 User Interrupts (Switch Table bits 6)
const int	INT_PU2_USER2		(1<<23);	// 23 PU #2 User Interrupts (Switch Table bits 7)


enum TimingSig		{
			TimingSigClock=0x01, TimingSigCycleStart=0x02, TimingSigCycleStop=0x04,
			TimingSigCalStart=0x08, TimingSigCalStop=0x10, TimingSigInjection=0x20,
			TimingSigHChange=0x40, TimingSigFRef=0x80
			};

#define	DEBUG	0

#if DEBUG
#define	dprintf(fmt, a...)       printf(fmt, ##a);
#else
#define	dprintf(fmt, a...)
#endif

char		driver[] = "/dev/admxrc0";
char		fpgafile[] = "tms-fpga.bit";

ADMXRC2_HANDLE		card;
ADMXRC2_STATUS		status;
ADMXRC2_DMADESC		dmaDesc;
ADMXRC2_CARD_INFO	ocardInfo;
ADMXRC2_SPACE_INFO	spInfo;
volatile uint*		fpga;

int			recvbufSize = 1024*1204*2;
char*			recvbuf;
int			continueOnError;

static int memcpy_64_mmx(void* to, const void* from, size_t len){
	size_t		i;

	// Check alignment and size
	if(((unsigned int)to & 0x07) || ((unsigned int)from & 0x07) || (len & 0x07)){
		fprintf(stderr, "memcpy_mmx: Memory pointers and length need to be aligned to 64bit boundary\n");
		return 1;
	}
	
	i = len / 8;
	while(i--){
		__asm__ __volatile__ (
			"movq (%0), %%mm0\n"
			"movq %%mm0, (%1)\n"
			:: "r" (from), "r" (to) : "memory");
		from = ((const unsigned char *)from) + 8;
		to = ((unsigned char *)to) + 8;
	}
	__asm__ __volatile__ ("emms":::"memory");
	return 0;
}

// Get current time in seconds
double getTime()
{
	struct timeval	tp;
	
	gettimeofday(&tp, NULL);
	return ((double) tp.tv_sec + (double) tp.tv_usec * 1e-6);
}

void hd8(void* data, int n){
	unsigned char* d = (unsigned char*)data;
	int	i;
	
	for(i = 0; i < n; i++){
		printf("%2.2x ", *d++);
		if((i & 0xF) == 0xF)
			printf("\n");
	}
	printf("\n");
}

void hd32(unsigned int a, void* data, int n){
	unsigned int* d = (unsigned int*)data;
	int	i;
	
	printf("%8.8x: ", a);
	for(i = 0; i < n; i++){
		printf("%8.8x ", *d++);
		if((i & 0x7) == 0x7)
			printf("\n%8.8x: ", a + i + 1);
	}
	printf("\n");
}

int readBlock(int numBytes){
	int		mode;
	ADMXRC2_STATUS	status;
	uint32_t	fpgaAddress;
	
	/* Build the DMA mode word. We want bursting, demand-mode, use ready and use bterm */
	mode = ADMXRC2_BuildDMAModeWord(ocardInfo.BoardType, ADMXRC2_IOWIDTH_64, 0,
		ADMXRC2_DMAMODE_USEREADY | ADMXRC2_DMAMODE_USEBTERM | ADMXRC2_DMAMODE_BURSTENABLE);

	// Setup memory area for access
	fpga[ADDR_REG] = 0;
	fpgaAddress = FPGA_MEM << 2;

	dprintf("Perform DMA\n");	
	status = ADMXRC2_DoDMA(card, dmaDesc, 0, numBytes, fpgaAddress, ADMXRC2_LOCALTOPCI, 0, mode, 0, 0, 0);
	if(status != ADMXRC2_SUCCESS){
		printf("DMA: fail: status: %x\n", status);
	}
	dprintf("DMA complete\n");

	return 0;
}

int writeBlock(int numBytes){
	int		mode;
	ADMXRC2_STATUS	status;
	uint32_t	fpgaAddress;
	
	/* Build the DMA mode word. We want bursting, demand-mode, use ready and use bterm */
	mode = ADMXRC2_BuildDMAModeWord(ocardInfo.BoardType, ADMXRC2_IOWIDTH_64, 0,
		ADMXRC2_DMAMODE_USEREADY | ADMXRC2_DMAMODE_USEBTERM | ADMXRC2_DMAMODE_BURSTENABLE);

	// Setup memory area for access
	fpga[ADDR_REG] = 0;
	
	fpgaAddress = FPGA_MEM << 2;

	dprintf("Perform DMA: To %x\n", fpgaAddress);	
	status = ADMXRC2_DoDMA(card, dmaDesc, 0, numBytes, fpgaAddress, ADMXRC2_PCITOLOCAL, 0, mode, 0, 0, 0);
	if(status != ADMXRC2_SUCCESS){
		printf("DMA: fail: status: %x\n", status);
	}
	dprintf("DMA complete\n");

	return 0;
}

void writeTestPattern(int numBytes, int base = 0){
	uint32_t*	p = (uint32_t*)recvbuf;
	uint32_t	n;
	
	for(n = 0; n < numBytes / sizeof(uint32_t); n++){
		p[n] = base + n;
	}
	
	writeBlock(numBytes);
//	memcpy_64_mmx((void*)&fpga[FPGA_MEM], recvbuf, BufferSize);
}

int readTestPattern(int numBytes, int base = 0){
	uint32_t*	p = (uint32_t*)recvbuf;
	uint32_t	n;
	
	readBlock(numBytes);
	
	for(n = 0; n < numBytes / sizeof(uint32_t); n++){
		if(p[n] != (base + n)){
			printf("Error at Address: %x Data was: %x Data should be: %x on test: %d\n", n, p[n], (base + n), base);
			if(n < 32)
				hd32(n, p, 64);
			else
				hd32(n - 32, &p[n - 32], 64);
			return 1;
		}
	}
	return 0;
}

void memDisplay(int numBytes){
	printf("Memory Display\n");
	readBlock(numBytes);
	hd32(0, recvbuf, 32);
}

void memTestSpeed(int numBytes){
	int		num = 100;
	int		n;
	double		ts, te, r;

	printf("Memory Test Read data rate: ChunkSize: %d\n", numBytes);
	ts = getTime();
	for(n = 0; n < num; n++){
		readBlock(numBytes);
	}
	te = getTime();
	
	r = (double(num) * numBytes) / (te - ts);
	printf("DataRate: %f MBytes/s\n", r / (1024*1024));
}

void memTestRead(int numBytes){
	int	i = 0;
	int	regBase = PU0_REG_BASE;

	fpga[TESTCTRL] = 0;
	fpga[MEM_RAO] = 0;
	fpga[MEM_RAS] = 1;

	printf("Memory test\n");
	writeTestPattern(numBytes, 0);

	while(1){
		if(!(i % 100))
			printf("%d\n", i);

		memset(recvbuf, 0, numBytes);
		if(continueOnError){
			readTestPattern(numBytes, 0);
		}
		else {
			if(readTestPattern(numBytes, 0))
				exit(1);
		}
		i++;
	}
}

void memTest(int numBytes){
	int	i = 0;
	int	regBase = PU0_REG_BASE;

	fpga[TESTCTRL] = 0;
	fpga[MEM_RAO] = 0;
	fpga[MEM_RAS] = 1;

	printf("Memory test\n");
	while(1){
		if(!(i % 100))
			printf("%d\n", i);

		writeTestPattern(numBytes, i);

		memset(recvbuf, 0, numBytes);
#ifdef ZAP
		readTestPattern(numBytes, i);
#else
		if(readTestPattern(numBytes, i))
			exit(1);
#endif
		i++;
	}
}

void memTestData(){
	int		i = 0;
	unsigned int	v = 0;
	int		regBase = PU0_REG_BASE;

	fpga[TESTCTRL] = 0;
	fpga[MEM_RAO] = 0;
	fpga[MEM_RAS] = 1;

	printf("Memory test\n");
	for(i = 0; i < 33; i++){
		if(i > 31)
			v = 0;
		else
			v = 1 << i;
		printf("%x\n", v);

		writeTestPattern(4, v);
		if(readTestPattern(4, v))
			exit(1);
	}
}

void usage(void) {
	fprintf(stderr, "Usage:\tpupeMemTest [options]\n");
	fprintf(stderr, " -help             - Help on command line parameters\n");
	fprintf(stderr, " -board <n>        - PUPE board number\n");
	fprintf(stderr, " -n <n>            - Number of bytes\n");
	fprintf(stderr, " -sdram <n>        - The PUPE SDRAM bank number\n");
	fprintf(stderr, " -bram <n>         - The PUPE BlockRAM bank number\n");
	fprintf(stderr, " -lclk <n>         - The LCLK frequency in megahertz\n");
	fprintf(stderr, " -bitfile <n>      - Set the bitfile name to load\n");
	fprintf(stderr, " -cont             - Continue on error\n");
}

static struct option options[] = {
		{ "?",			0, NULL, 0 },
		{ "h",			0, NULL, 0 },
		{ "help",		0, NULL, 0 },
		{ "board",		1, NULL, 0 },
		{ "n",			1, NULL, 0 },
		{ "sdram",		1, NULL, 0 },
		{ "bram",		1, NULL, 0 },
		{ "lclk",		1, NULL, 0 },
		{ "bitfile",		1, NULL, 0 },
		{ "cont",		0, NULL, 0 },
		{ 0,0,0,0 }
};

int main(int argc, char *argv[])
{
	int		optIndex = 0;
	int		c;
	const char*	s;
	float		clock_freq0 = 50.0e6f;
	float		clock_freq1 = 125.0e6f;
	ADMXRC2_STATUS	status;
	int		data;
	int		board = 0;
	int		numBytes = 0;
	int		sdram = 3;
	int		bram = -1;
	char*		bitFileName = fpgafile;
	
	while((c = getopt_long_only(argc, argv, "", options, &optIndex)) == 0){
		s = options[optIndex].name;
		if(!strcmp(s , "help") || !strcmp(s, "h") || !strcmp(s, "?")){
			usage();
			return 1;
		}
		else if(!strcmp(s, "board")){
			board = strtol(optarg, 0, 0);
		}
		else if(!strcmp(s, "n")){
			numBytes = strtol(optarg, 0, 0);
		}
		else if(!strcmp(s, "sdram")){
			sdram = strtol(optarg, 0, 0);
		}
		else if(!strcmp(s, "bram")){
			bram = strtol(optarg, 0, 0);
		}
		else if(!strcmp(s, "lclk")){
			clock_freq0 = 1e6 * strtol(optarg, 0, 0);
		}
		else if(!strcmp(s, "bitfile")){
			bitFileName = optarg;
		}
		else if(!strcmp(s, "cont")){
			continueOnError = 1;
		}
		else {
			usage();
			return 1;
		}
	}
	
	recvbuf = (char*)ADMXRC2_Malloc(recvbufSize);
	memset(recvbuf, 0x0, recvbufSize);
	
	printf("Testing Board: %d\n", board);

	status = ADMXRC2_OpenCardByIndex(board, &card);
	if(status != ADMXRC2_SUCCESS) {
		printf("Failed to open driver (%s) Status (%x)\n", driver, status);
		return 1;
	}

	if((status = ADMXRC2_GetCardInfo(card, &ocardInfo)) != ADMXRC2_SUCCESS){
		printf("Failed to get info: %s\n", ADMXRC2_GetStatusString(status));
		return 1;
	}

	if((status = ADMXRC2_GetSpaceInfo(card, 0, &spInfo)) != ADMXRC2_SUCCESS){
		printf("Failed to read info %s\n", ADMXRC2_GetStatusString(status));
		return 1;
	}
	fpga = (volatile uint*)spInfo.VirtualBase;

	// Set up Clocks
	status = ADMXRC2_SetClockRate(card, 0, clock_freq0, 0);
	if (status != ADMXRC2_SUCCESS){
		printf("Failed to set LCLK to %.1fMHz\n", clock_freq0 / 1.0e6f);
		return 1;
	}

	status = ADMXRC2_SetClockRate(card, 1, clock_freq1, 0);
	if (status != ADMXRC2_SUCCESS){
		printf("Failed to set MCLK to %.1fMHz\n", clock_freq1 / 1.0e6f);
		return 1;
	}

	printf("Set Clocks to: Lclk: %.2fMHz MClk: %.2fMHz\n", clock_freq0 / 1.0e6f, clock_freq1 / 1.0e6f);
        printf("Programming FPGA: %s\n", bitFileName);
        if((status = ADMXRC2_ConfigureFromFile(card, bitFileName)) != ADMXRC2_SUCCESS) {
                printf("Unable to load fpga: status(%x)\n", status);
                return 1;
        }

	status = ADMXRC2_SetupDMA(card, recvbuf, recvbufSize, 0, &dmaDesc);
	if(status != ADMXRC2_SUCCESS){
		printf("Failed to set up receive DMA buffer:\n");
		return 1;
	}

	// Wait a bit
//	sleep(1);

#ifdef ZAP
	// reset ADC PLL clock source
	printf("Reseting ADC clock source\n");
	fpga[ADC] = 0x200 | 0x000;
	usleep(1000);
	fpga[ADC] = 0x000 | 0x000;
	sleep(1);
#endif

#ifndef ZAP
	printf("Resetting Card\n");
	data = 1;
	ADMXRC2_Write(card, ADMXRC2_IOWIDTH_32, ADMXRC2_IOADAPTER, 0x28, &data, 4);
	usleep(1);
	data = 0;
	ADMXRC2_Write(card, ADMXRC2_IOWIDTH_32, ADMXRC2_IOADAPTER, 0x28, &data, 4);

	sleep(1);
#endif

	if(bram >= 0){
		fpga[MEM_REG]= SDRAM_BLOCKRAM;
		fpga[IMEM_REG]= bram;
		if(numBytes == 0)
			numBytes = 1024;
	}
	else {
		fpga[MEM_REG]= 0x08 + sdram;
		fpga[IMEM_REG]= 0;
		if(numBytes == 0)
			numBytes = 102400;
	}
	
	printf("Test MemoryArea: SDRAM: %x BLOCKRAM: %x BlockSize: %d\n", fpga[MEM_REG], fpga[IMEM_REG], numBytes);
	printf("FPGA Firmware: %x\n", fpga[FIRMWARE]);
	printf("FPGA TestCtrl: %x\n", fpga[TESTCTRL]);

//	memDisplay(numBytes);
//	memTestSpeed(numBytes);
	memTestRead(numBytes);
//	memTestData();
//	memTest(numBytes);
	
	// Close the device
	ADMXRC2_CloseCard(card);

	return 0;
}
