/*******************************************************************************
 *	BoapMc.cc	BoapMc RPC protocol
 *	T.Barnaby,	BEAM Ltd,	2012-11-14
 *	Copyright (c) 2012 All Right Reserved, Beam Ltd, http://www.beam.ltd.uk
 *******************************************************************************
 *
 * Simple multi-threaded bi-directional communications system.
 *
 * The BoapMcComms class allows for RPC calls defined using the BOAP IDL.
 * Only one RPC can be active at a time. this is enforced with a BMutex.
 * The BoapMcComms class also allows for serving RPC requests at the same time.
 * Only one request stream can be processed at a time (one packer buffer).
 *
 * proccessRx() should be called in a separate thread repeatidly. This reads RX
 * packets and queues them as either RPC replies or requests.
 */
#include <BoapMc.h>
#include <BCrc16.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#define DEBUG_LOCAL	0
#define DEBUG_LOCAL1	0

#if DEBUG_LOCAL
#include <BDebug.h>
#define	dlprintf1(fmt, a...)       {printf("%.3f: ", getTime()); printf(fmt, ##a);}
#define	dlprintf(fmt, a...)       {printf(fmt, ##a);}
#else
#define	dlprintf(fmt, a...)
#endif

#if DEBUG_LOCAL1
#include <BDebug.h>
#define	dl1printf(fmt, a...)       {printf("%.3f: ", getTime()); printf(fmt, ##a);}
#else
#define	dl1printf(fmt, a...)
#endif

/*******************************************************************************
 *	Base for all Client Classes
 *******************************************************************************
 */
BoapMcClientObject::BoapMcClientObject(BComms& comms) : ocomms(comms){
	oapiVersion = 0;
	oaddressTo = 0;
	oaddressFrom = 0;
}

BoapMcClientObject::~BoapMcClientObject(){
}

void BoapMcClientObject::setAddress(BUInt8 addressTo, BUInt8 addressFrom){
	oaddressTo = addressTo;
	oaddressFrom = addressFrom;
}

BUInt32 BoapMcClientObject::getApiVersion(){
	return oapiVersion;
}

BError BoapMcClientObject::performSend(){
	BError		err;
	BUInt32		nt;

	opacket.head.addressTo = oaddressTo;
	opacket.head.addressFrom = oaddressFrom;
	opacket.head.checksum = 0;
	opacket.head.checksum = bcrc16(&opacket, opacket.head.length);
	
	return ocomms.write(&opacket, opacket.head.length, nt);
}

BError BoapMcClientObject::performRecv(){
	BError		err;
	BUInt32		nt;
	BUInt16		checksum;

//	printf("BoapMcClientObject::performRecv: PacketMode: %d\n", ocomms.packetMode());
	if(ocomms.packetMode()){
		if(err = ocomms.read(&opacket, sizeof(opacket), nt))
			return err;
	}
	else {
		if(err = ocomms.read(&opacket, sizeof(opacket.head), nt))
			return err;
		if(err = ocomms.read(&opacket.data, opacket.head.length - sizeof(opacket.head), nt))
			return err;
	}
	
	// Validate checksum
	checksum = opacket.head.checksum;
	opacket.head.checksum = 0;
	if(checksum != bcrc16(&opacket, opacket.head.length))
		err.set(ErrorChecksum, "Checksum");
	opacket.head.checksum = checksum;

	return err;
}

BError BoapMcClientObject::performCall(){
	BError			err;

	if(err = performSend())
		return err;

	if(err = performRecv())
		return err;

	return err;
}


BoapMcSignalObject::BoapMcSignalObject(BComms& comms) : ocomms(comms){
}

BError BoapMcSignalObject::performSend(BoapMcPacket& tx){
	BError			err;

	return err;
}


/*******************************************************************************
 *	Base for all Server Objects
 *******************************************************************************
 */
BoapMcServiceObject::BoapMcServiceObject(){
	oapiVersion = 0;
}

BoapMcServiceObject::~BoapMcServiceObject(){
}

BError BoapMcServiceObject::process(BoapMcPacket& rx, BoapMcPacket& tx){
	BError			err;

	return err;
}

BError BoapMcServiceObject::sendEvent(BoapMcPacket& tx){
	BError		err;

	return err;
}

BError BoapMcServiceObject::processEvent(BoapMcPacket& rx){
	BError		err;

	return err;
}

/*******************************************************************************
 *	Base for all Comms Objects
 *******************************************************************************
 */

BoapMcComms::BoapMcComms(Bool threaded, BUInt rxQueueSize) : othreaded(threaded), opacketReqQueue(rxQueueSize), opacketTxQueue(0){
	ocomms = 0;
	oapiVersion = 0;
	oslave = 0;
	otimeout = 2000000;
	oaddressTo = 0;
	oaddressFrom = 0;
	opacketRx.head.cmd = 0;
	opacketReqRx.head.cmd = 0;
}

BoapMcComms::~BoapMcComms(){
}

void BoapMcComms::setCommsMode(Bool slave, BUInt txQueueSize){
	oslave = slave;
	opacketTxQueue.resize(txQueueSize + 1);
	opacketTxQueueWriteNum.setValue(txQueueSize);
}

void BoapMcComms::setComms(BComms& comms){
	ocomms = &comms;
}

void BoapMcComms::setComms(BComms* comms){
	ocomms = comms;
}

void BoapMcComms::setAddress(BUInt8 addressTo, BUInt8 addressFrom){
	oaddressTo = addressTo;
	oaddressFrom = addressFrom;
}

BUInt32 BoapMcComms::getApiVersion(){
	return oapiVersion;
}

BUInt32 BoapMcComms::setTimeout(BUInt32 timeoutUs){
	BUInt32	t = otimeout;
	
	otimeout = timeoutUs;
	
	return t;
}

// Process incoming packets. Normally called from a thread
BError BoapMcComms::processRx(BTimeout timeoutUs){
	BError	err;
	
	dlprintf("BoapMcComms::processRx: call packetRecv\n");
	if(!ocomms)
		return err.set(ErrorNotAvailable, "No comms available");

	/// !!! This should wait on comms for timeoutUs !!!
	if((timeoutUs == 0) && !ocomms->readAvailable()){
		return err.set(ErrorTimeout, "Timeout");
	}

	if(err = packetRecv(opacket)){
		dlprintf("BoapMcComms::processRx: error: %x %s\n", err.num(), err.str());
		return err;
	}
	
	dlprintf("BoapMcComms::processRx: packetRx: cmd: %x\n", opacket.head.cmd);
	if(opacket.head.cmd & BoapMcTypeReply){
//		printf("BoapMcComms::processRx: Gotreply: %x\n", opacket.head.cmd);
		memcpy(&opacketRx, &opacket, opacket.head.length);
		opacketRxSema.set();
	}
	else {
//		printf("BoapMcComms::processRx: GotReq: %x\n", opacket.head.cmd);
		opacketReqQueue.write(opacket);
	}
//	printf("BoapMcComms::processRx: End: status(%x, %x)\n", opacketRx.head.cmd, opacketReqRx.head.cmd);
	
	return err;
}

BError BoapMcComms::processRequests(BTimeout timeoutUs){
	BError	err;
	
	if(!ocomms)
		return err.set(ErrorNotAvailable, "No comms available");

	while(opacketReqQueue.readAvailable()){
		if(err = processRequest(timeoutUs))
			break;
	}
	
	return err;
}

BError BoapMcComms::processRequest(BTimeout timeoutUs){
	BError	err;
	BUInt32	nt;
	
	if(!ocomms)
		return err.set(ErrorNotAvailable, "No comms available");

	if(err = opacketReqQueue.read(opacketReqRx, timeoutUs))
		return err;

//	printf("BoapMcComms::processRequest: cmd: %d\n", opacketReqRx.head.cmd);

	// Process an incomming packet
	processPacket(opacketReqRx, opacketReqTx);

	if(opacketReqTx.head.length){
		opacketReqTx.head.addressTo = oaddressTo;
		opacketReqTx.head.addressFrom = oaddressFrom;

		if(oslave){
			// Send any requests from this end
			if(opacketTxQueue.size()){
//				printf("BoapMcComms::processRequest: num: %d numWrite: %d\n", opacketTxQueue.readAvailable(), opacketTxQueueWriteNum.value());
				while(opacketTxQueue.readAvailable()){
//					printf("BoapMcComms::processRequest: send: num: %d %d\n", opacketTxQueue.readAvailable(), opacketTxQueueWriteNum.value());
					ocomms->write(opacketTxQueue.readData(), opacketTxQueue.readData()->head.length, nt);
					opacketTxQueue.readDone(1);
					opacketTxQueueWriteNum.add(1);
				}
			}
			else {
				opacketTxSema.set();
#if TARGET_SYS_armsys
				bsysTaskYield();
#endif
			}
		}

//		printf("BoapMcComms::processRequest: cmd: %d reply: %d %d\n", opacketReqRx.head.cmd, opacketReqTx.head.length, opacketReqTx.head.cmd);
		err = packetSend(opacketReqTx);
	}
	
	return err;
}

BError BoapMcComms::processPacket(BoapMcPacket& rx, BoapMcPacket& tx){
	BError	err;

	// Ignore all packets
	tx.head.length = 0;
	tx.head.cmd = 0x00;

	return err;
}

BError BoapMcComms::performCall(){
	BError	err;

	dlprintf("BoapMcComms::performCall: cmd: %x\n", opacketTx.head.cmd);
	opacketRx.head.cmd = 0;
	opacketRxSema.wait(0);

	if(err = performSend())
		return err;

	do {
		if(othreaded){
			if(!opacketRxSema.wait(otimeout))
				return err.set(ErrorTimeout, "Timeout");
		}
		else {
			if(err = processRx(otimeout))
				return err;
		}
	} while((opacketRx.head.cmd & 0x7F) != opacketTx.head.cmd);

	dlprintf("BoapMcComms::performCall: end: cmd: %x\n", opacketRx.head.cmd);

	return err;
}

BError BoapMcComms::performSend(){
	dlprintf("BoapMcComms::performSend: cmd: %x\n", opacketTx.head.cmd);
	return packetSend(opacketTx);
}

BError BoapMcComms::packetSend(BoapMcPacket& packet){
	BError		err;
	BUInt32		nt;
	BMutexLock	lock(olockTx, 1);

	if(!ocomms)
		return err.set(ErrorNotAvailable, "No comms available");

//	printf("packetSend: "); hd8(&packet, packet.head.length);
	packet.head.addressTo = oaddressTo;
	packet.head.addressFrom = oaddressFrom;
	packet.head.checksum = 0;
	packet.head.checksum = bcrc16(&packet, packet.head.length);

	if(oslave && !(packet.head.cmd & BoapMcTypeReply)){
		// If a slave send request packets just before a reply to a masters RPC
		
		if(opacketTxQueue.size()){
//			printf("BoapMcComms::packetSend: num: %d WriteAvailable: %d timeout: %d\n", opacketTxQueue.writeAvailable(), opacketTxQueueWriteNum.value(), otimeout);
			if(!opacketTxQueueWriteNum.take(1, otimeout)){
				printf("BoapMcComms::packetSend: timeout\n");
				return err.set(ErrorTimeout, "Timeout");
			}
			else {
//				printf("BoapMcComms::packetSend: DoWrite: WriteAvailable: %d\n", opacketTxQueueWriteNum.value());
				opacketTxQueue.write(packet);
				return err;
			}
		}
		else {
//			printf("BoapMcComms::packetSend: Wait for TX slot\n");
			if(!opacketTxSema.wait(otimeout)){
				printf("BoapMcComms::packetSend: timeout\n");
				return err.set(ErrorTimeout, "Timeout");
			}
		}
//		printf("BoapMcComms::packetSend: Do TX\n");
	}
	
	err = ocomms->write(&packet, packet.head.length, nt);
	
	return err;
}

BError BoapMcComms::packetRecv(BoapMcPacket& packet){
	BError	err;
	BUInt32	nt;
	BUInt16	checksum;	

	if(!ocomms)
		return err.set(ErrorNotAvailable, "No comms available");

	dlprintf("BoapMcComms::packetRecv: PacketMode: %d\n", ocomms->packetMode());
	if(ocomms->packetMode()){
		if(err = ocomms->read(&packet, sizeof(packet), nt))
			return err;
	}
	else {
		if(err = ocomms->read(&packet, sizeof(packet.head), nt))
			return err;
		if(err = ocomms->read(&packet.data, packet.head.length - sizeof(packet.head), nt))
			return err;
	}

	// Validate checksum
	checksum = packet.head.checksum;
	packet.head.checksum = 0;
	if(checksum != bcrc16(&packet, packet.head.length)){
		err.set(ErrorChecksum, "Checksum");
		return err;
	}
	packet.head.checksum = checksum;

//	printf("packetRecv: "); hd8(&packet, packet.head.length);
	
	return err;
}

