/*******************************************************************************
 *	BTimeStamp.h	TimeStamp classes
 *			T.Barnaby,	BEAM Ltd,	2005-10-20
 *******************************************************************************
 */
#include <BTimeStamp.h>
#include <BTimeStampMs.h>
#include <sys/time.h>

static int mon_yday[2][13] = {
	/* Normal years.  */
	{ 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365 },
	/* Leap years.  */
	{ 0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366 }
};

BTimeStamp::BTimeStamp(){
	clear();
}

BTimeStamp::BTimeStamp(int year, int month, int day, int hour, int minute, int second, int microSecond){
	set(year, month, day, hour, minute, second, microSecond);
}

BTimeStamp::BTimeStamp(const BString str){
	if(str == "")
		clear();
	else
		setString(str);
}

BTimeStamp::~BTimeStamp(){
}

void BTimeStamp::clear(){
	setYDay(0, 0, 0, 0, 0, 0);
}

void BTimeStamp::setFirst(){
	setYDay(1, 0, 0, 0, 0, 0);
}

void BTimeStamp::setLast(){
	setYDay(9999, 0, 0, 0, 0, 0);
}

void BTimeStamp::setYDay(int year, int yday, int hour, int minute, int second, int microSecond){
	oyear = year;
	oyday = yday;
	ohour = hour;
	ominute = minute;
	osecond = second;
	omicroSecond = microSecond;
	ospare = 0;
}

void BTimeStamp::set(time_t time, int microSeconds){
	struct tm	tm;
	
	gmtime_r(&time, &tm);
	
	oyear = 1900 + tm.tm_year;
	oyday = tm.tm_yday;
	ohour = tm.tm_hour;
	ominute = tm.tm_min;
	osecond = tm.tm_sec;
	omicroSecond = 0;
	ospare = 0;
	addMicroSeconds(microSeconds);
}

void BTimeStamp::set(int year, int month, int day, int hour, int minute, int second, int microSecond){
	oyear = year;
	oyday = mon_yday[isLeap(year)][month-1] + day - 1;
	ohour = hour;
	ominute = minute;
	osecond = second;
	omicroSecond = microSecond;
	ospare = 0;
}

void BTimeStamp::set(const BTimeStampMs& timeStamp){
	oyear = timeStamp.year;
	oyday = timeStamp.yday;
	ohour = timeStamp.hour;
	ominute = timeStamp.minute;
	osecond = timeStamp.second;
	omicroSecond = timeStamp.milliSecond * 1000;
	ospare = 0;
}

void BTimeStamp::setTime(int hour, int minute, int second, int microSecond){
	ohour = hour;
	ominute = minute;
	osecond = second;
	omicroSecond = microSecond;
}

void BTimeStamp::setNow(){
	struct timeval	tv;
	struct tm	tm;
	
	gettimeofday(&tv, 0);
	gmtime_r(&tv.tv_sec, &tm);
	
	oyear = 1900 + tm.tm_year;
	oyday = tm.tm_yday;
	ohour = tm.tm_hour;
	ominute = tm.tm_min;
	osecond = tm.tm_sec;
	omicroSecond = tv.tv_usec;
	ospare = 0;
}

int BTimeStamp::year() const {
	return oyear;
}

int BTimeStamp::month() const {
	int		mon = 0;
	
	for(mon = 11; oyday < mon_yday[isLeap(oyear)][mon]; --mon);
	return mon + 1;
}

int BTimeStamp::day() const {
	int		mon = 0;
	int		day = 0;
	
	for(mon = 11; oyday < mon_yday[isLeap(oyear)][mon]; --mon);
	day = oyday - mon_yday[isLeap(oyear)][mon];
	
	return day + 1;
}

int BTimeStamp::yday() const {
	return oyday;
}

int BTimeStamp::hour() const {
	return ohour;
}

int BTimeStamp::minute() const {
	return ominute;
}

int BTimeStamp::second() const {
	return osecond;
}

int BTimeStamp::microSecond() const {
	return omicroSecond;
}


BString BTimeStamp::getString(BString separator) const {
	BString		s;
	int		mon = 0;
	int		day = 0;
	
	for(mon = 11; oyday < mon_yday[isLeap(oyear)][mon]; --mon);
	day = oyday - mon_yday[isLeap(oyear)][mon];

	s.printf("%04d-%02d-%02d%s%02d:%02d:%02d.%06d", oyear, mon + 1, day + 1, separator.retStr(), ohour, ominute, osecond, omicroSecond);
	return s;
}

BString BTimeStamp::getStringNoMs(BString separator) const {
	BString		s;
	int		mon = 0;
	int		day = 0;
	
	for(mon = 11; oyday < mon_yday[isLeap(oyear)][mon]; --mon);
	day = oyday - mon_yday[isLeap(oyear)][mon];

	s.printf("%04d-%02d-%02d%s%02d:%02d:%02d", oyear, mon + 1, day + 1, separator.retStr(), ohour, ominute, osecond);
	return s;
}

BString	BTimeStamp::getStringFormatted(BString format) const {
	struct tm	tm;
	int		mon = 0;
	int		day = 0;
	char		buf[64];
	
	buf[0] = '\0';
	
	for(mon = 11; oyday < mon_yday[isLeap(oyear)][mon]; --mon);
	day = oyday - mon_yday[isLeap(oyear)][mon];
	
	tm.tm_sec = osecond;
	tm.tm_min = ominute;
	tm.tm_hour = ohour;
	tm.tm_mday = day + 1;
	tm.tm_mon = mon;
	tm.tm_year = oyear - 1900;
	tm.tm_wday = 0;
	tm.tm_yday = oyday;
	tm.tm_isdst = 0;
	strftime(buf, sizeof(buf), format, &tm);
	
	return buf;
}

BError BTimeStamp::setString(const BString dateTime){
	BError		err;
	int		y = 0;
	int		mon = 0;
	int		day = 0;
	int		h = 0;
	int		m = 0;
	int		s = 0;
	int		ms = 0;
	int		n;
	int		yearDayFormat = 0;

	clear();
	if(dateTime == "first")
		setFirst();
	else if(dateTime == "last")
		setLast();
	else if(dateTime == "now")
		setNow();
	else {
		n = sscanf(dateTime, "%04u-%02u-%02u%*[ T]%02u:%02u:%02u.%06u", &y, &mon, &day, &h, &m, &s, &ms);
		if((n != 3) && (n != 5) && (n != 6) && (n != 7)){
			n = sscanf(dateTime, "%02u/%02u/%04u%*[ T]%02u:%02u:%02u.%06u", &day, &mon, &y, &h, &m, &s, &ms);
			if((n != 3) && (n != 5)  && (n != 6) && (n != 7)){
				n = sscanf(dateTime, "%04u/%02u/%02u%*[ T]%02u:%02u:%02u.%06u", &y, &mon, &day, &h, &m, &s, &ms);
				if((n != 3) && (n != 5)  && (n != 6) && (n != 7)){
					n = sscanf(dateTime, "%02u:%02u:%02u.%06u", &h, &m, &s, &ms);
					if((n != 2) && (n != 3) && (n != 4)){
						n = sscanf(dateTime, "%04u%3u%*[ T]%02u:%02u:%02u.%03u", &y, &day, &h, &m, &s, &ms);
						if((n != 6)){
							err.set(ErrorMisc, "Time format error\n");
						}
						else{
							ms *= 1000;
							yearDayFormat = 1;
						}
					}
				}
			}
		}

		if(!err){
			if(!yearDayFormat){
				if((mon < 1) || (mon > 12))
					return err.set(ErrorMisc, "Month value out of range");
				if((day < 1) || (day > 31))
					return err.set(ErrorMisc, BString("Day value out of range: ") + day);
			}
			if((h < 0) || (h > 23))
				return err.set(ErrorMisc, "Hour value out of range");
			if((m < 0) || (m > 59))
				return err.set(ErrorMisc, "Minute value out of range");
			if((s < 0) || (s > 59))
				return err.set(ErrorMisc, "Second value out of range");

			oyear = y;
			if(yearDayFormat)
				oyday = day - 1;
			else
				oyday = mon_yday[isLeap(oyear)][mon-1] + day - 1;
			ohour = h;
			ominute = m;
			osecond = s;
			omicroSecond = ms;
		}
	}
	return err;
}

void BTimeStamp::getDate(int& year, int& mon, int& day) const {
	year = oyear;
	for(mon = 11; oyday < mon_yday[isLeap(oyear)][mon]; --mon);
	day = oyday - mon_yday[isLeap(oyear)][mon];
}

void BTimeStamp::addMicroSeconds(int64_t microSeconds){
	addSeconds(microSeconds / 1000000);
	omicroSecond += microSeconds % 1000000;
	if(omicroSecond >= 1000000){
		if(microSeconds < 0){
			omicroSecond += 1000000;
			addSeconds(-1);
		}
		else {
			omicroSecond -= 1000000;
			addSeconds(1);
		}
	}
}

void BTimeStamp::addMilliSeconds(int milliSeconds){
	addSeconds(milliSeconds / 1000);
	omicroSecond += (1000 * (milliSeconds % 1000));
	if(omicroSecond >= 1000000){
		if(milliSeconds < 0){
			omicroSecond += 1000000;
			addSeconds(-1);
		}
		else {
			omicroSecond -= 1000000;
			addSeconds(1);
		}
	}
}

void BTimeStamp::addSeconds(int seconds){
	int	nyday = isLeap(oyear) ? 366 : 365;
	
	if(seconds >= 0){
		osecond += seconds % 60;
		if(osecond >= 60){
			ominute++;
			osecond -= 60;
		}

		ominute += (seconds / 60) % 60;
		if(ominute >= 60){
			ohour++;
			ominute -= 60;
		}

		ohour += (seconds / (60 * 60)) % 24;
		if(ohour >= 24){
			oyday++;
			ohour -= 24;
		}

		oyday += (seconds / (24 * 60 * 60)) % nyday;
		if(oyday >= nyday){
			oyear++;
			oyday -= nyday;
		}
	}
	else {
		seconds = -seconds;
		osecond -= seconds % 60;
		if(osecond >= 60){
			ominute--;
			osecond += 60;
		}

		ominute -= (seconds / 60) % 60;
		if(ominute >= 60){
			ohour--;
			ominute += 60;
		}

		ohour -= (seconds / (60 * 60)) % 24;
		if(ohour >= 24){
			oyday--;
			ohour += 24;
		}

		oyday -= (seconds / (24 * 60 * 60)) % nyday;
		if(oyday > 366){
			oyear--;
			oyday += isLeap(oyear) ? 366 : 365;
		}
	}
}

uint32_t BTimeStamp::getYearSeconds() const {
	uint32_t	s = 0;
	
	s += osecond;
	s += (60 * ominute);
	s += (60 * 60 * ohour);
	s += (60 * 60 * 24 * oyday);

	return s;
}

uint64_t BTimeStamp::getYearMicroSeconds() const {
	uint64_t	t = 0;
	
	t += omicroSecond;
	t += (1000000ULL * osecond);
	t += (1000000ULL * 60 * ominute);
	t += (1000000ULL * 60 * 60 * ohour);
	t += (1000000ULL * 60 * 60 * 24 * oyday);

	return t;
}

int BTimeStamp::compare(const BTimeStamp& timeStamp) const {
	if(oyear > timeStamp.oyear)
		return 1;
	else if(oyear < timeStamp.oyear)
		return -1;

	else if(oyday > timeStamp.oyday)
		return 1;
	else if(oyday < timeStamp.oyday)
		return -1;

	else if(ohour > timeStamp.ohour)
		return 1;
	else if(ohour < timeStamp.ohour)
		return -1;

	else if(ominute > timeStamp.ominute)
		return 1;
	else if(ominute < timeStamp.ominute)
		return -1;

	else if(osecond > timeStamp.osecond)
		return 1;
	else if(osecond < timeStamp.osecond)
		return -1;

	else if(omicroSecond > timeStamp.omicroSecond)
		return 1;
	else if(omicroSecond < timeStamp.omicroSecond)
		return -1;

	else
		return 0;
}

int BTimeStamp::isLeap(int year){
	return ((year) % 4 == 0 && ((year) % 100 != 0 || (year) % 400 == 0));
}


BInt64 BTimeStamp::difference(BTimeStamp t2, BTimeStamp t1){
	int	y;
	BInt64	t = 0;

	if(t1 > t2){
		for(y = t2.oyear; y < t1.oyear; y++){
			if(isLeap(y))
				t = t + (366 * (24 * 3600) * 1000000ULL);
			else
				t = t + (365 * (24 * 3600) * 1000000ULL);
		}
		t = -(t + t1.getYearMicroSeconds() - t2.getYearMicroSeconds());
	}
	else {
		for(y = t1.oyear; y < t2.oyear; y++){
			if(isLeap(y))
				t = t + (366 * (24 * 3600) * 1000000ULL);
			else
				t = t + (365 * (24 * 3600) * 1000000ULL);
		}

		t = t + t2.getYearMicroSeconds() - t1.getYearMicroSeconds();
	}

	return t;
}



void toBString(BTimeStamp& v, BString& s){
	s = v.getString();
}

void fromBString(BString& s, BTimeStamp& v){
	v.setString(s);
}
