/*******************************************************************************
 *	BFifoCirc.inc	FIFO Buffer Class
 *			T.Barnaby,	BEAM Ltd,	2006-02-22
 *******************************************************************************
 */
#include <BFifoCirc.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/mman.h>
#include <unistd.h>

#define	DEBUG_BFifoCirc	0

#if DEBUG_BFifoCirc
#define	BFifoCirc_dprintf(fmt, a...)       printf(fmt, ##a);
#else
#define	BFifoCirc_dprintf(fmt, a...)
#endif


template <class Type> Type* BFifoCirc<Type>::writeData(){
	BMutexLock	lock(olock);
	
	return &odata[owritePos];
}

template <class Type> BError BFifoCirc<Type>::write(const Type* data, uint32_t numTypes){
	BError	err;
	
	olock.lock();
	if(numTypes > osize)
		return BError(1, "Fifo to small for write");
	
	memcpy(&odata[owritePos], data, numTypes * sizeof(Type));
	olock.unlock();
	
	writeDone(numTypes);
	
	return err;
}

template <class Type> BError BFifoCirc<Type>::read(Type* data, uint32_t numTypes){
	BError	err;

	olock.lock();
	memcpy(data, &odata[oreadPos], numTypes * sizeof(Type));
	olock.unlock();

	readDone(numTypes);
	
	return err;
}

template <class Type> Type* BFifoCirc<Type>::readData(){
	BMutexLock	lock(olock);

	return &odata[oreadPos];
}

template <class Type> Type& BFifoCirc<Type>::operator[](int pos){
	BMutexLock	lock(olock);

	return odata[oreadPos + pos];
}

template <class Type> BFifoCirc<Type>::BFifoCirc(uint32_t size) : olock(BMutex::Recursive), owritePos(size), oreadPos(size){
	BMutexLock	lock(olock);
	BError		err;
	
	if(err = mapCircularBuffer(size)){
		fprintf(stderr, "%s\n", err.getString().retStr());
		exit(1);
	}
	owritePos.setSize(osize);
	oreadPos.setSize(osize);
}

template <class Type> BFifoCirc<Type>::~BFifoCirc(){
	BMutexLock	lock(olock);

	if(odata)
		unmapCircularBuffer();
	odata = 0;
	osize = 0;
}

template <class Type> uint32_t BFifoCirc<Type>::size(){
	BMutexLock	lock(olock);

	return osize;
}

template <class Type> void BFifoCirc<Type>::clear(){
	BMutexLock	lock(olock);
	
	owritePos.set(0);
	owriteNumFifoSamples.setValue(0);
	oreadPos.set(0);
}

template <class Type> BError BFifoCirc<Type>::mapCircularBuffer(uint32_t size){
	BError		err;
	int		shmFd = -1;
	BString		segName = BString("/BFifoCirc-") + BString(getpid());
	void*		m1;
	void*		m2;
	uint32_t	sizeBytes = size * sizeof(Type);

	// Round up to a multiple of the page size
	if(sizeBytes % 4096 != 0)
		sizeBytes = ((sizeBytes / 4096) + 1) * 4096;
	
	ovmSize = 2 * sizeBytes;
	
	if((shmFd = shm_open(segName, O_RDWR | O_CREAT | O_EXCL, 0600)) == -1){
		err.set(1, "Unable to open shared memory");
		return err;
	}
	
	// Set length of shared memory segment
	if(ftruncate(shmFd, ovmSize) == -1){
		close(shmFd);
		err.set(1, "Unable to set shared memory size");
		return err;
	}

	if((m1 = mmap(0, ovmSize, PROT_READ | PROT_WRITE, MAP_SHARED, shmFd, 0)) == MAP_FAILED){
		close(shmFd);
		err.set(1, BString("Unable to map shared memory: ") + strerror(errno));
		return err;
	}

	// Unmap the 2nd half
	if(munmap((char*)m1 + sizeBytes, sizeBytes) == -1){
		close(shmFd);
		err.set(1, "Unable to unmap shared memory1");
		return err;
	}

	// Map the first half into the now available hole where the second half used to be.
	if((m2 = mmap((char*)m1 + sizeBytes, sizeBytes, PROT_READ | PROT_WRITE, MAP_SHARED, shmFd, 0))  == MAP_FAILED){
		close(shmFd);
		err.set(1, "Unable to map shared memory");
		return err;
	}

	// fd no longer needed. The mapping is retained.
	close(shmFd);
	shm_unlink(segName);

	odata = (Type*)m1;
	osize = sizeBytes / sizeof(Type);
	
	return err;
}

template <class Type> void BFifoCirc<Type>::unmapCircularBuffer(){
	munmap(odata, ovmSize);
}

template <class Type> uint32_t BFifoCirc<Type>::writeAvailable(){
	BMutexLock	lock(olock);
	uint32_t	ns;
	
	if(oreadPos > owritePos)
		ns = oreadPos - owritePos - 1;
	else
		ns = osize - owritePos + oreadPos - 1;

	return ns;
}

template <class Type> BError BFifoCirc<Type>::writeWaitAvailable(uint32_t numFifoSamples){
	BError	err;

	owriteNumFifoSamples.waitLessThan(osize - numFifoSamples);
	
	return err;
}

template <class Type> BError BFifoCirc<Type>::readWaitAvailable(uint32_t numFifoSamples){
	BError	err;

	BFifoCirc_dprintf("BFifoCirc<Type>::readWaitAvailable: %d\n", numFifoSamples);
	owriteNumFifoSamples.waitMoreThanOrEqual(numFifoSamples);
	
	return err;
}

template <class Type> void BFifoCirc<Type>::writeDone(uint32_t numFifoSamples){
	BMutexLock	lock(olock);
	BIter	i;
	
	owritePos += numFifoSamples;
	owriteNumFifoSamples += numFifoSamples;

	BFifoCirc_dprintf("BFifoCirc<Type>::writeDone: %d (%d)\n", numFifoSamples, owriteNumFifoSamples.value());
}

template <class Type> BError BFifoCirc<Type>::readDone(uint32_t numFifoSamples){
	BMutexLock	lock(olock);
	BError		err;
	BIter		i;

	oreadPos += numFifoSamples;
	owriteNumFifoSamples -= numFifoSamples;
	
	return err;
}

template <class Type> uint32_t BFifoCirc<Type>::readAvailable(){
	BMutexLock	lock(olock);
	uint32_t	r;
	
	if(oreadPos == owritePos)
		r = 0;
	else
		r = owritePos.difference(oreadPos);
	
	return r;
}
