/*
** File: vpd.c  
** Project: ADMXRC2 module driver
** Purpose: OS-independent IOCTL handlers for reading and writing VPD (vital
**          Product Data).
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>
#include <core/coreif.h>
#include "device.h"
#include "vpd.h"

#define MAX_CHUNK (128)

static DfIoStatus
mapAcquireStatus(
  CoreAcquireStatus acqStatus)
{
  switch (acqStatus) {
  case CoreAcquireSuccess:
    return DfIoStatusSuccess;

  case CoreAcquireMustWait:
    return DfIoStatusError(ADMXRC2_DEVICE_BUSY);

  case CoreAcquireCancelled:
    return DfIoStatusError(ADMXRC2_CANCELLED);

  case CoreAcquireRemoved:
  case CoreAcquireFailed:
    return DfIoStatusError(ADMXRC2_FAILED);

  case CoreAcquireInvalidIndex:
    return DfIoStatusError(ADMXRC2_INVALID_INDEX);

  case CoreAcquireInvalid:
  case CoreAcquireUnexpectedError:
  case CoreAcquireWaiting:
  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }
}

static DfIoStatus
mapVpdStatus(
  CoreVpdStatus vpdStatus)
{
  switch (vpdStatus) {
  case CoreVpdSuccess:
    return DfIoStatusSuccess;
  case CoreVpdNotPresent:
  case CoreVpdTimeout:
  case CoreVpdVerifyFailed:
  case CoreVpdWriteFailed:
    return DfIoStatusError(ADMXRC2_FAILED);
  case CoreVpdInvalidRegion:
    return DfIoStatusError(ADMXRC2_INVALID_REGION);
  case CoreVpdInterrupted:
    return DfIoStatusError(ADMXRC2_CANCELLED);
  case CoreVpdInvalidParameter:
  case CoreVpdGeneralFailure:
  default:
    return DfIoStatusError(ADMXRC2_UNKNOWN_ERROR);
  }
}

void
syncVpdOnClose(
	Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx)
{
  CoreAcquireStatus acqStatus;
  CoreVpdStatus vpdStatus;
  CoreTicket ticket;

  ticket.count = 1;
  ticket.resources[0].code = CORE_RESOURCE_VPD;
  ticket.resources[0].index = 0;
  acqStatus = pDevCtx->coreInterface.pAcquireInitialize(pDevCtx->pCoreContext, &ticket);
  if (CoreAcquireSuccess != acqStatus) {
    dfDebugPrint(0, ("*** syncVpdOnClose: failed initialize ticket\n"));
    return;
  }
  acqStatus = pDevCtx->coreInterface.pAcquireSync(pDevCtx->pCoreContext, &ticket, 0);
  if (CoreAcquireSuccess != acqStatus) {
    dfDebugPrint(0, ("*** syncVpdOnClose: failed acquire ticket\n"));
    return;
  }
  vpdStatus = pDevCtx->coreInterface.pVpdSync(pDevCtx->pCoreContext, &ticket);
  if (CoreVpdSuccess != vpdStatus) {
    dfDebugPrint(0, ("*** syncVpdOnClose: failed to sync VPD\n"));
    goto done;
  }

done:
  pDevCtx->coreInterface.pRelease(pDevCtx->pCoreContext, &ticket);
}

DfIoStatus
ioctlReadConfig(
	Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize,
  unsigned int outSize)
{
  IOCTLS_ADMXRC2_READCONFIG* pIoctl = (IOCTLS_ADMXRC2_READCONFIG*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  CoreAcquireStatus acqStatus;
  CoreVpdStatus vpdStatus;
  CoreTicket ticket;
  uint8_t stagingBuffer[MAX_VPD_WIDTH];
  uint32_t index, value;
  size_t offset;
  unsigned int i;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize || sizeof(pIoctl->out) != outSize) {
    return DfIoStatusInvalid;
  }
  if (!admxrc2CheckAccessRights(pClCtx)) {
    return DfIoStatusError(ADMXRC2_ACCESS_DENIED);
  }

  index = pIoctl->in.index;
  offset = (size_t)index * pDevCtx->vpdWidth;

  ticket.count = 1;
  ticket.resources[0].code = CORE_RESOURCE_VPD;
  ticket.resources[0].index = 0;
  acqStatus = pDevCtx->coreInterface.pAcquireInitialize(pDevCtx->pCoreContext, &ticket);
  if (CoreAcquireSuccess != acqStatus) {
    return mapAcquireStatus(acqStatus);
  }
  acqStatus = pDevCtx->coreInterface.pAcquireSync(pDevCtx->pCoreContext, &ticket, 0);
  if (CoreAcquireSuccess != acqStatus) {
    return mapAcquireStatus(acqStatus);
  }

  vpdStatus = pDevCtx->coreInterface.pVpdRead(pDevCtx->pCoreContext, &ticket, offset, pDevCtx->vpdWidth, stagingBuffer);
  if (CoreVpdSuccess != vpdStatus) {
    status = mapVpdStatus(vpdStatus);
    goto done;
  }

  /* Convert little-endian buffer to value to be returned */
  value = 0U;
  for (i = 0; i < pDevCtx->vpdWidth; i++) {
    value = value | ((uint32_t)stagingBuffer[i] << (i << 3));
  }
  pIoctl->out.value = value;

done:
  pDevCtx->coreInterface.pRelease(pDevCtx->pCoreContext, &ticket);

  return status;
}

DfIoStatus
ioctlWriteConfig(
	Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx,
  void* pBuffer,
  unsigned int inSize)
{
  IOCTLS_ADMXRC2_WRITECONFIG* pIoctl = (IOCTLS_ADMXRC2_WRITECONFIG*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  CoreAcquireStatus acqStatus;
  CoreVpdStatus vpdStatus;
  CoreTicket ticket;
  uint8_t stagingBuffer[MAX_VPD_WIDTH];
  uint32_t index, value;
  size_t offset;
  unsigned int i;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }
  if (!admxrc2CheckAccessRights(pClCtx)) {
    return DfIoStatusError(ADMXRC2_ACCESS_DENIED);
  }

  index = pIoctl->in.index;
  value = pIoctl->in.value;
  offset = (size_t)index * pDevCtx->vpdWidth;

  /* Convert value to be written to a little-endian buffer */
  for (i = 0; i < pDevCtx->vpdWidth; i++) {
    stagingBuffer[i] = (uint8_t)(value >> (i << 3));
  }

  ticket.count = 1;
  ticket.resources[0].code = CORE_RESOURCE_VPD;
  ticket.resources[0].index = 0;
  acqStatus = pDevCtx->coreInterface.pAcquireInitialize(pDevCtx->pCoreContext, &ticket);
  if (CoreAcquireSuccess != acqStatus) {
    return mapAcquireStatus(acqStatus);
  }
  acqStatus = pDevCtx->coreInterface.pAcquireSync(pDevCtx->pCoreContext, &ticket, 0);
  if (CoreAcquireSuccess != acqStatus) {
    return mapAcquireStatus(acqStatus);
  }

  pClCtx->bNeedVpdSync = TRUE;
  vpdStatus = pDevCtx->coreInterface.pVpdWrite(pDevCtx->pCoreContext, &ticket, offset, pDevCtx->vpdWidth, stagingBuffer);
  if (CoreVpdSuccess != vpdStatus) {
    status = mapVpdStatus(vpdStatus);
    goto done;
  }

done:
  pDevCtx->coreInterface.pRelease(pDevCtx->pCoreContext, &ticket);

  return status;
}
