/*
** File: pio.c  
** Project: ADMXRC2 module driver
** Purpose: OS-independent IOCTL handlers for programmed I/O.
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>
#include <core/coreif.h>
#include "device.h"
#include "pio.h"
#include <admxrc2/types.h>

/* Length threshold at which we install a cancel callback for I/O requests */
#define CANCEL_THRESHOLD (0x1000U)

typedef struct _Context {
  boolean_t bCancel;
  DfIoStatus reason;
  DfEvent event;
} Context;

static const size_t maxChunk = 1024;

static ADMXRC2_STATUS
findWindow(
	Admxrc2DeviceContext* pDevCtx,
  uint32_t width,
  uint32_t flags,
  uint32_t local,
  size_t length,
  uint8_t** ppKernelAddr)
{
  unsigned int i;
  WindowAccessInfo* pAccessInfo;
  uint64_t localLimit;
  uint32_t offset;
  size_t accessLimit;
  uint8_t* pKernelAddr;

  if (flags & ADMXRC2_IOFIXED) {
    accessLimit = local + width;
  } else {
    accessLimit = local + length;
  }

  if (flags & ADMXRC2_IOADAPTER) {
    pKernelAddr = (uint8_t*)pDevCtx->window.adapter.pKernelBase;
    if (NULL == pKernelAddr) {
      /* Adapter BAR not found, shouldn't happen, but it did... */
      /* TO DO - decide if return ADMXRC_INVALID_REGION or keep old behavior of ADMXRC_INVALID_PARAMETER */
      return ADMXRC2_INVALID_REGION;
    }
    if (accessLimit > pDevCtx->window.adapter.size) {
      /* Access length extends past end of adapter register BAR */
      /* TO DO - decide if return ADMXRC2_REGION_TOO_LARGE or keep old behavior of ADMXRC_INVALID_PARAMETER */
      return ADMXRC2_REGION_TOO_LARGE;
    }
    /* Access length for adapter register BAR was validated OK */
    *ppKernelAddr = pKernelAddr + local;
    return ADMXRC2_SUCCESS;
  } else {
    for (i = 0; i < pDevCtx->info.numWindow; i++) {
      pAccessInfo = &pDevCtx->window.access[i];
      if (0 == pAccessInfo->size) {
        /* Must be adapter register BAR with no local bus presence - ignore */
        continue;
      }
      localLimit = pAccessInfo->localBase + pAccessInfo->size;
      if (pAccessInfo->localBase <= local && local < localLimit) {
        offset = local - (uint32_t)pAccessInfo->localBase;
        pKernelAddr = (uint8_t*)pAccessInfo->pKernelBase + offset;
        if (accessLimit > localLimit) {
          /* Local window containing 'local' was found, but access length extends past end of window */
          /* TO DO - decide if return ADMXRC2_REGION_TOO_LARGE or keep old behavior of ADMXRC_INVALID_PARAMETER */
          return ADMXRC2_REGION_TOO_LARGE;
        }
        /* Found a local window containing 'local' and access length was validated OK */
        *ppKernelAddr = pKernelAddr;
        return ADMXRC2_SUCCESS;
      }
    }
    /* Local window containing 'local' was not found */
    /* TO DO - decide if return ADMXRC_INVALID_REGION or keep old behavior of ADMXRC_INVALID_PARAMETER */
    return ADMXRC2_INVALID_REGION;
  }
}

static void
cancelCallback(
  DfClientRequest* pReq,
  DfIoStatus reason)
{
  Context* pContext = (Context*)dfRequestGetContext(pReq);

  pContext->reason = reason;
  pContext->bCancel = TRUE;
  dfEventSignal(&pContext->event);
}

DfIoStatus
ioctlRead(
	Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx,
  DfClientRequest* pReq,
  void* pBuffer,
  unsigned int inSize)
{
#if defined(ADMXRC2_CHECK_FLAGS)
  static const uint32_t legalFlagsMask = ADMXRC2_IOFIXED | ADMXRC2_IOADAPTER;
#endif
  IOCTLS_ADMXRC2_READ* pIoctl = (IOCTLS_ADMXRC2_READ*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  Context context;
  uint32_t widthOrder, local, flags;
  size_t length, chunk, remaining;
  unsigned int width;
  uint8_t* pUserBuf;
  uint8_t* pKernelAddr;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }
  if (!admxrc2CheckAccessRights(pClCtx)) {
    return DfIoStatusError(ADMXRC2_ACCESS_DENIED);
  }

  widthOrder = pIoctl->in.width;
  if (widthOrder > ADMXRC2_IOWIDTH_64) {
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
  }
  width = 1U << widthOrder;
  flags = pIoctl->in.flags;
  local = pIoctl->in.local;
  pUserBuf = pIoctl->in.pBuffer;
  length = pIoctl->in.length;
#if defined(ADMXRC2_CHECK_FLAGS)
  if (0 != (flags & ~legalFlagsMask)) {
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
  }
#endif

  dfDebugPrint(6, ("ioctlRead: widthOrder=%lu flags=0x%lx local=0x%lx pUserBuf=%p length=%p\n",
    (unsigned long)widthOrder, (unsigned long)flags, (unsigned long)local, pUserBuf, (uintptr_t)length));

  status = findWindow(pDevCtx, width, flags, local, length, &pKernelAddr);
  if (ADMXRC2_SUCCESS != status) {
    return status;
  }

  context.bCancel = FALSE;
  if (length >= CANCEL_THRESHOLD) {
    dfRequestSetContext(pReq, &context);
    dfEventInit(&context.event);
    if (!dfRequestSetCallback(pReq, cancelCallback)) {
      return DfIoStatusError(ADMXRC2_CANCELLED); 
    }
  }

  DF_USER_SPACE_BEGIN(TAG1)

  dfUserBufferValidate(pUserBuf, length, TRUE, TAG1);

  /* TO DO - honour 'widthOrder' parameter */

  remaining = length;
  while (remaining) {
    chunk = (remaining > maxChunk) ? maxChunk : remaining;
    dfCopyIU(pUserBuf, pKernelAddr, chunk, TAG1);
    pUserBuf += chunk;
    pKernelAddr += chunk;
    remaining -= chunk;
    if (context.bCancel) {
      break;
    }
  }

  DF_USER_SPACE_EXCEPT(TAG1)

  status = DfIoStatusError(ADMXRC2_INVALID_PARAMETER); /* TODO - could return better status code here */

  DF_USER_SPACE_END(TAG1)

  if (length >= CANCEL_THRESHOLD) {
    if (!dfRequestClearCallback(pReq)) {
      dfEventWait(&context.event);
      status = context.reason;
    }
    dfEventUninit(&context.event);
  }

  return status;
}

#if DF_NEED_THUNK
DfIoStatus
ioctlReadThunk(
	Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx,
  DfClientRequest* pReq,
  void* pBuffer,
  unsigned int inSize)
{
  IOCTLS32_ADMXRC2_READ* pIoctl32 = (IOCTLS32_ADMXRC2_READ*)pBuffer;
  IOCTLS_ADMXRC2_READ ioctl;

  if (NULL == pIoctl32 || sizeof(pIoctl32->in) != inSize) {
    return DfIoStatusInvalid;
  }

  ioctl.in.width = pIoctl32->in.width;
  ioctl.in.flags = pIoctl32->in.flags;
  ioctl.in.local = pIoctl32->in.local;
  ioctl.in.pBuffer = dfThunkPtr(pIoctl32->in.pBuffer);
  ioctl.in.length = (size_t)pIoctl32->in.length;
  return ioctlRead(pDevCtx, pClCtx, pReq, &ioctl, sizeof(ioctl.in));
}
#endif

DfIoStatus
ioctlWrite(
	Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx,
  DfClientRequest* pReq,
  void* pBuffer,
  unsigned int inSize)
{
#if defined(ADMXRC2_CHECK_FLAGS)
  static const uint32_t legalFlagsMask = ADMXRC2_IOFIXED | ADMXRC2_IOADAPTER;
#endif
  IOCTLS_ADMXRC2_WRITE* pIoctl = (IOCTLS_ADMXRC2_WRITE*)pBuffer;
  DfIoStatus status = DfIoStatusSuccess;
  Context context;
  uint32_t widthOrder, local, flags;
  size_t length, chunk, remaining;
  unsigned int width;
  const uint8_t* pUserBuf;
  uint8_t* pKernelAddr;

  if (NULL == pIoctl || sizeof(pIoctl->in) != inSize) {
    return DfIoStatusInvalid;
  }
  if (!admxrc2CheckAccessRights(pClCtx)) {
    return DfIoStatusError(ADMXRC2_ACCESS_DENIED);
  }

  widthOrder = pIoctl->in.width;
  if (widthOrder > ADMXRC2_IOWIDTH_64) {
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
  }
  width = 1U << widthOrder;
  flags = pIoctl->in.flags;
  local = pIoctl->in.local;
  pUserBuf = (const uint8_t*)pIoctl->in.pData;
  length = pIoctl->in.length;
#if defined(ADMXRC2_CHECK_FLAGS)
  if (0 != (flags & ~legalFlagsMask)) {
    return DfIoStatusError(ADMXRC2_INVALID_PARAMETER);
  }
#endif

  dfDebugPrint(6, ("ioctlWrite: widthOrder=%lu flags=0x%lx local=0x%lx pUserBuf=%p length=%p\n",
    (unsigned long)widthOrder, (unsigned long)flags, (unsigned long)local, pUserBuf, (uintptr_t)length));

  status = findWindow(pDevCtx, width, flags, local, length, &pKernelAddr);
  if (ADMXRC2_SUCCESS != status) {
    return status;
  }

  context.bCancel = FALSE;
  if (length >= CANCEL_THRESHOLD) {
    dfRequestSetContext(pReq, &context);
    dfEventInit(&context.event);
    if (!dfRequestSetCallback(pReq, cancelCallback)) {
      return DfIoStatusError(ADMXRC2_CANCELLED); 
    }
  }

  DF_USER_SPACE_BEGIN(TAG1)

  dfUserBufferValidate(pUserBuf, length, FALSE, TAG1);

  remaining = length;
  while (remaining) {
    chunk = (remaining > maxChunk) ? maxChunk : remaining;
    dfCopyUI(pKernelAddr, pUserBuf, chunk, TAG1);
    pUserBuf += chunk;
    pKernelAddr += chunk;
    remaining -= chunk;
    if (context.bCancel) {
      break;
    }
  }

  DF_USER_SPACE_EXCEPT(TAG1)

  status = DfIoStatusError(ADMXRC2_INVALID_PARAMETER); /* TODO - could return better status code here */

  DF_USER_SPACE_END(TAG1)

  if (length >= CANCEL_THRESHOLD) {
    if (!dfRequestClearCallback(pReq)) {
      dfEventWait(&context.event);
      status = context.reason;
    }
    dfEventUninit(&context.event);
  }

  return status;
}

#if DF_NEED_THUNK
DfIoStatus
ioctlWriteThunk(
	Admxrc2DeviceContext* pDevCtx,
	Admxrc2ClientContext* pClCtx,
  DfClientRequest* pReq,
  void* pBuffer,
  unsigned int inSize)
{
  IOCTLS32_ADMXRC2_WRITE* pIoctl32 = (IOCTLS32_ADMXRC2_WRITE*)pBuffer;
  IOCTLS_ADMXRC2_WRITE ioctl;

  if (NULL == pIoctl32 || sizeof(pIoctl32->in) != inSize) {
    return DfIoStatusInvalid;
  }

  ioctl.in.width = pIoctl32->in.width;
  ioctl.in.flags = pIoctl32->in.flags;
  ioctl.in.local = pIoctl32->in.local;
  ioctl.in.pData = dfThunkConstPtr(pIoctl32->in.pData);
  ioctl.in.length = (size_t)pIoctl32->in.length;
  return ioctlWrite(pDevCtx, pClCtx, pReq, &ioctl, sizeof(ioctl.in));
}
#endif