/*
** File: model_adcpxrc4lx.c
** Project: ADMXRC2 module driver
** Purpose: Code specific to ADCP-XRC-4LX, including the function to bootstrap
**          the device context.
**
** (C) Copyright Alpha Data 2014
*/

#include <df.h>

#include <core/adb1.h>
#include <core/adcpxrc4lx.h>
#include "device.h"
#include "info.h"
#include "model_boot.h"

static ADMXRC2_STATUS
getBankInfo(
  Admxrc2DeviceContext* pDevCtx,
  unsigned int bankIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETBANKINFO* pIoctl)
{
  VpdAdcpxrc4lxRev0* pVpd = (VpdAdcpxrc4lxRev0*)pVpdBuffer;
  unsigned int order;

  if (bankIndex < DF_ARRAY_LENGTH(pVpd->sdram)) {
    order = pVpd->sdram[bankIndex].order;
    pIoctl->out.width = 64U;
    pIoctl->out.typeMask = ADMXRC2_RAM_SDRAM_DDR2;
  } else {
    bankIndex -= (unsigned int)DF_ARRAY_LENGTH(pVpd->sdram);
    if (bankIndex < DF_ARRAY_LENGTH(pVpd->zbt)) {
      order = pVpd->zbt[bankIndex].order;
      pIoctl->out.width = 32U;
      pIoctl->out.typeMask = ADMXRC2_RAM_ZBTP;
    } else {
      return ADMXRC2_INVALID_INDEX;
    }
  }
  pIoctl->out.sizeWords = (uint32_t)(0x1U << order);
  pIoctl->out.bFitted = order ? TRUE : FALSE;

  return ADMXRC2_SUCCESS;
}

static void
getCardInfo(
	Admxrc2DeviceContext* pDevCtx,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETCARDINFO* pIoctl)
{
  VpdAdcpxrc4lxRev0* pVpd = (VpdAdcpxrc4lxRev0*)pVpdBuffer;
  unsigned int i, j;

  pIoctl->out.cardId = pVpd->cardId;
  pIoctl->out.fpgaType = pVpd->fpgaType;
  pIoctl->out.serialNumber = pVpd->serialNumber;
  pIoctl->out.numMemoryBank = 3;
  pIoctl->out.memoryBankFitted = 0;
  j = 0;
  for (i = 0; i < DF_ARRAY_LENGTH(pVpd->sdram); i++, j++) {
    if (pVpd->sdram[i].order) {
      /* Bank is fitted */
      pIoctl->out.memoryBankFitted |= (0x1U << j);
    }
  }
  for (i = 0; i < DF_ARRAY_LENGTH(pVpd->zbt); i++) {
    if (pVpd->zbt[i].order) {
      /* Bank is fitted */
      pIoctl->out.memoryBankFitted |= (0x1U << j);
    }
  }
  pIoctl->out.logicRevision = pVpd->cpldRev;
  pIoctl->out.boardRevision = pVpd->pcbRev;
}

static ADMXRC2_STATUS
getFpgaInfo(
	Admxrc2DeviceContext* pDevCtx,
  unsigned int targetIndex,
  const void* pVpdBuffer,
  IOCTLS_ADMXRC2_GETFPGAINFO* pIoctl)
{
  VpdAdcpxrc4lxRev0* pVpd = (VpdAdcpxrc4lxRev0*)pVpdBuffer;

  if (0 != targetIndex) {
    return ADMXRC2_INVALID_INDEX;
  }

  dfStrCpy(pIoctl->out.package, DF_ARRAY_LENGTH(pIoctl->out.package), "FF1148");
  pIoctl->out.type = (ADMXRC2_FPGA_TYPE)pVpd->fpgaType;
  pIoctl->out.speed = (uint32_t)-1;
  pIoctl->out.stepping = (uint32_t)-1;
  pIoctl->out.tempGrade = (uint32_t)-1;
  pIoctl->out.scd = pVpd->scd;

  return ADMXRC2_SUCCESS;
}

void
admxrc2BootstrapAdcpxrc4lx(
	Admxrc2DeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("admxrc2BootstrapAdcpxrc4lx: entered\n"));

  pDevCtx->methods.pGetBankInfo = getBankInfo;
  pDevCtx->methods.pGetCardInfo = getCardInfo;
  pDevCtx->methods.pGetFpgaInfo = getFpgaInfo;
  pDevCtx->bReverseSelectMapData = FALSE; /* Expecting already bit-reversed by user-space function ADMXRC2_LoadBitstream */
  pDevCtx->localAddressLimit = (uint64_t)(0x1ULL << 32U);
  pDevCtx->localWidth = 4U;
  pDevCtx->vpdWidth = 4U;
  pDevCtx->configureDmaAddress = 0x800020U; /* TODO - should make this dynamic, i.e. based on current value of LS1B register */
  pDevCtx->configureDmaMode =
    ADB1_DMA_CFG_32BIT | ADB1_DMA_CFG_USE_READY | ADB1_DMA_CFG_USE_BTERM |
    ADB1_DMA_CFG_ENABLE_BURST | ADB1_DMA_CFG_FIXED_LOCAL;
}
