/*
** File: main.c  
** Project: ADMXRC2 module driver
** Purpose: OS-independent entry point into ADMXRC2 module driver
**
** (C) Copyright Alpha Data 2013-2014
*/

#include "clock.h"
#include "combuf.h"
#include "coop.h"
#include "debugts.h"
#include "device.h"
#include "dma.h"
#include "event.h"
#include "flash.h"
#include "fpga.h"
#include "identify.h"
#include "info.h"
#include "lock.h"
#include "main.h"
#include "model_boot.h"
#include "pio.h"
#include "sensor.h"
#include "space.h"
#include "vpd.h"
#if defined(TODO)
#include "derror.h"
#include "diagnostic.h"
#include "dmabus.h"
#include "module.h"
#endif

#if !ADB3_MONOLITHIC
const char g_dfDriverName[] = "admxrc2";
#endif

/* User-mode interface for ADMXRC2_* API functions */
static const char g_interfaceClassName[] = "admxrc2";
static DfInterfaceClass* g_pInterfaceClass = NULL;
static DfInterfaceMethods g_interfaceMethods;

static DfIoStatus
onIoctl(
  DfDeviceObject* pDevObj,
  DfClientObject* pClObj,
  DfClientRequest* pReq,
  unsigned int ioctlCode,
  void* pIoctl,
  unsigned int inSize,
  unsigned int outSize)
{
  DfIoStatus status = DfIoStatusSuccess;
  Admxrc2ClientContext* pClCtx = (Admxrc2ClientContext*)dfClientGetContext(pClObj);
  Admxrc2DeviceContext* pDevCtx = (Admxrc2DeviceContext*)pClCtx->pDevCtx;

  dfDebugPrint(3, ("onIoctl(admxrc2): entered pDevObj=%p pClObj=%p pReq=%p ioctlCode=0x%lx pIoctl=%p inSize=0x%lx outSize=0x%lx\n",
    pDevObj, pClObj, pReq, (unsigned long)ioctlCode, pIoctl, (unsigned long)inSize, (unsigned long)outSize));

  switch (ioctlCode) {
  case ADMXRC2_IOCTLCODE_CANCELWAITFORINTERRUPT:
    status = ioctlCancelWaitForInterrupt(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_CONFIGURE:
    status = ioctlConfigure(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_CONFIGUREDMA:
    status = ioctlConfigureDma(pDevCtx, pClCtx, pReq, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_DODMA:
    status = ioctlDoDma(pDevCtx, pClCtx, pReq, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_DODMAIMMEDIATE:
    status = ioctlDoDmaImmediate(pDevCtx, pClCtx, pReq, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_ERASEFLASH:
    status = ioctlEraseFlash(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_GETBANKINFO:
    status = ioctlGetBankInfo(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETCARDINFO:
    status = ioctlGetCardInfo(pDevCtx, pIoctl, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETDRIVERVERSION:
    status = ioctlGetDriverVersion(pDevCtx, pIoctl, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETFLASHBLOCKINFO:
    status = ioctlGetFlashBlockInfo(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETFLASHINFO:
    status = ioctlGetFlashInfo(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETFPGAINFO:
    status = ioctlGetFpgaInfo(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETSENSORINFO:
    status = ioctlGetSensorInfo(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETSPACECONFIG:
    status = ioctlGetSpaceConfig(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETSPACEINFO:
    status = ioctlGetSpaceInfo(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETCLOCKFREQUENCY:
    status = ioctlGetClockFrequency(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_LOCK:
    status = ioctlLock(pDevCtx, pClCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_READ:
    status = ioctlRead(pDevCtx, pClCtx, pReq, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_READCONFIG:
    status = ioctlReadConfig(pDevCtx, pClCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_READFLASH:
    status = ioctlReadFlash(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_READSENSOR:
    status = ioctlReadSensor(pDevCtx, pIoctl, inSize, outSize);
    break;

#if DF_HAS_USER_EVENT_HANDLE
  case ADMXRC2_IOCTLCODE_REGISTEREVENT:
    status = ioctlRegisterEvent(pDevCtx, pClCtx, pIoctl, inSize);
    break;
#endif

  case ADMXRC2_IOCTLCODE_SETCLOCKFREQUENCY:
    status = ioctlSetClockFrequency(pDevCtx, pClCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_SETCOOPLEVEL:
    status = ioctlSetCoopLevel(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_SETSPACECONFIG:
    status = ioctlSetSpaceConfig(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_UNCONFIGURE:
    status = ioctlUnconfigure(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_UNLOCK:
    status = ioctlUnlock(pDevCtx, pClCtx, pIoctl, inSize);
    break;

#if DF_HAS_USER_EVENT_HANDLE
  case ADMXRC2_IOCTLCODE_UNREGISTEREVENT:
    status = ioctlUnregisterEvent(pDevCtx, pClCtx, pIoctl, inSize);
    break;
#endif

  case ADMXRC2_IOCTLCODE_WAITFORINTERRUPT:
    status = ioctlWaitForInterrupt(pDevCtx, pClCtx, pReq, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_WRITE:
    status = ioctlWrite(pDevCtx, pClCtx, pReq, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_WRITECONFIG:
    status = ioctlWriteConfig(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_WRITEFLASH:
    status = ioctlWriteFlash(pDevCtx, pClCtx, pReq, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_GETCOMMONBUFFERINFO:
    status = ioctlGetCommonBufferInfo(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_SYNCCOMMONBUFFER:
    status = ioctlSyncCommonBuffer(pDevCtx, pClCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETDEBUGTSINFO:
    status = ioctlGetDebugTSInfo(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETDEBUGTSSTATUS:
    status = ioctlGetDebugTSStatus(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_READDEBUGTS:
    status = ioctlReadDebugTS(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_RESETDEBUGTS:
    status = ioctlResetDebugTS(pDevCtx, pClCtx, pIoctl, inSize);
    break;

#if defined(TODO)
  case ADMXRC3_IOCTLCODE_GETCOMMONBUFFERCOUNT:
    status = ioctlGetCommonBufferCount(pDevCtx, pIoctl, outSize);
    break;

  case ADMXRC3_IOCTLCODE_GETMODULEINFO:
    status = ioctlGetModuleInfo(pDevCtx, pIoctl, inSize);
    break;

  case ADMXRC3_IOCTLCODE_SYNCFLASH:
    status = ioctlSyncFlash(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC3_IOCTLCODE_GETDEVICESTATUS:
    status = ioctlGetDeviceStatus(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC3_IOCTLCODE_CLEARDEVICEERRORS:
    status = ioctlClearDeviceErrors(pDevCtx, pClCtx);
    break;
#endif

  default:
    status = DfIoStatusInvalidCode;
    break;
  }

  if (DfIoStatusDeferred != status) {
    dfRequestComplete(pReq, status);
  }
  return status;
}

#if DF_NEED_THUNK
static DfIoStatus
onIoctlThunk(
  DfDeviceObject* pDevObj,
  DfClientObject* pClObj,
  DfClientRequest* pReq,
  unsigned int ioctlCode,
  void* pIoctl,
  unsigned int inSize,
  unsigned int outSize)
{
  DfIoStatus status = DfIoStatusSuccess;
  Admxrc2ClientContext* pClCtx = (Admxrc2ClientContext*)dfClientGetContext(pClObj);
  Admxrc2DeviceContext* pDevCtx = (Admxrc2DeviceContext*)pClCtx->pDevCtx;

  dfDebugPrint(3, ("onIoctlThunk: entered pDevObj=%p pClObj=%p pReq=%p ioctlCode=0x%lx pIoctl=%p inSize=0x%lx outSize=0x%lx\n",
    pDevObj, pClObj, pReq, (unsigned long)ioctlCode, pIoctl, (unsigned long)inSize, (unsigned long)outSize));

  switch (ioctlCode) {
  case ADMXRC2_IOCTLCODE_CANCELWAITFORINTERRUPT:
    status = ioctlCancelWaitForInterrupt(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_CONFIGURE:
    status = ioctlConfigureThunk(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_CONFIGUREDMA:
    status = ioctlConfigureDmaThunk(pDevCtx, pClCtx, pReq, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_DODMA:
    status = ioctlDoDmaThunk(pDevCtx, pClCtx, pReq, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_DODMAIMMEDIATE:
    status = ioctlDoDmaImmediateThunk(pDevCtx, pClCtx, pReq, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_ERASEFLASH:
    status = ioctlEraseFlash(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_GETBANKINFO:
    status = ioctlGetBankInfo(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETCARDINFO:
    status = ioctlGetCardInfo(pDevCtx, pIoctl, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETDRIVERVERSION:
    status = ioctlGetDriverVersion(pDevCtx, pIoctl, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETFLASHINFO:
    status = ioctlGetFlashInfo(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETFLASHBLOCKINFO:
    status = ioctlGetFlashBlockInfo(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETFPGAINFO:
    status = ioctlGetFpgaInfo(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETSPACECONFIG:
    status = ioctlGetSpaceConfig(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETSPACEINFO:
    status = ioctlGetSpaceInfo(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETCLOCKFREQUENCY:
    status = ioctlGetClockFrequency(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_LOCK:
    status = ioctlLockThunk(pDevCtx, pClCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_READ:
    status = ioctlReadThunk(pDevCtx, pClCtx, pReq, pIoctl, inSize);
    break;

# if DF_HAS_USER_EVENT_HANDLE
  case ADMXRC2_IOCTLCODE_REGISTEREVENT:
    status = ioctlRegisterEventThunk(pDevCtx, pClCtx, pIoctl, inSize);
    break;
# endif

  case ADMXRC2_IOCTLCODE_READCONFIG:
    status = ioctlReadConfig(pDevCtx, pClCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_READFLASH:
    status = ioctlReadFlashThunk(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_SETCLOCKFREQUENCY:
    status = ioctlSetClockFrequency(pDevCtx, pClCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_SETCOOPLEVEL:
    status = ioctlSetCoopLevel(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_SETSPACECONFIG:
    status = ioctlSetSpaceConfig(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_UNCONFIGURE:
    status = ioctlUnconfigure(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_UNLOCK:
    status = ioctlUnlock(pDevCtx, pClCtx, pIoctl, inSize);
    break;

# if DF_HAS_USER_EVENT_HANDLE
  case ADMXRC2_IOCTLCODE_UNREGISTEREVENT:
    status = ioctlUnregisterEventThunk(pDevCtx, pClCtx, pIoctl, inSize);
    break;
# endif

  case ADMXRC2_IOCTLCODE_WAITFORINTERRUPT:
    status = ioctlWaitForInterrupt(pDevCtx, pClCtx, pReq, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_WRITE:
    status = ioctlWriteThunk(pDevCtx, pClCtx, pReq, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_WRITECONFIG:
    status = ioctlWriteConfig(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_WRITEFLASH:
    status = ioctlWriteFlashThunk(pDevCtx, pClCtx, pReq, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_GETCOMMONBUFFERINFO:
    status = ioctlGetCommonBufferInfoThunk(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_SYNCCOMMONBUFFER:
    status = ioctlSyncCommonBufferThunk(pDevCtx, pClCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETDEBUGTSINFO:
    status = ioctlGetDebugTSInfoThunk(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_GETDEBUGTSSTATUS:
    status = ioctlGetDebugTSStatusThunk(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC2_IOCTLCODE_READDEBUGTS:
    status = ioctlReadDebugTSThunk(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC2_IOCTLCODE_RESETDEBUGTS:
    status = ioctlResetDebugTS(pDevCtx, pClCtx, pIoctl, inSize);
    break;

#if defined(TODO)
  case ADMXRC3_IOCTLCODE_GETCOMMONBUFFERCOUNT:
    status = ioctlGetCommonBufferCount(pDevCtx, pIoctl, outSize);
    break;

  case ADMXRC3_IOCTLCODE_GETMODULEINFO:
    status = ioctlGetModuleInfoThunk(pDevCtx, pIoctl, inSize);
    break;

  case ADMXRC3_IOCTLCODE_SYNCFLASH:
    status = ioctlSyncFlash(pDevCtx, pClCtx, pIoctl, inSize);
    break;

  case ADMXRC3_IOCTLCODE_GETDEVICESTATUS:
    status = ioctlGetDeviceStatus(pDevCtx, pIoctl, inSize, outSize);
    break;

  case ADMXRC3_IOCTLCODE_CLEARDEVICEERRORS:
    status = ioctlClearDeviceErrors(pDevCtx, pClCtx);
    break;
#endif

  default:
    status = DfIoStatusInvalidCode;
    break;
  }

  if (DfIoStatusDeferred != status) {
    dfRequestComplete(pReq, status);
  }
  return status;
}
#endif

static DfIoStatus
onOpen(
	DfDeviceObject* pDevObj,
  DfClientObject* pClObj,
  DfClientRequest* pReq,
  DfInterface* pInterface,
  boolean_t bPrivileged)
{
  Admxrc2DeviceContext* pDevCtx;
  Admxrc2ClientContext* pClCtx = NULL;
  unsigned int i;
  DfSpinLockFlags f;
  DfIoStatus status = DfIoStatusSuccess;

  pDevCtx = (Admxrc2DeviceContext*)dfInterfaceGetContext(pInterface);

  pClCtx = (Admxrc2ClientContext*)dfMalloc(sizeof(Admxrc2ClientContext));
  if (NULL == pClCtx) {
    status = DfIoStatusNoMemory;
    goto done;
  }
  
  pClCtx->pDevCtx = pDevCtx;
  pClCtx->bPrivileged = bPrivileged;
  adb3CoreClientInit(&pClCtx->coreClient);
  pClCtx->bCoopLevelIsSet = FALSE;
  for (i = 0; i < pDevCtx->info.numFlashBank; i++) {
    pClCtx->bNeedFlashSync[i] = FALSE;
  }
  pClCtx->bNeedVpdSync = FALSE;

#if !defined(ADMXRC2_LOCKED_BUFFERS_GLOBAL)
  /* Initialize DMA buffer stuff */
  dfSpinLockInit(&pClCtx->lockedBuffers.lock);
  pClCtx->lockedBuffers.stack.top = 0;
  for (i = 0; i < MAX_NUM_LOCKED_USER_SPACE_BUFFER; i++) {
    /* Handle actually returned to user-space app is 'i + 1' */
    pClCtx->lockedBuffers.stack.free[i] = i;
  }
#endif

  pClCtx->hDisableSleep = dfSystemSleepDisable(pDevCtx->pDevObj);
  if (NULL == pClCtx->hDisableSleep) {
    status = DfIoStatusNoMemory;
    goto done;
  }

  f = dfSpinLockGet(&pDevCtx->powerMgmt.lock);
  if (pDevCtx->powerMgmt.bDeferOpen) {
    dfDebugPrint(1, ("onOpen: deferring open\n"));
    status = DfIoStatusDeferred;
    dfRequestPending(pReq);
    pClCtx->pOpenRequest = pReq;
    dfListAddToTail(&pDevCtx->powerMgmt.waitingOpen, &pClCtx->waitNode);
    dfSpinLockPut(&pDevCtx->powerMgmt.lock, f);
  } else {
    pDevCtx->powerMgmt.openCount++;
    dfSpinLockPut(&pDevCtx->powerMgmt.lock, f);
  }

done:
  if (status != DfIoStatusDeferred) {
    if (DfIoStatusIsError(status)) {
      if (NULL != pClCtx->hDisableSleep) {
        dfSystemSleepReenable(pDevCtx->pDevObj, pClCtx->hDisableSleep);
      }
      if (NULL != pClCtx) {
        dfFree(pClCtx);
      }
    } else {
      dfClientSetContext(pClObj, pClCtx);
    }
    dfRequestComplete(pReq, status);
  }
  return status;
}

static void
onCleanup(
	DfDeviceObject* pDevObj,
  DfClientObject* pClObj)
{
  Admxrc2ClientContext* pClCtx = (Admxrc2ClientContext*)dfClientGetContext(pClObj);
  Admxrc2DeviceContext* pDevCtx = (Admxrc2DeviceContext*)pClCtx->pDevCtx;
  unsigned int i;
  DfSpinLockFlags f;

  /* Clean up DMA transfers, i.e. cancel any that are still ongoing */
  cleanupDma(pDevCtx, pClCtx);

  /* If this client still owns any target FPGAs, release ownership */
  for (i = 0; i < pDevCtx->info.numTargetFpga; i++) {
    f = dfSpinLockGet(&pDevCtx->fpga[i].owner.lock);
    if (pClCtx == pDevCtx->fpga[i].owner.pClCtx) {
      pDevCtx->fpga[i].owner.pClCtx = NULL;
    }
    dfSpinLockPut(&pDevCtx->fpga[i].owner.lock, f);
  }
}

static void
onClose(
	DfDeviceObject* pDevObj,
  DfClientObject* pClObj)
{
  Admxrc2ClientContext* pClCtx = (Admxrc2ClientContext*)dfClientGetContext(pClObj);
  Admxrc2DeviceContext* pDevCtx = (Admxrc2DeviceContext*)pClCtx->pDevCtx;
  unsigned int i;
  DfSpinLockFlags f;

  for (i = 0; i < pDevCtx->info.numFlashBank; i++) {
    if (pClCtx->bNeedFlashSync[i]) {
      syncFlashOnClose(pDevCtx, pClCtx, i);
      pClCtx->bNeedFlashSync[i] = FALSE;
    }
  }
  if (pClCtx->bNeedVpdSync) {
    syncVpdOnClose(pDevCtx, pClCtx);
    pClCtx->bNeedVpdSync = FALSE;
  }

#if DF_HAS_USER_EVENT_HANDLE
  /* Clean up lists of registered user events */
  cleanupUserEvents(pDevCtx, pClCtx);
#endif

  /* Clean up locked user-space buffers */
  cleanupLockedBuffers(pDevCtx, pClCtx);

  /* Inform core device that it's no longer in use by this client. */
  if (pClCtx->bPrivileged) {
    pDevCtx->coreInterface.pSetCoopLevel(pDevCtx->pCoreContext, &pClCtx->coreClient, CoreCoopLevelUnset);
  }

  f = dfSpinLockGet(&pDevCtx->powerMgmt.lock);
  pDevCtx->powerMgmt.openCount--;
  dfSpinLockPut(&pDevCtx->powerMgmt.lock, f);

  dfSystemSleepReenable(pDevCtx->pDevObj, pClCtx->hDisableSleep);

  /* Free the memory associated with the client's context */
  if (NULL != pClCtx) {
    dfFree(pClCtx);
    dfClientSetContext(pClObj, NULL);
  }
}

static void
freeDeviceContext(
	Admxrc2DeviceContext* pDevCtx)
{
  if (pDevCtx->userMode.bRegistered) {
    if (!dfInterfaceUnregister(pDevCtx->pDevObj, &pDevCtx->userMode.iface)) {
      dfDebugPrint(0, ("*** freeDeviceContext: failed to unregister user-mode interface\n"));
    } else {
      dfDebugPrint(1, ("freeDeviceContext: unregistered user-mode interface\n"));
      pDevCtx->userMode.bRegistered = FALSE;
    }
  }

  if (pDevCtx->interruptWait.bTicketPoolValid) {
    dfDebugPrint(1, ("freeDeviceContext: uninitializing interrupt wait ticket pool\n"));
    dfPoolUninit(&pDevCtx->interruptWait.ticketPool);
    pDevCtx->interruptWait.bTicketPoolValid = FALSE;
  }

  if (pDevCtx->dma.bTicketPoolValid) {
    dfDebugPrint(1, ("freeDeviceContext: uninitializing DMA ticket pool\n"));
    dfPoolUninit(&pDevCtx->dma.ticketPool);
    pDevCtx->dma.bTicketPoolValid = FALSE;
  }

  if (pDevCtx->clockGenerator.bTicketPoolValid) {
    dfDebugPrint(1, ("freeDeviceContext: uninitializing clock generator ticket pool\n"));
    dfPoolUninit(&pDevCtx->clockGenerator.ticketPool);
    pDevCtx->clockGenerator.bTicketPoolValid = FALSE;
  }
}

static void
stopDevice(
	Admxrc2DeviceContext* pDevCtx)
{
  unsigned int i;
  CoreWindowInfo windowInfo;

  if (NULL != pDevCtx->powerMgmt.hDisableSleep) {
    dfSystemSleepReenable(pDevCtx->pDevObj, pDevCtx->powerMgmt.hDisableSleep);
    pDevCtx->powerMgmt.hDisableSleep = NULL;
    dfDebugPrint(1, ("stopDevice: removed system sleep disable hint\n"));
  }

  if (pDevCtx->userMode.bEnabled) {
    if (!dfInterfaceEnable(pDevCtx->pDevObj, &pDevCtx->userMode.iface, FALSE)) {
      dfDebugPrint(0, ("*** stopDevice: failed to disable user-mode interface\n"));
    } else {
      dfDebugPrint(1, ("stopDevice: disabled user-mode interface\n"));
      pDevCtx->userMode.bEnabled = FALSE;
    }
  }

  if (pDevCtx->bCoreInterfaceImpd) {
    for (i = 0; i < pDevCtx->info.numCommonBuffer; i++) {
      dfMmappableRegionUnregister(pDevCtx->pDevObj, i + ADMXRC2_REGION_TAG_COMMONBUFFER);
      dfDebugPrint(1, ("stopDevice: unregistered mappable common buffer %u", i));
    }

    for (i = 0; i < pDevCtx->info.numWindow; i++) {
      uint64_t base;

      pDevCtx->coreInterface.pGetWindowInfo(pDevCtx->pCoreContext, i, &windowInfo);
      base = (uint64_t)(uintptr_t)windowInfo.pKernelBase;
      dfMmappableRegionUnregister(pDevCtx->pDevObj, i + ADMXRC2_REGION_TAG_WINDOW);
      dfDebugPrint(1, ("stopDevice: unregistered mappable region %u, base=0x%08lx_%08lx\n", i, dfSplitUint64(base)));
    }

    for (i = 0; i < pDevCtx->info.numTargetFpga; i++) {
#if defined(TODO)
      if (pDevCtx->fpga[i].alert.bNotificationRegistered) {
        if (pDevCtx->coreInterface.pNotificationUnreg(pDevCtx->pCoreContext, &pDevCtx->fpga[i].alert.notification.coreNotification)) {
          pDevCtx->fpga[i].alert.bNotificationRegistered = FALSE;
          dfDebugPrint(1, ("stopDevice: unregistered for FPGA_ALERT(%u) notification\n", i));
        } else {
          dfDebugPrint(1, ("*** stopDevice: failed to unregister for FPGA_ALERT(%u) notification\n", i));
        }
      }
#endif

      if (pDevCtx->fpga[i].interrupt.bNotificationRegistered) {
        if (pDevCtx->coreInterface.pNotificationUnreg(pDevCtx->pCoreContext, &pDevCtx->fpga[i].interrupt.notification.coreNotification)) {
          pDevCtx->fpga[i].interrupt.bNotificationRegistered = FALSE;
          dfDebugPrint(1, ("stopDevice: unregistered for FPGA_INTERRUPT(%u) notification\n", i));
        } else {
          dfDebugPrint(1, ("*** stopDevice: failed to unregister for FPGA_INTERRUPT(%u) notification\n", i));
        }
      }
    }

    if (pDevCtx->sensor.bNotificationRegistered) {
      if (pDevCtx->coreInterface.pNotificationUnreg(pDevCtx->pCoreContext, &pDevCtx->sensor.notification.coreNotification)) {
        pDevCtx->sensor.bNotificationRegistered = FALSE;
        dfDebugPrint(1, ("stopDevice: unregistered for SENSOR_POLL notification\n"));
      } else {
        dfDebugPrint(1, ("*** stopDevice: failed to unregister for SENSOR_POLL notification\n"));
      }
    }

    for (i = 0; i < pDevCtx->info.numFlashBank; i++) {
      if (NULL != pDevCtx->flash[i].pStagingBuffer) {
        dfFree(pDevCtx->flash[i].pStagingBuffer);
        pDevCtx->flash[i].pStagingBuffer = NULL;
        dfDebugPrint(1, ("stopDevice: freed Flash bank %lu staging buffer\n", (unsigned long)i));
      }
    }

    dfReleaseInterface(pDevCtx->pDevObj, g_coreInterfaceName);
    pDevCtx->bCoreInterfaceImpd = FALSE;
    dfDebugPrint(1, ("stopDevice: released core interface\n"));
  }
}

boolean_t
admxrc2OnAdd(
	DfDeviceObject* pDevObj,
	Admxrc2DeviceContext* pDevCtx)
{
  unsigned int i;
  uint32_t val32;

  dfDebugPrint(1, ("admxrc2OnAdd: entered\n"));

  /* Initialize the device context */
	pDevCtx->pDevObj = pDevObj;
  pDevCtx->bCoreInterfaceImpd = FALSE;
  pDevCtx->userMode.bRegistered = FALSE;
  pDevCtx->userMode.bEnabled = FALSE;
  pDevCtx->dma.bTicketPoolValid = FALSE;
  pDevCtx->clockGenerator.bTicketPoolValid = FALSE;
  pDevCtx->interruptWait.bTicketPoolValid = FALSE;
  for (i = 0; i < MAX_NUM_TARGET_FPGA; i++) {
    dfSpinLockInit(&pDevCtx->fpga[i].owner.lock);
    pDevCtx->fpga[i].owner.pClCtx = NULL;
    pDevCtx->fpga[i].interrupt.bNotificationRegistered = FALSE;
#if DF_HAS_USER_EVENT_HANDLE
    dfSpinLockInit(&pDevCtx->fpga[i].interrupt.eventList.lock);
    pDevCtx->fpga[i].interrupt.eventList.pHead = NULL;
    pDevCtx->fpga[i].interrupt.eventList.pTail = NULL;
#endif
    dfSpinLockInit(&pDevCtx->fpga[i].interrupt.waitList.lock);
    dfListInit(&pDevCtx->fpga[i].interrupt.waitList.header);
    pDevCtx->fpga[i].interrupt.bInterruptFlag = FALSE;
#if defined(TODO)
    pDevCtx->fpga[i].alert.bNotificationRegistered = FALSE;
#if DF_HAS_USER_EVENT_HANDLE
    dfSpinLockInit(&pDevCtx->fpga[i].alert.eventList.lock);
    pDevCtx->fpga[i].alert.eventList.pHead = NULL;
    pDevCtx->fpga[i].alert.eventList.pTail = NULL;
#endif
    dfSpinLockInit(&pDevCtx->fpga[i].alert.waitList.lock);
    dfListInit(&pDevCtx->fpga[i].alert.waitList.header);
#endif
    pDevCtx->sensor.bNotificationRegistered = FALSE;
  }
  for (i = 0; i < MAX_NUM_FLASH_BANK; i++) {
    pDevCtx->flash[i].pStagingBuffer = NULL;
  }
  dfSpinLockInit(&pDevCtx->powerMgmt.lock);
  pDevCtx->powerMgmt.openCount = 0;
  pDevCtx->powerMgmt.scheme = PowerMgmtNotActive;
  pDevCtx->powerMgmt.bDeferOpen = FALSE;
  dfListInit(&pDevCtx->powerMgmt.waitingOpen);

  /* From here on, things may fail */

  if (!dfPoolInit(&pDevCtx->clockGenerator.ticketPool, pDevObj, "clockTkt", ClockGeneratorTicket)) {
    dfDebugPrint(0, ("*** admxrc2OnAdd: failed to initialize clock generator ticket pool\n"));
    goto failed;
  }
  pDevCtx->clockGenerator.bTicketPoolValid = TRUE;
  dfDebugPrint(1, ("admxrc2OnAdd: initialized clock generator ticket pool\n"));
  
  if (!dfPoolInit(&pDevCtx->dma.ticketPool, pDevObj, "dmaTkt", DmaTicketUnion)) {
    dfDebugPrint(0, ("*** admxrc2OnAdd: failed to initialize DMA ticket pool\n"));
    goto failed;
  }
  pDevCtx->dma.bTicketPoolValid = TRUE;
  dfDebugPrint(1, ("admxrc2OnAdd: initialized DMA ticket pool\n"));
  
  if (!dfPoolInit(&pDevCtx->interruptWait.ticketPool, pDevObj, "intWtTkt", InterruptWaitTicket)) {
    dfDebugPrint(0, ("*** admxrc2OnAdd: failed to initialize notification ticket pool\n"));
    goto failed;
  }
  pDevCtx->interruptWait.bTicketPoolValid = TRUE;
  dfDebugPrint(1, ("admxrc2OnAdd: initialized notification ticket pool\n"));
  
  if (!dfInterfaceRegister(pDevObj, &pDevCtx->userMode.iface, g_pInterfaceClass, &g_interfaceMethods)) {
    dfDebugPrint(0, ("*** admxrc2OnAdd: failed to register user-mode interface\n"));
    goto failed;
  }
  dfInterfaceSetContext(&pDevCtx->userMode.iface, pDevCtx);
  pDevCtx->userMode.bRegistered = TRUE;
  dfDebugPrint(1, ("admxrc2OnAdd: registered user-mode interface\n"));

  val32 = pDevCtx->powerMgmt.scheme;
  dfParameterGetUint32(pDevCtx->pDevObj->pDrvObj, "PowerManagementScheme", &val32);
  if (val32 < PowerMgmtInvalidValue) {
    pDevCtx->powerMgmt.scheme = (PowerMgmtScheme)val32; /* Cast is safe because of above check */
  }

  return TRUE;

failed:
  freeDeviceContext(pDevCtx);
  return FALSE;
}

void
admxrc2OnRemove(
	DfDeviceObject* pDevObj,
	Admxrc2DeviceContext* pDevCtx)
{
	dfDebugPrint(1, ("admxrc2OnRemove: entered, pDevObj=%p pDevCtx=%p\n", pDevObj, pDevCtx));

	freeDeviceContext(pDevCtx);
}

boolean_t
admxrc2OnStart(
	DfDeviceObject* pDevObj,
  Admxrc2DeviceContext* pDevCtx,
	DfBusResources* pRawRes,
	DfBusResources* pXlatRes)
{
  const size_t maxSizeT = (size_t)-1;
  unsigned int i;
  CoreWindowInfo windowInfo;
  CoreFlashInfo flashInfo;
  CoreFlashStatus flashStatus;

	dfDebugPrint(1, ("admxrc2OnStart: entered, pDevObj=%p pDevCtx=%p\n", pDevObj, pDevCtx));

	if (!dfImportInterface(pDevObj, g_coreInterfaceName, sizeof(Adb3CoreInterface), &pDevCtx->coreInterface, &pDevCtx->pCoreContext)) {
		dfDebugPrint(0, ("*** admxrc2OnStart: failed to import core interface\n"));
		goto failed;
	}
	pDevCtx->bCoreInterfaceImpd = TRUE;
	dfDebugPrint(1, ("admxrc2OnStart: imported core interface\n"));

  pDevCtx->info.model = pDevCtx->coreInterface.pGetModel(pDevCtx->pCoreContext);
  if (!admxrc2Identify(pDevCtx, pDevCtx->info.model)) {
    dfDebugPrint(0, ("*** admxrc2OnStart: unknown model type %lu\n", (unsigned long)pDevCtx->info.model));
    goto failed;
  }

  if (NULL == pDevCtx->methods.pGetBankInfo || NULL == pDevCtx->methods.pGetCardInfo || NULL == pDevCtx->methods.pGetFpgaInfo) {
    dfDebugPrint(0, ("*** admxrc2OnStart: not all methods have been established\n"));
    goto failed;
  }

  pDevCtx->info.numTargetFpga = pDevCtx->coreInterface.pGetNumTargetFpga(pDevCtx->pCoreContext);
  for (i = 0; i < pDevCtx->info.numTargetFpga; i++) {
    pDevCtx->fpga[i].interrupt.notification.targetIndex = i;
    if (!pDevCtx->coreInterface.pNotificationReg(pDevCtx->pCoreContext, CORE_NOTIFY_FPGA_INTERRUPT(i), &pDevCtx->fpga[i].interrupt.notification.coreNotification, &fpgaInterruptNotifyCallback, pDevCtx)) {
      dfDebugPrint(1, ("*** admxrc2OnStart: failed to register for FPGA_INTERRUPT(%u) notification\n", i));
      goto failed;
    } else {
      dfDebugPrint(1, ("admxrc2OnStart: registered for FPGA_INTERRUPT(%u) notification\n", i));
      pDevCtx->fpga[i].interrupt.bNotificationRegistered = TRUE;
    }

#if defined(TODO)
    pDevCtx->fpga[i].alert.notification.targetIndex = i;
    if (!pDevCtx->coreInterface.pNotificationReg(pDevCtx->pCoreContext, CORE_NOTIFY_FPGA_ALERT(i), &pDevCtx->fpga[i].alert.notification.coreNotification, &fpgaAlertNotifyCallback, pDevCtx)) {
      dfDebugPrint(1, ("*** admxrc2OnStart: failed to register for FPGA_ALERT(%u) notification\n", i));
      goto failed;
    } else {
      dfDebugPrint(1, ("admxrc2OnStart: registered for FPGA_ALERT(%u) notification\n", i));
      pDevCtx->fpga[i].alert.bNotificationRegistered = TRUE;
    }
#endif
  }

#if 0
  if (!pDevCtx->coreInterface.pNotificationReg(pDevCtx->pCoreContext, CORE_NOTIFY_SENSOR_POLL, &pDevCtx->sensor.notification.coreNotification, &sensorPollNotifyCallback, pDevCtx)) {
    dfDebugPrint(1, ("*** admxrc2OnStart: failed to register for SENSOR_POLL notification\n", i));
    goto failed;
  } else {
    dfDebugPrint(1, ("admxrc2OnStart: registered for SENSOR_POLL notification\n", i));
    pDevCtx->sensor.bNotificationRegistered = TRUE;
  }
#endif

  pDevCtx->info.numWindow = pDevCtx->coreInterface.pGetNumWindow(pDevCtx->pCoreContext);
  for (i = 0; i < pDevCtx->info.numWindow; i++) {
    DfMmappableStatus status;

    pDevCtx->coreInterface.pGetWindowInfo(pDevCtx->pCoreContext, i, &windowInfo);
    if (windowInfo.localSize == 0) {
      /* Must be adapter register BAR */
      pDevCtx->window.adapter.pKernelBase = windowInfo.pKernelBase;
      pDevCtx->window.adapter.size = windowInfo.kernelSize;
    } else {
      pDevCtx->window.access[i].pKernelBase = windowInfo.pKernelBase;
      pDevCtx->window.access[i].localBase = windowInfo.localBase;
      pDevCtx->window.access[i].size = windowInfo.kernelSize;
    }
    status = dfMmappableRegionRegister(pDevObj, i + ADMXRC2_REGION_TAG_WINDOW, FALSE, windowInfo.pKernelBase, windowInfo.kernelSize, windowInfo.translatedBase, windowInfo.translatedSize);
    if (DfMmappableSuccess == status) {
      dfDebugPrint(1, ("admxrc2OnStart: registered mappable I/O region, pKernelBase=%p kernelSize=%p translatedBase=0x%08lx_%08lx translatedSize=0x%08lx_%08lx\n",
        windowInfo.pKernelBase, windowInfo.kernelSize, dfSplitUint64(windowInfo.translatedBase), dfSplitUint64(windowInfo.translatedSize)));
    } else {
      dfDebugPrint(0, ("*** admxrc2OnStart: failed to register mappable I/O region, pKernelBase=%p kernelSize=%p translatedBase=0x%08lx_%08lx translatedSize=0x%08lx_%08lx\n",
        windowInfo.pKernelBase, windowInfo.kernelSize, dfSplitUint64(windowInfo.translatedBase), dfSplitUint64(windowInfo.translatedSize)));
      break;
    }
  }
  
  pDevCtx->info.numCommonBuffer = pDevCtx->coreInterface.pGetNumCommonBuffer(pDevCtx->pCoreContext);
  for (i = 0; i < pDevCtx->info.numCommonBuffer; i++) {
    DfMmappableStatus status;
    CoreCommonBufferInfo info;

    pDevCtx->coreInterface.pGetCommonBufferInfo(pDevCtx->pCoreContext, i, &info);
    status = dfMmappableRegionRegister(pDevObj, i + ADMXRC2_REGION_TAG_COMMONBUFFER, TRUE, info.pKernelBase, info.size, 0, 0);
    if (DfMmappableSuccess == status) {
      dfDebugPrint(1, ("admxrc2OnStart: registered mappable common buffer, pKernelBase=%p size=%p\n",
        info.pKernelBase, info.size));
    } else {
      dfDebugPrint(0, ("*** admxrc2OnStart: failed to register mappable common buffer, pKernelBase=%p size=%p\n",
        info.pKernelBase, info.size));
      break;
    }
  }
  
  pDevCtx->info.numFlashBank = pDevCtx->coreInterface.pGetNumFlashBank(pDevCtx->pCoreContext);
  for (i = 0; i < pDevCtx->info.numFlashBank; i++) {
    flashStatus = pDevCtx->coreInterface.pGetFlashInfo(pDevCtx->pCoreContext, 0, &flashInfo);
    if (CoreFlashSuccess == flashStatus) {
      if (flashInfo.largestBlock > maxSizeT) {
        dfDebugPrint(1, ("*** admxrc2OnStart: staging buffer of 0x%08lx_%08lx bytes for Flash bank %lu is too large for this architecture\n",
          dfSplitUint64(flashInfo.largestBlock), (unsigned long)i));
        goto failed;
      }
      pDevCtx->flash[i].pStagingBuffer = (uint8_t*)dfMalloc((size_t)flashInfo.largestBlock); /* Cast safe due to check immediately above */
      if (NULL == pDevCtx->flash[i].pStagingBuffer) {
        dfDebugPrint(1, ("*** admxrc2OnStart: failed to allocate 0x%08lx_%08lx bytes for Flash bank %lu staging buffer\n",
          dfSplitUint64(flashInfo.largestBlock), (unsigned long)i));
        goto failed;
      }
    }
  }
  
  if (!dfInterfaceEnable(pDevObj, &pDevCtx->userMode.iface, TRUE)) {
    dfDebugPrint(0, ("*** admxrc2OnStart: failed to enable user-mode interface\n"));
    goto failed;
  } else {
    dfDebugPrint(1, ("admxrc2OnStart: enabled user-mode interface\n"));
    pDevCtx->userMode.bEnabled = TRUE;
  }

  if (pDevCtx->powerMgmt.scheme == PowerMgmtNone) {
    pDevCtx->powerMgmt.hDisableSleep = dfSystemSleepDisable(pDevCtx->pDevObj);
    if (NULL == pDevCtx->powerMgmt.hDisableSleep) {
      dfDebugPrint(0, ("*** admxrc2OnStart: failed to add system sleep disable hint\n"));
      goto failed;
    }
  } else {
    dfDebugPrint(1, ("admxrc2OnStart: power management scheme is 'none', added system sleep disable hint\n"));
  }

  return TRUE;

failed:
	stopDevice(pDevCtx);
	return FALSE;
}

void
admxrc2OnStop(
	DfDeviceObject* pDevObj,
	Admxrc2DeviceContext* pDevCtx,
  boolean_t bCanTouchHardware)
{
	dfDebugPrint(1, ("admxrc2OnStop: entered, pDevObj=%p pDevCtx=%p bCanTouchHardware=%s\n",
    pDevObj, pDevCtx, bCanTouchHardware ? "TRUE" : "FALSE"));

	stopDevice(pDevCtx);
}

boolean_t
admxrc2OnPowerQuery(
	DfDeviceObject* pDevObj,
	Admxrc2DeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int requestedState,
  boolean_t bIsShutdown)
{
  DfSpinLockFlags f;
  boolean_t bVeto = FALSE;

  dfDebugPrint(1, ("admxrc2OnPowerQuery: entered, pDevObj=%p pDevCtx=%p currentState=D%u requestedState=D%u bIsShutdown=%s\n",
    pDevObj, pDevCtx, currentState, requestedState, bIsShutdown ? "TRUE" : "FALSE"));

  dfAssert(requestedState != currentState);

  if (requestedState <= currentState) {
    /* Wants to power up - always allow */
    dfAssert(!bIsShutdown);
    return FALSE;
  }

  if (bIsShutdown) {
    /* If shutting down, don't bother to veto power change */
    return FALSE;
  }

  /* Powering down */
  f = dfSpinLockGet(&pDevCtx->powerMgmt.lock);
  switch (pDevCtx->powerMgmt.scheme) {
  case PowerMgmtNotActive:
    if (pDevCtx->powerMgmt.openCount) {
      bVeto = TRUE; /* Veto power state change because device is open somewhere */
    } else {
      /* Start refusing open attempts */
      pDevCtx->powerMgmt.bDeferOpen = TRUE;
    }
    break;

  case PowerMgmtNone:
  default:
    bVeto = TRUE; /* Veto power state change */
    break;
  }
  dfSpinLockPut(&pDevCtx->powerMgmt.lock, f);

  return bVeto;
}

static void
completeWaitingOpens(
  Admxrc2DeviceContext* pDevCtx,
  DfIoStatus status)
{
  DfListNode* p;
  Admxrc2ClientContext* pClCtx;
  unsigned int count = 0;

  dfDebugPrint(1, ("completeWaitingOpens: completing opens\n"));

  while (!dfListIsEmpty(&pDevCtx->powerMgmt.waitingOpen)) {
    p = dfListGetHead(&pDevCtx->powerMgmt.waitingOpen);
    dfListRemove(p);
    pClCtx = DF_CONTAINER_OF(p, Admxrc2ClientContext, waitNode);
    dfRequestComplete(pClCtx->pOpenRequest, status);
    count++;
  }

  dfDebugPrint(1, ("completeWaitingOpens: completed %u open(s)\n", count));
}

void
admxrc2OnPowerSet(
	DfDeviceObject* pDevObj,
	Admxrc2DeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState,
  boolean_t bIsShutdown)
{
  DfSpinLockFlags f;

  dfDebugPrint(1, ("admxrc2OnPowerSet: entered, pDevObj=%p pDevCtx=%p currentState=D%u newState=D%u bIsShutdown=%s\n",
    pDevObj, pDevCtx, currentState, newState, bIsShutdown ? "TRUE" : "FALSE"));

  if (newState <= currentState) {
    /* Powering up or reaffirming current working state */
    dfAssert(!bIsShutdown);
    f = dfSpinLockGet(&pDevCtx->powerMgmt.lock);
    switch (pDevCtx->powerMgmt.scheme) {
    case PowerMgmtNotActive:
      /* Stop deferring open attempts */
      pDevCtx->powerMgmt.bDeferOpen = FALSE;
      dfSpinLockPut(&pDevCtx->powerMgmt.lock, f);
      /* Complete all waiting open attempts */
      completeWaitingOpens(pDevCtx, DfIoStatusSuccess);
      break;

    case PowerMgmtNone:
    default:
      dfSpinLockPut(&pDevCtx->powerMgmt.lock, f);
      /* Nothing to do */
      break;
    }
  }
}

boolean_t
admxrc2DriverEntry(
  DfDriverObject* pDrvObj)
{
  dfDebugPrint(1, ("admxrc2DriverEntry: entered\n"));

#if defined(ADMXRC2_LOCKED_BUFFERS_GLOBAL)
  initLockedBuffers(pDrvObj);
#endif

  g_interfaceMethods.pOnCleanup = onCleanup;
  g_interfaceMethods.pOnClose = onClose;
  g_interfaceMethods.pOnIoctl = onIoctl;
#if DF_NEED_THUNK
  g_interfaceMethods.pOnIoctlThunk = onIoctlThunk;
#endif
  g_interfaceMethods.pOnOpen = onOpen;

  if (!dfInterfaceClassRegister(pDrvObj, "admxrc2i", &g_pInterfaceClass)) {
    dfDebugPrint(0, ("*** admxrc2DriverEntry: failed to register interface class '%s'\n", g_interfaceClassName));
    return FALSE;
  }
  dfDebugPrint(1, ("admxrc2DriverEntry: registered interface class '%s', pClass=%p\n", g_interfaceClassName, g_pInterfaceClass));

  return TRUE;
}

extern void
admxrc2DriverExit(
  DfDriverObject* pDrvObj)
{ 
  dfDebugPrint(1, ("admxrc2DriverExit: entered\n"));

  if (NULL != g_pInterfaceClass) {
    dfDebugPrint(1, ("admxrc2DriverExit: unregistering interface class '%s', pClass=%p\n", g_interfaceClassName, g_pInterfaceClass));
    dfInterfaceClassUnregister(g_pInterfaceClass);
    g_pInterfaceClass = NULL;
  }
}

/*
** ---------------------------------------------------------------------------------
** These functions are built into the modular driver, but not the monolithic driver
** ---------------------------------------------------------------------------------
*/

#if !ADB3_MONOLITHIC
static boolean_t
onAdd(
	DfDeviceObject* pDevObj)
{
	Admxrc2DeviceContext* pDevCtx;

  dfDebugPrint(1, ("onAdd: entered, pDevObj=%p\n", pDevObj));

  pDevCtx = (Admxrc2DeviceContext*)dfDeviceGetContext(pDevObj);
	pDevCtx->pDevObj = pDevObj;

  return admxrc2OnAdd(pDevObj, pDevCtx);
}
#endif

#if !ADB3_MONOLITHIC
static void
onRemove(
	DfDeviceObject* pDevObj)
{
	Admxrc2DeviceContext* pDevCtx;

	dfDebugPrint(1, ("onRemove: entered, pDevObj=%p\n", pDevObj));

	pDevCtx = (Admxrc2DeviceContext*)dfDeviceGetContext(pDevObj);
  admxrc2OnRemove(pDevObj, pDevCtx);
}
#endif

#if !ADB3_MONOLITHIC
static boolean_t
onStart(
	DfDeviceObject* pDevObj,
	DfBusResources* pRawRes,
	DfBusResources* pXlatRes)
{
  Admxrc2DeviceContext* pDevCtx;

	dfDebugPrint(1, ("onStart: entered, pDevObj=%p\n", pDevObj));

	pDevCtx = (Admxrc2DeviceContext*)dfDeviceGetContext(pDevObj);
  return admxrc2OnStart(pDevObj, pDevCtx, pRawRes, pXlatRes);
}
#endif

#if !ADB3_MONOLITHIC
static void
onStop(
	DfDeviceObject* pDevObj)
{
	Admxrc2DeviceContext* pDevCtx;

	dfDebugPrint(1, ("onStop: entered, pDevObj=%p\n", pDevObj));

	pDevCtx = (Admxrc2DeviceContext*)dfDeviceGetContext(pDevObj);
  admxrc2OnStop(pDevObj, pDevCtx);
}
#endif

#if !ADB3_MONOLITHIC
static boolean_t
onPowerQuery(
	DfDeviceObject* pDevObj,
  unsigned int currentState,
  unsigned int requestedState,
  boolean_t bIsShutdown)
{
	Admxrc2DeviceContext* pDevCtx;

	pDevCtx = (Admxrc2DeviceContext*)dfDeviceGetContext(pDevObj);

  dfDebugPrint(1, ("onPowerQuery: entered, pDevObj=%p pDevCtx=%p currentState=D%u requestedState=D%u bIsShutdown=%s\n",
    pDevObj, pDevCtx, currentState, requestedState, bIsShutdown ? "TRUE" : "FALSE"));

  return admxrc2OnPowerQuery(pDevObj, pDevCtx, currentState, requestedState, bIsShutdown);
}

static void
onPowerSet(
	DfDeviceObject* pDevObj,
  unsigned int currentState,
  unsigned int newState,
  boolean_t bIsShutdown)
{
	Admxrc2DeviceContext* pDevCtx;

	pDevCtx = (Admxrc2DeviceContext*)dfDeviceGetContext(pDevObj);

  dfDebugPrint(1, ("onPowerSet: entered, pDevObj=%p pDevCtx=%p currentState=D%u newState=D%u  bIsShutdown=%s\n",
    pDevObj, pDevCtx, currentState, newState, bIsShutdown ? "TRUE" : "FALSE"));

  admxrc2OnPowerSet(pDevObj, pDevCtx, currentState, newState, bIsShutdown);
}
#endif

#if !ADB3_MONOLITHIC
boolean_t
dfDriverEntry(
	DfDriverObject* pDrvObj)
{
	dfDebugPrint(1, ("dfDriverEntry: ADMXRC2 Module Driver, pDrvObj=%p version=%lu.%lu.%lu.%lu\n",
    pDrvObj,
    (unsigned long)ADB3_VERSION_0, (unsigned long)ADB3_VERSION_1, (unsigned long)ADB3_VERSION_2, (unsigned long)ADB3_VERSION_3));

  pDrvObj->deviceContextSize = sizeof(Admxrc2DeviceContext);
  pDrvObj->methods.pOnAdd = onAdd;
  pDrvObj->methods.pOnRemove = onRemove;
  pDrvObj->methods.pOnStart = onStart;
  pDrvObj->methods.pOnStop = onStop;
  pDrvObj->methods.pOnPowerQuery = onPowerQuery;
  pDrvObj->methods.pOnPowerSet = onPowerSet;

  return admxrc2DriverEntry(pDrvObj);
}
#endif

#if !ADB3_MONOLITHIC
extern void
dfDriverExit(
	DfDriverObject* pDrvObj)
{
	dfDebugPrint(1, ("dfDriverExit: ADMXRC2 Module Driver exiting\n"));

  admxrc3DriverExit(pDrvObj);
}
#endif
