/*
** File: buffer_desc.c  
** Project: Linux Driver Framework
** Purpose: Implements buffer description functions.
**
** (C) Copyright Alpha Data 2010
*/

#include "df.h"
#include "dflinux.h"

#include <linux/pagemap.h> /* For page_cache_release / put_page */

DfDescriptionResult
dfBufferDescriptionCreateUser(
  const void* pUserAddress,
  size_t length,
  DfBufferDescription** ppDescription)
{
  const unsigned int pageOffsetMask = ~PAGE_MASK;
  const uintptr_t pageAddressMask = PAGE_MASK;
  DfDescriptionResult result = DfDescriptionSuccess;
  DfBufferDescription* pDescription = NULL;
  uintptr_t pageAddress, startPage, endPage;
  size_t numPage, pageRemaining;
  size_t numRegion, regionIndex, regionPages;
  size_t allocSize;
  long numMapped;
  unsigned int pageOffset;
  struct page** pPages;

  if (0 == length) {
    return DfDescriptionInvalidBuffer;
  }
  if (NULL == pUserAddress) {
    /* Trap NULL pointer */
    return DfDescriptionNullPointer;
  }
  if ((uintptr_t)pUserAddress + length < (uintptr_t)pUserAddress) {
    /* Region wraps back to 0 - not permitted */
    return DfDescriptionInvalidBuffer;
  }

  pageOffset = (unsigned int)(uintptr_t)pUserAddress & pageOffsetMask;
  pageAddress = (uintptr_t)pUserAddress & pageAddressMask;
  startPage = (uintptr_t)pUserAddress >> PAGE_SHIFT;
  endPage = ((uintptr_t)pUserAddress + length - 1) >> PAGE_SHIFT;
  numPage = (size_t)(endPage - startPage + 1);
  numRegion = (numPage + DF_BUFFER_DESCRIPTION_REGION_SIZE - 1) / DF_BUFFER_DESCRIPTION_REGION_SIZE;

  pDescription = (DfBufferDescription*)dfMalloc(sizeof(*pDescription) + (numRegion - 1) * sizeof(struct page**));
  if (NULL == pDescription) {
    return DfDescriptionNoMemory;
  }
  pDescription->pAddress = pUserAddress;
  pDescription->length = length;
  pDescription->system.bUserMode = TRUE;
  pDescription->system.bLocked = FALSE;
  pDescription->system.numRegion = numRegion;
  pDescription->system.numPage = numPage;

  regionIndex = 0;
  regionPages = 0;
  pageRemaining = numPage;
  while (pageRemaining) {
    regionPages = (pageRemaining > DF_BUFFER_DESCRIPTION_REGION_SIZE) ? DF_BUFFER_DESCRIPTION_REGION_SIZE : pageRemaining;
    allocSize = sizeof(struct page*) * regionPages;
    pPages = (struct page**)dfMalloc(allocSize);
    if (NULL == pPages) {
      result = DfDescriptionNoMemory;
      goto done;
    }
    pDescription->system.region[regionIndex] = pPages;
    down_read(&current->mm->mmap_sem);
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4, 10, 0))
    numMapped = get_user_pages_remote(current, current->mm, pageAddress, regionPages, FOLL_WRITE, pPages, NULL, NULL);
#elif (LINUX_VERSION_CODE >= KERNEL_VERSION(4, 9, 0))
    numMapped = get_user_pages_remote(current, current->mm, pageAddress, regionPages, FOLL_WRITE, pPages, NULL);
#elif (LINUX_VERSION_CODE >= KERNEL_VERSION(4, 6, 0))
    numMapped = get_user_pages_remote(current, current->mm, pageAddress, regionPages, 1, 0, pPages, NULL);
#else
    numMapped = get_user_pages(current, current->mm, pageAddress, regionPages, 1, 0, pPages, NULL);
#endif
    up_read(&current->mm->mmap_sem);
    if (numMapped != regionPages) {
      result = DfDescriptionInvalidBuffer;
      goto done;
    }
    pageAddress += regionPages << PAGE_SHIFT;
    pageRemaining -= regionPages;
    regionIndex++;
  }

 done:
  if (DfDescriptionSuccess != result) {
    dfBufferDescriptionDelete(pDescription);
  } else {
    *ppDescription = pDescription;
  }
  return result;
}

void
dfBufferDescriptionDelete(
  DfBufferDescription* pDescription)
{
  size_t regionIndex, pageIndex, numPage, n;
  struct page** pPages;
  struct page* p;

  dfAssert(!pDescription->system.bLocked);
  numPage = pDescription->system.numPage;
  regionIndex = 0;
  while (numPage) {
    n = (numPage > DF_BUFFER_DESCRIPTION_REGION_SIZE) ? DF_BUFFER_DESCRIPTION_REGION_SIZE : numPage;
    pPages = pDescription->system.region[regionIndex];
    if (NULL == pPages) {
      break;
    }
    pageIndex = 0;
    while (pageIndex < n) {
      p = pPages[pageIndex];
      if (NULL != p) {
        if (!PageReserved(p)) {
          SetPageDirty(p);
        }
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4, 6, 0))
        put_page(p);
#else
        page_cache_release(p);
#endif
      }
      pageIndex++;
    }
    dfFree(pPages);
    numPage -= n;
    regionIndex++;
  }
  dfFree(pDescription);
}

boolean_t
dfBufferDescriptionLock(
  DfBufferDescription* pDescription)
{
  /* This is a no operation in Linux */
  dfAssert(!pDescription->system.bLocked);
  pDescription->system.bLocked = TRUE;
  return TRUE;
}

void
dfBufferDescriptionUnlock(
  DfBufferDescription* pDescription)
{
  /* This is a no operation in Linux */
  dfAssert(pDescription->system.bLocked);
  pDescription->system.bLocked = FALSE;
}
