/*
** File: ucd90120buf.c  
** Project: ADB3 core driver
** Purpose: Functions for accessing buffered rail readings of the
**          UCD90120 system monitor, in models that use it.
**
** (C) Copyright Alpha Data 2014
**
*/

#include <df.h>

#include "ucd90120buf.h"

/*
** Buffered readings of a UCD902120 rail are 128 bits / 4 DWORDs each,
** with fields as follows (little-endian):
**
**   4:0   d[0][4:0]   = 5-bit two's complement exponent of Rail Voltage
**  47:32  d[1][15:0]  = 16-bit integer mantissa of Rail Voltage
**  75:64  d[2][11:0]  = 11-bit integer mantissa of Rail Current
**  79:76  d[2][15:12] = 5-bit two's complement exponent of Rail Current
** 106:96  d[3][10:0]  = 11-bit integer mantissa of Chip Temperature
** 111:97  d[3][15:11] = 5-bit two's complement exponent of Chip Temperature 
*/
typedef struct _Ucd90120BufRawRailReading {
  uint32_t d[4];
} Ucd90120BufRawRailReading;

/* Portably shift int32_t 'value' by 'nbits', performing sign-extension. */
/* 'nbits' must be in range 0 .. 31 */
static int32_t
sra32(
  int32_t value,
  int nbits)
{
  dfAssert(nbits >= 0 && nbits <= 31);
  if (nbits == 0) {
    return value;
  }
  if (value >= 0) {
    return value >> nbits;
  } else {
    uint32_t mask = (0x1U << (32 - nbits)) - 1U;
    return (int32_t)((((uint32_t)value >> nbits) & mask) | ~mask);
  }
}

boolean_t
ucd90120BufReadSensor(
  void* pUcd90120Buf,
  DfMemoryHandle hUcd90120Buf,
  unsigned int railIndex,
  unsigned int type,
  CoreSensorValue* pReading)
{
  const unsigned int railIndexLimit = 0x100U / sizeof(Ucd90120BufRawRailReading);
  Ucd90120BufRawRailReading* pRawReading = (Ucd90120BufRawRailReading*)pUcd90120Buf + railIndex;
  uint32_t tmp32;
  int32_t mantissa;
  int8_t exponent;

  if (railIndex >= railIndexLimit) {
    dfAssert(FALSE);
    return FALSE;
  }

  pRawReading = (Ucd90120BufRawRailReading*)pUcd90120Buf + railIndex;

  switch (type) {
  case UCD90120SENSORTYPE_VOLTAGE: /* "LINEAR16" format */
    tmp32 = dfPciMemRead32(hUcd90120Buf, &pRawReading->d[0]);
    exponent = (int8_t)(tmp32 & 0x1FU);
    if (exponent & 0x10) {
      /* Sign-extend exponent from 5 to 8 bits. */
      exponent = (int8_t)(exponent | 0xE0);
    }
    tmp32 = dfPciMemRead32(hUcd90120Buf, &pRawReading->d[1]);
    mantissa = (int32_t)(tmp32 & 0xFFFFU);
    break;

  case UCD90120SENSORTYPE_CURRENT: /* "LINEAR11" format */
    tmp32 = dfPciMemRead32(hUcd90120Buf, &pRawReading->d[2]);
    exponent = (int8_t)((tmp32 >> 11) & 0x1FU);
    if (exponent & 0x10) {
      /* Sign-extend exponent from 5 to 8 bits. */
      exponent = (int8_t)(exponent | 0xE0);
    }
    mantissa = (int32_t)(tmp32 & 0x7FFU);
    if (mantissa & 0x400) {
      /* Sign-extend mantissa from 11 to 32 bits. */
      mantissa |= 0xFFFFF800;
    }
    break;

  case UCD90120SENSORTYPE_TEMP: /* "LINEAR11" format */
    tmp32 = dfPciMemRead32(hUcd90120Buf, &pRawReading->d[3]);
    exponent = (int8_t)((tmp32 >> 11) & 0x1FU);
    if (exponent & 0x10) {
      /* Sign-extend exponent from 5 to 8 bits. */
      exponent = (int8_t)(exponent | 0xE0);
    }
    mantissa = (int32_t)(tmp32 & 0x7FFU);
    if (mantissa & 0x400) {
      /* Sign-extend mantissa from 11 to 32 bits. */
      mantissa |= 0xFFFFF800;
    }
    break;

  default:
    /* Not a valid sensor type for UCD90120. */
    dfAssert(FALSE);
    return FALSE;
  }

  /* Compute final value (assumed to use 'doubleValue' member of union). */
  if (exponent >= 0) {
    pReading->doubleValue.intPart = mantissa << exponent;
    pReading->doubleValue.fracPart = 0U;
  } else {
    pReading->doubleValue.intPart = sra32(mantissa, (int)-exponent);
    pReading->doubleValue.fracPart = (uint32_t)mantissa << ((int)exponent + 32);
  }
  return TRUE; /* OK */
}
