/*
** File: si5338.h  
** Project: ADB3 core driver
** Purpose: Exports definitions for the the SI5338 clock generator.
**
** (C) Copyright Alpha Data 2011, 2015
*/

#ifndef _ADATA_CORE_SI5338_H
#define _ADATA_CORE_SI5338_H

#include "coreif.h"
#include "serializer.h"
#include "i2c.h"

struct _Si5338Request;
typedef struct _Si5338Request Si5338Request;
struct _Si5338ProgramRequest;
typedef struct _Si5338ProgramRequest Si5338ProgramRequest;
struct _Si5338ProgramRequestSync;
typedef struct _Si5338ProgramRequestSync Si5338ProgramRequestSync;

/* Register addresses in SI5338 */
#define SI5338_REGIDX_REV_ID            (0U)
#define SI5338_REGIDX_DEV_CONFIG2       (2U)
#define SI5338_REGIDX_DEV_CONFIG3       (3U)
#define SI5338_REGIDX_DEV_CONFIG4       (4U)
#define SI5338_REGIDX_DEV_CONFIG5       (5U)
#define SI5338_REGIDX_I2C_ADDR         (27U)
#define SI5338_REGIDX_PFD_IN_SEL       (29U)
#define SI5338_REGIDX_FCAL_OVRD0       (45U)
#define SI5338_REGIDX_FCAL_OVRD1       (46U)
#define SI5338_REGIDX_FCAL_OVRD2       (47U)
#define SI5338_REGIDX_FCAL_OVRD_EN     (49U)
#define SI5338_REGIDX_MSx_FIDCTL(x)    ((uint16_t)(52U + 11U * (x))) /* Multisynth x frequency increment/decrement control */
#define SI5338_REGIDX_MSx_P1(x)        ((uint16_t)(53U + 11U * (x))) /* Multisynth x registers */
#define SI5338_REGIDX_MS0_P1          SI5338_REGIDX_MSx_P1(0)
#define SI5338_REGIDX_MS1_P1          SI5338_REGIDX_MSx_P1(1)
#define SI5338_REGIDX_MS2_P1          SI5338_REGIDX_MSx_P1(2)
#define SI5338_REGIDX_MS3_P1          SI5338_REGIDX_MSx_P1(3)
#define SI5338_REGIDX_MSN_P1           (97U)                    /* VCO multisynth registers */
#define SI5338_REGIDX_MS0_FIDP1(byte) ((uint16_t)(123U + byte))
#define SI5338_REGIDX_MS0_FIDP2(byte) ((uint16_t)(136U - byte)) /* NOTE: byte order is reversed */
#define SI5338_REGIDX_MS0_FIDP3(byte) ((uint16_t)(137U + byte))
#define SI5338_REGIDX_MS1_FIDP1(byte) ((uint16_t)(152U + byte))
#define SI5338_REGIDX_MS1_FIDP2(byte) ((uint16_t)(165U - byte)) /* NOTE: byte order is reversed */
#define SI5338_REGIDX_MS1_FIDP3(byte) ((uint16_t)(166U + byte))
#define SI5338_REGIDX_MS2_FIDP1(byte) ((uint16_t)(174U + byte))
#define SI5338_REGIDX_MS2_FIDP2(byte) ((uint16_t)(187U - byte)) /* NOTE: byte order is reversed */
#define SI5338_REGIDX_MS2_FIDP3(byte) ((uint16_t)(188U + byte))
#define SI5338_REGIDX_MS3_FIDP1(byte) ((uint16_t)(196U + byte))
#define SI5338_REGIDX_MS3_FIDP2(byte) ((uint16_t)(209U - byte)) /* NOTE: byte order is reversed */
#define SI5338_REGIDX_MS3_FIDP3(byte) ((uint16_t)(210U + byte))
#define SI5338_REGIDX_LOL_STATUS      (218U)
#define SI5338_REGIDX_DRIVER_DISABLE  (230U)
#define SI5338_REGIDX_FCAL0           (235U)
#define SI5338_REGIDX_FCAL1           (236U)
#define SI5338_REGIDX_FCAL2           (237U)
#define SI5338_REGIDX_LOL_PAUSE       (241U)
#define SI5338_REGIDX_DCLK_DIS        (242U)
#define SI5338_REGIDX_PLL_SOFT_RESET  (246U)
#define SI5338_REGIDX_PAGE            (255U)

/* Number of 8-bit wide registers in SI5338 */
#define SI5338_NUM_REG (351U)

typedef struct _Si5338DeviceContext {
  I2cContext* pI2cCtx;
  SerializerQueue pageBitSerializer; /* Serializes page register access */
  uint8_t busIndex;      /* I2C bus index where the SI5338 lives */
  uint8_t deviceIndex;   /* I2C device index of the SI5338 */
  uint8_t pageBit;       /* Current page register setting in SI5338 */
  struct {
    uint32_t frequency; /* Calculated SI5338 VCO frequency */
  } vco;
  struct {
    uint32_t p1, p2, p3;     /* Multisynth n values */
    uint32_t frequency;      /* Calculated frequency (from p{1,2,3}) of multisynth n, Hz */
    uint64_t fidp1, fidp2, fidp3; /* MSx_FIDP1/2/3 values corresponding to "frequencyStep" */
    uint32_t frequencyStep;  /* Frequency step size corresponding to fidp{1,2,3} */
    int32_t currentStep;     /* Signed number of steps that are currently applied to multisynth */
  } multisynth[4];
  boolean_t bIsSi5338a;  /* TRUE if SI5338A, FALSE if SI5338B */
  struct { /* Used to save & restore state when powering down & up */
    CoreClockWord last;
    boolean_t bLastValid;
  } save[4];
} Si5338DeviceContext;

typedef struct _Si5338ConfigurationRecord {
  uint8_t addrLo;
  uint8_t addrHi;
  uint8_t mask;
  uint8_t data;
} Si5338ConfigurationRecord;

typedef enum _Si5338ClockWordStatus {
  Si5338ClockWordStatusSuccess = 0,
  Si5338ClockWordStatusOutOfRange = 1,
  Si5338ClockWordStatusGeneralFailure = 2
} Si5338ClockWordStatus;

typedef enum _Si5338Status {
  Si5338StatusSuccess = 0,
  Si5338StatusI2CTimeout = 1,
  Si5338StatusI2CError = 2,
  Si5338StatusInvalidRegister = 3,
  Si5338StatusGeneralFailure = 4
} Si5338Status;

typedef void Si5338Callback(Si5338DeviceContext* pSi5338Ctx, Si5338Request* pRequest, uint8_t data, Si5338Status status, void* pContext);

struct _Si5338Request { /* Client must not touch */
  Si5338DeviceContext* pSi5338Ctx;  /* Essential information about the SI5338's operating environment */
  SerializerRequest pageBitRequest; /* For serializing page bit updates */
  I2cRequest i2cRequest;            /* For serializing I2C accesses */
  Si5338Callback* pCallback;        /* Called on completion of SI5338 register access */
  uint16_t regIndex;                /* Byte address of SI5338 register within device */
  uint8_t data;                     /* Value to write to SI5338 register if writing, otherwise filled in for a read by I2C core */
  uint8_t mask;                     /* Mask for writing data to SI5338 register */
  boolean_t bWrite;                 /* TRUE for writing to SI5338, FALSE for reading */
  void* pContext;                   /* Passed to pCallback */
};

typedef enum _Si5338ProgramStatus {
  Si5338ProgramStatusSuccess = 0,
  Si5338ProgramStatusI2CError = 1,
  Si5338ProgramStatusInvalidIndex = 2,
  Si5338ProgramStatusGeneralFailure = 3
} Si5338ProgramStatus;

typedef void Si5338ProgramCallback(Si5338DeviceContext* pSi5338Ctx, Si5338ProgramRequest* pRequest, unsigned int msIndex, Si5338ProgramStatus status, void* pContext);

struct _Si5338ProgramRequest {
  Si5338Request request;
  Si5338ProgramCallback* pCallback;
  CoreClockWord clockWord;
  unsigned int msIndex;
  unsigned int stage;
  void* pContext;                   /* Passed to pCallback */
};

struct _Si5338ProgramRequestSync {
  Si5338ProgramRequest request;
  DfEvent ev;
  Si5338ProgramStatus status;
};

#endif
