/*
** File: pci9xxx_93x6.c  
** Project: ADB3 core driver
** Purpose: Functions for accessing the 93x6 EEPROM on PCI9080-based and PCI9656-based cards.
**
** (C) Copyright Alpha Data 2009-2010
**
** These functions are compatible with both the PCI9080 and PCI9656, provided that the
** value passed for the bIsPci9656 parameter is correct. The bit fields within the CNTRL
** register are the same on the PCI9080 and PCI9656, apart from one extra field on the
** PCI9656 that controls the driver for the serial data line.
*/

#include <df.h>

#include "pci9xxx.h"
#include "pci9xxx_93x6.h"

#define USE_BUSY_DELAY (0)

#if USE_BUSY_DELAY
# define BUSY_DELAY(us) dfDelayMicroseconds(us)
#else
# define BUSY_DELAY(us) /* Do nothing */
#endif

/*
** Receive up to 32 bits from a 93x6 EEPROM
*/
static void
receive93x6Bits(
	Pci9080Registers* pRegs,
  DfMemoryHandle hRegs,
	uint32_t* pVal32,
	unsigned int n,
	uint32_t* pCntrlValue)
{
	uint32_t val32 = 0;
	unsigned int i;

	for (i = 0; i < n; i++) {
		val32 <<= 1U;
		dfPciMemWrite32(hRegs, &pRegs->CNTRL, *pCntrlValue);
		dfPciMemRead32(hRegs, &pRegs->CNTRL);
    BUSY_DELAY(1);
		dfPciMemWrite32(hRegs, &pRegs->CNTRL, *pCntrlValue | PLX_CNTRL_EESK);
		dfPciMemRead32(hRegs, &pRegs->CNTRL);
    BUSY_DELAY(1);
		if (dfPciMemRead32(hRegs, &pRegs->CNTRL) & PLX_CNTRL_EEDO) {
			val32 |= 0x1U;
		}
		dfPciMemWrite32(hRegs, &pRegs->CNTRL, *pCntrlValue);
    BUSY_DELAY(1);
	}

	*pVal32 = val32;
}

/*
** Send up to 32 bits to a 93x6 EEPROM
**
** If bIsPci9656, leaves EEDO pin disabled on exit.
*/
static void
send93x6Bits(
	Pci9080Registers* pRegs,
  DfMemoryHandle hRegs,
	boolean_t bIsPci9656,
	uint32_t val32,
	unsigned int n,
	uint32_t* pCntrlValue)
{
  const uint32_t bitMask = 0x1U << (n - 1);
	unsigned int i;

	if (bIsPci9656) {
		/* Enable driver on EEDI pin */
		*pCntrlValue &= ~PLX_CNTRL_PCI9656_EEDI_OD;
	}
	dfPciMemWrite32(hRegs, &pRegs->CNTRL, *pCntrlValue);
	dfPciMemRead32(hRegs, &pRegs->CNTRL);
	BUSY_DELAY(1);

	/* Send instruction to EEPROM one bit at a time */
	for (i = 0; i < n; i++) {
		/* Pre-clear the EEPROM data bit */
		*pCntrlValue &= ~PLX_CNTRL_EEDI;
		/* Set the EEPROM data bit if necessary */
		*pCntrlValue |= (val32 & bitMask) ? PLX_CNTRL_EEDI : 0x00000000;
		
		/* Clock the data bit to the EEPROM */
		dfPciMemWrite32(hRegs, &pRegs->CNTRL, *pCntrlValue);
		dfPciMemRead32(hRegs, &pRegs->CNTRL);
		BUSY_DELAY(1);
		dfPciMemWrite32(hRegs, &pRegs->CNTRL, *pCntrlValue | PLX_CNTRL_EESK);
		dfPciMemRead32(hRegs, &pRegs->CNTRL);
		BUSY_DELAY(1);
		dfPciMemWrite32(hRegs, &pRegs->CNTRL, *pCntrlValue);
		dfPciMemRead32(hRegs, &pRegs->CNTRL);
		BUSY_DELAY(1);

		val32 <<= 1;
	}

	if (bIsPci9656) {
		/* Disable driver on EEDI pin */
		*pCntrlValue |= PLX_CNTRL_PCI9656_EEDI_OD;
		dfPciMemWrite32(hRegs, &pRegs->CNTRL, *pCntrlValue);
		dfPciMemRead32(hRegs, &pRegs->CNTRL);
		BUSY_DELAY(1);
	}
}

/*
** Send a command to a 93x6 EEPROM
*/
static void
send93x6Command(
	Pci9080Registers* pRegs,
  DfMemoryHandle hRegs,
	boolean_t bIsPci9656,
	uint32_t command,
	unsigned int n,
	uint32_t* pCntrlValue)
{	
	/* Toggle EEPROM's chip select low then high to reset state machine inside chip */

  /* Clear data, chip select and clock bits */
	*pCntrlValue &= ~(PLX_CNTRL_EESK | PLX_CNTRL_EEDI | PLX_CNTRL_EECS);
	dfPciMemWrite32(hRegs, &pRegs->CNTRL, *pCntrlValue);
	dfPciMemRead32(hRegs, &pRegs->CNTRL);
	BUSY_DELAY(1);
	/* Chip select asserted */
	*pCntrlValue |= PLX_CNTRL_EECS;
	dfPciMemWrite32(hRegs, &pRegs->CNTRL, *pCntrlValue);
	dfPciMemRead32(hRegs, &pRegs->CNTRL);
	BUSY_DELAY(1);

	/* Send the instruction to the EEPROM */
	send93x6Bits(pRegs, hRegs, bIsPci9656, command, n, pCntrlValue);
}

void
read93x6(
	Pci9080Registers* pRegs,
  DfMemoryHandle hRegs,
	boolean_t bIsPci9656,
	unsigned int addrLen,
	unsigned int instrLen,
	uint8_t location,
	uint16_t* pVal16)
{
	uint8_t addrMask = (uint8_t)(location & ((1U << addrLen) - 1U));
	uint32_t val32;
	uint32_t cntrlReg;

	cntrlReg = dfPciMemRead32(hRegs, &pRegs->CNTRL);

	/* Send read command to EEPROM */
	send93x6Command(pRegs, hRegs, bIsPci9656, (EE_READ << addrLen) | (location & addrMask), instrLen, &cntrlReg);

	/* Get 16-bit word from EEPROM */
	receive93x6Bits(pRegs, hRegs, &val32, 16, &cntrlReg);

	*pVal16 = (uint16_t)val32;
}

boolean_t
write93x6(
	Pci9080Registers* pRegs,
  DfMemoryHandle hRegs,
	boolean_t bIsPci9656,
	unsigned int addrLen,
	unsigned int instrLen,
	uint8_t location,
	uint16_t val16)
{
	uint8_t addrMask = (uint8_t)(location & ((1U << addrLen) - 1U));
	uint32_t cntrlReg;
	unsigned int i;
  boolean_t bSuccess = FALSE; /* Assume write times out, to begin with */

	cntrlReg = dfPciMemRead32(hRegs, &pRegs->CNTRL);

	/* Enable writing to EEPROM */
	send93x6Command(pRegs, hRegs, bIsPci9656, EE_WREN << (addrLen - 2), instrLen, &cntrlReg);

	/* Send write command to EEPROM */
	send93x6Command(pRegs, hRegs, bIsPci9656, (EE_WRITE << addrLen) | (location & addrMask),	instrLen, &cntrlReg);

	/* Send data word to EEPROM */
	send93x6Bits(pRegs, hRegs, bIsPci9656, (uint32_t)val16, 16, &cntrlReg);

	/* Activate EEPROM's status mechanism: deselect and reselect */
	dfPciMemWrite32(hRegs, &pRegs->CNTRL, cntrlReg & ~PLX_CNTRL_EECS);
	dfPciMemRead32(hRegs, &pRegs->CNTRL);
	BUSY_DELAY(1);
	dfPciMemWrite32(hRegs, &pRegs->CNTRL, cntrlReg);
	dfPciMemRead32(hRegs, &pRegs->CNTRL);
	BUSY_DELAY(1);

	/* Poll chip status until write cycle completes (or times out) */
	for (i = 0; i < 10; i++) {
		if (dfPciMemRead32(hRegs, &pRegs->CNTRL) & PLX_CNTRL_EEDO) {
			bSuccess = TRUE;
      break;
		}
    dfDelayThreadFor(dfMicrosecondsToTime(10000));
	}

	/* Disable writing to EEPROM */
	send93x6Command(pRegs, hRegs, bIsPci9656, EE_WDS << (addrLen - 2), instrLen, &cntrlReg);

	return bSuccess;
}

CoreVpdStatus
read93x6Buffer(
	Adb3CoreDeviceContext* pDevCtx,
	boolean_t bIsPci9656,
  unsigned int addrLen,
  unsigned int instrLen,
	uint32_t address,
	size_t length,
	void* pData)
{
  CoreVpdStatus status = CoreVpdSuccess;
  uint16_t val16;
  uint8_t* p = (uint8_t*)pData;

  if (length) {
    if (address & 0x1U) {
      read93x6(pDevCtx->hardware.bridge.pPci9080, pDevCtx->hardware.hBridge, bIsPci9656, addrLen, instrLen, (uint8_t)(address >> 1), &val16);
      *p++ = (uint8_t)(val16 >> 8);
      length--;
      address++;
    }
    while (length >= 2) {
      read93x6(pDevCtx->hardware.bridge.pPci9080, pDevCtx->hardware.hBridge, bIsPci9656, addrLen, instrLen, (uint8_t)(address >> 1), &val16);
      *p++ = (uint8_t)val16;
      *p++ = (uint8_t)(val16 >> 8);
      length -= 2;
      address += 2;
    }
    if (length & 0x1U) {
      read93x6(pDevCtx->hardware.bridge.pPci9080, pDevCtx->hardware.hBridge, bIsPci9656, addrLen, instrLen, (uint8_t)(address >> 1), &val16);
      *p++ = (uint8_t)val16;
    }
  }

  return status;
}

CoreVpdStatus
write93x6Buffer(
	Adb3CoreDeviceContext* pDevCtx,
	boolean_t bIsPci9656,
  unsigned int addrLen,
  unsigned int instrLen,
	uint32_t address,
	size_t length,
	const void* pData)
{
  uint16_t val16;
  uint8_t* p = (uint8_t*)pData;

  if (length) {
    if (address & 0x1U) {
      read93x6(pDevCtx->hardware.bridge.pPci9080, pDevCtx->hardware.hBridge, bIsPci9656, addrLen, instrLen, (uint8_t)(address >> 1), &val16);
      val16 = (uint16_t)(val16 & 0x00ffU);
      val16 = (uint16_t)(val16 | ((uint16_t)*p++ << 8));
      if (!write93x6(pDevCtx->hardware.bridge.pPci9080, pDevCtx->hardware.hBridge, bIsPci9656, addrLen, instrLen, (uint8_t)(address >> 1), val16)) {
        return CoreVpdTimeout;
      }
      length--;
      address++;
    }
    while (length >= 2) {
      val16 = (uint16_t)*p++;
      val16 = (uint16_t)(val16 | ((uint16_t)*p++ << 8));
      if (!write93x6(pDevCtx->hardware.bridge.pPci9080, pDevCtx->hardware.hBridge, bIsPci9656, addrLen, instrLen, (uint8_t)(address >> 1), val16)) {
        return CoreVpdTimeout;
      }
      length -= 2;
      address += 2;
    }
    if (length & 0x1U) {
      read93x6(pDevCtx->hardware.bridge.pPci9080, pDevCtx->hardware.hBridge, bIsPci9656, addrLen, instrLen, (uint8_t)(address >> 1), &val16);
      val16 = (uint16_t)(val16 & 0xff00U);
      val16 = (uint16_t)(val16 | (uint16_t)*p++);
      if (!write93x6(pDevCtx->hardware.bridge.pPci9080, pDevCtx->hardware.hBridge, bIsPci9656, addrLen, instrLen, (uint8_t)(address >> 1), val16)) {
        return CoreVpdTimeout;
      }
    }
  }

  return CoreVpdSuccess;
}
