/*
** File: model_admxrc5lx.c  
** Project: ADB3 core driver
** Purpose: ADM-XRC-5LX-specific code, including function to bootstrap device context.
**
** NOTES
**
** 1. This code applies to the ADM-XRC-5LX. The ADM-XRC-5LXA supersedes
**    the ADM-XRC-5LX, and is effectively board revision 2 of the
**    ADM-XRC-5LX. Differences between the boards are:
**
**    - Clock generators is different:
**      -5LX has ICS307 (LCLK) + ICS8430-61 (MCLK)
**      -5LXA has V4 clock synth. in ADB1 Bridge (LCLK) + ICS8430-61 (MCLK).
**
**    - VPD structures in Flash for -5LX and -5LXA are slightly different,
**      due to clocking generator differences.
**
**    - Model-specific registers in ADB1 Bridge BAR2 related to clocking are
**      different for -5LX and -5LXA.
**
** 2. The model code set here is CoreModelAdmxrc5lx, which is the same as
**    for the ADM-XRC-5LXA.
**
** 2. The VPD structure used in the device context structure for this model
**    is that of the ADM-XRC-5LXA, NOT the ADM-XRC-5LX. The VPD data read
**    from Flash, which is laid out as per struct VpdAdmxrc5lxRev0, and
**    converted to a VpdAdmxrc5lxaRev0 struct. This is done so that the
**    ADMXRC2 module driver (layered on top of this driver) can use struct
**    VpdAdmxrc5lxaRev0 regardless of whether the board is an ADM-XRC-5LX
**    (revision 1 board) or an ADM-XRC-5LXA (revision 2 board).
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>

#include "adb1_common.h"
#include "adb1v4v5_common.h"
#include "admxrc5lx.h"
#include "admxrc5lxa.h"
#include "coreclock.h"
#include "coredma.h"
#include "device.h"
#include "flash.h"
#include "flash_cfi.h"
#include "ics307_common.h"
#include "ics8430_61_common.h"
#include "model_boot.h"

static struct {
  unsigned int regOffset; /* LM87 register offset */
  boolean_t bSigned;      /* TRUE => register value is int8_t; otherwise uint8_t */
  int32_t factor;         /* Fixed point, where 0x00100000 => 1, 0x00080000 => 0.5, 0xFFF80000 => -0.5 etc. */
} g_lm87Scaling[] = {
  { 0x21U, FALSE, 14785,   /* 0.0141 */ },
  { 0x28U, FALSE, 10276,   /* 0.0098 */ },
  { 0x29U, FALSE, 10276,   /* 0.0098 */ },
  { 0x20U, FALSE, 13631    /* 0.0130 */ },
  { 0x22U, FALSE, 18036,   /* 0.0172 */ },
  { 0x23U, FALSE, 27263,   /* 0.0260 */ },
  { 0x24U, FALSE, 65536,   /* 0.0625 */ },
  { 0x25U, FALSE, 14785,   /* 0.0141 */ },
  { 0x27U, TRUE,  1048576, /* 1.0000 */ },
  { 0x26U, TRUE,  1048576, /* 1.0000 */ }
};

static CoreSensorInfo g_lm87SensorInfo[] = {
  /* LM87 sensors */
  { "1V supply rail",              0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x21, Vccp1 reading */
  { "1.2V supply rail",            0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x28, FAN1/AIN1 reading */
  { "1.8V supply rail",            0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x29, FAN2/AIN2 reading */
  { "2.5V supply rail",            0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x20, +2.5V / ext temp. 2 reading */
  { "3.3V supply rail",            0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x22, +Vcc reading */
  { "5V supply rail",              0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x23, 5V reading */
  { "PMC connector VIO",           0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x24, +12V reading */
  { "Front panel I/O power rail",  0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x25, Vccp2 reading */
  { "LM87 internal temperature",   0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* LM87 register 0x27, Int temp reading */
  { "Target FPGA ext. temp.",      0, { 0,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }  /* LM87 register 0x26, Ext temp. 1 reading */
};

static unsigned int
initSensorInfo(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int sensorIndex)
{
  unsigned int i, n = 0;

  dfAssert(sensorIndex + DF_ARRAY_LENGTH(g_lm87SensorInfo) <= MAX_NUM_SENSOR);
  for (i = 0; i < DF_ARRAY_LENGTH(g_lm87SensorInfo) && sensorIndex < MAX_NUM_SENSOR; i++, sensorIndex++) {
    pDevCtx->info.sensor[sensorIndex] = g_lm87SensorInfo[i];
    n++;
  }
  return n; 
}

static CoreSensorStatus
readSensor(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int lm87SensorIndex,
  CoreSensorValue* pReading)
{
  ModelRegsAdb1V4V5* pModelRegs;
  DfMemoryHandle hModel;
  int16_t val16;
  int32_t factor, value;

  dfAssert(DF_ARRAY_LENGTH(g_lm87SensorInfo) == DF_ARRAY_LENGTH(g_lm87Scaling));

  pModelRegs = pDevCtx->hardware.model.pAdb1V4V5;
  hModel = pDevCtx->hardware.hModel;

  if (lm87SensorIndex < DF_ARRAY_LENGTH(g_lm87SensorInfo)) {
    /* LM87 sensors */
    if (g_lm87Scaling[lm87SensorIndex].bSigned) {
      val16 = (int16_t)(int8_t)dfPciMemRead8(hModel, pModelRegs->sysmonbuf + g_lm87Scaling[lm87SensorIndex].regOffset);
    } else {
      val16 = (int16_t)dfPciMemRead8(hModel, pModelRegs->sysmonbuf + g_lm87Scaling[lm87SensorIndex].regOffset);
    }
    factor = g_lm87Scaling[lm87SensorIndex].factor;
    value = (val16 * factor) / 16;
    pReading->doubleValue.intPart = (int32_t)(int16_t)((value >> 16) & 0xFFFFU);
    pReading->doubleValue.fracPart = ((uint32_t)value & 0xFFFFU) << 16;
    return CoreSensorSuccess;
  }

  return CoreSensorInvalidIndex;
}

static boolean_t
readVpd(
  Adb3CoreDeviceContext* pDevCtx)
{
  VpdAdmxrc5lxaRev0* pVpd = &pDevCtx->info.vpd.pData->admxrc5lxa.rev0;
  VpdAdmxrc5lxRev0 vpdRaw;
  uint16_t vpdVersion = 0xffffU;
  int i;

  flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdVersion), &vpdVersion);
  vpdVersion = dfLe16ToCpu(vpdVersion);
  switch (vpdVersion) {
  case 0:
    /* Ok */
    break;

  case 0xffffU:
  	dfDebugPrint(0, ("*** readVpd: invalid VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    return FALSE;

  default:
  	dfDebugPrint(0, ("+++ readVpd: unrecognised VPD version %lu(0x%lx)\n",
      (unsigned long)vpdVersion, (unsigned long)vpdVersion));
    break;
  }

  flashRead(pDevCtx, 0, pDevCtx->info.bootstrap.vpdOffset, sizeof(vpdRaw), &vpdRaw);
  DF_ZERO_OBJECT(pVpd);
  pVpd->version = vpdVersion;
  pVpd->boardType = dfLe16ToCpu(vpdRaw.boardType);
  pVpd->modifications = dfLe32ToCpu(vpdRaw.modifications);
  pVpd->stepping = dfLe8ToCpu(vpdRaw.stepping);
  pVpd->tempGrade = dfLe8ToCpu(vpdRaw.tempGrade);
  pVpd->scd = dfLe32ToCpu(vpdRaw.scd);
  pVpd->lclkRef = dfLe32ToCpu(vpdRaw.lclkRef);
  pVpd->mclkRef = dfLe32ToCpu(vpdRaw.mclkRef);
  for (i = 0; i < DF_ARRAY_LENGTH(vpdRaw.sdram); i++) {
    pVpd->sdram[i].order = dfLe8ToCpu(vpdRaw.sdram[i].order);
  }
  pVpd->cardId = dfLe32ToCpu(vpdRaw.cardId);
  pVpd->serialNumber = dfLe32ToCpu(vpdRaw.serialNumber);
  pVpd->fpgaType = dfLe16ToCpu(vpdRaw.fpgaType);
  pVpd->pcbRev = dfLe8ToCpu(vpdRaw.pcbRev);
  pVpd->cpldRev = (uint8_t)(0x10U + pDevCtx->pDevObj->hardwareId.variant.pci.revision);

  pDevCtx->info.vpd.bValid = TRUE;

  return TRUE;
}

static void
onIcs307ProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  Ics307DeviceContext* pIgnored,
  Ics307ProgramRequest* pRequest,
  Ics307ProgramStatus status)
{
  onClockProgramDone(pDevCtx, 0, ics307MapProgramStatus(status));
}

static void
onIcs8430_61ProgramDone(
  Adb3CoreDeviceContext* pDevCtx,
  Ics8430_61DeviceContext* pIgnored,
  Ics8430_61ProgramRequest* pRequest,
  Ics8430_61ProgramStatus status)
{
  onClockProgramDone(pDevCtx, 1, ics8430_61MapProgramStatus(status));
}

static ClockProgramStatus
clockProgram(
	Adb3CoreDeviceContext*	pDevCtx,
  unsigned int clockIndex,
	const CoreClockWord* pClockWord)
{
  switch (clockIndex) {
  case 0:
    return ics307MapProgramStatus(ics307Program(pDevCtx, &pDevCtx->model.admxrc5lx.ics307.context, &pDevCtx->model.admxrc5lx.ics307.programming, pClockWord, onIcs307ProgramDone));

  case 1:
    return ics8430_61MapProgramStatus(ics8430_61Program(pDevCtx, &pDevCtx->model.admxrc5lx.ics8430_61.context, &pDevCtx->model.admxrc5lx.ics8430_61.programming, pClockWord, onIcs8430_61ProgramDone));

  default:
    return ClockProgramInvalidIndex;
  }
}

static ClockWordStatus
clockWord(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int clockIndex,
  uint32_t flags,
  uint64_t frequencyReq,
  uint64_t frequencyMin,
  uint64_t frequencyMax,
  CoreClockWord* pClockWord)
{
  VpdAdmxrc5lxaRev0* pVpd = &pDevCtx->info.vpd.pData->admxrc5lxa.rev0;

  if (clockIndex >= pDevCtx->info.bootstrap.numClockGen) {
    return ClockWordInvalidIndex;
  }

  switch (clockIndex) {
  case 0:
    return ics307MapClockWordStatus(clockWordIcs307(flags, pVpd->lclkRef, frequencyReq, frequencyMin, frequencyMax, FALSE, pClockWord));

  case 1:
    return ics8430_61MapClockWordStatus(clockWordIcs8430_61(flags, pVpd->mclkRef, frequencyReq, frequencyMin, frequencyMax, pClockWord));

  default:
    return ClockWordInvalidIndex;
	}
}

static FpgaControlStatus
fpgaControl(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreFpgaControlOp opCode,
  boolean_t* pbValue)
{
  return adb1V4V5FpgaControl(pDevCtx, index, opCode, &pDevCtx->model.admxrc5lx.shadow.fpctl, pbValue);
}

static boolean_t
initHardware(
  Adb3CoreDeviceContext* pDevCtx,
  boolean_t bInit,
  unsigned int phase)
{
  Adb1BridgeRegisters* pRegs;
  DfMemoryHandle hBridge;
  ModelRegsAdb1V4V5* pModelRegs;
  DfMemoryHandle hModel;
  VpdAdmxrc5lxaRev0* pVpd = &pDevCtx->info.vpd.pData->admxrc5lxa.rev0;
  uint32_t ls0b, ls0r, ls1b, ls1r;
  uint32_t fOut;

  dfDebugPrint(1, ("initHardware: bInit=%s\n", bInit ? "TRUE" : "FALSE"));

  pRegs = pDevCtx->hardware.bridge.pAdb1;
  hBridge = pDevCtx->hardware.hBridge;
  pModelRegs = pDevCtx->hardware.model.pAdb1V4V5;
  hModel = pDevCtx->hardware.hModel;

  if (bInit) {
    switch (phase) {
    case 0:
      pDevCtx->info.model = CoreModelAdmxrc5lx;

      /* Ensure that all interrupts are disabled in Bridge */
      pDevCtx->hardware.shadow.bridge.adb1.pictlMaster = ADB1_PICTL_ENMODEL | ADB1_PICTL_ENDMA_ALL | ADB1_PICTL_ENINTA;
      dfPciMemWrite32(hBridge, &pRegs->pictl, 0);
      (void)dfPciMemRead32(hBridge, &pRegs->pictl);

      /* Enable target FPGA interrupt */
      pDevCtx->model.admxrc5lx.shadow.ictl = ADB1V4V5_ICTL_ENFPGA;
      dfPciMemWrite32(hModel, &pModelRegs->ictl, pDevCtx->model.admxrc5lx.shadow.ictl);
      (void)dfPciMemRead32(hModel, &pModelRegs->ictl);

      /* Enable DMA engines */
      adb1EnableDmaEngines(pDevCtx, TRUE);

      /* Establish FPGA control register shadow */
      pDevCtx->model.admxrc5lx.shadow.fpctl = 0;
      dfPciMemWrite32(hModel, &pModelRegs->fpctl, pDevCtx->model.admxrc5lx.shadow.fpctl);

      /* Reset Flash and identify it */
      adb1V4V5FlashReset(pDevCtx);
      if (flashCfiIdentify(pDevCtx, 0, 0)) {
        pDevCtx->info.flash[0].bPresent = TRUE;
        pDevCtx->info.flash[0].targetArea.start = ADMXRC5LX_FLASH_TARGET_AREA_START;
        pDevCtx->info.flash[0].targetArea.length = pDevCtx->info.flash[0].totalSize - ADMXRC5LX_FLASH_TARGET_AREA_START;
        readVpd(pDevCtx);
      } else {
        dfDebugPrint(0, ("*** initHardware: failed to identify Flash\n"));
      }

      /* Initialize ICS307 stuff */
      if (!ics307Init(
        pDevCtx,
        &pDevCtx->model.admxrc5lx.ics307.context,
        pVpd->lclkRef,
        Ics307InterfaceTypeXpl,
        FALSE,
        &pDevCtx->hardware.model.pAdb1V4V5->clocking.admxrc5lx.lclkctl,
        pDevCtx->hardware.hModel,
        &fOut))
      {
        dfDebugPrint(0, ("*** initHardware: failed to initialize ICS307 clock synthesizer\n"));
      } else {
        pDevCtx->clockGenerator[0].currentFrequency = fOut;
      }

      /* Initialize ICS8430-61 synthesizer stuff */
      if (!ics8430_61Init(
        pDevCtx,
        &pDevCtx->model.admxrc5lx.ics8430_61.context,
        pVpd->mclkRef,
        &pDevCtx->hardware.model.pAdb1V4V5->clocking.admxrc5lx.mclkctl,
        pDevCtx->hardware.hModel,
        &fOut))
      {
        dfDebugPrint(0, ("*** initHardware: failed to initialize ICS8430-61 clock synthesizer\n"));
      } else {
        pDevCtx->clockGenerator[1].currentFrequency = fOut;
      }

      /* Initialize information about sensors */
      (void)initSensorInfo(pDevCtx, 0U);

      /* Find base addresses and sizes of local bus windows */
      pDevCtx->info.window[0].busBase = pDevCtx->rawResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].busSize = pDevCtx->rawResources.variant.pci.memBar[1].size;
      ls0b = dfPciMemRead32(hBridge, &pRegs->ls0b);
      ls0r = dfPciMemRead32(hBridge, &pRegs->ls0r);
      ls0r &= 0xfffffff0U;
      ls0b &= ls0r;
      pDevCtx->info.window[0].localBase = ls0b;
      pDevCtx->info.window[0].localSize = ~ls0r + 1;
      pDevCtx->info.window[0].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[1].base;
      pDevCtx->info.window[0].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[0].pKernelBase = pDevCtx->hardware.pBridged;
      pDevCtx->info.window[0].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[1].size;
      pDevCtx->info.window[1].busBase = pDevCtx->rawResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].busSize = pDevCtx->rawResources.variant.pci.memBar[2].size;
      ls1b = dfPciMemRead32(hBridge, &pRegs->ls1b);
      ls1r = dfPciMemRead32(hBridge, &pRegs->ls1r);
      ls1r &= 0xfffffff0U;
      ls1b &= ls1r;
      pDevCtx->info.window[1].localBase = ls1b;
      pDevCtx->info.window[1].localSize = ~ls1r + 1;
      pDevCtx->info.window[1].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[2].base;
      pDevCtx->info.window[1].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[1].pKernelBase = pDevCtx->hardware.model.pAdb1V4V5;
      pDevCtx->info.window[1].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[2].size;
      pDevCtx->info.window[2].busBase = pDevCtx->rawResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].busSize = pDevCtx->rawResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].localBase = 0;
      pDevCtx->info.window[2].localSize = 0;
      pDevCtx->info.window[2].translatedBase = pDevCtx->translatedResources.variant.pci.memBar[0].base;
      pDevCtx->info.window[2].translatedSize = pDevCtx->translatedResources.variant.pci.memBar[0].size;
      pDevCtx->info.window[2].pKernelBase = pDevCtx->hardware.bridge.pGeneric;
      pDevCtx->info.window[2].kernelSize = (size_t)pDevCtx->translatedResources.variant.pci.memBar[0].size;
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  } else {
    switch (phase) {
    case 0:
      /* Deinitialize ICS8430-61 stuff */
      ics8430_61Uninit(pDevCtx, &pDevCtx->model.admxrc5lx.ics8430_61.context);

      /* Deinitialize ICS307 stuff */
      ics307Uninit(pDevCtx, &pDevCtx->model.admxrc5lx.ics307.context);

      if (pDevCtx->hardware.bCanTouch) {
        /* Disable DMA engines */
        adb1EnableDmaEngines(pDevCtx, FALSE);
      }
      break;

    case 1:
      /* Nothing to do in phase 1 */
      break;

    default:
      dfAssert(FALSE);
      break;
    }
  }

  dfDebugPrint(1, ("initHardware: done\n"));
  return TRUE;
}

static boolean_t
isr(
  DfInterruptObject* pInterruptObj,
  void* pInterruptContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterruptContext;

  return adb1V4V5HandleInterrupt(pDevCtx, pDevCtx->hardware.shadow.bridge.adb1.pictl, pDevCtx->model.admxrc5lx.shadow.ictl);
}

static void
powerChange(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int currentState,
  unsigned int newState)
{
  Ics8430_61ProgramStatus ics8430_61Status;
  Ics307ProgramStatus ics307Status;
  ModelRegsAdb1V4V5* pModelRegs;
  DfMemoryHandle hModel;

  dfAssert(newState == 0 || newState == 3);

  pModelRegs = pDevCtx->hardware.model.pAdb1V4V5;
  hModel = pDevCtx->hardware.hModel;

  dfDebugPrint(1, ("powerChange: D%u->D%u\n", currentState, newState));
  if (currentState != newState) {
    if (currentState < newState) { /* Powering down (to D0) */
      /* Save ICS307 state */
      ics307SaveState(pDevCtx, &pDevCtx->model.admxrc5lx.ics307.context);

      /* Save ICS8430-61 state */
      ics8430_61SaveState(pDevCtx, &pDevCtx->model.admxrc5lx.ics8430_61.context);
    } else { /* Powering up (from D3) */
      /* Reset Flash device */
      adb1V4V5FlashReset(pDevCtx);

      /* Reenable target FPGA interrupt */
      dfPciMemWrite32(hModel, &pModelRegs->ictl, pDevCtx->model.admxrc5lx.shadow.ictl);
      (void)dfPciMemRead32(hModel, &pModelRegs->ictl);

      /* Restore ICS8430-61 state */
      ics8430_61Status = ics8430_61RestoreState(pDevCtx, &pDevCtx->model.admxrc5lx.ics8430_61.context);
      if (Ics8430_61ProgramStatusSuccess != ics8430_61Status) {
        dfDebugPrint(0, ("*** powerChange: failed to restore ICS8430-61 state, status=%u\n", ics8430_61Status));
      }

      /* Restore ICS307 state */
      ics307Status = ics307RestoreState(pDevCtx, &pDevCtx->model.admxrc5lx.ics307.context);
      if (Ics307ProgramStatusSuccess != ics307Status) {
        dfDebugPrint(0, ("*** powerChange: failed to restore ICS307 state, status=%u\n", ics307Status));
      }
    }
  }
}

void
adb3CoreBootstrapAdmxrc5lx(
  Adb3CoreDeviceContext* pDevCtx)
{
  dfDebugPrint(1, ("adb3CoreBootstrapAdmxrc5lx: entered\n"));

  pDevCtx->methods.pClockProgram = clockProgram;
  pDevCtx->methods.pClockWord = clockWord;
  pDevCtx->methods.pDmaList = adb1DmaList;
  pDevCtx->methods.pDmaTransfer = adb1DmaTransfer;
  pDevCtx->methods.pFlash = adb1V4V5Flash;
  pDevCtx->methods.pEnableInterrupts = adb1EnableInterrupts;
  pDevCtx->methods.pFpgaControl = fpgaControl;
  pDevCtx->methods.pFpgaSelectMap = adb1V4V5FpgaSelectMap;
  pDevCtx->methods.pInitHardware = initHardware;
  pDevCtx->methods.pIsr = isr;
  pDevCtx->methods.pMapBusResources = adb1V4V5MapBusResources;
  pDevCtx->methods.pPowerChange = powerChange;
  pDevCtx->methods.pReadSensor = readSensor;
  pDevCtx->methods.pReadWriteVpd = adb1V4V5ReadWriteVpd;
  pDevCtx->methods.pSyncVpd = adb1V4V5SyncVpd;
  pDevCtx->methods.pValidateBusResources = adb1V4V5ValidateBusResources;
  pDevCtx->methods.pWindowConfig = adb1WindowConfig;

  pDevCtx->info.bootstrap.numClockGen = 2;
  pDevCtx->info.bootstrap.numDmaChannel = ADMXRC5LX_NUM_DMA_CHANNEL;
  pDevCtx->info.bootstrap.numIoModule = 1;
  pDevCtx->info.bootstrap.numSensor = ADMXRC5LX_NUM_LM87_SENSOR;
  pDevCtx->info.bootstrap.numTargetFpga = 1;
  pDevCtx->info.bootstrap.numWindow = 3;
  pDevCtx->info.bootstrap.numFlashBank = 1;
  pDevCtx->info.bootstrap.bVpdInFlash = TRUE;
  pDevCtx->info.bootstrap.vpdLength = ADMXRC5LX_VPD_SIZE_IN_FLASH;
  pDevCtx->info.bootstrap.vpdOffset = ADMXRC5LX_VPD_OFFSET_IN_FLASH;
  pDevCtx->info.bootstrap.vpdBufferLength = sizeof(VpdAdmxrc5lxRev0);
  pDevCtx->info.bootstrap.flash[0].deviceWidth = 2;
  pDevCtx->info.bootstrap.flash[0].width = 2;
  pDevCtx->info.bootstrap.b64BitDmaAddress = FALSE;
  pDevCtx->info.bootstrap.dmaDescriptorSize = 32;
  pDevCtx->info.bootstrap.dmaTransferMaxSize = 0x800000U;
}
