/*
** File: device.h  
** Project: ADB3 core driver
** Purpose: Data structures and definitions private to ADB3 core driver,
**          including the device context structure.
**
** (C) Copyright Alpha Data 2009-2015
*/

#if !defined(ADATA_CORE_DEVICE_H)
#define ADATA_CORE_DEVICE_H

#include <df.h>

#include "adb3.h"
#include "cfi.h"
#include "i2c.h"
#include "icap.h"
#include "serializer.h"
#include "shared_defs.h"
#include "device_defs.h"
#include "si5338.h"
#include "icd2061.h"
#include "ics843034_01.h"
#include "pci9xxx.h"
#if defined(ADMXRC_DRIVER)
#include "ics307.h"
#include "ics8430_61.h"
#include "admxrc.h"
#include "admxrc2l.h"
#include "admxrc2.h"
#include "admxpl.h"
#include "admxp.h"
#include "admxrc4.h"
#include "admxrc4fx.h"
#include "adcpxrc4lx.h"
#include "adpexrc4fx.h"
#include "admamc5a2.h"
#include "admxrc5lx.h"
#include "admxrc5lxa.h"
#include "admxrc5t1.h"
#include "admxrc5t2.h"
#include "admxrc5tda1.h"
#include "admxrc5tz.h"
#include "adcbbp.h"
#include "admpcie6s1.h"
#include "adpdrc2.h"
#include "v4clksynth.h"
#endif
#if defined(ADB3_DRIVER)
#include "admxrc6tadv8.h"
#include "admxrc6tl.h"
#include "admxrc6t1.h"
#include "admxrc6tda1.h"
#include "admxrc6tge.h"
#include "admxrc6tx.h"
#include "admxrc7k1.h"
#include "admxrc7v1.h"
#include "adpexrc5t.h"
#include "adpexrc6t.h"
#include "adpexrc6tadv.h"
#include "admxrc7z1.h"
#include "admxrcku1.h"
#include "admpcie7v3.h"
#include "admpcie8v3.h"
#include "admpcie8k5.h"
#include "admpcieku3.h"
#include "avr.h"
#include "avr_clock.h"
#include "avr_common.h"
#include "avr2.h"
#include "avr2_clock.h"
#include "avr2_common.h"
#include "cfi.h"
#include "pci9xxx.h"
#include "shared_defs.h"
#include "device_defs.h"
#include "coreif.h"
#include "i2c.h"
#include "icap.h"
#include "icd2061.h"
#include "ics843034_01.h"
#include "serializer.h"
#include "si5338.h"
#include "v5clksynth.h"
#include "v6clksynth.h"
#endif
#if defined(ADB3_CUSTOM)
#include <core/device_custom.h>
#endif

#if defined(ADB3_DRIVER)
# if DF_NEED_THUNK
#  define ADB3_NEED_THUNK (1)
# endif
# include <ioctl_adb3.h>
# include <adb3/status.h>
#endif

struct _Adb3CoreClientContext;
typedef struct _Adb3CoreClientContext Adb3CoreClientContext;
struct _Adb3CoreDeviceContext;
typedef struct _Adb3CoreDeviceContext Adb3CoreDeviceContext;
struct _Adb3CoreNotificationList;
typedef struct _Adb3CoreNotificationList Adb3CoreNotificationList;
struct _Adb3CoreFlashInfo;
typedef struct _Adb3CoreFlashInfo Adb3CoreFlashInfo;
struct _Adb3CoreFlashControl;
typedef struct _Adb3CoreFlashControl Adb3CoreFlashControl;
union _Adb3CoreVpd;
typedef union _Adb3CoreVpd Adb3CoreVpd;
struct _ClockGeneratorContext;
typedef struct _ClockGeneratorContext ClockGeneratorContext;
struct _DmaBusMapperSlice;
typedef struct _DmaBusMapperSlice DmaBusMapperSlice;
struct _DmaChannelContext;
typedef struct _DmaChannelContext DmaChannelContext;
struct _DmaDescriptors;
typedef struct _DmaDescriptors DmaDescriptors;
struct _DmaMappingInfo;
typedef struct _DmaMappingInfo DmaMappingInfo;
struct _DmaNodeTablePosition;
typedef struct _DmaNodeTablePosition DmaNodeTablePosition;
enum _DmaRequestType;
typedef enum _DmaRequestType DmaRequestType;
struct _WindowInfo;
typedef struct _WindowInfo WindowInfo;
struct _LegacyFlashInformation;
typedef struct _LegacyFlashInformation LegacyFlashInformation;

typedef struct _ResourceQueue {
  CoreResourceDesc* pHead;
  CoreResourceDesc* pTail;
  unsigned int      length;
  unsigned int      users;
} ResourceQueue;

typedef enum _ClockProgramStatus {
  ClockProgramSuccess = 0,
  ClockProgramInvalidIndex = 1,
  ClockProgramOutOfRange = 2,
  ClockProgramHardwareError = 3,
  ClockProgramGeneralFailure = 4
} ClockProgramStatus;

typedef enum _ClockWordStatus {
  ClockWordSuccess = 0,
  ClockWordInvalidIndex = 1,
  ClockWordOutOfRange = 2,
  ClockWordGeneralFailure = 4
} ClockWordStatus;

typedef enum _FpgaControlStatus {
  FpgaControlSuccess = 0,
  FpgaControlInvalidIndex = 1,
  FpgaControlInvalidOp = 2,
  FpgaControlNotSupported = 3
} FpgaControlStatus;

typedef enum _FpgaSelectMapStatus {
  FpgaSelectMapSuccess = 0,
  FpgaSelectMapInvalidIndex = 1,
  FpgaSelectMapNotSupported = 2
} FpgaSelectMapStatus;

struct _ClockGeneratorContext {
  unsigned int index;
  uint64_t currentFrequency; /* 0 means 'not valid / not programmed / last program failed' */
  struct {
    unsigned int state;
    CoreClockWord word;
    CoreClockProgCallback* pCallback;
    CoreTicket* pCallbackTicket;
    void* pCallbackContext;
  } program;
};

struct _DmaNodeTablePosition {
  unsigned int index;
  size_t offset;
};

/*
** Holds information about completed setups:
** - If 'type' field of DmaContext is 'DmaRequestTypeUser', 'pElements' gets
**   set to the scatter-gather table handed to us by framework when our
**   mapping callback is invoked.
** - If 'type' field of DmaContext is 'DmaRequestTypeBus', 'pElements' gets
**   set to the scatter-gather table we generate ourselves in the
**   DmaContext::dmaBus[]::pTable tables.
*/
struct _DmaMappingInfo {
  DfDmaMapperNode* pTable;       /* Points to scatter-gather table */
  unsigned int length;           /* Number of elements in scatter-gather table */
  DmaNodeTablePosition position; /* Current position in scatter-gather table for current DMA chunk */
};

enum _DmaRequestType {
  DmaRequestTypeUser = 0, /* Transferring from/to device to/from user-space buffer */
  DmaRequestTypeBus  = 1  /* Transferring from/to device to/from PCI-E address */
};

/* Structure containing DMA descriptor information; two per DMA engine to allow double-buffering */
struct _DmaDescriptors {
  DfDmaBuffer* pDmaBuffer;     /* Driver framework object representing buffer holding DMA descriptors */
  union {                      /* Kernel address of DMA descriptor buffer */
    void* pGeneric;
    Adb3DmaDescriptor* pAdb3;  /* Specific to ADB3-based hardware */
  } buffer;
  uint64_t busAddress;         /* Bus address of DMA descriptor buffer */
  size_t length;               /* Length of DMA descriptor buffer in bytes */
};

struct _DmaBusMapperSlice {
  uint64_t busAddress;
  uint64_t localAddress;
  size_t length;
};

struct _DmaChannelContext {
  DfSpinLock lock;                 /* Used to effect state transitions atomically */
  DmaDescriptors descriptor[2];    /* Structure containing DMA descriptor information; two per DMA engine to allow double-buffering */
  struct {                         /* Used ONLY when 'type' field below is 'DmaRequestTypeBus' */
    DfDmaMapperNode* pTable;         /* Analogous to the scatter-gather tables handed to us by framework */
  } dmaBus[2];
  struct {                         /* Information for currently executing DMA transfer */
    CoreTicket* pTicket;             /* Ticket of current DMA transfer */
    CoreDmaStatus status;            /* Status of current DMA transfer (success, failure etc.) */
    CoreDmaCallback* pCallback;      /* Completion callback of current DMA transfer */
    void* pCallbackContext;          /* Client-specified value of current DMA transfer */
    union {                          /* Holds client-supplied parameters of current DMA transfer */
      CoreDmaParameters user;          /* Valid if DMA to/from user-mode buffer (see 'type' field below) */
      CoreDmaBusParameters bus;        /* Valid if DMA to/from PCI-E address (see 'type' field below) */
    } parameters;
    union {
      DfDmaMapperSlice user;
      DmaBusMapperSlice bus;
    } slice;
    DfDmaMapperPosition position;
    DmaMappingInfo table[2];
    DmaRequestType type;             /* Type of DMA transfer: targetting user-mode buffer, or targetting device on bus */
    struct {                         /* Holds state information for current DMA transfer */
      boolean_t bCancel;               /* TRUE if DMA transfer is being cancelled */
      unsigned int main;               /* Overall state: { doing a DMA transfer, not doing a DMA transfer } */
      struct {                         /* DMA chunk setup state */
        unsigned int state;              /* Possible states: { doing setup, not doing setup } */
        unsigned int tableIndex;         /* Index into 'table'; can be 0 or 1 */
        unsigned int count;              /* Number of completely setup descriptor buffers; incremented when a setup completes, decremented when a teardown completes */
#if defined(BUILD_DEBUGTS)
        unsigned int chunkIndex;
#endif
      } setup;
      struct {                         /* DMA chunk hardware state */
        unsigned int state;              /* Possible states: { hardware is doing DMA, hardware is not doing DMA } */
        unsigned int tableIndex;         /* Index into 'table' and 'descriptor'; can be 0 or 1 */
        unsigned int count;              /* Number of descriptor buffers ready; incremented when a setup completes, decremented when a DMA chunk completes on hardware */
#if defined(BUILD_DEBUGTS)
        unsigned int chunkIndex;
#endif
      } transfer;
      struct {                         /* DMA chunk teardown state */
        unsigned int state;              /* Possible states: { doing teardown, not doing teardown } */
        unsigned int tableIndex;         /* Index into 'table' and 'descriptor'; can be 0 or 1 */
        unsigned int count;              /* Number of descriptor buffers ready for teardown; incremented when a DMA chunk completes on hardware; decremented by when a teardown completes */
#if defined(BUILD_DEBUGTS)
        unsigned int chunkIndex;
#endif
      } teardown;
    } state;
  } curr;
#if defined(BUILD_DEBUGTS)
  DfTimestampBuffer timestampBuffer;
  uint64_t isrTimestamp;
#endif
};

typedef enum _DmaCompletionCode {
  DmaCompletionNormal = 0,
  DmaCompletionFailed = 1,
  DmaCompletionEot = 2
} DmaCompletionCode;

struct _WindowInfo {
  DfSpinLock lock;
  uint64_t busSize;
  uint64_t busBase;
  uint64_t localSize;
  uint64_t localBase;
  uint64_t translatedSize;
  uint64_t translatedBase;
  void* pKernelBase;
  size_t kernelSize;
};

struct _Adb3CoreNotificationList {
  DfSpinLock lock;
  DfList header;
};

union _Adb3CoreVpd {
  uint8_t generic[1];
#if defined(ADMXRC_DRIVER)
  union {
    VpdAdmxrcRev0 rev0;
  } admxrc;
  union {
    VpdAdmxrc2lRev0 rev0;
  } admxrc2l;
  union {
    VpdAdmxrc2Rev0 rev0;
  } admxrc2;
  union {
    VpdAdpdrc2Rev0 rev0;
  } adpdrc2;
  union {
    VpdAdpwrc2Rev0 rev0;
  } adpwrc2;
  union {
    VpdAdmxplRev0 rev0;
  } admxpl;
  union {
    VpdAdmxpRev0 rev0;
  } admxp;
  union {
    VpdAdpxpiRev0 rev0;
  } adpxpi;
  union {
    VpdAdmxrc4Rev0 rev0;
  } admxrc4;
  union {
    VpdAdcpxrc4lxRev0 rev0;
  } adcpxrc4lx;
  union {
    VpdAdmxrc4fxRev0 rev0;
  } admxrc4fx;
  union {
    VpdAdpexrc4fxRev1 rev1;
  } adpexrc4fx;
  union {
    VpdAdmamc5a2Rev0 rev0;
  } admamc5a2;
  union {
    VpdAdmxrc5lxaRev0 rev0;
  } admxrc5lxa;
  union {
    VpdAdmxrc5t1Rev0 rev0;
  } admxrc5t1;
  union {
    VpdAdmxrc5t2Rev0 rev0;
  } admxrc5t2;
  union {
    VpdAdmxrc5tda1Rev0 rev0;
  } admxrc5tda1;
  union {
    VpdAdmxrc5tzRev0 rev0;
  } admxrc5tz;
  union {
    VpdAdcbbpRev0 rev0;
  } adcbbp;
  union {
    VpdAdmpcie6s1Rev0 rev0;
  } admpcie6s1;
#endif
#if defined(ADB3_DRIVER)
  union {
    VpdAdpexrc5tRev0 rev0;
  } adpexrc5t;
  union {
    VpdAdmxrc6tlRev1 rev1;
  } admxrc6tl;
  union {
    VpdAdmxrc6t1Rev0 rev0;
  } admxrc6t1;
  union {
    VpdAdmxrc6tgeRev0 rev0;
  } admxrc6tge;
  union {
    VpdAdmxrc6tadv8Rev0 rev0;
  } admxrc6tadv8;
  union {
    VpdAdpexrc6tRev0 rev0;
  } adpexrc6t;
  union {
    VpdAdpexrc6tadvRev0 rev0;
  } adpexrc6tadv;
  union {
    VpdAdmxrc6tda1Rev0 rev0;
  } admxrc6tda1;
  union {
    VpdAdmxrc7k1Rev0 rev0;
  } admxrc7k1;
  union {
    VpdAdmxrc7v1Rev0 rev0;
  } admxrc7v1;
  union {
    VpdAdmxrc7z1Rev0 rev0;
  } admxrc7z1;
  union {
    VpdAdmxrcku1Rev0 rev0;
  } admxrcku1;
  VpdAdmpcie7v3 admpcie7v3;
  VpdAdmpcieku3 admpcieku3;
  VpdAdmpcie8v3 admpcie8v3;
  VpdAdmpcie8k5 admpcie8k5;
#endif
};

struct _LegacyFlashInformation {
  struct {                      /* Special blocks */
    uint32_t address;
    uint32_t size;
  } specialBlock[FLASH_LEGACY_MAX_NUM_SPECIAL_BLOCK];
  uint32_t blockSize;           /* Normal block size */
  uint32_t largestBlockSize;    /* Largest block size */
  uint16_t vendorId;
  uint16_t deviceId;
  unsigned int numSpecialBlock; /* Number of special blocks */
};

struct _Adb3CoreFlashInfo {
  boolean_t bPresent;
  boolean_t bLegacyFlash;
  struct {
    void* pGeneric;
    CfiInformation* pCfi;
    LegacyFlashInformation* pLegacy;
  } detail;
  uint64_t totalSize; /* total size (of all dies) */
  struct { /* defines start and size of target FPGA bitstream area */
    uint64_t start;
    uint64_t length;
  } targetArea;
};

typedef enum _FlashCacheState {
  FlashCacheInvalid,  /* We don't have a cached copy of any Flash block */
  FlashCacheValid,    /* We have an unmodified copy of a Flash block */
  FlashCacheDirty     /* We have a modified copy of a Flash block */
} FlashCacheState;

struct _Adb3CoreFlashControl {
  struct {
    FlashCacheState state; /* Invalid, Valid or Dirty */
    uint64_t address;      /* This member is valid when state is Valid or Dirty */
    uint64_t length;       /* This member is valid when state is Valid or Dirty */
    uint8_t* pBuffer;      /* Data in this buffer is valid when state is Valid or Dirty */
  } cache;
};

typedef enum _IprogControlOp {
  IprogControlOpScheduleFromNow,
  IprogControlOpScheduleOnStop,
  IprogControlOpAbort,
} IprogControlOp;

/*
** ----------------------------------------------------------------
** The core device context structure
** ----------------------------------------------------------------
*/

struct _Adb3CoreDeviceContext {
	DfDeviceObject* pDevObj;
	boolean_t bCoreInterfaceRegd;
  DfBusResources rawResources;
  DfBusResources translatedResources;
  boolean_t bDmaMapperAllocated;

  struct { /* Support for SetCoopLevel */
    DfSpinLock lock;        /* Protects this struct. */
    CoreClient* pExclusiveClient; /* Handle to whoever has exclusivity for this device, or NULL if nobody. */
    DfList clients;
  } cooperation;

#if defined(ADB3_DRIVER)
  struct {
    DfInterface iface;
    boolean_t bEnabled;
    boolean_t bRegistered;
  } userMode;
#endif

	/* This structure contains method pointers for performing model-specific actions */
	struct {
		/* The following method is responsible for initializing all of the other method pointers */
		void (*pModelBootstrap)(Adb3CoreDeviceContext* pDevCtx);

		/* These following method pointers are initialized by the ModelBootstrap method */
    boolean_t (*pAlertPoll)(Adb3CoreDeviceContext* pDevCtx, unsigned int targetIndex);
    void (*pClearErrors)(Adb3CoreDeviceContext* pDevCtx);
    ClockProgramStatus (*pClockProgram)(Adb3CoreDeviceContext* pDevCtx, unsigned int index, const CoreClockWord* pClockWord);
    ClockWordStatus (*pClockWord)(Adb3CoreDeviceContext* pDevCtx, unsigned int index, uint32_t flags, uint64_t frequencyReq, uint64_t frequencyMin, uint64_t frequencyMax, CoreClockWord* pClockWord);
    void (*pDmaList)(Adb3CoreDeviceContext* pDevCtx, boolean_t bWriteToDevice, boolean_t bFixedLocal, DfDmaMapperNode* pTable, unsigned int tableLength, DmaNodeTablePosition* pTablePosition, void* pDescriptorBuffer, unsigned int maxDescriptor, uint64_t descriptorBusAddress);
    void (*pDmaTransfer)(Adb3CoreDeviceContext* pDevCtx, unsigned int channel, boolean_t bStart, void* pDescriptorBuffer, uint64_t descriptorBusAddress);
		void (*pEnableInterrupts)(Adb3CoreDeviceContext* pDevCtx, boolean_t bDisable);
		void (*pFlash)(Adb3CoreDeviceContext* pDevCtx, unsigned int bankIndex, boolean_t bWrite, uint64_t address, uint32_t* pItem);
		void (*pFlashRemap)(Adb3CoreDeviceContext* pDevCtx, unsigned int bankIndex, uint64_t logicalAddr, uint64_t length, uint64_t* pPhysicalAddr, uint64_t* pLengthOut);
		FpgaControlStatus (*pFpgaControl)(Adb3CoreDeviceContext* pDevCtx, unsigned int index, CoreFpgaControlOp opCode, boolean_t* pbValue);
		FpgaSelectMapStatus (*pFpgaSelectMap)(Adb3CoreDeviceContext* pDevCtx, unsigned int index, boolean_t bWrite, size_t length, void* pBuffer);
    void (*pGetDeviceStatus)(Adb3CoreDeviceContext* pDevCtx, CoreDeviceStatus* pDeviceStatus);
    IprogResult (*pIprogControl)(Adb3CoreDeviceContext* pDevCtx, uint32_t targetIndex, IprogControlOp, uint64_t address, uint32_t delayMs);
    IprogResult (*pIprogStatus)(Adb3CoreDeviceContext* pDevCtx, uint32_t targetIndex, IprogScheduleState* pState, uint64_t* pAddress, uint32_t* pDelayMs);
		boolean_t (*pInitHardware)(Adb3CoreDeviceContext* pDevCtx, boolean_t bInit, unsigned int phase);
    boolean_t (*pIsr)(DfInterruptObject* pInterruptObj, void* pContext);
		boolean_t (*pMapBusResources)(Adb3CoreDeviceContext* pDevCtx, DfBusResources* pRaw, DfBusResources* pTranslated, boolean_t bUnmap);
		void (*pPowerChange)(Adb3CoreDeviceContext* pDevCtx, unsigned int currentState, unsigned int newState);
		CoreSensorStatus (*pReadSensor)(Adb3CoreDeviceContext* pDevCtx, unsigned int sensorIndex, CoreSensorValue* pReading);
		CoreVpdStatus (*pReadWriteVpd)(Adb3CoreDeviceContext* pDevCtx, boolean_t bWrite, size_t address, size_t length, void* pData);
		CoreVpdStatus (*pSyncVpd)(Adb3CoreDeviceContext* pDevCtx);
		boolean_t (*pValidateBusResources)(Adb3CoreDeviceContext* pDevCtx, DfBusResources* pRaw, DfBusResources* pTranslated);
    CoreWindowConfigStatus (*pWindowConfig)(Adb3CoreDeviceContext* pDevCtx, unsigned int windowIndex, boolean_t bConfigure, uint32_t flags, CoreWindowConfig* pConfig);
	} methods;

	struct {
    boolean_t bInitializedPhase0;
    boolean_t bInitializedPhase1;
    boolean_t bMapped;
    boolean_t bCanTouch;

		union {
			void* pGeneric;
#if defined(ADB3_DRIVER)
      Adb3GenericRegs* pAdb3;
#endif
      Pci9080Registers* pPci9080;
      Pci9656Registers* pPci9656;
#if defined(ADMXRC_DRIVER)
      Adb1BridgeRegisters* pAdb1;
#endif
		} bridge;
    DfMemoryHandle hBridge;

		struct {
      union {
#if defined(ADMXRC_DRIVER)
        struct {
          uint32_t pictl;
          uint32_t pictlMaster;
          uint32_t dmaCtl[ADB1_MAX_NUM_DMA_ENGINE];
        } adb1;
        struct {
          uint32_t intcsrMask;
          uint32_t intcsrMaskMaster;
          uint32_t intcsr;
          uint32_t intcsrMaster;
          uint16_t dmacsr;
        } pci9xxx;
#endif
#if defined(ADB3_DRIVER)
        struct {
          uint32_t irqEnable;
          uint32_t irqEnableMaster;
        } adb3;
#endif
      } bridge;
		} shadow;

		union {
			void* pGeneric;
#if defined(ADMXRC_DRIVER)
      ModelRegsAdmxrc* pAdmxrc;
      ModelRegsAdmxrc2l* pAdmxrc2l;
      ModelRegsAdmxrc2* pAdmxrc2;
      ModelRegsAdpdrc2* pAdpdrc2;
      ModelRegsAdmxrc4* pAdmxrc4;
      ModelRegsAdmxpl* pAdmxpl;
      ModelRegsAdmxp* pAdmxp;
      ModelRegsAdb1V4V5* pAdb1V4V5;
      ModelRegsAdpexrc4fx* pAdpexrc4fx;
      ModelRegsAdcbbp* pAdcbbp;
      ModelRegsAdmpcie6s1* pAdmpcie6s1;
#endif
#if defined(ADB3_DRIVER)
      ModelRegsAdpexrc5t* pAdpexrc5t;
      ModelRegsAdmxrc6tx* pAdmxrc6tx;
      ModelRegsAdpexrc6tadv* pAdpexrc6tadv;
#endif
		} model;
    DfMemoryHandle hModel;

    void* pBridged;
    DfMemoryHandle hBridged;
    void* pBridgedPrefetch;
    DfMemoryHandle hBridgedPrefetch;
  } hardware;

  struct {
    DfSpinLock lock;
    struct {
      ResourceQueue queue;
      unsigned int users[MAX_NUM_CLOCK_GENERATOR];
#if defined(DF_DBG_BUILD)
      CoreTicket* pCurrentTicket[MAX_NUM_CLOCK_GENERATOR];
#endif
    } clockGen;
    struct {
      ResourceQueue queue;
      unsigned int users[MAX_NUM_DMA_ENGINE];
#if defined(DF_DBG_BUILD)
      CoreTicket* pCurrentTicket[MAX_NUM_DMA_ENGINE];
#endif
    } dmaEngine;
    struct {
      ResourceQueue queue;
      unsigned int users[1];
#if defined(DF_DBG_BUILD)
      CoreTicket* pCurrentTicket[1];
#endif
    } eeprom;
    struct {
      ResourceQueue queue;
      unsigned int users[MAX_NUM_FLASH_BANK];
#if defined(DF_DBG_BUILD)
      CoreTicket* pCurrentTicket[MAX_NUM_FLASH_BANK];
#endif
    } flash;
    struct {
      ResourceQueue queue;
      unsigned int users[MAX_NUM_TARGET_FPGA];
#if defined(DF_DBG_BUILD)
      CoreTicket* pCurrentTicket[MAX_NUM_TARGET_FPGA];
#endif
    } targetFpga;
    struct {
      ResourceQueue queue;
      unsigned int users[1];
#if defined(DF_DBG_BUILD)
      CoreTicket* pCurrentTicket[1];
#endif
    } sysMon;
  } resource;

  struct {
    struct { /* The following information must be valid when the ModelBootstrap method returns */
      unsigned int numDmaChannel;
      unsigned int numClockGen;
      unsigned int numIoModule;
      unsigned int numSensor;
      boolean_t bIncludeDiagSensors;
      unsigned int numTargetFpga;
      unsigned int numWindow;
      unsigned int numFlashBank;
      boolean_t bVpdInFlash;
      uint32_t vpdLength;
      uint32_t vpdOffset;
      unsigned int vpdBufferLength;
      struct {
        unsigned int deviceWidth; /* Width of Flash device data bus, in bytes */ 
        unsigned int width;       /* Width of Flash data actually used on the board */ 
      } flash[MAX_NUM_FLASH_BANK];
      boolean_t b64BitDmaAddress;
      unsigned int dmaDescriptorSize;
      size_t dmaTransferMaxSize;
      boolean_t bWorkAround4kCrossing;
      struct {
        unsigned int bNoFpgaToHost : 1; /* DMA engine doesn't support FPGA to Host transfers */
        unsigned int bNoHostToFpga : 1; /* DMA engine doesn't support Host to FPGA transfers */
      } dmaRestrictions[MAX_NUM_DMA_ENGINE];
      boolean_t bI2cInterruptDriven;
    } bootstrap;

    /* The following are initialized by identifyFlash */
    Adb3CoreFlashInfo flash[MAX_NUM_FLASH_BANK];

    /* The following are initialized by initHardware */
    CoreModelType model;

    struct {
      Adb3CoreVpd* pData;
      boolean_t bValid;
      boolean_t bChecksumError;
    } vpd;

    WindowInfo window[MAX_NUM_LOCAL_BUS_WINDOW];

    CoreSensorInfo sensor[MAX_NUM_SENSOR];

    CoreIoModuleInfo ioModule[MAX_NUM_IO_MODULE];

    struct {
      uint32_t serialNumber;
      uint32_t bridgeDate;
      uint32_t bridgeTime;
      uint8_t cpldRevision;
      uint8_t pcbRevision;
    } prodTest;
  } info;

  Adb3CoreFlashControl flashControl[MAX_NUM_FLASH_BANK];

  ClockGeneratorContext clockGenerator[MAX_NUM_CLOCK_GENERATOR];

  struct {
    Adb3CoreNotificationList fpgaInterrupt[MAX_NUM_TARGET_FPGA];
    Adb3CoreNotificationList fpgaAlert[MAX_NUM_TARGET_FPGA];
    Adb3CoreNotificationList sensorPoll;
  } notification;

  struct {
    boolean_t bConnected;
    boolean_t bEnabled;
    boolean_t bPolledMode;
    boolean_t bUsingMsi;
    DfInterruptObject object;
    struct {
      DfDpc dpc;
      boolean_t bDpcInitialized;
#ifdef ADB3_DBG_IRQ_COUNT
      uint32_t count;
      uint32_t expected;
#endif
    } dma[MAX_NUM_DMA_ENGINE];
    struct {
      DfDpc dpc;
      boolean_t bDpcInitialized;
    } fpga[MAX_NUM_TARGET_FPGA];
    struct {
      DfDpc dpc;
      boolean_t bDpcInitialized;
    } hwMon[MAX_NUM_TARGET_FPGA];
  } interrupt;

  struct {
    unsigned int mapLength;
    DmaChannelContext channel[MAX_NUM_DMA_ENGINE];
  } dma;

  struct {
    DfSpinLock lock;
    struct {
      unsigned int enable;
      DfTimer timer;
      boolean_t bPolling[MAX_NUM_TARGET_FPGA];
      boolean_t bTimerRunning;
      boolean_t bShutdown;
      DfEvent shutdownEvent;
    } polling;
  } hwMon;

  struct {
    DfDmaBuffer** pComBufs;
    unsigned int count;
  } comBuf;

  /* Model-specific stuff here */
  union {
#if defined(ADMXRC_DRIVER)
    struct {
      struct {
        Icd2061DeviceContext context;
        Icd2061ProgramRequest programming[2];
      } icd2061;
      struct {
        uint8_t fcon;
        uint8_t ienab;
      } shadow;
    } admxrc, admxrc2l, admxrc2, adpdrc2;

    struct {
      struct {
        Ics307DeviceContext context;
        Ics307ProgramRequest programming;
      } ics307;
      struct {
        uint32_t ictl;
        uint32_t fpctl;
      } shadow;
    } admxpl, admxp;

    struct {
      struct {
        Ics307DeviceContext context;
        Ics307ProgramRequest programming;
      } ics307;
      struct {
        Ics8430_61DeviceContext context;
        Ics8430_61ProgramRequest programming;
      } ics8430_61;
      struct {
        uint32_t ictl;
        uint32_t fpctl;
      } shadow;
    } admxrc4, adpexrc4fx, admxrc5lx;

    struct {
      struct {
        uint32_t ictl;
        uint32_t fpctl;
      } shadow;
    } adcpxrc4lx;

    struct {
      struct {
        V4ClkSynthDeviceContext context;
        V4ClkSynthProgramRequest programming;
      } v4ClkSynth;
      struct {
        Ics8430_61DeviceContext context;
        Ics8430_61ProgramRequest programming;
      } ics8430_61;
      struct {
        uint32_t ictl;
        uint32_t fpctl;
      } shadow;
    } admxrc5lxa;

    struct {
      struct {
        V4ClkSynthDeviceContext context;
        V4ClkSynthProgramRequest programming;
      } v4ClkSynth;
      struct {
        uint32_t ictl;
        uint32_t fpctl;
      } shadow;
    } admamc5a2;

    struct {
      struct {
        V4ClkSynthDeviceContext context;
        V4ClkSynthProgramRequest programming;
      } v4ClkSynth;
      struct {
        Ics843034_01DeviceContext context;
        Ics843034_01ProgramRequest programming;
      } ics843034_01;
      struct {
        uint32_t ictl;
        uint32_t fpctl;
      } shadow;
    } admxrc4fx, admxrc5t1, admxrc5t2, admxrc5tda1, admxrc5tz;

    struct {
      struct {
        V4ClkSynthDeviceContext context;
        V4ClkSynthProgramRequest programming;
      } v4ClkSynth;
      struct {
        uint32_t ictl;
        uint32_t fpctl[2];
      } shadow;
    } adcbbp;

    struct {
      struct {
        I2cContext context;
        I2cStatus status;  /* Result of last I2C access */
        uint8_t readValue; /* Data read in last I2C read */
      } i2c;
      struct {
        Si5338DeviceContext context;
        Si5338ProgramRequest programming[4];
        boolean_t bRefClk200GlitchFree;
        boolean_t bMClkGlitchFree;
      } si5338;
      struct {
        uint8_t fcon;
        uint8_t ienab;
      } shadow;
    } admpcie6s1;
#endif

#if defined(ADB3_DRIVER)
    struct {
      boolean_t bNeedMsiWorkaround;
    } genericAdb3;

    struct {
      boolean_t bNeedMsiWorkaround;
      struct {
        V5ClkSynthDeviceContext context;
        V5ClkSynthProgramRequest programming;
        uint32_t refClkFrequency;
      } v5ClkSynth;
      struct {
        Ics843034_01DeviceContext context;
        Ics843034_01ProgramRequest programming;
        uint32_t refClk0Frequency, refClk1Frequency;
      } ics843034_01;
      struct {
        uint32_t fpgaCtl;
        uint32_t intCtl;
      } shadow;
    } adpexrc5t;

    struct {
      boolean_t bNeedMsiWorkaround;
      struct {
        I2cContext context;
      } i2c;
      struct {
        V5ClkSynthDeviceContext context;
        V5ClkSynthProgramRequest programming;
        uint32_t refClkFrequency;
      } v5ClkSynth;
      struct {
        uint32_t fpgaCtl;
        uint32_t intCtl;
      } shadow;
    } admxrc6tl;

    struct {
      boolean_t bNeedMsiWorkaround;
      struct {
        I2cContext context;
      } i2c;
      struct {
        V6ClkSynthDeviceContext context;
        V6ClkSynthProgramRequest programming;
        uint32_t refClkFrequency;
      } v6ClkSynth;
      struct {
        uint32_t fpgaCtl;
        uint32_t intCtl;
      } shadow;
    } admxrc6t1;

    struct {
      boolean_t bNeedMsiWorkaround;
      struct {
        I2cContext context;
      } i2c;
      struct {
        Si5338DeviceContext context;
        Si5338ProgramRequest programming[4];
      } si5338;
      struct {
        V6ClkSynthDeviceContext context;
        V6ClkSynthProgramRequest programming;
        uint32_t refClkFrequency;
      } v6ClkSynth;
      struct {
        uint32_t fpgaCtl;
        uint32_t intCtl;
      } shadow;
    } admxrc6tge;

    struct {
      boolean_t bNeedMsiWorkaround;
      struct {
        I2cContext context;
      } i2c;
      struct {
        AvrDeviceContext context;
        unsigned int numVoltageSensor;
      } avr;
      boolean_t bAvrInitialized;
      struct {
        uint32_t intCtl;
      } shadow;
    } admxrc6tadv8;

    struct {
      boolean_t bNeedMsiWorkaround;
#if ADPEXRC6T_INCLUDE_SI5338_INTERFACE
      struct {
        I2cContext context;
      } i2c;
      struct {
        Si5338DeviceContext context;
        Si5338ProgramRequest programming[4];
      } si5338;
#endif
      struct {
        V6ClkSynthDeviceContext context;
        V6ClkSynthProgramRequest programming;
        uint32_t refClkFrequency;
      } v6ClkSynth;
      struct {
        AvrDeviceContext context;
      } avr;
      boolean_t bAvrInitialized;
      struct {
        uint32_t fpgaCtl;
        uint32_t intCtl;
      } shadow;
      unsigned int forcePcieGen;
    } adpexrc6t;

    struct {
      boolean_t bNeedMsiWorkaround;
      boolean_t bIsController;
      struct {
        AvrDeviceContext context;
      } avr;
      boolean_t bAvrInitialized;
      struct {
        uint32_t fpgaCtl[ADPEXRC6TADVCTL_NUM_TARGET_FPGA];
        uint32_t intCtl;
      } shadow;
    } adpexrc6tadv;

    struct {
      boolean_t bNeedMsiWorkaround;
      struct {
        I2cContext context;
      } i2c;
      struct {
        V6ClkSynthDeviceContext context;
        V6ClkSynthProgramRequest programming;
        uint32_t refClkFrequency;
      } v6ClkSynth;
      struct {
        uint32_t fpgaCtl;
        uint32_t intCtl;
      } shadow;
    } admxrc6tda1;

    struct {
      boolean_t bNeedMsiWorkaround;
      struct {
        V6ClkSynthDeviceContext context;
        V6ClkSynthProgramRequest programming;
        uint32_t refClkFrequency;
      } v6ClkSynth;
      struct {
        AvrDeviceContext context;
      } avr;
      boolean_t bAvrInitialized;
      struct {
        AvrClockContext context;
        AvrClockRequest request[ADMXRC7K1_NUM_AVR_CLOCK];
      } avrClock;
      boolean_t bAvrClockInitialized;
      struct {
        uint32_t fpgaCtl;
        uint32_t intCtl;
      } shadow;
    } admxrc7k1;

    struct {
      boolean_t bNeedMsiWorkaround;
      struct {
        V6ClkSynthDeviceContext context;
        V6ClkSynthProgramRequest programming;
        uint32_t refClkFrequency;
      } v6ClkSynth;
      struct {
        AvrDeviceContext context;
      } avr;
      boolean_t bAvrInitialized;
      struct {
        AvrClockContext context;
        AvrClockRequest request[ADMXRC7V1_NUM_AVR_CLOCK];
      } avrClock;
      boolean_t bAvrClockInitialized;
      struct {
        uint32_t fpgaCtl;
        uint32_t intCtl;
      } shadow;
    } admxrc7v1;

    struct {
      struct {
        uint32_t intCtl;
      } shadow;
    } admxrc7z1;

    struct {
      boolean_t bBoardRevision1;
      struct {
        I2cContext context;
      } i2c;
      struct {
        Si5338DeviceContext context;
        Si5338ProgramRequest programming[4];
      } si5338[2];
      struct {
        uint32_t intCtl;
      } shadow;
      struct {
        boolean_t bPresent;
        IcapContext context;
      } icap;
    } admpcie7v3;

    struct {
      struct {
        I2cContext context;
      } i2c;
      struct {
        Si5338DeviceContext context;
        Si5338ProgramRequest programming[4];
      } si5338[2];
      struct {
        uint32_t intCtl;
      } shadow;
      struct {
        boolean_t bPresent;
        IcapContext context;
      } icap;
      boolean_t bIs2ndEndpoint;
    } admpcieku3;

    struct {
      struct {
        V6ClkSynthDeviceContext context;
        V6ClkSynthProgramRequest programming;
        uint32_t refClkFrequency;
      } v6ClkSynth;
      struct {
        Avr2DeviceContext context;
      } avr2;
      boolean_t bAvr2Initialized;
      struct {
        Avr2ClockContext context;
        Avr2ClockRequest request[ADMXRCKU1_NUM_AVR2_CLOCK];
      } avr2Clock;
      boolean_t bAvr2ClockInitialized;
      struct {
        uint32_t fpgaCtl;
        uint32_t intCtl;
      } shadow;
    } admxrcku1;

    struct {
      struct {
        Avr2DeviceContext context;
      } avr2;
      boolean_t bAvr2Initialized;
      struct {
        Avr2ClockContext context;
        Avr2ClockRequest request[ADMPCIE8V3_NUM_AVR2_CLOCK];
      } avr2Clock;
      boolean_t bAvr2ClockInitialized;
      struct {
        uint32_t intCtl;
      } shadow;
      struct {
        boolean_t bPresent;
        IcapContext context;
      } icap;
      boolean_t bIs2ndEndpoint;
    } admpcie8v3;

    struct {
      struct {
        Avr2DeviceContext context;
      } avr2;
      boolean_t bAvr2Initialized;
      struct {
        Avr2ClockContext context;
        Avr2ClockRequest request[ADMPCIE8K5_NUM_AVR2_CLOCK];
      } avr2Clock;
      boolean_t bAvr2ClockInitialized;
      struct {
        uint32_t intCtl;
      } shadow;
      struct {
        boolean_t bPresent;
        IcapContext context;
      } icap;
    } admpcie8k5;
#endif

#if defined(ADB3_CUSTOM)
ADB3_CORE_DEVICE_CONTEXT_CUSTOM
#endif
  } model;
};

struct _Adb3CoreClientContext {
  Adb3CoreDeviceContext* pDevCtx;
  boolean_t bPrivileged;
};

/* Returns true if the range of items specified by [ START , START + N ) is not a subset of [ 0 , LENGTH ) */
#define CHECK_BOUNDS(START, N, LENGTH) \
	((START) + (N) < (START) || (START) + (N) > (LENGTH))

/* Returns true if the range of items specified by [ START , START + N ) is not a subset of [ OFFSET , OFFSET + LENGTH ) */
#define CHECK_BOUNDS_OFFSET(START, N, OFFSET, LENGTH) \
	((START) + (N) < (START) || (START) < (OFFSET) || (START) + (N) > (LENGTH) + (OFFSET))

#endif
