/*
** File: corecoop.c  
** Project: ADB3 core driver
** Purpose: Core Interface functions relating to a device's cooperative
**          level.
**
** (C) Copyright Alpha Data 2013
*/

#include <df.h>
#include "device.h"
#include "corecoop.h"

boolean_t
coreSetCoopLevel(
  void* pInterfaceContext,
  CoreClient* pClient,
  CoreCoopLevel coopLevel)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pInterfaceContext;
  DfSpinLockFlags f;
  boolean_t bSuccess;

  f = dfSpinLockGet(&pDevCtx->cooperation.lock);

  switch (coopLevel) {
  case CoreCoopLevelUnset:
    if (pDevCtx->cooperation.pExclusiveClient == pClient) {
      pDevCtx->cooperation.pExclusiveClient = NULL; /* Device no longer in exclusive mode. */
    }
    /* Unsetting cooperative level always succeeds */
    bSuccess = TRUE;
    break;

  case CoreCoopLevelNormal:
    if (NULL == pDevCtx->cooperation.pExclusiveClient) {
      /* Succeed if device not currently exclusive. */
      bSuccess = TRUE;
    } else {
      if (pDevCtx->cooperation.pExclusiveClient == pClient) {
        /* Succeed because client handle is current exclusive owner. */
        bSuccess = TRUE;
        pDevCtx->cooperation.pExclusiveClient = NULL; /* Device no longer in exclusive mode. */
      } else {
        /* Fail, because this client handle is not current exclusive owner. */
        bSuccess = FALSE;
      }
    }
    break;

  case CoreCoopLevelExclusive:
    if (dfListIsEmpty(&pDevCtx->cooperation.clients)) {
      /* Nobody has opened the device yet, so succeed */
      bSuccess = TRUE;
      pDevCtx->cooperation.pExclusiveClient = pClient; /* Device now in exclusive mode. */
    } else {
      if (pDevCtx->cooperation.pExclusiveClient == pClient) {
        /* Reaffirming exclusive cooperative level; succeeds, but no operation. */
        bSuccess = TRUE;
      } else {
        /* Fail, because this client handle is not current exclusive owner. */
        bSuccess = FALSE;
      }
    }
    break;

  default:
    /* Invalid cooperative level specified. */
    bSuccess = FALSE;
    break;
  }

  if (bSuccess) {
    if (coopLevel == CoreCoopLevelUnset) {
      if (pClient->bInList) {
        dfListRemove(&pClient->node);
        pClient->bInList = FALSE;
      }
    } else {
      if (!pClient->bInList) {
        dfListAddToTail(&pDevCtx->cooperation.clients, &pClient->node);
        pClient->bInList = TRUE;
      }
    }
  }

  dfSpinLockPut(&pDevCtx->cooperation.lock, f);

  return bSuccess;
}
