/*
** File: avr2_common.h  
** Project: ADB3 core driver
** Purpose: Exports functions for accessing the AVR2 uC interface
**
** (C) Copyright Alpha Data 2015
*/

#ifndef _ADATA_CORE_AVR2_COMMON_H
#define _ADATA_CORE_AVR2_COMMON_H

#include "avr2.h"
#include "serializer.h"

struct _Adb3CoreDeviceContext;
struct _Avr2DeviceContext;
typedef struct _Avr2DeviceContext Avr2DeviceContext;
struct _Avr2Request;
typedef struct _Avr2Request Avr2Request;
struct _Avr2TransRequest;
typedef struct _Avr2TransRequest Avr2TransRequest;

/*
** ----------------------------------------------------------------
** AVR2 uC access byte transport layer *
** ----------------------------------------------------------------
*/

typedef enum _Avr2RequestStatus {
  Avr2RequestStatusSuccess   = 0,
  Avr2RequestStatusCancelled = 1
} Avr2RequestStatus;

typedef enum _Avr2DataFlag {
  Avr2DataFlagNone = 0, /* Neither start nor end of packet */
  Avr2DataFlagSOH  = 1, /* Start of packet */
  Avr2DataFlagEOT  = 2  /* End of packet */
} Avr2DataFlag;

typedef void Avr2Callback(Avr2Request* pRequest, Avr2RequestStatus status, uint8_t data, Avr2DataFlag dataFlag, void* pContext);

typedef enum _Avr2RequestState {
  Avr2RequestStateQueued   = 0, /* Queued */
  Avr2RequestStateActive   = 1, /* Waiting for transmission / reception */
  Avr2RequestStateComplete = 2  /* Finished; callback is executing or has executed */
} Avr2RequestState;

struct _Avr2Request { /* Client must not touch */
  SerializerRequest serRequest;
  Avr2DeviceContext* pAvr2Ctx;
  Avr2Callback* pCallback;
  void* pContext;
  uint8_t data;
  Avr2DataFlag dataFlag;
  boolean_t bWrite; /* TRUE for transmitting to AVR, FALSE for receiving */
  Avr2RequestState state;
  boolean_t bCancel;
};

/* Must be called in thread context */
extern Avr2RequestStatus
avr2ByteReadSync(
  Avr2DeviceContext* pAvr2Ctx,
  uint8_t* pData8,
  Avr2DataFlag* pDataFlag);

/* Must be called in thread context */
extern Avr2RequestStatus
avr2ByteWriteSync(
  Avr2DeviceContext* pAvr2Ctx,
  uint8_t val8,
  Avr2DataFlag dataFlag);

extern void
avr2ByteRead(
  Avr2DeviceContext* pAvr2Ctx,
  Avr2Request* pRequest,
  Avr2Callback* pCallback,
  void* pContext);

extern void
avr2ByteWrite(
  Avr2DeviceContext* pAvr2Ctx,
  Avr2Request* pRequest,
  uint8_t data,
  Avr2DataFlag dataFlag,
  Avr2Callback* pCallback,
  void* pContext);

extern boolean_t
avr2ByteCancel(
  Avr2DeviceContext* pAvr2Ctx,
  Avr2Request* pRequest);

/*
** ----------------------------------------------------------------
** AVR2 uC access transaction layer
** ----------------------------------------------------------------
*/

typedef enum _Avr2TransStatus {
  Avr2TransStatusSuccess      = 0, /* Success */
  Avr2TransStatusQueueTimeout = 1, /* Timed out while queued */
  Avr2TransStatusRXTimeout    = 2, /* Timed out receiving response */
  Avr2TransStatusTXTimeout    = 3  /* Timed out sending command */
} Avr2TransStatus;

typedef void Avr2TransCallback(Avr2TransRequest* pRequest, Avr2TransStatus status, size_t actualResponseLength, void* pContext);

typedef enum _Avr2TransactionState {
  Avr2TransactionStateQueued   = 0, /* Queued */
  Avr2TransactionStateCommand  = 1, /* Sending the command */
  Avr2TransactionStateResponse = 2, /* Receiving the response */
  Avr2TransactionStateComplete = 3  /* Finished; callback is executing or has executed */
} Avr2TransactionState;

struct _Avr2TransRequest {  /* Client must not touch */
  SerializerRequest serRequest;
  Avr2DeviceContext* pAvr2Ctx;
  Avr2TransCallback* pCallback;
  void* pContext;
  const uint8_t* pCommand; /* Command to uC */
  size_t commandLength;    /* Command length, in bytes */
  uint8_t* pResponse;      /* Filled in with response from uC */
  size_t responseLength;   /* Expected response length, in bytes */
  unsigned int timeout;    /* In microseconds */
  Avr2TransactionState state;
  boolean_t bTimeout;
};

/* Must be called in thread context */
extern Avr2TransStatus
avr2TransactionSync(
  Avr2DeviceContext* pAvr2Ctx,
  const uint8_t* pCommand,
  size_t commandLength,
  uint8_t* pResponse,
  size_t responseLength,
  unsigned int timeout,
  size_t* pActualResponseLength); /* In microseconds */

extern void
avr2Transaction(
  Avr2DeviceContext* pAvr2Ctx,
  Avr2TransRequest* pRequest,
  const uint8_t* pCommand,
  size_t commandLength,
  uint8_t* pResponse,
  size_t responseLength,
  unsigned int timeout, /* In microseconds */
  Avr2TransCallback* pCallback,
  void* pContext);

/*
** ----------------------------------------------------------------
** AVR2 uC access device context
** ----------------------------------------------------------------
*/

typedef union _Avr2Status {
  struct {
    uint32_t serviceMode     : 1;
    uint32_t rxProtocolError : 1;
    uint32_t txFramingError  : 1;
    uint32_t _reserved1      : 29;
  } flags;
  uint32_t asUint32;
} Avr2Status;

typedef void Avr2RxIntEnableCallback(Avr2DeviceContext* pAvr2Ctx);
typedef void Avr2TxIntEnableCallback(Avr2DeviceContext* pAvr2Ctx);

struct _Avr2DeviceContext {
  uint32_t flashSize;                        /* Size of AVR uC Flash memory */
  uint32_t flashPageSize;                    /* Size of a page AVR uC Flash memory */
  uint8_t flashPageOrder;                    /* log2(flashPageSize) */
  uint32_t* pCtlStat;                        /* Pointer to AVR2 interface ctl/status register */
  DfMemoryHandle hCtlStat;                   /* Handle to AVR2 interface ctl/status register */
  DfInterruptObject* pInterruptObject;       /* Interrupt object associated with DPCs */
  struct {
    DfSpinLock lock;                         /* Spinlock that guards this struct */
    struct {
      SerializerQueue serializer;            /* Serializes reception from AVR uC */
      Avr2RxIntEnableCallback* pEnableInt;   /* Method called to enable RX interrupt */
      DfDpc dpc;                             /* RX interrupt DPC */
    } rx;
    struct {
      SerializerQueue serializer;            /* Serializes transmission to AVR uC */
      Avr2TxIntEnableCallback* pEnableInt;   /* Method called to enable TX interrupt */
      DfDpc dpc;                             /* TX interrupt DPC */
    } tx;
  } transport; /* byte transport layer */
  struct {
    DfSpinLock lock;                         /* Spinlock that guards this struct */
    SerializerQueue serializer;              /* Serializes transactions with AVR uC */
    DfTimer timer;                           /* Timeout timer */
    Avr2Request avr2Request;                 /* Used for sending/receiving each byte of a transaction */
    size_t position;                         /* Byte index into command/response, for current transaction */
  } transaction; /* transaction layer */
  struct {
    uint8_t* pMergeBuffer;                   /* Used in AVR flash writes */
  } vpd;
};

/* Must be called in thread context */
extern boolean_t
avr2Init(
  Avr2DeviceContext* pAvr2Ctx,
  DfInterruptObject* pInterruptObject,
  uint32_t* pCtlStat,
  DfMemoryHandle hCtlStat,
  uint32_t flashSize,                     /* uC Flash memory size */
  uint8_t flashPageOrder,                 /* log2(uC Flash memory page size) */
  Avr2RxIntEnableCallback* pEnableRxInt,  /* Function that enables AVR2 RX interrupt */
  Avr2TxIntEnableCallback* pEnableTxInt); /* Function that enables AVR2 TX interrupt */

/* Must be called in thread context */
extern void
avr2Uninit(
  Avr2DeviceContext* pAvr2Ctx);

extern boolean_t
avr2IsServiceMode(
  Avr2DeviceContext* pAvr2Ctx);

extern void
avr2GetStatus(
  Avr2DeviceContext* pAvr2Ctx,
  Avr2Status* pAvr2Status);

#endif
