/*
** File: avr.c  
** Project: ADB3 core driver
** Purpose: Functions for accessing AVR microcontroller interface.
**
** (C) Copyright Alpha Data 2011
*/

#include "device.h"
#include "avr.h"
#include "avr_common.h"

#define DEBUGLEVEL_INIT    (2) /* Level of debug messages relating to initialization / uninitialization */
#define DEBUGLEVEL_CANCEL  (1) /* Level of debug messages relating to cancellation / cleanup */
#define DEBUGLEVEL_ERROR   (0) /* Level of debug messages relating to driver logic errors */
#define DEBUGLEVEL_WARNING (1) /* Level of debug messages relating to driver logic errors */
#define DEBUGLEVEL_NORMAL  (6) /* Level of debug messages relating to normal operation */

/* AVR uC "Request firmware version" command */
static const uint8_t g_vnCommand[] = { 'V', 'N', '?' };

/*
** ----------------------------------------------------------------
** AVR uC access byte transport layer *
** ----------------------------------------------------------------
*/

typedef struct _AvrRequestSync {
  AvrRequest request;
  DfEvent* pEvent;
  AvrRequestStatus status;
} AvrRequestSync;

static void
avrRxSerializerAcquired(
  SerializerQueue* pQueue,
  SerializerRequest* pRequest,
  void* pContextIgnored)
{
  AvrRequest* pAvrRequest = DF_CONTAINER_OF(pRequest, AvrRequest, serRequest);
  AvrDeviceContext* pAvrCtx = pAvrRequest->pAvrCtx;
  DfSpinLockFlags f;

  f = dfSpinLockGet(&pAvrCtx->transport.lock);
  if (pAvrRequest->bCancel) {
    dfSpinLockPut(&pAvrCtx->transport.lock, f);
    serializerPut(&pAvrCtx->transport.rx.serializer);
    pAvrRequest->pCallback(pAvrRequest, AvrRequestStatusCancelled, 0, pAvrRequest->pContext);
  } else {
    pAvrRequest->state = AvrRequestStateActive;
    dfSpinLockPut(&pAvrCtx->transport.lock, f);
    /* Enable AVR RX not empty interrupt */
    pAvrCtx->transport.rx.pEnableInt(pAvrCtx);
  }
}

static void
avrTxSerializerAcquired(
  SerializerQueue* pQueue,
  SerializerRequest* pRequest,
  void* pContext)
{
  AvrRequest* pAvrRequest = DF_CONTAINER_OF(pRequest, AvrRequest, serRequest);
  AvrDeviceContext* pAvrCtx = pAvrRequest->pAvrCtx;
  DfSpinLockFlags f;

  f = dfSpinLockGet(&pAvrCtx->transport.lock);
  if (pAvrRequest->bCancel) {
    dfSpinLockPut(&pAvrCtx->transport.lock, f);
    serializerPut(&pAvrCtx->transport.tx.serializer);
    pAvrRequest->pCallback(pAvrRequest, AvrRequestStatusCancelled, 0, pAvrRequest->pContext);
  } else {
    pAvrRequest->state = AvrRequestStateActive;
    dfSpinLockPut(&pAvrCtx->transport.lock, f);
    /* Enable AVR TX empty interrupt */
    pAvrCtx->transport.tx.pEnableInt(pAvrCtx);
  }
}

static void
avrCallbackSync(
  AvrRequest* pRequest,
  AvrRequestStatus status,
  uint8_t data,
  void* pContext /* ignored */)
{
  AvrRequestSync* pRequestSync = DF_CONTAINER_OF(pRequest, AvrRequestSync, request);

  pRequestSync->status = status;
  dfEventSignal(pRequestSync->pEvent);
}

static void
avrOnRxNotEmpty(
  DfDpc* pDpc,
  void* pContextIgnored,
  void* pArgIgnored)
{
  AvrDeviceContext* pAvrCtx = DF_CONTAINER_OF(pDpc, AvrDeviceContext, transport.rx.dpc);
  AvrRequest* pRequest = NULL;
  SerializerRequest* pSerRequest;
  uint32_t val32;
  uint8_t data8;
  DfSpinLockFlags f;

  f = dfSpinLockGet(&pAvrCtx->transport.lock);
  pSerRequest = serializerGetHead(&pAvrCtx->transport.rx.serializer);
  if (NULL != pSerRequest) {
    pRequest = DF_CONTAINER_OF(pSerRequest, AvrRequest, serRequest);
  }
  if (NULL != pRequest && !pRequest->bWrite) {
    dfAssert(pRequest->state == AvrRequestStateActive);
    val32 = dfPciMemRead32(pAvrCtx->hCtlStat, pAvrCtx->pCtlStat);
    if (!(val32 & ADMXRC6Tx_V_AVRSTAT_RX_EMPTY)) {
      /* Extract the byte received */
      data8 = (uint8_t)ADMXRC6Tx_E_AVRSTAT_DATA(val32);
      dfDebugPrint(DEBUGLEVEL_NORMAL, ("avrOnRxNotEmpty: data8=0x%x('%c')\n", (unsigned int)data8, isprint(data8) ? data8 : '.'));
      /* Advance the RX FIFO */
      dfPciMemWrite32(pAvrCtx->hCtlStat, pAvrCtx->pCtlStat, pAvrCtx->ctlStat | ADMXRC6Tx_V_AVRCTL_RX | ADMXRC6Tx_V_AVRCTL_FLASH);
      val32 = dfPciMemRead32(pAvrCtx->hCtlStat, pAvrCtx->pCtlStat);
      pRequest->state = AvrRequestStateComplete;
      dfSpinLockPut(&pAvrCtx->transport.lock, f);
      serializerPut(&pAvrCtx->transport.rx.serializer);
      pRequest->pCallback(pRequest, AvrRequestStatusSuccess, data8, pRequest->pContext);
    } else {
      dfSpinLockPut(&pAvrCtx->transport.lock, f);
      /* Shouldn't get here, but if we do, enable AVR RX not empty interrupt in order to try again */
      pAvrCtx->transport.rx.pEnableInt(pAvrCtx);
    }
  } else {
    dfSpinLockPut(&pAvrCtx->transport.lock, f);
    dfDebugPrint(DEBUGLEVEL_ERROR, ("+++ avrOnRxNotEmpty: nothing to do\n"));
  }
}

static void
avrOnTxEmpty(
  DfDpc* pDpc,
  void* pContextIgnored,
  void* pArgIgnored)
{
  AvrDeviceContext* pAvrCtx = DF_CONTAINER_OF(pDpc, AvrDeviceContext, transport.tx.dpc);
  AvrRequest* pRequest = NULL;
  SerializerRequest* pSerRequest;
  uint32_t val32;
  uint8_t data8;
  DfSpinLockFlags f;

  f = dfSpinLockGet(&pAvrCtx->transport.lock);
  pSerRequest = serializerGetHead(&pAvrCtx->transport.tx.serializer);
  if (NULL != pSerRequest) {
    pRequest = DF_CONTAINER_OF(pSerRequest, AvrRequest, serRequest);
  }
  if (NULL != pRequest && pRequest->bWrite) {
    dfAssert(pRequest->state == AvrRequestStateActive);
    val32 = dfPciMemRead32(pAvrCtx->hCtlStat, pAvrCtx->pCtlStat);
    if (val32 & ADMXRC6Tx_V_AVRSTAT_TX_EMPTY) {
      /* Transmit the byte */
      data8 = pRequest->data;
      dfDebugPrint(DEBUGLEVEL_NORMAL, ("avrOnTxEmpty: data8=0x%x('%c')\n", (unsigned int)data8, isprint(data8) ? data8 : '.'));
      val32 = pAvrCtx->ctlStat | ADMXRC6Tx_V_AVRCTL_TX | ADMXRC6Tx_V_AVRCTL_FLASH | ADMXRC6Tx_V_AVRCTL_DATA(data8);
      dfPciMemWrite32(pAvrCtx->hCtlStat, pAvrCtx->pCtlStat, val32);
      val32 = dfPciMemRead32(pAvrCtx->hCtlStat, pAvrCtx->pCtlStat);
      pRequest->state = AvrRequestStateComplete;
      dfSpinLockPut(&pAvrCtx->transport.lock, f);
      serializerPut(&pAvrCtx->transport.tx.serializer);
      pRequest->pCallback(pRequest, AvrRequestStatusSuccess, 0, pRequest->pContext);
    } else {
      dfSpinLockPut(&pAvrCtx->transport.lock, f);
      /* Shouldn't get here, but if we do, enable AVR TX empty interrupt in order to try again */
      pAvrCtx->transport.tx.pEnableInt(pAvrCtx);
    }
  } else {
    dfSpinLockPut(&pAvrCtx->transport.lock, f);
    dfDebugPrint(DEBUGLEVEL_WARNING, ("+++ avrOnTxEmpty: nothing to do\n"));
  }
}

void
avrByteRead(
  AvrDeviceContext* pAvrCtx,
  AvrRequest* pRequest,
  AvrCallback* pCallback,
  void* pContext)
{
  dfDebugPrint(DEBUGLEVEL_NORMAL, ("avrByteRead: pRequest=%p pCallback=%p\n", pRequest, pCallback));

  pRequest->pAvrCtx = pAvrCtx;
  pRequest->bWrite = FALSE;
  pRequest->pCallback = pCallback;
  pRequest->pContext = pContext;
  pRequest->state = AvrRequestStateQueued;
  pRequest->bCancel = FALSE;
  serializerGet(&pAvrCtx->transport.rx.serializer, &pRequest->serRequest, avrRxSerializerAcquired, NULL);
}

void
avrByteWrite(
  AvrDeviceContext* pAvrCtx,
  AvrRequest* pRequest,
  uint8_t data,
  AvrCallback* pCallback,
  void* pContext)
{
  dfDebugPrint(DEBUGLEVEL_NORMAL, ("avrByteWrite: pRequest=%p data=0x%02x pCallback=%p\n", pRequest, data, pCallback));

  pRequest->pAvrCtx = pAvrCtx;
  pRequest->data = data;
  pRequest->bWrite = TRUE;
  pRequest->pCallback = pCallback;
  pRequest->pContext = pContext;
  pRequest->state = AvrRequestStateQueued;
  pRequest->bCancel = FALSE;
  serializerGet(&pAvrCtx->transport.tx.serializer, &pRequest->serRequest, avrTxSerializerAcquired, NULL);
}

boolean_t
avrByteCancel(
  AvrDeviceContext* pAvrCtx,
  AvrRequest* pRequest)
{
  DfSpinLockFlags f;
  boolean_t bCancelled = FALSE;

  dfDebugPrint(DEBUGLEVEL_CANCEL, ("avrByteCancel: pRequest=%p\n", pRequest));

  f = dfSpinLockGet(&pAvrCtx->transport.lock);
  pRequest->bCancel = TRUE;
  switch (pRequest->state) {
  case AvrRequestStateQueued:
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("avrByteCancel: cancelling AVR request in Queued state\n"));
    if (pRequest->bWrite) {
      bCancelled = serializerCancel(&pAvrCtx->transport.tx.serializer, &pRequest->serRequest);
    } else {
      bCancelled = serializerCancel(&pAvrCtx->transport.rx.serializer, &pRequest->serRequest);
    }
    dfSpinLockPut(&pAvrCtx->transport.lock, f);
    break;

  case AvrRequestStateActive:
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("avrByteCancel: cancelling AVR request in Active state\n"));
    if (pRequest->bWrite) {
      serializerPut(&pAvrCtx->transport.tx.serializer);
    } else {
      serializerPut(&pAvrCtx->transport.rx.serializer);
    }
    dfSpinLockPut(&pAvrCtx->transport.lock, f);
    bCancelled = TRUE;
    break;

  case AvrRequestStateComplete:
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("avrByteCancel: cancelling AVR request in Complete state\n"));
    dfSpinLockPut(&pAvrCtx->transport.lock, f);
    break;

  default:
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("*** avrByteCancel: cancelling AVR request in unknown state (%u)\n", pRequest->state));
    dfSpinLockPut(&pAvrCtx->transport.lock, f);
    dfAssert(FALSE);
    break;
  }

  return bCancelled;
}

AvrRequestStatus
avrByteReadSync(
  AvrDeviceContext* pAvrCtx,
  uint8_t* pData)
{
  AvrRequestSync request;
  DfEvent event;

  dfEventInit(&event);
  request.pEvent = &event;
  avrByteRead(pAvrCtx, &request.request, avrCallbackSync, NULL);
  dfEventWait(&event);
  dfEventUninit(&event);
  *pData = request.request.data;
  return request.status;
}

AvrRequestStatus
avrByteWriteSync(
  AvrDeviceContext* pAvrCtx,
  uint8_t data)
{
  AvrRequestSync request;
  DfEvent event;

  dfEventInit(&event);
  request.pEvent = &event;
  avrByteWrite(pAvrCtx, &request.request, data, avrCallbackSync, NULL);
  dfEventWait(&event);
  dfEventUninit(&event);
  return request.status;
}

/*
** ----------------------------------------------------------------
** AVR uC access transaction layer
** ----------------------------------------------------------------
*/

typedef struct _AvrTransRequestSync {
  AvrTransRequest request;
  DfEvent* pEvent;
  AvrTransStatus status;
} AvrTransRequestSync;

static void
avrTransByteCallback(
  AvrRequest* pAvrRequest,
  AvrRequestStatus reqStatus,
  uint8_t data,
  void* pContext)
{
  AvrTransRequest* pRequest = (AvrTransRequest*)pContext;
  AvrDeviceContext* pAvrCtx = pRequest->pAvrCtx;
  size_t position;
  DfSpinLockFlags f;
  unsigned int timeout;
  AvrTransStatus status = AvrTransStatusSuccess;
  boolean_t bInvoke = FALSE;

  f = dfSpinLockGet(&pAvrCtx->transaction.lock);
  position = pAvrCtx->transaction.position;
  switch (pRequest->state) {
  case AvrTransactionStateCommand:
    if (pRequest->bTimeout) {
      /* Timeout function has already executed; we can invoke the callback */
      pRequest->state = AvrTransactionStateComplete;
      dfSpinLockPut(&pAvrCtx->transaction.lock, f);
      status = AvrTransStatusTimeout;
      bInvoke = TRUE;
    } else {
      dfAssert(reqStatus == AvrRequestStatusSuccess);
      position++;
      pAvrCtx->transaction.position = position;
      if (position >= pRequest->commandLength) {
        if (pRequest->responseLength) {
          /* Finished sending command, start receiving response */
          pAvrCtx->transaction.position = 0;
          pRequest->state = AvrTransactionStateResponse;
          avrByteRead(pAvrCtx, &pAvrCtx->transaction.avrRequest, avrTransByteCallback, pRequest);
          dfSpinLockPut(&pAvrCtx->transaction.lock, f);
        } else {
          /* Finished sending command, and no response expected */
          pRequest->state = AvrTransactionStateComplete;
          timeout = pRequest->timeout;
          if (0 == timeout || dfTimerCancel(&pAvrCtx->transaction.timer)) {
            /* Successfully cancelled the timer function; we can invoke the callback */
            dfSpinLockPut(&pAvrCtx->transaction.lock, f);
            bInvoke = TRUE;
          } else {
            /* Let the timer function invoke the callback */
            dfSpinLockPut(&pAvrCtx->transaction.lock, f);
          }
        }
      } else {
        /* Send next byte of command command */
        avrByteWrite(pAvrCtx, &pAvrCtx->transaction.avrRequest, pRequest->pCommand[position], avrTransByteCallback, pRequest);
        dfSpinLockPut(&pAvrCtx->transaction.lock, f);
      }
    }
    break;

  case AvrTransactionStateResponse:
    if (pRequest->bTimeout) {
      /* Timeout function has already executed; we can invoke the callback */
      pRequest->state = AvrTransactionStateComplete;
      dfSpinLockPut(&pAvrCtx->transaction.lock, f);
      status = AvrTransStatusTimeout;
      bInvoke = TRUE;
    } else {
      dfAssert(reqStatus == AvrRequestStatusSuccess);
      pRequest->pResponse[position] = data;
      position++;
      pAvrCtx->transaction.position = position;
      if (position >= pRequest->responseLength) {
        /* Finished receiving response */
        pRequest->state = AvrTransactionStateComplete;
        timeout = pRequest->timeout;
        if (0 == timeout || dfTimerCancel(&pAvrCtx->transaction.timer)) {
          /* Successfully cancelled the timer function; we can invoke the callback */
          dfSpinLockPut(&pAvrCtx->transaction.lock, f);
          bInvoke = TRUE;
        } else {
          /* Let the timer function invoke the callback */
          dfSpinLockPut(&pAvrCtx->transaction.lock, f);
        }
      } else {
        /* Receive next byte of response */
        avrByteRead(pAvrCtx, &pAvrCtx->transaction.avrRequest, avrTransByteCallback, pRequest);
        dfSpinLockPut(&pAvrCtx->transaction.lock, f);
      }
    }
    break;

  case AvrTransactionStateQueued:
  case AvrTransactionStateComplete:
  default:
    /* Should never get here */
    dfSpinLockPut(&pAvrCtx->transaction.lock, f);
    dfAssert(FALSE);
    break;
  }

  if (bInvoke) {
    serializerPut(&pAvrCtx->transaction.serializer);
    pRequest->pCallback(pRequest, status, pRequest->pContext);
  }
}

static void
avrTransactionTimeout(
  DfTimer* pTimer,
  void* pTimerContextIgnored,
  void* pCallbackContext)
{
  AvrTransRequest* pRequest = (AvrTransRequest*)pCallbackContext;
  AvrDeviceContext* pAvrCtx = pRequest->pAvrCtx;
  DfSpinLockFlags f;
  boolean_t bInvoke = FALSE;

  dfDebugPrint(DEBUGLEVEL_ERROR, ("avrTransactionTimeout: timed out, cancelling AVR transaction\n"));

  f = dfSpinLockGet(&pAvrCtx->transaction.lock);
  pRequest->bTimeout = TRUE;
  switch (pRequest->state) {
  case AvrTransactionStateQueued:
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("avrTransactionTimeout: cancelling AVR transaction in Queued state\n"));
    bInvoke = serializerCancel(&pAvrCtx->transaction.serializer, &pRequest->serRequest);
    dfSpinLockPut(&pAvrCtx->transaction.lock, f);
    break;

  case AvrTransactionStateCommand:
  case AvrTransactionStateResponse:
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("avrTransactionTimeout: cancelling AVR transaction in Command/Response state\n"));
    bInvoke = avrByteCancel(pAvrCtx, &pAvrCtx->transaction.avrRequest);
    dfSpinLockPut(&pAvrCtx->transaction.lock, f);
    break;

  case AvrTransactionStateComplete:
    dfDebugPrint(DEBUGLEVEL_CANCEL, ("avrTransactionTimeout: cancelling AVR transaction in Complete state\n"));
    dfSpinLockPut(&pAvrCtx->transaction.lock, f);
    bInvoke = TRUE;
    break;

  default:
    /* Should never get here */
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** avrTransactionTimeout: cancelling AVR transaction in unknown state (%u)\n", pRequest->state));
    dfSpinLockPut(&pAvrCtx->transaction.lock, f);
    break;
  }

  if (bInvoke) {
    serializerPut(&pAvrCtx->transaction.serializer);
    pRequest->pCallback(pRequest, AvrTransStatusTimeout, pRequest->pContext);
  }
}

static void
avrTransactionSerializerAcquired(
  SerializerQueue* pQueue,
  SerializerRequest* pSerRequest,
  void* pContextIgnored)
{
  AvrTransRequest* pRequest = DF_CONTAINER_OF(pSerRequest, AvrTransRequest, serRequest);
  AvrDeviceContext* pAvrCtx = pRequest->pAvrCtx;
  DfSpinLockFlags f;
  unsigned int timeout;

  pAvrCtx->transaction.position = 0;
  f = dfSpinLockGet(&pAvrCtx->transaction.lock);
  if (pRequest->bTimeout) {
    dfSpinLockPut(&pAvrCtx->transaction.lock, f);
    serializerPut(pQueue);
    pRequest->pCallback(pRequest, AvrTransStatusTimeout, pRequest->pContext);
  } else {
    if (pRequest->commandLength) {
      pRequest->state = AvrTransactionStateCommand;
      avrByteWrite(pAvrCtx, &pAvrCtx->transaction.avrRequest, pRequest->pCommand[pAvrCtx->transaction.position], avrTransByteCallback, pRequest);
    } else {
      pRequest->state = AvrTransactionStateResponse;
      avrByteRead(pAvrCtx, &pAvrCtx->transaction.avrRequest, avrTransByteCallback, pRequest);
    }
    timeout = pRequest->timeout;
    if (timeout) {
      dfTimerSchedule(&pAvrCtx->transaction.timer, dfTimeGet() + dfMicrosecondsToTime(timeout), avrTransactionTimeout, pRequest);
    }
    dfSpinLockPut(&pAvrCtx->transaction.lock, f);
  }
}

static void
avrTransactionCallbackSync(
  AvrTransRequest* pRequest,
  AvrTransStatus status,
  void* pContextIgnored)
{
  AvrTransRequestSync* pRequestSync = DF_CONTAINER_OF(pRequest, AvrTransRequestSync, request);

  pRequestSync->status = status;
  dfEventSignal(pRequestSync->pEvent);
}

AvrTransStatus
avrTransactionSync(
  AvrDeviceContext* pAvrCtx,
  const uint8_t* pCommand,
  size_t commandLength,
  uint8_t* pResponse,
  size_t responseLength,
  unsigned int timeout)
{
  AvrTransRequestSync request;
  DfEvent event;

  dfEventInit(&event);
  request.pEvent = &event;
  avrTransaction(pAvrCtx, &request.request, pCommand, commandLength, pResponse, responseLength, timeout, avrTransactionCallbackSync, NULL);
  dfEventWait(&event);
  dfEventUninit(&event);
  return request.status;
}

void
avrTransaction(
  AvrDeviceContext* pAvrCtx,
  AvrTransRequest* pRequest,
  const uint8_t* pCommand,
  size_t commandLength,
  uint8_t* pResponse,
  size_t responseLength,
  unsigned int timeout,
  AvrTransCallback* pCallback,
  void* pContext)
{
  pRequest->pAvrCtx = pAvrCtx;
  pRequest->pCommand = pCommand;
  pRequest->commandLength = commandLength;
  pRequest->pResponse = pResponse;
  pRequest->responseLength = responseLength;
  pRequest->timeout = timeout;
  pRequest->pCallback = pCallback;
  pRequest->pContext = pContext;
  pRequest->bTimeout = FALSE;
  pRequest->state = AvrTransactionStateQueued;
  serializerGet(&pAvrCtx->transaction.serializer, &pRequest->serRequest, avrTransactionSerializerAcquired, NULL);
}

/*
** ----------------------------------------------------------------
** AVR uC access device context
** ----------------------------------------------------------------
*/

boolean_t
avrInit(
  AvrDeviceContext* pAvrCtx,
  DfInterruptObject* pInterruptObject,
  uint32_t* pCtlStat,
  DfMemoryHandle hCtlStat,
  uint32_t flashSize,
  uint32_t flashPageSize,
  AvrRxIntEnableCallback* pEnableRxInt,
  AvrTxIntEnableCallback* pEnableTxInt)
{
  static const unsigned int maxFlushCount = 10000U; /* max. number of RX FIFO reads before giving up flushing */
  static const unsigned int versionTimeout = 200000U; /* in microseconds */
  uint8_t vnResponse[11];
  AvrResponseVersion version;
  AvrTransStatus status;
  uint32_t val32;
  unsigned int count;
  size_t mergeBufferSize;

  dfAssert(flashSize <= 0x20000U);
  dfAssert(flashPageSize <= flashSize);

  /* Cannot fail */

  DF_ZERO_OBJECT(pAvrCtx);
  pAvrCtx->pInterruptObject = pInterruptObject;
  pAvrCtx->pCtlStat = pCtlStat;
  pAvrCtx->hCtlStat = hCtlStat;
  pAvrCtx->flashSize = flashSize;
  pAvrCtx->flashPageSize = flashPageSize;
  dfSpinLockInit(&pAvrCtx->transport.lock);
  serializerQueueInit(&pAvrCtx->transport.rx.serializer);
  pAvrCtx->transport.rx.pEnableInt = pEnableRxInt;
  dfDpcInit(&pAvrCtx->transport.rx.dpc, pInterruptObject, avrOnRxNotEmpty, pAvrCtx);
  serializerQueueInit(&pAvrCtx->transport.tx.serializer);
  pAvrCtx->transport.tx.pEnableInt = pEnableTxInt;
  dfDpcInit(&pAvrCtx->transport.tx.dpc, pInterruptObject, avrOnTxEmpty, pAvrCtx);
  dfSpinLockInit(&pAvrCtx->transaction.lock);
  serializerQueueInit(&pAvrCtx->transaction.serializer);
  dfTimerInit(&pAvrCtx->transaction.timer, pAvrCtx);

  /* From here on can fail */

  /* Establish AVR ctl/status register shadow */
  val32 = dfPciMemRead32(hCtlStat, pCtlStat);
  pAvrCtx->ctlStat = val32 & 0xFFFF8000U;
  if (val32 & ADMXRC6Tx_V_AVRSTAT_SERVICEMODE) {
    dfDebugPrint(DEBUGLEVEL_WARNING, ("avrInit: in service mode, ctlStat=0x%08lx\n", (unsigned long)val32));
  }

  mergeBufferSize = flashPageSize + 5U; /* 5 bytes is enough for old (3) or new (5) AVR Flash commands */
  pAvrCtx->vpd.pMergeBuffer = dfMalloc(mergeBufferSize);
  if (NULL == pAvrCtx->vpd.pMergeBuffer) {
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** avrInit: failed to allocate Flash merge buffer of %u(0x%x) bytes\n",
      (unsigned int)mergeBufferSize, (unsigned int)mergeBufferSize));
    goto failed;
  }
  dfDebugPrint(DEBUGLEVEL_INIT, ("avrInit: allocated Flash merge buffer of %u(0x%x) bytes\n",
    (unsigned int)mergeBufferSize, (unsigned int)mergeBufferSize));

  /* Flush any preexisting data out of the RX FIFO */
  count = 0;
  val32 = dfPciMemRead32(hCtlStat, pCtlStat);
  while (!(val32 & ADMXRC6Tx_V_AVRSTAT_RX_EMPTY)) {
    dfPciMemWrite32(hCtlStat, pCtlStat, pAvrCtx->ctlStat | ADMXRC6Tx_V_AVRCTL_RX | ADMXRC6Tx_V_AVRCTL_FLASH);
    val32 = dfPciMemRead32(hCtlStat, pCtlStat);
    count++;
    if (count == maxFlushCount) {
      dfDebugPrint(DEBUGLEVEL_ERROR, ("*** avrInit: RX FIFO still not empty after %u reads - assuming broken\n", count));
      goto failed;
    }
  };
  if (count) {
    dfDebugPrint(DEBUGLEVEL_WARNING, ("+++ avrInit: flushed %u character(s) out of RX FIFO\n", count));
  }

  /* Get AVR uC firmware version */
  status = avrTransactionSync(pAvrCtx, g_vnCommand, DF_ARRAY_LENGTH(g_vnCommand), vnResponse, sizeof(vnResponse), versionTimeout);
  if (AvrTransStatusSuccess != status) {
    dfDebugPrint(DEBUGLEVEL_ERROR, ("*** avrInit: failed to get AVR uC firmware version\n"));
  } else {
    dfCopyKK(&version, &vnResponse[3], sizeof(version));
    dfDebugPrint(0, ("avrInit: AVR firmware version %u.%u.%u.%u\n",
      (unsigned int)dfLe16ToCpu(version.nMajorRelease),
      (unsigned int)dfLe16ToCpu(version.nMinorRelease),
      (unsigned int)dfLe16ToCpu(version.nMajorBuild),
      (unsigned int)dfLe16ToCpu(version.nMinorBuild)));
  }

  /* Success */
  return TRUE;

failed: 
  avrUninit(pAvrCtx);
  return FALSE;
}

void
avrUninit(
  AvrDeviceContext* pAvrCtx)
{
  if (NULL != pAvrCtx->vpd.pMergeBuffer) {
    dfFree(pAvrCtx->vpd.pMergeBuffer);
    pAvrCtx->vpd.pMergeBuffer = NULL;
  }
  dfDpcUninit(&pAvrCtx->transport.rx.dpc);
  dfDpcUninit(&pAvrCtx->transport.tx.dpc);
  dfTimerUninit(&pAvrCtx->transaction.timer);
}

boolean_t
avrIsServiceMode(
  AvrDeviceContext* pAvrCtx)
{
  uint32_t val32;

  val32 = dfPciMemRead32(pAvrCtx->hCtlStat, pAvrCtx->pCtlStat);
  return (val32 & ADMXRC6Tx_V_AVRSTAT_SERVICEMODE) ? TRUE : FALSE;
}
