/*
** File: admxrc6tx_common.c  
** Project: ADB3 core driver
** Purpose: ADM-XRC-6TL/6T1/6TGE/6TGEL/6TDA1-specific code for various hardware functions.
**
** (C) Copyright Alpha Data 2010-2013
*/

#include <df.h>

#include "adb3.h"
#include "admxrc6tx_common.h"
#include "flash.h"
#include "i2c_common.h"

static CoreSensorInfo g_lm87SensorInfo[] = {
  /* LM87 sensors */
  { "1V supply rail",              0, { 0,  1311 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x21, Vccp1 reading */
  { "1.5V supply rail",            0, { 0,  1966 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x28, FAN1/AIN1 reading */
  { "1.8V supply rail",            0, { 0,  2359 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x29, FAN2/AIN2 reading */
  { "2.5V supply rail",            0, { 0,  3277 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x20, +2.5V / ext temp. 2 reading */
  { "3.3V supply rail",            0, { 0,  4325 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x22, +Vcc reading */
  { "5V supply rail",              0, { 0,  6554 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x23, 5V reading */
  { "XMC variable power rail",     0, { 0, 15729 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x24, +12V reading */
  { "XRM I/O voltage",             0, { 0,  3277 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* LM87 register 0x25, Vccp2 reading */
  { "LM87 internal temperature",   0, { 3,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* LM87 register 0x27, Int temp reading */
  { "Target FPGA ext. temp.",      0, { 4,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }  /* LM87 register 0x26, Ext temp. 1 reading */
};

/* These correspond to the elements of 'g_sensorInfo' above, to convert the raw ADC values into Volt / deg. C */
static struct {
  unsigned int regOffset; /* LM87 register offset */
  boolean_t bSigned;      /* TRUE => register value is int8_t; otherwise uint8_t */
  int32_t factor;         /* Fixed point, where 0x00100000 => 1, 0x00080000 => 0.5, 0xFFF80000 => -0.5 etc. */
} g_lm87Scaling[] = {
  { 0x21U, FALSE, 14785    /* 0.0141f */ },
  { 0x28U, FALSE, 10276,   /* 0.0098f */ },
  { 0x29U, FALSE, 10276,   /* 0.0098f */ },
  { 0x20U, FALSE, 13631,   /* 0.0130f */ },
  { 0x22U, FALSE, 18036,   /* 0.0172f */ },
  { 0x23U, FALSE, 27263,   /* 0.0260f */ },
  { 0x24U, FALSE, 65536,   /* 0.0625f */ },
  { 0x25U, FALSE, 14785,   /* 0.0141f */ },
  { 0x27U, TRUE,  1048576, /* 1.0000f */ },
  { 0x26U, TRUE,  1048576, /* 1.0000f */ }
};

/* Xilinx System Monitor sensors, available with firmware 1.4 (PCI rev ID 0x04) and later */
static CoreSensorInfo g_xsmonSensorInfo[] = {
  { "Bridge temperature",          0, { 5,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* XSMON internal FPGA temp. reading */
  { "Bridge VCCINT",               0, { 0,  3932 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* XSMON internal FPGA VINT reading */
  { "Bridge VCCAUX",               0, { 0,  3932 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }  /* XSMON internal FPGA VAUX reading */
};
static CoreSensorInfo g_xsmonSensorInfoAlt[] = {
  { "XSMON Bridge temperature",          0, { 5,     0 }, CoreSensorDataDouble, CoreSensorUnitC,    0 }, /* XSMON internal FPGA temp. reading */
  { "XSMON Bridge VCCINT",               0, { 0,  3932 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }, /* XSMON internal FPGA VINT reading */
  { "XSMON Bridge VCCAUX",               0, { 0,  3932 }, CoreSensorDataDouble, CoreSensorUnitV,    0 }  /* XSMON internal FPGA VAUX reading */
};

/* These correspond to the elements of 'g_xsmonSensorInfo' above, to convert the raw ADC values into Volt / deg. C */
static struct {
  int32_t factor;
  int32_t offset;
} g_xsmonScaling[] = {
  { 32254, -17901158 }, /* T(C)  = x * (503.975 / 1024) - 273.15 */
  { 192,    0 },        /* V(mV) = x * (3 / 1024) */
  { 192,    0 }         /* V(mV) = x * (3 / 1024) */
};

/* Extra diagnostic sensors that have to be enabled by setting driver parameter EnableDiagSensor to TRUE */
static CoreSensorInfo g_diagSensorInfo[] = {
  /* Sensors that are fields in ADB3 PWRSTAT register */
  { "1.5V & 2.5V supply good",     0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 PWRSTAT.PSU1_GOOD */
  { "1.0V supply good",            0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 PWRSTAT.PSU2_GOOD */
  { "1.8V supply good",            0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 PWRSTAT.DIG_1V8_OK */
  { "DDR-3 SDRAM VRef A good",     0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 PWRSTAT.VREF_DRAM_A_OK */
  { "DDR-3 SDRAM VRef B good",     0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 PWRSTAT.VREF_DRAM_B_OK */
  { "Bridge MGT 1.0V supply good", 0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 PWRSTAT.BRG_AVCC_1V0_OK */
  { "Bridge MGT 1.2V supply good", 0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 PWRSTAT.BRG_AVCC_1V2_OK */
  { "Target MGT 1.0V supply good", 0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 PWRSTAT.TGT_AVCC_1V0_OK */
  { "Target MGT 1.2V supply good", 0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 PWRSTAT.TGT_AVCC_1V2_OK */
  { "XRM VI/O supply good",        0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 PWRSTAT.XRM_VIO_OK */
  { "XRM VI/O select lines [2:0]", 0, { 0,     0 }, CoreSensorDataUInt32, CoreSensorUnitNone, 0 }, /* ADB3 PWRSTAT.XRMVSEL */
  { "XRM VI/O supply enabled",     0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 PWRSTAT.XRMVEN */
  { "XRM presence detect",         0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 PWRSTAT.XRM_PRESENT */
  { "XRM force 2.5V",              0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 PWRSTAT.FORCE2V5 */
  { "XRM ROM VI/O(Nom) valid",     0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 PWRSTAT.NOMV_VD */
  { "XRM ROM VI/O(Nom) OOB",       0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 PWRSTAT.OOB_VREQ */
  { "XRM ROM error",               0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 PWRSTAT.ERROR */
  /* Sensors that are fields in ADB3 PWRSTAT register */
  { "Target FPGA PROG# asserted",  0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 FPCTL.PROG */
  { "Target FPGA INIT# asserted",  0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 FPCTL.INIT */
  { "Target FPGA DONE asserted",   0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 FPCTL.DONE */
  { "Target FPGA BUSY asserted",   0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 FPCTL.BUSY */
  { "Target FPGA MODE[2:0]",       0, { 0,     0 }, CoreSensorDataUInt32, CoreSensorUnitNone, 0 }, /* ADB3 FPCTL.MODE */
  { "Target FPGA FBS# asserted",   0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 FPCTL.FBS */
  { "Target FPGA ONECONFIG",       0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 FPCTL.ONECONFIG */
  { "Target FPGA INDGRADE",        0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 FPCTL.INDGRADE */
  { "Target FPGA TGT_CLEARED",     0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 FPCTL.TGT_CLEARED */
  { "Target FPGA RELOAD",          0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 FPCTL.RELOAD */
  /* Sensors that are fields in ADB3 MPTL_RECOVER register */
  { "MPTL_RECOVER.AR_COUNT",       0, { 0,     0 }, CoreSensorDataUInt32, CoreSensorUnitNone, 0 }, /* ADB3 MPTL_RECOVER.AR_COUNT */
  { "MPTL_RECOVER.OCP_INFLIGHT",   0, { 0,     0 }, CoreSensorDataUInt32, CoreSensorUnitNone, 0 }, /* ADB3 MPTL_RECOVER.OCP_RECOVERY_INFLIGHT */
  /* Sensors that are fields in ADB3 MPTL_STAT register */
  { "MPTL_STAT.TARGET_CONFIGURED", 0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 MPTL_STAT.TARGET_CONFIGURED */
  { "MPTL_STAT.LOCAL_GTX_ONLINE",  0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 MPTL_STAT.LOCAL_GTX_ONLINE */
  { "MPTL_STAT.TARGET_GTX_ONLINE", 0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 MPTL_STAT.TARGET_GTX_ONLINE */
  { "MPTL_STAT.GTX_CLK_LOCKED",    0, { 0,     0 }, CoreSensorDataBool,   CoreSensorUnitNone, 0 }, /* ADB3 MPTL_STAT.GTX_CLK_LOCKED */
  { "MPTL_STAT.SYNC_LOSS_COUNT",   0, { 0,     0 }, CoreSensorDataUInt32, CoreSensorUnitNone, 0 }  /* ADB3 MPTL_STAT.SYNC_LOSS_COUNT */
};

static struct {
  unsigned int offset;
  unsigned int startBit;
  unsigned int length;
} g_diagSensorFields[] = {
  /* PWRSTAT fields */
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, pwrStat),       0, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, pwrStat),       1, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, pwrStat),       2, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, pwrStat),       3, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, pwrStat),       4, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, pwrStat),       5, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, pwrStat),       6, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, pwrStat),       7, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, pwrStat),       8, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, pwrStat),       9, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, pwrStat),      16, 3 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, pwrStat),      19, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, pwrStat),      24, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, pwrStat),      25, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, pwrStat),      26, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, pwrStat),      27, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, pwrStat),      31, 1 }, 
  /* FPCTL fields */
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, fpgaCtl),       0, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, fpgaCtl),       1, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, fpgaCtl),       2, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, fpgaCtl),       3, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, fpgaCtl),       8, 3 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, fpgaCtl),      27, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, fpgaCtl),      28, 1 }, 
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, fpgaCtl),      29, 1 },
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, fpgaCtl),      30, 1 },
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, fpgaCtl),      31, 1 },
  /* MPTL_RECOVER fields */
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, mptlRecovery),  4, 4 },
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, mptlRecovery),  8, 8 },
  /* MPTL_STAT fields */
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, mptlStat),      0, 1 },
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, mptlStat),      1, 1 },
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, mptlStat),      2, 1 },
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, mptlStat),      3, 1 },
  { DF_OFFSET_OF(ModelRegsAdmxrc6tx, mptlStat),      8, 24 }
};

unsigned int g_admxrc6txNumDiagSensor = DF_ARRAY_LENGTH(g_diagSensorInfo);

boolean_t
admxrc6txI2cXrmIdentify(
	Adb3CoreDeviceContext* pDevCtx,
  I2cContext* pI2cCtx)
{
  CoreIoModuleInfo* pInfo = &pDevCtx->info.ioModule[0];
	ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  boolean_t bAssumeXrmPresent = FALSE, bSuccess, bForce2V5;
  uint8_t val8;
  uint32_t pwrStat;
  unsigned int i;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  /* This driver parameter can be used to override presence detection for XRMs that don't drive the presence pin correctly */
  dfParameterGetBoolean(pDevCtx->pDevObj->pDrvObj, "AssumeXrmPresent", &bAssumeXrmPresent);

  pInfo->bIsFmc = FALSE;
  pwrStat = dfPciMemRead32(hModel, &pModelRegs->pwrStat);
  pInfo->bPresent = (pwrStat & ADMXRC6Tx_V_PWRSTAT_XPRSNT) == ADMXRC6Tx_V_PWRSTAT_XPRSNT ? TRUE : FALSE;
  if (bAssumeXrmPresent) {
    pInfo->bPresent = TRUE;
  }
  bForce2V5 = (pwrStat & ADMXRC6Tx_V_PWRSTAT_XFORCE2V5) == ADMXRC6Tx_V_PWRSTAT_XFORCE2V5 ? TRUE : FALSE;
  bSuccess = admxrc6txXrmReadByte(pI2cCtx, 0, &val8);
  if (bSuccess) {
    if (pInfo->bPresent) {
      dfDebugPrint(2, ("admxrc6txI2cXrmIdentify: XRM has FRU ROM\n"));
    } else {
      /* Well behaved XRMs should assert the presence pin, but some don't */
      pInfo->bPresent = TRUE;
      dfDebugPrint(0, ("+++ admxrc6txI2cXrmIdentify: XRM has FRU ROM, but doesn't assert presence pin\n"));
    }
    if (val8 != 0x1U) {
      dfDebugPrint(0, ("+++ admxrc6txI2cXrmIdentify: Invalid first byte of FRU ROM, expected=0x%02x actual=0x%02x\n",
        (unsigned int)val8, (unsigned int)val8));
    }
    pInfo->bLegacyXrm = FALSE;
    pInfo->bForce2v5 = bForce2V5;
    pInfo->pVpdBuffer = dfMalloc(ADMXRC6TX_FRU_ROM_SIZE);
    if (NULL == pInfo->pVpdBuffer) {
      dfDebugPrint(0, ("*** admxrc6txI2cXrmIdentify: Failed to allocate FRU ROM buffer, size %p bytes\n", (void*)(uintptr_t)ADMXRC6TX_FRU_ROM_SIZE));
      return FALSE;
    }
    pInfo->vpdLength = ADMXRC6TX_FRU_ROM_SIZE;
    pInfo->pVpdBuffer[0] = val8;
    for (i = 1; i < ADMXRC6TX_FRU_ROM_SIZE; i++) {
      bSuccess = admxrc6txXrmReadByte(pI2cCtx, (uint8_t)i, &pInfo->pVpdBuffer[i]);
      if (!bSuccess) {
        dfDebugPrint(0, ("*** admxrc6txI2cXrmIdentify: Failed to read byte 0x%x from FRU ROM\n", i));
        return FALSE;
      }
    }
  } else {
    if (pInfo->bPresent) {
      dfDebugPrint(2, ("+++ admxrc6txI2cXrmIdentify: FRU ROM not detected - assuming legacy XRM\n"));
      pInfo->bLegacyXrm = TRUE;
      pInfo->bForce2v5 = bForce2V5;
    } else {
      dfDebugPrint(2, ("admxrc6txI2cXrmIdentify: XRM not present\n"));
      pInfo->bPresent = FALSE;
    }
  }

  return TRUE;
}

unsigned int
admxrc6txInitSensorInfoLm87(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int sensorIndex)
{
  unsigned int i, n = 0;

  dfAssert(sensorIndex + DF_ARRAY_LENGTH(g_lm87SensorInfo) <= MAX_NUM_SENSOR);
  for (i = 0; i < DF_ARRAY_LENGTH(g_lm87SensorInfo) && sensorIndex < MAX_NUM_SENSOR; i++, sensorIndex++) {
    pDevCtx->info.sensor[sensorIndex] = g_lm87SensorInfo[i];
    n++;
  }
  return n; 
}

unsigned int
admxrc6txInitSensorInfoXsmon(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int sensorIndex,
  boolean_t bAlternateNaming)
{
  unsigned int i, n = 0;

  if (bAlternateNaming) {
    dfAssert(sensorIndex + DF_ARRAY_LENGTH(g_xsmonSensorInfoAlt) <= MAX_NUM_SENSOR);
    for (i = 0; i < DF_ARRAY_LENGTH(g_xsmonSensorInfoAlt) && sensorIndex < MAX_NUM_SENSOR; i++, sensorIndex++) {
      pDevCtx->info.sensor[sensorIndex] = g_xsmonSensorInfoAlt[i];
      n++;
    }
  } else {
    dfAssert(sensorIndex + DF_ARRAY_LENGTH(g_xsmonSensorInfo) <= MAX_NUM_SENSOR);
    for (i = 0; i < DF_ARRAY_LENGTH(g_xsmonSensorInfo) && sensorIndex < MAX_NUM_SENSOR; i++, sensorIndex++) {
      pDevCtx->info.sensor[sensorIndex] = g_xsmonSensorInfo[i];
      n++;
    }
  }
  return n; 
}

unsigned int
admxrc6txInitSensorInfoDiag(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int sensorIndex)
{
  unsigned int i, n = 0;

  dfAssert(sensorIndex + DF_ARRAY_LENGTH(g_diagSensorInfo) <= MAX_NUM_SENSOR);
  for (i = 0; i < DF_ARRAY_LENGTH(g_diagSensorInfo) && sensorIndex < MAX_NUM_SENSOR; i++, sensorIndex++) {
    pDevCtx->info.sensor[sensorIndex] = g_diagSensorInfo[i];
    n++;
  }
  return n; 
}

CoreSensorStatus
admxrc6txReadSensorLm87(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int lm87SensorIndex,
  CoreSensorValue* pReading)
{
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  int16_t val16;
  int32_t factor, value;

  dfAssert(DF_ARRAY_LENGTH(g_lm87SensorInfo) == DF_ARRAY_LENGTH(g_lm87Scaling));

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  if (lm87SensorIndex < DF_ARRAY_LENGTH(g_lm87SensorInfo)) {
    /* LM87 sensors */
    if (g_lm87Scaling[lm87SensorIndex].bSigned) {
      val16 = (int16_t)(int8_t)dfPciMemRead8(hModel, pModelRegs->sysMonBuf + g_lm87Scaling[lm87SensorIndex].regOffset);
    } else {
      val16 = (int16_t)dfPciMemRead8(hModel, pModelRegs->sysMonBuf + g_lm87Scaling[lm87SensorIndex].regOffset);
    }
    factor = g_lm87Scaling[lm87SensorIndex].factor;
    value = (val16 * factor) / 16;
    pReading->doubleValue.intPart = (int32_t)(int16_t)((value >> 16) & 0xFFFFU);
    pReading->doubleValue.fracPart = ((uint32_t)value & 0xFFFFU) << 16;
    return CoreSensorSuccess;
  }
  return CoreSensorInvalidIndex;
}

CoreSensorStatus
admxrc6txReadSensorXsmon(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int xsMonSensorIndex,
  CoreSensorValue* pReading)
{
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  int32_t factor, offset, value;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  if (xsMonSensorIndex < DF_ARRAY_LENGTH(g_xsmonSensorInfo)) {
    uint32_t* pRegister;
    uint16_t val10;

    pRegister = (uint32_t*)&pModelRegs->xsmonTemp1 + (xsMonSensorIndex << 1);
    /* Get 10-bit A/D reading */
    val10 = (uint16_t)((dfPciMemRead32(hModel, pRegister) >> 6) & 0x3ffU);
    /* Get scaling factor and offset in i15.u16 fixed-point format */
    factor = g_xsmonScaling[xsMonSensorIndex].factor;
    offset = g_xsmonScaling[xsMonSensorIndex].offset;
    /* Apply formula using i15.u16 fixed-point arithmetic */
    value = (int32_t)val10 * factor + offset;
    pReading->doubleValue.intPart = (value >> 16) & 0xFFFFU;
    pReading->doubleValue.fracPart = ((uint32_t)value & 0xFFFFU) << 16;
    return CoreSensorSuccess;
  }
  return CoreSensorInvalidIndex;
}

CoreSensorStatus
admxrc6txReadSensorDiag(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int diagSensorIndex,
  CoreSensorValue* pReading)
{
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t* pReg;
  uint32_t val32;

  dfAssert(DF_ARRAY_LENGTH(g_diagSensorInfo) == DF_ARRAY_LENGTH(g_diagSensorFields));

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  if (diagSensorIndex < DF_ARRAY_LENGTH(g_diagSensorInfo)) {
    pReg = (uint8_t*)pModelRegs + g_diagSensorFields[diagSensorIndex].offset;
    val32 = dfPciMemRead32(hModel, (uint32_t*)pReg);
    val32 >>= g_diagSensorFields[diagSensorIndex].startBit;
    val32 &= DF_MASK32U(g_diagSensorFields[diagSensorIndex].length);
    switch (g_diagSensorInfo[diagSensorIndex].datatype) {
    case CoreSensorDataBool:
      pReading->booleanValue = val32 ? TRUE : FALSE;
      break;

    case CoreSensorDataUInt32:
      pReading->uint32Value = val32;
      break;

    case CoreSensorDataInt32:
      pReading->int32Value = (int32_t)val32;
      break;

    case CoreSensorDataDouble:
      pReading->doubleValue.intPart = (int16_t)val32;
      pReading->doubleValue.fracPart = 0;
      break;

    default:
      return CoreSensorUnexpectedError;
    }
    return CoreSensorSuccess;
  }
  return CoreSensorInvalidIndex;
}

CoreSensorStatus
admxrc6txReadSensor(
	Adb3CoreDeviceContext* pDevCtx,
  unsigned int index,
  CoreSensorValue* pReading)
{
  CoreSensorStatus status = CoreSensorInvalidIndex;

  if (index >= pDevCtx->info.bootstrap.numSensor) {
    return CoreSensorInvalidIndex;
  }

  status = admxrc6txReadSensorLm87(pDevCtx, index, pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_lm87SensorInfo);
  }

  if (pDevCtx->pDevObj->hardwareId.variant.pci.revision >= 0x04U) {
    status = admxrc6txReadSensorXsmon(pDevCtx, index, pReading);
    if (status == CoreSensorSuccess) {
      return status;
    } else if (status != CoreSensorInvalidIndex) {
      return status;
    } else {
      index -= (unsigned int)DF_ARRAY_LENGTH(g_xsmonSensorInfo);
    }
  }

  status = admxrc6txReadSensorDiag(pDevCtx, index, pReading);
  if (status == CoreSensorSuccess) {
    return status;
  } else if (status != CoreSensorInvalidIndex) {
    return status;
  } else {
    index -= (unsigned int)DF_ARRAY_LENGTH(g_diagSensorInfo);
  }

  return status;
}

void
admxrc6txI2cStart(
	I2cContext* pI2cCtx,
  uint8_t bus,
  uint8_t slot,
  uint8_t address,
  boolean_t bWrite,
  uint8_t data,
  void* pContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pContext;
	ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t cmd;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  if (bWrite) {
    dfDebugPrint(6, ("admxrc6txI2cStart: write, bus=0x%x slot=0x%02x address=0x%02x data=0x%02x\n", bus, slot, address, data));
  } else {
    dfDebugPrint(6, ("admxrc6txI2cStart: read, bus=0x%x slot=0x%02x address=0x%02x\n", bus, slot, address));
  }
  cmd = ADMXRC6Tx_V_I2CCTL_DISADDR(address) |
        ADMXRC6Tx_V_I2CCTL_IFADDR(bus) |
        ADMXRC6Tx_V_I2CCTL_I2CADDR(slot) |
        ADMXRC6Tx_V_I2CCTL_ERROR; /* Clear any existing error status */
  cmd |= bWrite ? ADMXRC6Tx_V_I2CCTL_DISWRITE : ADMXRC6Tx_V_I2CCTL_DISREAD;
  if (bWrite) {
    cmd |= ADMXRC6Tx_V_I2CCTL_DISDATA(data);
  }
  dfPciMemWrite32(hModel, &pModelRegs->i2cCtl, cmd);
  dfPciMemRead32(hModel, &pModelRegs->i2cCtl);
}

boolean_t
admxrc6txI2cPoll(
	I2cContext* pI2cCtx,
  boolean_t bWrite,
  uint8_t* pData,
  boolean_t* pbSuccess,
  void* pContext)
{
  Adb3CoreDeviceContext* pDevCtx = (Adb3CoreDeviceContext*)pContext;
	ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint32_t val32;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  val32 = dfPciMemRead32(hModel, &pModelRegs->i2cCtl);
  if (val32 & ADMXRC6Tx_V_I2CCTL_READY) {
    if (val32 & ADMXRC6Tx_V_I2CCTL_ERROR) {
      *pbSuccess = FALSE;
      dfDebugPrint(1, ("+++ admxrc6txI2cPoll: error detected by I2C interface\n"));
    } else {
      *pbSuccess = TRUE;
      if (!bWrite) {
        dfDebugPrint(6, ("admxrc6txI2cPoll: read completed, data=0x%02x\n", (unsigned int)(val32 & 0xFFU)));
        *pData = (uint8_t)val32;
      } else {
        dfDebugPrint(6, ("admxrc6txI2cPoll: write completed\n"));
      }
    }
    return TRUE;
  } else {
    return FALSE;
  }
}

boolean_t
admxrc6txXrmReadByte(
	I2cContext* pI2cCtx,
  uint8_t address,
  uint8_t* pVal8)
{
  boolean_t bSuccess;
  uint8_t val8;

  bSuccess = (i2cReadSync(pI2cCtx, 0, ADMXRC6Tx_I2C_SLOT_XRM, address, &val8) == I2cStatusSuccess) ? TRUE : FALSE;
  if (bSuccess) {
    *pVal8 = val8;
  }
  return bSuccess;
}

boolean_t
admxrc6txXrmWriteByte(
	I2cContext* pI2cCtx,
  uint8_t address,
  uint8_t val8)
{
  boolean_t bSuccess;

  bSuccess = (i2cWriteSync(pI2cCtx, 0, ADMXRC6Tx_I2C_SLOT_XRM, address, val8) == I2cStatusSuccess) ? TRUE : FALSE;
  return bSuccess;
}

CoreVpdStatus
admxrc6txReadWriteVpd(
	Adb3CoreDeviceContext* pDevCtx,
	boolean_t bWrite,
	size_t address,
	size_t length,
	void* pData)
{
  CoreFlashStatus flashStatus;

  if (CHECK_BOUNDS(address, length, pDevCtx->info.flash[0].totalSize)) {
    return CoreVpdInvalidRegion;
  }
	if (bWrite) {
    flashStatus = flashWrite(pDevCtx, 0, address, length, pData);
	} else {
		flashStatus = flashRead(pDevCtx, 0, address, length, pData);
	}
  if (CoreFlashInvalidIndex == flashStatus) {
    /* Should never happen, but map CoreFlashInvalidIndex to CoreVpdGeneralFailure */
    return CoreVpdGeneralFailure;
  } else {
    return (CoreVpdStatus)flashStatus;
  }
}

CoreVpdStatus
admxrc6txSyncVpd(
	Adb3CoreDeviceContext* pDevCtx)
{
  CoreFlashStatus flashStatus;

  flashStatus = flashSync(pDevCtx, 0);
  if (CoreFlashInvalidIndex == flashStatus) {
    /* Should never happen, but map CoreFlashInvalidIndex to CoreVpdGeneralFailure */
    return CoreVpdGeneralFailure;
  } else {
    return (CoreVpdStatus)flashStatus;
  }
}

void
admxrc6txFlashResetBank(
  Adb3CoreDeviceContext* pDevCtx,
  FlashRegsAdmxrc6tx* pFlashRegs)
{
  uint32_t flctl;
  DfMemoryHandle hModel;

  dfDebugPrint(9, ("admxrc6txFlashReset: entered\n"));

  hModel = pDevCtx->hardware.hModel;

  /* Drive fl_rst_n, assert fl_rst_n */
  flctl = (0x1U << 27) | (0x1U << 31);
  dfPciMemWrite32(hModel, &pFlashRegs->ctl, flctl);
  dfPciMemRead32(hModel, &pFlashRegs->ctl);

  /* Deassert fl_rst_n */
  flctl &= ~(0x1U << 31);
  dfPciMemWrite32(hModel, &pFlashRegs->ctl, flctl);
  dfPciMemRead32(hModel, &pFlashRegs->ctl);

  /* Stop driving fl_rst_n */
  flctl &= ~(0x1U << 27);
  dfPciMemWrite32(hModel, &pFlashRegs->ctl, flctl);
  dfPciMemRead32(hModel, &pFlashRegs->ctl);

  /* Allow device time to recover */
  dfDelayThreadFor(dfMicrosecondsToTime(20));
}

void
admxrc6txFlashBank(
  Adb3CoreDeviceContext* pDevCtx,
  FlashRegsAdmxrc6tx* pFlashRegs,
  boolean_t bWrite,
  uint64_t location,
  uint32_t* pValue)
{
  DfMemoryHandle hModel;
  uint32_t flctl, flpage;

  dfDebugPrint(9, ("admxrc6txFlashBank: pFlashRegs=%p bWrite=%lu location=0x%08lx_%08lx *pValue=0x%lx\n",
    pFlashRegs, (unsigned long)bWrite, dfSplitUint64(location), (unsigned long)*pValue));

  if (location >= 0x10000000U) {
    return;
  }

  hModel = pDevCtx->hardware.hModel;

  flpage = (uint32_t)location & 0xffe0000U;
  dfPciMemWrite32(hModel, &pFlashRegs->addr, flpage);
  (void)dfPciMemRead32(hModel, &pFlashRegs->addr);

  if (bWrite) {
    /* Select mode 0, set fl_a, drive fl_a, fl_d, fl_ce_n, fl_oe_n, fl_we_n, assert fl_ce_n */
    flctl = ((uint32_t)location & 0x1fffeU) | (0x7U << 24) | (0x1U << 28);
    dfPciMemWrite32(hModel, &pFlashRegs->ctl, flctl);
    (void)dfPciMemRead32(hModel, &pFlashRegs->ctl);

    /* Set fl_d */
    dfPciMemWrite32(hModel, &pFlashRegs->data, (uint16_t)*pValue);
    (void)dfPciMemRead32(hModel, &pFlashRegs->data);

    /* Assert fl_we_n */
    flctl |= (0x1U << 30);
    dfPciMemWrite32(hModel, &pFlashRegs->ctl, flctl);
    (void)dfPciMemRead32(hModel, &pFlashRegs->ctl);

    /* Deassert fl_we_n */
    flctl &= ~(0x1U << 30);
    dfPciMemWrite32(hModel, &pFlashRegs->ctl, flctl);
    (void)dfPciMemRead32(hModel, &pFlashRegs->ctl);

    /* Deassert fl_ce_n */
    flctl &= ~(0x1U << 28);
    dfPciMemWrite32(hModel, &pFlashRegs->ctl, flctl);
    (void)dfPciMemRead32(hModel, &pFlashRegs->ctl);

    /* Stop driving fl_a, fl_d, fl_ce_n, fl_oe_n, fl_we_n */
    flctl &= ~(0x7U << 24);
    dfPciMemWrite32(hModel, &pFlashRegs->ctl, flctl);
    (void)dfPciMemRead32(hModel, &pFlashRegs->ctl);
	} else {
    /* Select mode 0, set fl_a, Drive fl_a, fl_ce_n, fl_oe_n, fl_we_n, assert fl_ce_n */
    flctl = ((uint32_t)location & 0x1fffeU) | (0x3U << 25) | (0x1U << 28);
    dfPciMemWrite32(hModel, &pFlashRegs->ctl, flctl);
    (void)dfPciMemRead32(hModel, &pFlashRegs->ctl);

    /* Assert fl_oe_n */
    flctl |= (0x1U << 29);
    dfPciMemWrite32(hModel, &pFlashRegs->ctl, flctl);
    (void)dfPciMemRead32(hModel, &pFlashRegs->ctl);

    /* Read fl_d */
    *pValue = (uint16_t)dfPciMemRead32(hModel, &pFlashRegs->data);

    /* Deassert fl_oe_n */
    flctl &= ~(0x1U << 29);
    dfPciMemWrite32(hModel, &pFlashRegs->ctl, flctl);
    (void)dfPciMemRead32(hModel, &pFlashRegs->ctl);

    /* Deassert fl_ce_n */
    flctl &= ~(0x1U << 28);
    dfPciMemWrite32(hModel, &pFlashRegs->ctl, flctl);
    (void)dfPciMemRead32(hModel, &pFlashRegs->ctl);

    /* Stop driving fl_a, fl_ce_n, fl_oe_n, fl_we_n */
    flctl &= ~(0x3U << 25);
    dfPciMemWrite32(hModel, &pFlashRegs->ctl, flctl);
    (void)dfPciMemRead32(hModel, &pFlashRegs->ctl);

    dfDebugPrint(9, ("admxrc6txFlashBank: *value=0x%lx\n", (unsigned long)*pValue));
	}
}

void
admxrc6txFlashSingleBank(
  Adb3CoreDeviceContext* pDevCtx,
  unsigned int bankIndex,
  boolean_t bWrite,
  uint64_t location,
  uint32_t* pValue)
{
  dfDebugPrint(9, ("admxrc6txFlashSingleBank: bankIndex=%u bWrite=%lu location=0x%08lx_%08lx *pValue=0x%lx\n",
    bankIndex, (unsigned long)bWrite, dfSplitUint64(location), (unsigned long)*pValue));

  if (bankIndex != 0) {
    return;
  }

  admxrc6txFlashBank(pDevCtx, &pDevCtx->hardware.model.pAdmxrc6tx->flash, bWrite, location, pValue);
}

CoreVpdStatus
admxrc6txReadAvrVpdData(
  Adb3CoreDeviceContext* pDevCtx,
  uint16_t address,
  size_t n,
  void* pBuffer)
{
  ModelRegsAdmxrc6tx* pModelRegs;
  DfMemoryHandle hModel;
  uint8_t* p = (uint8_t*)pBuffer;
  uint16_t mask;

  pModelRegs = pDevCtx->hardware.model.pAdmxrc6tx;
  hModel = pDevCtx->hardware.hModel;

  mask = sizeof(pModelRegs->model3.avr.vpd) - 1U;
  while (n) {
    address = (uint16_t)(address & mask);
    *p++ = dfPciMemRead8(hModel, &pModelRegs->model3.avr.vpd[address]);
    address++;
    n--;
  }
  return CoreVpdSuccess;
}

boolean_t
admxrc6txVerifyAvrVpdChecksum(
	Adb3CoreDeviceContext* pDevCtx,
  uint16_t vpdLength)
{
  ModelRegsAdmxrc6tx* pModelRegs;
  CoreVpdStatus status;
  uint8_t buff[16];
  uint8_t sum = 0, checksum;
  uint16_t offset = 0;
  uint16_t remaining = (uint16_t)(vpdLength - 1U);
  uint16_t i, chunk;

  if (vpdLength < 5 || vpdLength > sizeof(pModelRegs->model3.avr.vpd)) {
  	dfDebugPrint(0, ("*** admxrc6txVerifyAvrVpdChecksum: invalid VPD length %lu(0x%lx)\n",
      (unsigned long)vpdLength, (unsigned long)vpdLength));
    pDevCtx->info.vpd.bChecksumError = TRUE;
    return FALSE;
  }

  /* Get checksum from VPD memory; always last byte of structure */
  status = admxrc6txReadAvrVpdData(pDevCtx, remaining, sizeof(checksum), &checksum);
  if (CoreVpdSuccess != status) {
  	dfDebugPrint(0, ("*** admxrc6txVerifyAvrVpdChecksum: failed to read checksum from VPD area\n"));
    pDevCtx->info.vpd.bChecksumError = TRUE;
    return FALSE;
  }

  /* Compute checksum over all but last byte of structure */
  while (remaining) {
    chunk = (uint16_t)((remaining > sizeof(buff)) ? sizeof(buff) : remaining);
    status = admxrc6txReadAvrVpdData(pDevCtx, offset, chunk, buff);
    if (CoreVpdSuccess != status) {
  	  dfDebugPrint(0, ("*** admxrc6txVerifyAvrVpdChecksum: failed to data chunk from VPD area for checksumming\n"));
      pDevCtx->info.vpd.bChecksumError = TRUE;
      return FALSE;
    }

    for (i = 0; i < chunk; i++) {
      sum = (uint8_t)(sum + buff[i]);
    }
    offset = (uint16_t)(offset + chunk);
    remaining = (uint16_t)(remaining - chunk);
  }
  sum = (uint8_t)-sum;

  if (sum == checksum) {
    return TRUE;
  } else {
  	dfDebugPrint(0, ("*** admxrc6txVerifyAvrVpdChecksum: checksum invalid, computed 0x%02lx found 0x%02lx\n",
      (unsigned long)sum, (unsigned long)checksum));
    pDevCtx->info.vpd.bChecksumError = TRUE;
    return FALSE;
  }
}
