/*
** File: adb1.h  
** Project: ADB1 core driver
** Purpose: Definitions for ADB1 PCI/PCI-X to Local Bus Bridge.
**
** (C) Copyright Alpha Data 2013
*/

#ifndef _ADATA_CORE_ADB1_H
#define _ADATA_CORE_ADB1_H

#include <df.h>

/* ---------------------------------------------------- */
/* BAR0 registers (generic)                             */
/* ---------------------------------------------------- */

/* Flags for NDL field of ADB1_DMA_DESCRIPTOR */
#define ADB1_DMA_NDL_LOCALSPACE (0x0U << 0)
#define ADB1_DMA_NDL_PCISPACE   (0x1U << 0)
#define ADB1_DMA_NDL_ENDCHAIN   (0x1U << 1)
#define ADB1_DMA_NDL_INTERRUPT  (0x1U << 2)

/* Flags for SIZ field of ADB1_DMA_DESCRIPTOR */
#define ADB1_DMA_SIZ_LOCALTOPCI (0x1U << 31)
#define ADB1_DMA_SIZ_PCITOLOCAL (0x0U << 31)

/* Linked-list DMA node structure, used by a DMA engine. */
#pragma pack(1)
typedef struct _Adb1DmaDescriptor {
  uint32_t pal;
  uint32_t pah;
  uint32_t la;
  uint32_t siz;
  uint32_t ndl;
  uint32_t ndh;
  uint32_t reserved1[2];
} Adb1DmaDescriptor;
#pragma pack()

/* Fields for Adb1DmaRegisters::cfg register */
#define ADB1_DMA_CFG_WIDTHMASK      (0x3U << 0)
#define ADB1_DMA_CFG_32BIT          (0x2U << 0)
#define ADB1_DMA_CFG_64BIT          (0x3U << 0)
#define ADB1_DMA_CFG_USE_READY      (0x1U << 6)
#define ADB1_DMA_CFG_USE_BTERM      (0x1U << 7)
#define ADB1_DMA_CFG_ENABLE_BURST   (0x1U << 8)
#define ADB1_DMA_CFG_CHAINED        (0x1U << 9)
#define ADB1_DMA_CFG_DONE_INTERRUPT (0x1U << 10)
#define ADB1_DMA_CFG_FIXED_LOCAL    (0x1U << 11)
#define ADB1_DMA_CFG_DEMAND         (0x1U << 12)
#define ADB1_DMA_CFG_USE_EOT        (0x1U << 14)
#define ADB1_DMA_CFG_INTERRUPT_PCI  (0x1U << 17)

/* Fields for Adb1DmaRegisters::ctl register */
#define ADB1_DMA_CTL_GO      (0x1U << 0)
#define ADB1_DMA_CTL_ABORT   (0x1U << 1)
#define ADB1_DMA_CTL_ENABLE  (0x1U << 2)
#define ADB1_DMA_CTL_IDLE    (0x1U << 4)
#define ADB1_DMA_CTL_ENABLED (0x1U << 5)
#define ADB1_DMA_CTL_CABORT  (0x1U << 6)
#define ADB1_DMA_CTL_MABORT  (0x1U << 7)
#define ADB1_DMA_CTL_TABORT  (0x1U << 8)
#define ADB1_DMA_CTL_LEOT    (0x1U << 9)
#define ADB1_DMA_CTL_SCM     (0x1U << 10)
#define ADB1_DMA_CTL_RESET   (0x1U << 11)

/* Registers for a DMA engine. */
#pragma pack(1)
typedef struct _Adb1DmaRegisters {
  uint32_t pal;
  uint32_t pah;
  uint32_t la;
  uint32_t siz;
  uint32_t ndl;
  uint32_t ndh;
  uint32_t cfg;
  uint32_t ctl;
} Adb1DmaRegisters;
#pragma pack()

#define ADB1_MAX_NUM_DMA_ENGINE (4U)

/* BAR0 registers. */
#pragma pack(1)
typedef struct _Adb1BridgeRegisters {
  uint32_t ls0b;           /* 0x00 */
  uint32_t ls0r;           /* 0x04 */
  uint32_t ls0cfg;         /* 0x08 */
  uint32_t _reserved1[1];  /* 0x0C */
  uint32_t ls1b;           /* 0x10 Bridge rev. 1+ */
  uint32_t ls1r;           /* 0x14 Bridge rev. 1+ */
  uint32_t ls1cfg;         /* 0x18 Bridge rev. 1+ */
  uint32_t _reserved2[1];  /* 0x1C */
  uint32_t lsto;           /* 0x20 */
  uint32_t lstat;          /* 0x24 */
  uint32_t lctl;           /* 0x28 */
  uint32_t lerc;           /* 0x2C */
  uint32_t lera;           /* 0x30 */
  uint32_t lictl;          /* 0x34 */
  uint32_t listat;         /* 0x38 */
  uint32_t _reserved3[5];  /* 0x3C */
  uint32_t pictl;          /* 0x50 */
  uint32_t pistat;         /* 0x54 */
  uint32_t _reserved4[10]; /* 0x58 */
  Adb1DmaRegisters dma[ADB1_MAX_NUM_DMA_ENGINE]; /* 0x80 Bridge rev. 1+ */
} Adb1BridgeRegisters;
#pragma pack()

/* Fields for Adb1BridgeRegisters::pictl */
#define ADB1_PICTL_ENMODEL          (0x1U)                /* Enable model-specific interrupt */
#define ADB1_PICTL_ENDMA(channel)   (0x100U << (channel)) /* Enable DMA engine interrupt */
#define ADB1_PICTL_ENDMA_ALL        (0xFF00U)             /* Enable all DMA engine interrupts (up to 8) */
#define ADB1_PICTL_ENINTA           (0x1U << 31)          /* Allows PCI INTA# to be asserted for any unmasked interrupt. */

/* Fields for Adb1BridgeRegisters::pistat */
#define ADB1_PISTAT_IRQMODEL        (0x1U)                /* Model-specific interrupt */
#define ADB1_PISTAT_IRQDMA(channel) (0x100U << (channel)) /* DMA engine interrupt */

/* ---------------------------------------------------- */
/* BAR1 registers (model-specific)                      */
/* ---------------------------------------------------- */

/* ADM-XRC-5LX (but not ADM-XRC-5LXA) */
#pragma pack(1)
typedef struct _Admxrc5lxClockRegisters {
  uint32_t _reserved1[4];
  uint32_t lclkctl;
  uint32_t mclkctl;
  uint32_t _reserved2[2];
} Admxrc5lxClockRegisters;
#pragma pack()

/* All ADB1-based V4 and V5 models, except ADM-XRC-5LX */
#pragma pack(1)
typedef struct _Adb1V4V5ClockRegisters {
  uint32_t _reserved1[2];
  uint32_t mclkctl;
  uint32_t _reserved2[1];
  uint32_t lclkctl;
  uint32_t lclkstat1;
  uint32_t lclkstat2;
  uint32_t lclkstat3;
} Adb1V4V5ClockRegisters;
#pragma pack()

#pragma pack(1)
/* BAR1 registers */
typedef struct _ModelRegsAdb1V4V5 {
  union {
    Admxrc5lxClockRegisters admxrc5lx;   /* 0x000000 */
    Adb1V4V5ClockRegisters others;       /* 0x000000 */
  } clocking;
  uint8_t fpsmap[4];                     /* 0x000020 */
  uint32_t fpctl;                        /* 0x000024 */
  uint32_t ictl;                         /* 0x000028 */
  uint32_t istat;                        /* 0x00002C */
  uint32_t pwrctl;                       /* 0x000030 */
  uint32_t pwrstat;                      /* 0x000034 */
  uint32_t _reserved3[50];               /* 0x000038 */
  uint32_t flctl;                        /* 0x000100 */
  uint32_t flpage;                       /* 0x000104 */
  uint32_t fldata;                       /* 0x000108 */
  uint32_t _reserved4[13];               /* 0x00010C */
  uint8_t flbuf[64];                     /* 0x000140 */
  uint32_t _reserved5[32];               /* 0x000180 */
  uint32_t xrmid[17];                    /* 0x000200 */
  uint32_t _reserved6[111];              /* 0x000244 */
  uint32_t sysmonctl;                    /* 0x000400 */
  uint32_t _reserved7[63];               /* 0x000404 */
  uint8_t sysmonbuf[0x100];              /* 0x000500 */
  uint8_t i2c[0x200];                    /* 0x000600 */
  uint8_t _reserved8[0x3F800];           /* 0x000800 */
  uint8_t parjtag[0x20000];              /* 0x040000 */
  uint8_t _reserved9[0x20000];           /* 0x060000 */
  uint8_t flash[0x40000];                /* 0x080000 */
  uint8_t _reserved10[0x40000];          /* 0x0C0000 */
} ModelRegsAdb1V4V5;                     /* 0x100000 */
#pragma pack()

/* Fields for ModelRegsAdb1V4V5::fpctl */
#define ADB1V4V5_FPCTL_PROG   (0x1U << 0)
#define ADB1V4V5_FPCTL_INIT   (0x1U << 1)
#define ADB1V4V5_FPCTL_DONE   (0x1U << 2)
#define ADB1V4V5_FPCTL_RELOAD (0x1U << 31)

/* Fields for ModelRegsAdb1V4V5::ictl */
#define ADB1V4V5_ICTL_ENFPGA   (0x1U << 0) /* Enable target FPGA interrupt */

/* Fields for ModelRegsAdb1V4V5::istat */
#define ADB1V4V5_ISTAT_IRQFPGA (0x1U << 0) /* Target FPGA interrupt */

#endif
