/*
** File: portability.c
** Project: ADMXRC3 API module library
** Purpose: Linux-specific portability functions used by ADMXRC3 API module library
**
** (C) Copyright Alpha Data 2010
*/

#include "portability.h"
#include "../portable_file.h"
#include "../portable_string.h"
#include "../portable_wstring.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>
#include <assert.h>
#include <errno.h>

portable_file_t
portableFileOpenA(
  const char* pFileName,
  bool_t bWrite)
{
  return fopen(pFileName, "rt");
}

portable_file_t
portableFileOpenW(
  const wchar_t* pFileName,
  bool_t bWrite)
{
  const wchar_t* pSrc = pFileName;
  char* pUtf8Str = NULL;
  mbstate_t state;
  size_t n, m;

  pSrc = pFileName;
  memset(&state, 0, sizeof(state));
  n = wcsrtombs(NULL, &pSrc, 0, &state);
  if (n == (size_t)-1) {
    /* Conversion not possible */
    return NULL;
  }
  pUtf8Str = (char*)malloc((n + 1) * sizeof(char));
  if (NULL == pUtf8Str) {
    return NULL;
  }
  pSrc = pFileName;
  memset(&state, 0, sizeof(state));
  m = wcsrtombs(pUtf8Str, &pSrc, n + 1, &state);
  if (m == (size_t)-1) {
    /* Conversion not possible */
    return NULL;
  }
  assert(m == n);
  return portableFileOpenA(pUtf8Str, bWrite);
  free(pUtf8Str);
}

void
portableFileClose(
  portable_file_t hFile)
{
  fclose(hFile);
}

bool_t
portableFileGetSize(
  portable_file_t hFile,
  uint64_t* pSize)
{
  int fd;
  struct stat info;

  fd = fileno(hFile);
  if (fstat(fd, &info)) {
    /*
    ** Probably hit the GLIBC overflow bug that affects 32-bit processes running under a 64-bit kernel;
    ** Fall back to using fseek() & ftell() to get the file's size.
    */

    long initPos, sizePos;
    int result;

    /* Save original file position */
    initPos = ftell(hFile);
    if (initPos < 0) {
      return FALSE;
    }

    /* Position to end of file */
    result = fseek(hFile, 0, SEEK_END);
    if (result) {
      return FALSE;
    }

    /* Get file size */
    sizePos = ftell(hFile);
    if (sizePos < 0) {
      return FALSE;
    }

    /* Restore original file position */
    result = fseek(hFile, initPos, SEEK_SET);
    if (result) {
      return FALSE;
    }

    *pSize = (uint64_t)sizePos;
  } else {
    *pSize = info.st_size;
  }
  return TRUE;
}

bool_t
portableFileRead(
  portable_file_t hFile,
  void* pBuffer,
  uint64_t n,
  uint64_t* pNumRead)
{
  size_t chunk, numRead;
  uint64_t totalRead;

  totalRead = 0;
  while (n) {
    if (n > (size_t)-1) {
      chunk = (size_t)-1;
    } else {
      chunk = (size_t)n;
    }
    numRead = fread(pBuffer, 1, chunk, hFile);
    if (numRead == 0) {
      break;
    }
    totalRead += numRead;
    n -= numRead;
  }
  *pNumRead = totalRead;
  return TRUE;
}

bool_t
portableFileSetPointer(
  portable_file_t hFile,
  uint64_t offset)
{
  if (offset > (off_t)-1) {
    return FALSE;
  }
  if (fseeko(hFile, (off_t)offset, SEEK_SET)) {
    return FALSE;
  }
  return TRUE;
}

size_t
portableStringLengthA(
  const char* pDst,
  size_t max)
{
  char c;
  size_t n = 0;

  if (!max) {
    return 0;
  }
  max--;
  while (max) {
    c = *pDst++;
    if ('\0' == c) {
      break;
    }
    max--;
    n++;
  }
  return n;
}

size_t
portableStringLengthW(
  const wchar_t* pDst,
  size_t max)
{
  wchar_t c;
  size_t n = 0;

  if (!max) {
    return 0;
  }
  max--;
  while (max) {
    c = *pDst++;
    if (L'\0' == c) {
      break;
    }
    max--;
    n++;
  }
  return n;
}

void
portableStringCat(
  char* pDst,
  size_t max,
  const char* pSrc)
{
  size_t n;

  if (!max) {
    return;
  }
  n = portableStringLengthA(pDst, max);
  max--;
  max -= n;
  pDst += n;
  if (max) {
    memcpy(pDst, pSrc, max);
  }
  pDst[max] = '\0';
}

void
portableStringCopy(
  char* pDst,
  size_t max,
  const char* pSrc)
{
  char c;

  if (!max) {
    return;
  }
  max--;
  while (max) {
    c = *pSrc;
    if ('\0' == c) {
      break;
    }
    *pDst++ = *pSrc++;
    max--;
  }
  *pDst++ = '\0';
}

void
portableStringLower(
  char* pString,
  size_t bufSize)
{
  char c;

  if (!bufSize) {
    return;
  }
  bufSize--;
  while (bufSize) {
    c = *pString;
    if ('\0' == c) {
      break;
    }
    *pString++ = tolower(c);
    bufSize--;
  }
}

void
portableStringUpper(
  char* pString,
  size_t bufSize)
{
  char c;

  if (!bufSize) {
    return;
  }
  bufSize--;
  while (bufSize) {
    c = *pString;
    if ('\0' == c) {
      break;
    }
    *pString++ = toupper(c);
    bufSize--;
  }
}

int
portableStringCompareA(
  const char* pString1,
  const char* pString2)
{
  return strcmp(pString1, pString2);
}

int
portableStringCompareW(
  const wchar_t* pString1,
  const wchar_t* pString2)
{
  return wcscmp(pString1, pString2);
}

bool_t
portableUTF8ToWideChar(
  wchar_t* pBuffer,
  size_t bufferSize,
  const char* pAnsiString)
{
  const char* pSrc = pAnsiString;
  mbstate_t state;
  size_t n;

  if (!bufferSize) {
    return FALSE;
  }
  memset(&state, 0, sizeof(state));
  n = mbsrtowcs(pBuffer, &pSrc, bufferSize, &state);
  if (n == (size_t)-1) {
    return FALSE;
  }
  pBuffer[bufferSize - 1] = L'\0'; /* Ensure WNUL-terminated */
  return TRUE; /* Success */
}
