-- =============================================================================================
--                             	                                                            **
-- =============================================================================================
--	Title: sampck_synth_init.vhd
--	Description:	VHDL  design file for initialisation of AD9510 on adp_pupe
--	Sets the AD9510 for bypassed dividers and to ignore the chip level syncs

--	Register 00 is left as default (0x10)
--	Registers O1-03 do not exist
--	Register 04-06 set the N divider to 25 for a 125 MHz reference, 5MHz channel spacing
--	Register 07 (Loss of reference) is left as default (0x00)
--	Register 08-0C enable the charge pump normal operation,charge pump current,PLL enabled and ref= divide by 2
--	Register 0D (Lock Detect) is left as default (0x00)
--	Registers OE-33 do not exist
--	Registers 34-3B ( delay control) are left as default
--	Registers 3C-3F have the LVPECL outputs disabled to preserve power which requires
--	  that D0 and D1 of these is set to 1; all other bits can use the default
--	Registers 40-43 defaults to the LVDS mode enabled with 3.5 mA drive but are explicitly set anyway.
--	Register 44 does not exist
--	Register 45 defaults to CK1 as the input and enabled. D2 could be set to disable CK2. Similarly
--	  D3 and D4 could be set to power down the prescaler and ref in inputs, THOUGH NOT STRICTLY NECESSARY
--	  the remainder default states are ok.
--	Registers 46-47 do not exist
--	Registers 48-56 (even)become don't cares if the divider is bypassed; could be set to 11 hex
--	  if the 200 MHz clock is used at 100 MHz
--	Registers 49-57 (odd)are set to 80 ( bypass divider) or possibly C0 ( disable sync as well)
--    if the internal dividers aren't used. Note that since only out 4-7 used, only really need
--	  to do this for 51-57.
--	Registers 58-59 do not exist
--	Registers 5a lsb must be toggled to update the registers - either the odd or even registers must be programmed
--	  so this must be pulsed
--
--	Hence the modified registers are 04,05,06,08,09,0a,0b,0c,
--	                                 3c,3d,3e,3f,
--	                                 40,41,42,43,
--                                   45,49,4b,4d,4f,
--                                   51,53,55,57
--	(25 total)



--	Mod Record
--	===========
--	1)08/05/2007 15:50:52		sampck_synth_init created
--
--	2)14/06/2007 17:42:11		Amended setup to use 4.8 mA drive, pos polarity, staus pins = dig lock
--													LOR enabled and correct values for N and R ( note only P has N for N-1 programmed)
-- (C) Copyright Alpha Data Parallel Systems Ltd. 1999-2003
--
--
-------------------------------------------------------------------------------------------------




--********************************************************************************************








--********************************************************************************************
--			Entity Definition
--
-- Notes:
--********************************************************************************************
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;



--Required for 'attribute' use
--library synplify;
--use synplify.attributes.all;


entity sampck_synth_init is
port
(
		reset			:in		std_logic;
		clock			:in		std_logic;
		trigger		:in		std_logic;

		init_done		:out		std_logic;
		wr_strobe	:out		std_logic;
		busy			:in		std_logic;


		address		:out		std_logic_vector( 7 downto 0);
		wr_data		:out		std_logic_vector( 7 downto 0)
);
end sampck_synth_init;
--********************************************************************************************













--********************************************************************************************
--	Architecture Definition
--
--********************************************************************************************
architecture synth of sampck_synth_init is









--********************************************************************************************
--	Constant/Type Definitions
--********************************************************************************************
constant END_VAL	:std_logic_vector := X"FF";

--No. of initialisation writes
constant INIT_LEN	:positive := 28;


--Note the definition - order of array specification determines order of writes
-- actual length is arbitrary - must accomodate the largest write sequence
-- and last pair must be FF.
-- INIT_LEN ensures that there is always 1 pair for FF
type SLV_ARRAY_INIT8W is array(0 to 2*INIT_LEN) of std_logic_vector(7 downto 0);

--Values as address/data pairs
constant init_sequence :SLV_ARRAY_INIT8W :=
(
	--set counters - A=1,P=1,B=25 so VCXO is divided by 25
	x"04",x"00",	-- A is bypassed for P in FD mode
	x"05",x"00",	--B msbyte
	x"06",x"19",	--B lsbyte; 25 required, N for N programmed
	x"07",x"04",	--LOR enabled


	x"08",x"47",  --charge pump normal op, status pin = DLDHI  (2B outputs LOR on status pin), pos polarity
	x"09",x"70",	-- 4.8 mA for CP current
	x"0A",x"00",	--Prescale in FD mode, divide by 1, B enabled, PLL normal operation
	x"0B",x"00",
	x"0C",x"02",	-- Reference counter program N for divide by N
								--Lock detect window is default, LOR is enabled by default ( address D)

	-- turn off LVPECL
	x"3C",x"0A",
	x"3D",x"0A",
	x"3E",x"0A",
	x"3F",x"0A",


	-- enable all LVDS out
	x"40",x"02",
	x"41",x"02",
	x"42",x"02",
	x"43",x"02",

	 --VCXO (ck2), enable ref in
	 X"45",x"00",

	-- set all channels to divider bypass
	x"49",x"C0",
	x"4b",x"C0",
	x"4d",x"C0",
	x"4f",x"C0",
	x"51",x"C0",
	x"53",x"C0",
	x"55",x"C0",
	x"57",x"C0",


	others =>x"FF"
);

























--State machine variable
  type state_type is
  (
  	idle_state,
  	wrreg_state,
  	wrreg_strblo_state,
  	wrreg_busy_state,
  	wr5ahi_state,
  	wr5ahi_busy_state
  );


--********************************************************************************************


--********************************************************************************************
--	Component Definitions
--********************************************************************************************


--********************************************************************************************







--********************************************************************************************
--	Signal Definitions
--
--********************************************************************************************

signal 	clock_sig :std_logic;
signal 	reset_sig :std_logic;
signal 	wr_strobe_sig  :std_logic;
signal 	busy_sig  :std_logic;
signal 	addr_sig   :std_logic_vector(7 downto 0);
signal 	wr_data_sig    :std_logic_vector(7 downto 0);

signal 	state_adv_sig  :std_logic;
signal	trig_sig :std_logic_vector(1 downto 0);

signal 	ser_state  :state_type;
signal 	next_ser_state  :state_type;

signal 	next_wr_strobe_sig  :std_logic;
signal 	next_addr_sig   :std_logic_vector(7 downto 0);
signal 	next_wr_data_sig    :std_logic_vector(7 downto 0);





signal init_addr_sig 	:std_logic_vector( 7 downto 0);
signal init_data_sig 	:std_logic_vector( 7 downto 0);
signal init_trig_sig 	:std_logic;
signal instr_cnt_sig 	:std_logic_vector( 7 downto 0);
signal active_sig 	:std_logic;
signal active_pl_sig 	:std_logic;
signal send_data_sig 	:std_logic;

signal next_active_sig 	:std_logic;


signal init_done_sig 	:std_logic;





--********************************************************************************************





--********************************************************************************************
--	Attributes
--********************************************************************************************


--********************************************************************************************



























--********************************************************************************************
--	Architectural Statements
--********************************************************************************************

begin
--==============================================================================================
-- Port mappings
--==============================================================================================
	clock_sig <= clock;
	reset_sig <= reset;
	wr_strobe <= wr_strobe_sig;

	address<= addr_sig;
	wr_data <= wr_data_sig;

	busy_sig <=busy;

 	state_adv_sig  <='1';
	init_done <= init_done_sig;
------------------------------------------------------------------------------------------------



--==============================================================================================
-- Wait for rising edge of trigger sig
--==============================================================================================
process (clock_sig ,reset_sig)
begin
	if reset_sig = '1' then
			trig_sig <=(others =>'0');
	elsif rising_edge (clock_sig) then
			trig_sig(0) <=trigger;
			trig_sig(1)	<=	trig_sig(0);
	end if;
end process;
------------------------------------------------------------------------------------------------








--==============================================================================================
-- Signal a write
--==============================================================================================
process (clock_sig ,reset_sig)
begin
	if reset_sig = '1' then
			init_addr_sig <= init_sequence(0);
			init_data_sig <= init_sequence(1);

			init_trig_sig <= '0';
			init_done_sig <= '0';

			instr_cnt_sig <=(others =>'0');
			active_pl_sig <= '0';

			send_data_sig <= '0';
	elsif rising_edge (clock_sig) then

		init_addr_sig <= init_sequence(2*CONV_INTEGER(instr_cnt_sig));
		init_data_sig <= init_sequence(1+2*CONV_INTEGER(instr_cnt_sig));
		active_pl_sig <= active_sig;

		--enable the sequence following a trigger rising edge, disable when the end of sequence is reached
		if trig_sig="01" then
				init_trig_sig <= '1';
		else
			if init_sequence(2*CONV_INTEGER(instr_cnt_sig)) = END_VAL then
				init_trig_sig <= '0';
				init_done_sig <= '1';
		end if;
		end if;

		--update the count on a falling edge of the active signal
		-- cleared if sequencenot triggered
		if init_trig_sig = '0' then
				instr_cnt_sig <=(others =>'0');
		else
			if active_sig= '0'  and active_pl_sig= '1' then	--falling edge
					instr_cnt_sig <= instr_cnt_sig +'1';
			end if;
		end if;

		-- if sequence enabled and active pline clear, trigger a write
		if init_trig_sig = '1' then
			if init_sequence(2*CONV_INTEGER(instr_cnt_sig)) = END_VAL then
					send_data_sig <= '0';
			else

				if active_sig= '0'  and active_pl_sig= '0' then
						send_data_sig <= '1';
				else
						send_data_sig <= '0';
				end if;
			end if;
		else
				send_data_sig <= '0';
		end if;

	end if;
end process;
------------------------------------------------------------------------------------------------







--==============================================================================================
-- State machine update
--==============================================================================================
process (clock_sig ,reset_sig)
begin
	if reset_sig = '1' then
		addr_sig <=  (others => '0');
		wr_data_sig <=  (others => '0');
		wr_strobe_sig <= '0';
		active_sig <= '0';
		ser_state <= idle_state;
	elsif rising_edge (clock_sig) then
		addr_sig <=  next_addr_sig;
		wr_data_sig <=  next_wr_data_sig;
		wr_strobe_sig <= next_wr_strobe_sig;
		active_sig <= next_active_sig;
		if state_adv_sig='1' then
			ser_state <=next_ser_state;
		end if;
	end if;
end process;
------------------------------------------------------------------------------------------------







--==============================================================================================
-- State machine muxing
--==============================================================================================
	statemux :
	process(	send_data_sig)
	begin
		case ser_state is

		-----------------------------------------------
		-- wait for a trigger
		--
		--
		-----------------------------------------------
		when idle_state =>
			next_wr_strobe_sig <='0';
			next_addr_sig<= init_addr_sig;
			next_wr_data_sig<= init_data_sig;
			next_active_sig <='0';

			if send_data_sig='1' then
				next_ser_state <= wrreg_state;
			else
				next_ser_state <= idle_state;
			end if;
		-----------------------------------------------




		-----------------------------------------------
		-- Setup register - strobe low state not really needed
		-----------------------------------------------
		when wrreg_state =>
			next_wr_strobe_sig <='1';
			next_addr_sig<= init_addr_sig;
			next_wr_data_sig<= init_data_sig;
			next_ser_state <= wrreg_strblo_state;
			next_active_sig <='1';

		when wrreg_strblo_state =>
			next_wr_strobe_sig <='0';
			next_addr_sig<= init_addr_sig;
			next_wr_data_sig<= init_data_sig;
			next_active_sig <='1';

			if busy_sig ='1' then
				next_ser_state <= wrreg_busy_state;
			else
				next_ser_state <= wrreg_strblo_state;
			end if;
		-----------------------------------------------

		-----------------------------------------------
		-- Wait for write to finish
		-----------------------------------------------
		when wrreg_busy_state =>
			next_wr_strobe_sig <='0';
			next_addr_sig<= init_addr_sig;
			next_wr_data_sig<= init_data_sig;
			next_active_sig <='1';

			if busy_sig ='0' then
				next_ser_state <= wr5ahi_state;
			else
				next_ser_state <= wrreg_busy_state;
			end if;
		-----------------------------------------------




		-----------------------------------------------
		-- Strobe 5a hi- self clears so no need oto write it low
		-----------------------------------------------
		when wr5ahi_state =>
			next_wr_strobe_sig <='1';
			next_addr_sig<= X"5a";
			next_wr_data_sig<= X"01";
			next_active_sig <='1';

			if busy_sig ='1' then
				next_ser_state <= wr5ahi_busy_state;
			else
				next_ser_state <= wr5ahi_state;
			end if;
		-----------------------------------------------

		-----------------------------------------------
		-- Wait for write to finish
		-----------------------------------------------
		when wr5ahi_busy_state =>
			next_wr_strobe_sig <='0';
			next_addr_sig<= X"5a";
			next_wr_data_sig<= X"01";
			next_active_sig <='1';

			if busy_sig ='0' then
				next_ser_state <= idle_state;
			else
				next_ser_state <= wr5ahi_busy_state;
			end if;
		-----------------------------------------------



		-----------------------------------------------
		-- catch any other states
		-----------------------------------------------
		when others =>
			next_wr_strobe_sig <='0';
			next_addr_sig<= X"49";
			next_wr_data_sig<= X"C0";
			next_active_sig <='0';

			next_ser_state <= idle_state;
		-----------------------------------------------
		end case;
	end process statemux;


------------------------------------------------------------------------------------------------







end; -- architecture  sampck_synth_init
--********************************************************************************************


