-- =============================================================================================
--                             	                                                            **
-- =============================================================================================
--	Title: sampck_synth_cntrl.vhd
--	Description:	VHDL  design file for control of AD9510 on adp_pupe
--	Communication cycles comprise of sending an instruction word of 16 bits to the AD9510
--  followed by reading or writing a data word of 8, 16 or 32 bits
--
--  The default data organisation is MSB first, with register addresses decrementing. Data is byte orientated and isn't loaded
--  until address 5a is strobed. Each byte is mapped to the bottom 8 bits of a 32 bit address to avoid problems with
--  address sense.
--  All reads and writes are 8 bit values.
--
--  A maximum of 90 locations exist so requires 7 address bits - port set to 8 bits wide and msb ignored

--  AD9510 TIMING
--  Clock Rate (SCLK, 1/tSCLK) 25 MHz
--  Pulse Width High, tPWH 16 ns
--  Pulse Width Low, tPWL 16 ns
--  SDIO to SCLK Setup, tDS 2 ns (before rising edge of clock on device)
--  SCLK to SDIO Hold, tDH 1 ns (after rising edge of clock on device)
--  SCLK to Valid SDIO and SDO, tDV 6 ns (falling edge of clock on device)
--  CSB to SCLK Setup and Hold, tS, tH 2 ns
--  CSB Minimum Pulse Width High, tPWH 3 ns
--  This code runs at half the LCLK  rate (max 50 MHz)
--	Mod Record
--	===========
--	1)26/03/07		sampck_synth_cntrl created
--
-- (C) Copyright Alpha Data Parallel Systems Ltd. 1999-2003
--
--
-------------------------------------------------------------------------------------------------




--********************************************************************************************








--********************************************************************************************
--			Entity Definition
--
-- Notes:
--********************************************************************************************
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;



entity sampck_synth_cntrl is
generic
(
	END_CNT	:positive := 2
);
port
(
		reset		:in		std_logic;
		clock		:in		std_logic;
		wr_data		:in		std_logic_vector( 7 downto 0);
		rd_data		:out		std_logic_vector( 7 downto 0);
		rd_status	:out	std_logic;
		loc_func		 :in		std_logic;

		address		:in		std_logic_vector( 7 downto 0);

		rd_strobe		:in		std_logic;
		wr_strobe		:in		std_logic;
		busy		:out		std_logic;


		synth_status	:in	std_logic;		--ad9510 monitor output, fpga input
		synth_sclk	:out	std_logic;		--ad9510 serial clock,   fpga output
		synth_sdio	:out	std_logic;		--ad9510 data in, 		   fpga output
		synth_sdo	:in	std_logic;				--ad 9510 data out,		   fpga input
		synth_csb	:out	std_logic;			--ad9510 cs, active low, fpga output

		synth_func	:out	std_logic			--ad9510 multipurpose input
);
end sampck_synth_cntrl;
--********************************************************************************************













--********************************************************************************************
--	Architecture Definition
--
--********************************************************************************************
architecture synth of sampck_synth_cntrl is









--********************************************************************************************
--	Constant/Type Definitions
--********************************************************************************************

--State machine variable
  type state_type is
  (
  	idle_state,
  	csb_lo_state,
  	csb_done_state,
  	instr_cklo_state,
  	instr_ckhi_state,
  	wrdata_cklo_state,
  	wrdata_ckhi_state,
  	rddata_cklo_state,
  	rddata_ckhi_state
  );

constant	INSTR_LEN :positive := 23;
constant	DATA_LEN :positive := 7;

--********************************************************************************************


--********************************************************************************************
--	Component Definitions
--********************************************************************************************
constant USE_OLD : BOOLEAN := FALSE;

--********************************************************************************************







--********************************************************************************************
--	Signal Definitions
--
--********************************************************************************************

signal 	clock_sig :std_logic;
signal 	reset_sig :std_logic;


signal 	rd_strobe_sig  :std_logic;
signal 	wr_strobe_sig  :std_logic;

signal 	busy_sig  :std_logic;
signal 	bitcnt_sig  :std_logic_vector(7 downto 0);
signal 	ckdiv_cnt_sig  :std_logic_vector(7 downto 0);
signal 	state_adv_sig  :std_logic;


signal 	ser_ckop_sig   :std_logic;
signal 	ser_ckop_pl_sig   :std_logic;
signal 	ser_wrdat_sig  :std_logic;
signal 	ser_sdoin_sig  :std_logic;
signal 	ser_csbop_sig  :std_logic;
signal	synth_status_sig :std_logic;
signal	synth_func_sig   :std_logic;

signal 	ser_addr_sig   :std_logic_vector(7 downto 0);
signal 	wr_data_sig    :std_logic_vector(7 downto 0);

signal 	ser_instr_sig   :std_logic_vector(23 downto 0);
signal	ser_if_trig_sig :std_logic_vector(1 downto 0);

signal 	next_ser_ckop_sig   :std_logic;
signal 	next_ser_sdoin_sig  :std_logic;
signal 	next_ser_csbop_sig  :std_logic;


signal 	next_busy_sig  :std_logic;
signal 	next_bitcnt_sig  :std_logic_vector(7 downto 0);



signal 	ser_state  :state_type;
signal 	next_ser_state  :state_type;



signal	ser_trig_flag_sig	:std_logic;



signal	instr_bit_sig :std_logic;
signal	wr_bit_sig :std_logic;
signal	rd_bits_sig :std_logic_vector(7 downto 0);



signal 	ser_sdio_sig   :std_logic_vector(23 downto 0);
signal 	rd_shiften_sig   :std_logic;
--********************************************************************************************




--********************************************************************************************
--	Attributes
--********************************************************************************************


--********************************************************************************************



























--********************************************************************************************
--	Architectural Statements
--********************************************************************************************

begin
--==============================================================================================
-- Port mappings
--==============================================================================================
	clock_sig <= clock;
	reset_sig <= reset;
	busy <=busy_sig;
	ser_addr_sig <= address;
	wr_data_sig <=wr_data;
	rd_strobe_sig <= rd_strobe;
	wr_strobe_sig <= wr_strobe;

	rd_data <= rd_bits_sig;
	rd_status <= synth_status_sig ;


	synth_sclk    <= ser_ckop_sig;    --ad9510 serial clock   fpga output
	synth_sdio    <= ser_wrdat_sig;	  --ad9510 data in        fpga output
	ser_sdoin_sig <= synth_sdo;       --ad 9510 data out      fpga input
	synth_csb     <= ser_csbop_sig;   --ad9510 cs, active low fpga output

	synth_func_sig <=loc_func;
------------------------------------------------------------------------------------------------







--==============================================================================================
-- Latch the status bit output by the AD9510
-- and the mutli purpose func bit ( reset low, sync low or PWDN low depending on programmed
-- configuration)
--==============================================================================================
process (clock_sig ,reset_sig)
begin
	if reset_sig = '1' then
			synth_func	<=	'0';
	elsif rising_edge (clock_sig) then
			synth_func	<=	synth_func_sig;          -- fpga output
	end if;
end process;
--status sig is async to local clock and can be pulsed
synth_status_sig <=synth_status;
------------------------------------------------------------------------------------------------











--==============================================================================================
-- Divider for serial clock rate
--==============================================================================================
process (clock_sig ,reset_sig)
begin
	if reset_sig = '1' then
		ckdiv_cnt_sig <= (others => '0');
		state_adv_sig <= '0';
	elsif rising_edge (clock_sig) then
		if( ckdiv_cnt_sig= END_CNT) then
			ckdiv_cnt_sig <= (others => '0');
			state_adv_sig <= '1';
		else
			ckdiv_cnt_sig <=ckdiv_cnt_sig+1;
			state_adv_sig <= '0';
		end if;
	end if;
end process;
------------------------------------------------------------------------------------------------



--==============================================================================================
-- Generate the instruction word
-- MSB (D23) is determined by the sense ( 1= read, 0 = write)
-- Width bits (D22,D21) are always 0 for a single byte transfer
-- D20 to D16 are always 0
-- D7 to D0 = register address
--==============================================================================================
process (clock_sig ,reset_sig)
begin
	if reset_sig = '1' then
		ser_instr_sig <= (others => '0');
	elsif rising_edge (clock_sig) then
		ser_instr_sig(22 downto 16) <= (others => '0');
		ser_instr_sig(15 downto 8)<=ser_addr_sig;

		if wr_strobe_sig= '1' then
			ser_instr_sig(23) <='0';
			ser_instr_sig(7 downto 0)<=wr_data_sig;
		elsif rd_strobe_sig= '1' then
			ser_instr_sig(23) <='1';
			--read sdio is don't care but keep it tidy
			ser_instr_sig(7 downto 0)<=(others =>'0');
		end if;

	end if;
end process;
------------------------------------------------------------------------------------------------







--==============================================================================================
-- Trigger a read or write of the data from the rising edge of the strobe
-- when busy is clear
--==============================================================================================
process (clock_sig ,reset_sig)
begin
	if reset_sig = '1' then
		ser_if_trig_sig <= (others => '0');
		ser_trig_flag_sig <= '0';
	elsif rising_edge (clock_sig) then
		ser_if_trig_sig(0) <= rd_strobe_sig or wr_strobe_sig;
		ser_if_trig_sig(1) <= ser_if_trig_sig(0);

		if busy_sig= '1' then
			ser_trig_flag_sig <= '0';
		else
			if ser_if_trig_sig="01" then
				ser_trig_flag_sig <= '1';
			end if;
		end if;

	end if;
end process;
------------------------------------------------------------------------------------------------

















--==============================================================================================
-- State machine update
--==============================================================================================
process (clock_sig ,reset_sig)
begin
	if reset_sig = '1' then
		ser_ckop_sig <= '0';
		ser_ckop_pl_sig <= '0';
		ser_csbop_sig <= '1';	--defaults to high for no transfers active

		busy_sig <= '0';
		bitcnt_sig  <=(others => '0');

		ser_state <= idle_state;
	elsif rising_edge (clock_sig) then
		ser_ckop_sig <= next_ser_ckop_sig;
		ser_csbop_sig <= next_ser_csbop_sig;
		ser_ckop_pl_sig <=ser_ckop_sig;

		busy_sig <= next_busy_sig;

		if state_adv_sig='1' then
			bitcnt_sig  <=next_bitcnt_sig ;
			ser_state <=next_ser_state;
		end if;
	end if;
end process;
------------------------------------------------------------------------------------------------




--==============================================================================================
--Data goes out MSB first
--
--==============================================================================================
process (clock_sig ,reset_sig)
begin
	if reset_sig = '1' then
		ser_wrdat_sig <= '0';
		rd_bits_sig <= (others => '0');
		rd_shiften_sig <='0';
	elsif rising_edge (clock_sig) then
		--Latch incoming data following a rising edge of the ser clock o/p
		-- and correct bits are available in the frame
		if rd_shiften_sig ='1'  and  ser_ckop_pl_sig='0' and ser_ckop_sig ='1' then
			rd_bits_sig <= rd_bits_sig(6 downto 0)& ser_sdoin_sig;
		end if;

		case bitcnt_sig is
		when x"17" =>
			ser_wrdat_sig <= ser_instr_sig(0);
			rd_shiften_sig <=	ser_instr_sig(23);
		when x"16" =>
			ser_wrdat_sig <= ser_instr_sig(1);
			rd_shiften_sig <=	ser_instr_sig(23);
		when x"15" =>
			ser_wrdat_sig <= ser_instr_sig(2);
			rd_shiften_sig <=	ser_instr_sig(23);
		when x"14" =>
			ser_wrdat_sig <= ser_instr_sig(3);
			rd_shiften_sig <=	ser_instr_sig(23);
		when x"13" =>
			ser_wrdat_sig <= ser_instr_sig(4);
			rd_shiften_sig <=	ser_instr_sig(23);
		when x"12" =>
			ser_wrdat_sig <= ser_instr_sig(5);
			rd_shiften_sig <=	ser_instr_sig(23);
		when x"11" =>
			ser_wrdat_sig <= ser_instr_sig(6);
			rd_shiften_sig <=	ser_instr_sig(23);
		when x"10" =>
			ser_wrdat_sig <= ser_instr_sig(7);
			rd_shiften_sig <=	ser_instr_sig(23);
		when x"0F" =>
			ser_wrdat_sig <= ser_instr_sig(8);
			rd_shiften_sig <='0';
		when x"0E" =>
			ser_wrdat_sig <= ser_instr_sig(9);
			rd_shiften_sig <='0';
		when x"0D" =>
			ser_wrdat_sig <= ser_instr_sig(10);
			rd_shiften_sig <='0';
		when x"0C" =>
			ser_wrdat_sig <= ser_instr_sig(11);
			rd_shiften_sig <='0';
		when x"0B" =>
			ser_wrdat_sig <= ser_instr_sig(12);
			rd_shiften_sig <='0';
		when x"0A" =>
			ser_wrdat_sig <= ser_instr_sig(13);
			rd_shiften_sig <='0';
		when x"09" =>
			ser_wrdat_sig <= ser_instr_sig(14);
			rd_shiften_sig <='0';
		when x"08" =>
			ser_wrdat_sig <= ser_instr_sig(15);
			rd_shiften_sig <='0';
		when x"07" =>
			ser_wrdat_sig <= ser_instr_sig(16);
			rd_shiften_sig <='0';
		when x"06" =>
			ser_wrdat_sig <= ser_instr_sig(17);
			rd_shiften_sig <='0';
		when x"05" =>
			ser_wrdat_sig <= ser_instr_sig(18);
			rd_shiften_sig <='0';
		when x"04" =>
			ser_wrdat_sig <= ser_instr_sig(19);
			rd_shiften_sig <='0';
		when x"03" =>
			ser_wrdat_sig <= ser_instr_sig(20);
			rd_shiften_sig <='0';
		when x"02" =>
			ser_wrdat_sig <= ser_instr_sig(21);
			rd_shiften_sig <='0';
		when x"01" =>
			ser_wrdat_sig <= ser_instr_sig(22);
			rd_shiften_sig <='0';
		when x"00" =>
			ser_wrdat_sig <= ser_instr_sig(23);
			rd_shiften_sig <='0';
		when others =>
			ser_wrdat_sig <= ser_instr_sig(23);
			rd_shiften_sig <='0';
		end case;
	end if;
end process;
------------------------------------------------------------------------------------------------




--==============================================================================================
-- State machine muxing
--==============================================================================================
	statemux :
	process(ser_trig_flag_sig	)
	begin
		case ser_state is

		-----------------------------------------------
		-- wait for a rd/wr cycle to be triggered
		--
		--
		-----------------------------------------------
		when idle_state =>
      next_ser_ckop_sig<='0';
			next_ser_csbop_sig<='1';

			next_busy_sig <='0';
			next_bitcnt_sig <= ( others =>'0');

			if ser_trig_flag_sig='1' then
				next_ser_state <= csb_lo_state;
			else
				next_ser_state <= idle_state;
			end if;
		-----------------------------------------------



		-----------------------------------------------
		-- Set the csb bit low, clock low
		--
		--
		-----------------------------------------------
		when csb_lo_state =>
      next_ser_ckop_sig<='0';
			next_ser_csbop_sig<='0';

			next_busy_sig <='1';
			next_bitcnt_sig <= (others => '0');

			next_ser_state <= instr_cklo_state;
		-----------------------------------------------


		-----------------------------------------------
		-- Set the instruction bit, clock low
		-- write msb first
		--
		-----------------------------------------------
		when instr_cklo_state =>
      next_ser_ckop_sig<='0';
			next_ser_csbop_sig<='0';

			next_busy_sig <='1';
			next_bitcnt_sig <= bitcnt_sig;

			next_ser_state <= instr_ckhi_state;
		-----------------------------------------------


		-----------------------------------------------
		-- Set the clock line high, hold the instruction bit
		--
		--
		-----------------------------------------------
		when instr_ckhi_state =>
      next_ser_ckop_sig<='1';
			next_ser_csbop_sig<='0';

			next_busy_sig <='1';
			next_bitcnt_sig <= bitcnt_sig + '1';
			if bitcnt_sig = INSTR_LEN then
				next_ser_state <= csb_done_state;
			else
				next_ser_state <= instr_cklo_state;
			end if;
		-----------------------------------------------






		-----------------------------------------------
		-- Set the ck low for the last time, csb is low
		--
		--
		-----------------------------------------------
		when csb_done_state =>
      next_ser_ckop_sig<='0';
			next_ser_csbop_sig<='0';

			next_busy_sig <='1';
			next_bitcnt_sig <= (others => '0');

			next_ser_state <= idle_state;
		-----------------------------------------------





		-----------------------------------------------
		-- catch any other states
		-----------------------------------------------
		when others =>
      next_ser_ckop_sig<='0';
			next_ser_csbop_sig<='1';

			next_busy_sig <='0';
			next_bitcnt_sig <= ( others =>'0');

			next_ser_state <= idle_state;
		-----------------------------------------------
		end case;
	end process statemux;
------------------------------------------------------------------------------------------------




end; -- architecture  sampck_synth_cntrl
--********************************************************************************************


