-- =============================================================================================
--                             	                                                            **
-- =============================================================================================
--	Title: sampck_synth.vhd
--	Description:	VHDL  design file for control of AD9510 on adp_pupe
--	The 'cntrl' module handles the seria communications with the AD9510
--  The 'init' module forces the defualt setup to be loaded following a reset
--	Once intialisation has completed the host can use the read/write strobe ports and the data/address ports
--	to perform read/write cycles.
--  All reads and writes are 8 bit values.
--
--  A maximum of 90 locations exist so requires 7 address bits - port set to 8 bits wide and msb ignored
--	Mod Record
--	===========
--	1)26/03/07		sampck_synth created
--
-- (C) Copyright Alpha Data Parallel Systems Ltd. 1999-2003
--
--
-------------------------------------------------------------------------------------------------



--ToDo
--Fix 'gated clock' message - mux o/p bits in a separate process
--********************************************************************************************








--********************************************************************************************
--			Entity Definition
--
-- Notes:
--********************************************************************************************
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;



--Required for 'attribute' use
--library synplify;
--use synplify.attributes.all;


entity sampck_synth is
generic
(
	END_CNT	:positive := 2
);
port
(
		reset		:in		std_logic;
		clock		:in		std_logic;
		wr_data		:in		std_logic_vector( 7 downto 0);
		rd_data		:out		std_logic_vector( 7 downto 0);
		rd_status	:out	std_logic;
		loc_func		 :in		std_logic;

		address		:in		std_logic_vector( 7 downto 0);

		rd_strobe		:in		std_logic;
		wr_strobe		:in		std_logic;
		busy		:out		std_logic;
		busy_flag		:out		std_logic;
		init_busy		:out		std_logic;


		synth_status	:in	std_logic;		--ad9510 monitor output, fpga input
		synth_sclk	:out	std_logic;		--ad9510 serial clock,   fpga output
		synth_sdio	:out	std_logic;		--ad9510 data in, 		   fpga output
		synth_sdo	:in	std_logic;				--ad 9510 data out,		   fpga input
		synth_csb	:out	std_logic;			--ad9510 cs, active low, fpga output

		synth_func	:out	std_logic			--ad9510 multipurpose input
);
end sampck_synth;
--********************************************************************************************













--********************************************************************************************
--	Architecture Definition
--
--********************************************************************************************
architecture synth of sampck_synth is









--********************************************************************************************
--	Constant/Type Definitions
--********************************************************************************************

constant CNTMSB 	:positive := 7;
--********************************************************************************************


--********************************************************************************************
--	Component Definitions
--********************************************************************************************
component sampck_synth_cntrl is
generic
(
	END_CNT	:positive := 2
);
port
(
		reset		:in		std_logic;
		clock		:in		std_logic;
		wr_data		:in		std_logic_vector( 7 downto 0);
		rd_data		:out		std_logic_vector( 7 downto 0);
		rd_status	:out	std_logic;

		address		:in		std_logic_vector( 7 downto 0);
		loc_func 		:in		std_logic;
		rd_strobe		:in		std_logic;
		wr_strobe		:in		std_logic;
		busy		:out		std_logic;

		synth_status	:in	std_logic;		--ad9510 monitor output, fpga input
		synth_sclk	:out	std_logic;		--ad9510 serial clock,   fpga output
		synth_sdio	:out	std_logic;		--ad9510 data in, 		   fpga output
		synth_sdo	:in	std_logic;				--ad 9510 data out,		   fpga input
		synth_csb	:out	std_logic;			--ad9510 cs, active low, fpga output

		synth_func	:out	std_logic			--ad9510 multipurpose input
);
end component sampck_synth_cntrl;







component sampck_synth_init is
port
(
		reset			:in		std_logic;
		clock			:in		std_logic;
		trigger		:in		std_logic;

		init_done	:out		std_logic;
		wr_strobe	:out		std_logic;
		busy			:in		std_logic;


		address		:out		std_logic_vector( 7 downto 0);
		wr_data		:out		std_logic_vector( 7 downto 0)
);
end component sampck_synth_init;


--********************************************************************************************







--********************************************************************************************
--	Signal Definitions
--
--********************************************************************************************

signal 	clock_sig :std_logic;
signal 	reset_sig :std_logic;

signal	host_rdstatus_sig :std_logic;


signal 	host_wrdat_sig    :std_logic_vector(7 downto 0);
signal 	host_addr_sig    :std_logic_vector(7 downto 0);
signal 	host_wr_strobe_sig  :std_logic;
signal 	host_rd_strobe_sig  :std_logic;
signal 	host_func_sig  :std_logic;


signal 	synth_init_done_sig  :std_logic;
signal 	synth_init_strobe_sig  :std_logic;
signal 	synth_init_trig_sig  :std_logic;
signal 	synth_init_cnt_sig  :std_logic_vector(CNTMSB downto 0);
signal 	synth_init_data_sig    :std_logic_vector(7 downto 0);
signal 	synth_init_addr_sig    :std_logic_vector(7 downto 0);


signal 	sm_busy_sig  :std_logic;
signal 	sm_busy_flag_sig  :std_logic;
signal 	sm_busypl_sig  :std_logic;
signal 	mux_wrdat_sig    :std_logic_vector(7 downto 0);
signal 	mux_wraddr_sig    :std_logic_vector(7 downto 0);
signal 	host_rddat_sig    :std_logic_vector(7 downto 0);
signal 	mux_wr_strobe_sig  :std_logic;
signal 	mux_rd_strobe_sig  :std_logic;






signal	synth_sclk_sig	:std_logic;
signal	synth_sdio_sig	:std_logic;
signal	synth_sdo_sig	:std_logic;
signal 	synth_csb_sig  :std_logic;
signal	synth_status_sig :std_logic;
signal	synth_func_sig   :std_logic;





--********************************************************************************************





--********************************************************************************************
--	Attributes
--********************************************************************************************

--********************************************************************************************



























--********************************************************************************************
--	Architectural Statements
--********************************************************************************************

begin
--==============================================================================================
-- Port mappings
--==============================================================================================
	clock_sig <= clock;
	reset_sig <= reset;

	rd_data<=host_rddat_sig;
	rd_status <= host_rdstatus_sig;

	host_wrdat_sig <=wr_data;
	host_addr_sig <=address;
	host_rd_strobe_sig <= rd_strobe;
	host_wr_strobe_sig <= wr_strobe;
	host_func_sig <= loc_func;


	--Synth ports
	synth_status_sig<=synth_status;
	synth_sclk	  <=synth_sclk_sig;
	synth_sdio	  <= synth_sdio_sig;
	synth_sdo_sig <= synth_sdo;
	synth_csb	    <=synth_csb_sig;
	synth_func	  <=synth_func_sig;
------------------------------------------------------------------------------------------------






--==============================================================================================
--Latch the rising edge of the busy flag for host readback; any read or write strobe
--clears this flag
--==============================================================================================
process(reset_sig,clock_sig)
begin
	if reset_sig = '1' then
		init_busy <= '0';
		busy <= '0';
		busy_flag <= '0';

		sm_busypl_sig <= '0';
		sm_busy_flag_sig <= '0';
	elsif rising_edge(clock_sig) then
		busy        <= sm_busy_sig;
		busy_flag <= sm_busy_flag_sig;
		init_busy		<= not synth_init_done_sig;

		sm_busypl_sig <= sm_busy_sig;

		if host_rd_strobe_sig= '1' or host_rd_strobe_sig ='1' then
			sm_busy_flag_sig <= '0';
		else
			if sm_busypl_sig='0' and sm_busy_sig ='1' then
				sm_busy_flag_sig <= '1';
			end if;
		end if;

	end if;
end process;
------------------------------------------------------------------------------------------------







--==============================================================================================
--Counter to provide a delay then trigger ( rising edge) following a reset
--stops incrementing when MSB is set
--==============================================================================================
process(reset_sig,clock_sig)
begin
	if reset_sig = '1' then
		synth_init_cnt_sig <= ( others => '0');
		synth_init_trig_sig <= '0';
	elsif rising_edge(clock_sig) then
		synth_init_trig_sig <= synth_init_cnt_sig(CNTMSB);

		if synth_init_trig_sig ='0' then
			synth_init_cnt_sig <=synth_init_cnt_sig+ '1';
		end if;
	end if;
end process;
------------------------------------------------------------------------------------------------

--==============================================================================================
-- Mux the host and init signals
--==============================================================================================
process(reset_sig,clock_sig)
begin
	if reset_sig = '1' then
		mux_wrdat_sig  <= (others =>'0');
		mux_wraddr_sig <= (others =>'0');
		mux_rd_strobe_sig <= '0';
		mux_wr_strobe_sig <= '0';
	elsif rising_edge(clock_sig) then
		if synth_init_done_sig ='0' then
			mux_wrdat_sig  <= synth_init_data_sig;
			mux_wraddr_sig <= synth_init_addr_sig;
			mux_rd_strobe_sig <= '0';
			mux_wr_strobe_sig <= synth_init_strobe_sig;
		else
			mux_wrdat_sig  <= host_wrdat_sig;
			mux_wraddr_sig <= host_addr_sig;
			mux_rd_strobe_sig <= host_rd_strobe_sig;
			mux_wr_strobe_sig <= host_wr_strobe_sig;
		end if;
	end if;
end process;
------------------------------------------------------------------------------------------------





--==============================================================================================
-- Runs the initialisation sequence following a reset
--==============================================================================================
initsynth: sampck_synth_init
port map
(
		reset		=> reset_sig,
		clock		=> clock_sig,
		trigger		=>synth_init_cnt_sig(CNTMSB),
		wr_strobe	=>synth_init_strobe_sig,
		init_done		=> synth_init_done_sig,

		busy			=> sm_busy_sig,
		wr_data		=>synth_init_data_sig,
		address		=> synth_init_addr_sig
);


------------------------------------------------------------------------------------------------






--==============================================================================================
-- State machine to generate read and write cycles
--==============================================================================================
cksynth:sampck_synth_cntrl
generic map
(
	END_CNT	=>END_CNT
)
port map
(
		reset		=> reset_sig,
		clock		=> clock_sig,

		wr_data		=> mux_wrdat_sig,
		address		=> mux_wraddr_sig,
		rd_data		=> host_rddat_sig,

		rd_status	=> host_rdstatus_sig,
		busy		 => sm_busy_sig,
		loc_func     => host_func_sig,
		rd_strobe		=>mux_rd_strobe_sig,
		wr_strobe	  =>mux_wr_strobe_sig,

		synth_status	=>synth_status_sig,
		synth_sclk	=>synth_sclk_sig,
		synth_sdio	=> synth_sdio_sig,
		synth_sdo	 => synth_sdo_sig,
		synth_csb	=>synth_csb_sig,
		synth_func	=>synth_func_sig
);
--------------------------------------------------------------------------------------------------







end; -- architecture  sampck_synth
--********************************************************************************************


