--
-- pupe.vhd
--
-- Pick Up Processing Engine
-- Module for single set of 3 channels
-- Connects up the 4 main sub-modules in the system:
-- Timing         : Edge detection and test register control of Timing Signals
-- PLL/SPU        : Main Phase Lock Loop and Signal Processing
-- Data Logger    : Data Capture State Machines
-- Host Interface : Host Accessible Registers and Memories
--
-- Version 1.0 5/1/06 A.McCormick (Alpha Data)



library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_misc.all;

library work;
use work.common_interface.all;


entity pupe is
  
  generic (
    index : integer := 0);
  port (
    -- Host Abstract Interface Connection
    clk         : in  std_logic;
    rst         : in  std_logic;
    system_in   : in  std_logic_vector(system_width-1 downto 0);
    system_out  : out std_logic_vector(system_width-1 downto 0);
    debug       : in  std_Logic_vector(63 downto 0);
    -- Interrupts
    irq         : out std_logic_vector(7 downto 0);
    -- IO Output
    phase_table_msb_out : out std_logic;
    -- Timing Inputs
    cycle_start : in  std_logic;
    cycle_stop  : in  std_logic;
    cal_start   : in  std_logic;
    cal_stop    : in  std_logic;
    injection   : in  std_logic;
    hchange     : in  std_logic;
    fref        : in  std_logic;
    fref_test         : in  std_logic;
    ten_mhz_clk : in  std_logic;
    -- Cross connections to other pupes plls
    pll_mixed_sigma0_out : out std_logic_vector(13 downto 0);
    pll_mixed_sigma1_out : out std_logic_vector(13 downto 0);
    pll_mixed_sigma0a_in : in std_logic_vector(13 downto 0);
    pll_mixed_sigma1a_in : in std_logic_vector(13 downto 0);
    pll_mixed_sigma0b_in : in std_logic_vector(13 downto 0);
    pll_mixed_sigma1b_in : in std_logic_vector(13 downto 0);  
    -- ADC Inputs
    sigma       : in  std_logic_vector(13 downto 0);
    deltax      : in  std_logic_vector(13 downto 0);
    deltay      : in  std_logic_vector(13 downto 0));

end pupe;

architecture rtl of pupe is

  component timing
    port (
      clk                     : in  std_logic;
      rst                     : in  std_logic;
      cycle_start_in          : in  std_logic;
      cycle_stop_in           : in  std_logic;
      cal_start_in            : in  std_logic;
      cal_stop_in             : in  std_logic;
      injection_in            : in  std_logic;
      hchange_in              : in  std_logic;
      fref_in                 : in  std_logic;
      fref_test         : in  std_logic;
      ten_mhz_clk             : in  std_logic;
      test_timing_in          : in  std_logic_vector(31 downto 0);
      test_timing_wstb        : in  std_logic;
      cycle_start_out         : out std_logic;
      cycle_start_out_ten_mhz : out std_logic;
      cycle_stop_out          : out std_logic;
      cal_start_out           : out std_logic;
      cal_stop_out            : out std_logic;
      injection_out           : out std_logic;
      hchange_out             : out std_logic;
      fref_out                : out std_logic);
  end component;


  component pllspu
    port (
      clk                 : in  std_logic;
      rst                 : in  std_logic;
      ten_mhz_clk         : in  std_logic;
      -- Timing Inputs (already edge detected)
      cycle_start         : in  std_logic;
      cycle_start_ten_mhz : in  std_logic;
      cycle_stop          : in  std_logic;
      cal_start           : in  std_logic;
      cal_stop            : in  std_logic;
      injection           : in  std_logic;
      hchange             : in  std_logic;
      fref                : in  std_logic;
      frev_in             : in  std_logic;
      -- PLL Registers
      init                : in  std_logic;
      loop_ctrl           : in  std_logic;
      dds_freq_limit_en   : in  std_logic;
      blr_disable         : in  std_Logic;
      pll_frequency       : in  std_logic_vector(31 downto 0);
      pll_freqdelay       : in  std_logic_vector(31 downto 0);
      pll_phasedelay      : in  std_logic_vector(31 downto 0);
      pll_gain            : in  std_logic_vector(31 downto 0);
      dds_freq_min        : in  std_logic_vector(31 downto 0);
      dds_freq_max        : in  std_logic_vector(31 downto 0);
      use_neighbour_pll : in  std_logic_vector(3 downto 0);
      -- Cross connections to other pupes plls
      pll_mixed_sigma0_out : out std_logic_vector(13 downto 0);
      pll_mixed_sigma1_out : out std_logic_vector(13 downto 0);
      pll_mixed_sigma0a_in : in std_logic_vector(13 downto 0);
      pll_mixed_sigma1a_in : in std_logic_vector(13 downto 0);
      pll_mixed_sigma0b_in : in std_logic_vector(13 downto 0);
      pll_mixed_sigma1b_in : in std_logic_vector(13 downto 0); 
      -- ADC Inputs
      sigma               : in  std_logic_vector(13 downto 0);
      deltax              : in  std_logic_vector(13 downto 0);
      deltay              : in  std_logic_vector(13 downto 0);
      -- Switch Table Read Only Interface
      switch_table_addr   : out std_logic_vector(3 downto 0);
      switch_table_data   : in  std_logic_vector(31 downto 0);
      -- Phase Table Read Only Interface
      phase_table_addr    : out std_logic_vector(12 downto 0);
      phase_table_data    : in  std_logic_vector(7 downto 0);
      -- Outputs to Data Logging Module
      result_valid        : out std_logic;
      result_0            : out std_logic_vector (23 downto 0);
      result_1            : out std_logic_vector (23 downto 0);
      result_2            : out std_logic_vector (23 downto 0);
      acq                 : out std_logic;
      cycle               : out std_logic_vector(31 downto 0);
      ctime               : out std_logic_vector(31 downto 0);
      ms_pulse_out        : out std_logic;
      phase_table_sync  : out std_Logic;
      hchange_sync      : out std_logic;
      injection_sync      : out std_logic;
      cal_start_sync    : out std_logic;
      -- IO Output
      phase_table_msb_out : out std_logic;
      -- State Machine PCI Control
      cycle_init_val      : in  std_logic_vector(31 downto 0);
      cycle_val_set       : in  std_logic;
      switch_state_reg    : out std_logic_vector(3 downto 0);
      -- Diagnostics Output
      diagnostics         : out std_logic_vector(255 downto 0));
  end component;

  component data_logger
    port (
      clk               : in  std_logic;
      rst               : in  std_logic;
      ten_mhz_clk       : in  std_logic;
      -- Inputs from PLL
      ms_pulse          : in  std_logic;
      acq               : in  std_logic;
      cycle             : in  std_logic_vector(31 downto 0);
      ctime             : in  std_logic_vector(31 downto 0);
      sigma_avg         : in  std_logic_vector(15 downto 0);
      deltax_avg        : in  std_logic_vector(15 downto 0);
      deltay_avg        : in  std_logic_vector(15 downto 0);
      avg_valid         : in  std_logic;
      init              : in  std_logic;
      bunchmean_acc     : in  std_logic_vector(1 downto 0);
      phase_table_sync  : in  std_Logic;
      hchange_sync      : in std_logic;
      injection_sync      : in std_logic;
      cal_start_sync    : in std_logic;
      -- Timing Inputs (already edge detected)
      cycle_start_edge       : in  std_logic;
      cycle_stop_edge        : in  std_logic;
      cal_start_edge         : in  std_logic;
      cal_stop_edge          : in  std_logic;
      injection_edge         : in  std_logic;
      hchange_edge           : in  std_logic;
      -- Timing Inputs 
      cycle_start       : in  std_logic;
      cycle_start_ten_mhz       : in  std_logic;
      cycle_stop        : in  std_logic;
      cal_start         : in  std_logic;
      cal_stop          : in  std_logic;
      injection         : in  std_logic;
      hchange           : in  std_logic;
      fref              : in  std_logic;
      -- Diagnostics from PLL
      diagnostics       : in  std_logic_vector(255 downto 0);
      -- Host Control Registers
      diag_control      : in  std_logic_vector(31 downto 0);
      diag_control_set  : in  std_logic;
      diag_delay      : in  std_logic_vector(31 downto 0);
      diag_delay_set  : in  std_logic;
      diag_trigger      : in  std_logic_vector(31 downto 0);
      diag_status       : out   std_logic_vector(31 downto 0);
      diag_irq          : out std_Logic; 
      -- Write Signals to Memory
      sdram_addr        : out std_logic_vector(24 downto 0);
      sdram_data        : out std_logic_vector(63 downto 0);
      sdram_write       : out std_logic;
      timing_addr       : out std_logic_vector(11 downto 0);
      timing_data       : out std_logic_vector(63 downto 0);
      timing_write      : out std_logic;
      info_addr         : out std_logic_vector(5 downto 0);
      info_data         : out std_logic_vector(63 downto 0);
      info_write        : out std_logic;
      diagnostics_data  : out std_logic_vector(63 downto 0);
      diagnostics_addr  : out std_logic_vector(11 downto 0);
      diagnostics_write : out std_logic;
      bunchmean0_data   : out std_logic_vector(63 downto 0);
      bunchmean0_addr   : out std_logic_vector(11 downto 0);
      bunchmean0_write  : out std_logic;
      bunchmean1_data   : out std_logic_vector(63 downto 0);
      bunchmean1_addr   : out std_logic_vector(11 downto 0);
      bunchmean1_write  : out std_logic);
  end component;

  component host_interface
    generic (
      index : integer := 0);
    port (
      -- Host Abstract Interface
      clk               : in  std_logic;
      rst               : in  std_logic;
      system_in         : in  std_logic_vector(system_width-1 downto 0);
      system_out        : out std_logic_vector(system_width-1 downto 0);
      -- Connection to Timing, PLL and Data Logger
      control     : out std_Logic_vector(31 downto 0);
      status      : in  std_Logic_vector(31 downto 0); 
      test_timing       : out std_logic_vector(31 downto 0);
      test_timing_wstb  : out std_logic;
      cycle             : in  std_logic_vector(31 downto 0);
      cycle_init_val    : out std_logic_vector(31 downto 0);
      cycle_val_set     : out std_logic;
      ctime             : in  std_logic_vector(31 downto 0);
      time_tbladdr      : in  std_logic_vector(31 downto 0);
      pll_frequency     : out std_logic_vector(31 downto 0);
      pll_freqdelay     : out std_logic_vector(31 downto 0);
      pll_phasedelay    : out std_logic_vector(31 downto 0);
      pll_gain          : out std_logic_vector(31 downto 0);
      dds_freq_min      : out std_logic_vector(31 downto 0);
      dds_freq_max      : out std_logic_vector(31 downto 0);
      diag_control      : out  std_logic_vector(31 downto 0);
      diag_control_set  : out  std_logic;
      diag_delay        : out std_logic_vector(31 downto 0);
      diag_delay_set    : out  std_logic;
      diag_trigger      : out  std_logic_vector(31 downto 0);
      diag_status       : in   std_logic_vector(31 downto 0);
      sdram_addr        : in  std_logic_vector(24 downto 0);
      sdram_data        : in  std_logic_vector(63 downto 0);
      sdram_write       : in  std_logic;
      fifo_irq          : out std_Logic;
      timing_addr       : in  std_logic_vector(11 downto 0);
      timing_data       : in  std_logic_vector(63 downto 0);
      timing_write      : in  std_logic;
      info_addr         : in  std_logic_vector(5 downto 0);
      info_data         : in  std_logic_vector(63 downto 0);
      info_write        : in  std_logic;
      phase_table_addr  : in  std_logic_vector(12 downto 0);
      phase_table_data  : out std_logic_vector(7 downto 0);
      switch_table_addr : in  std_logic_vector(3 downto 0);
      switch_table_data : out std_logic_vector(31 downto 0);
      diagnostics_data  : in  std_logic_vector(63 downto 0);
      diagnostics_addr  : in  std_logic_vector(11 downto 0);
      diagnostics_write : in  std_logic;
      bunchmean0_data   : in  std_logic_vector(63 downto 0);
      bunchmean0_addr   : in  std_logic_vector(11 downto 0);
      bunchmean0_write  : in  std_logic;
      bunchmean1_data   : in  std_logic_vector(63 downto 0);
      bunchmean1_addr   : in  std_logic_vector(11 downto 0);
      bunchmean1_write  : in  std_logic
      );
  end component;

  signal test_timing      : std_logic_vector(31 downto 0);
  signal control      : std_logic_vector(31 downto 0);
  signal status      : std_logic_vector(31 downto 0);
  signal time_tbladdr      : std_logic_vector(31 downto 0);
  signal test_timing_wstb : std_logic;

  signal cycle_start_edge, cycle_stop_edge, cal_start_edge, cal_stop_edge, injection_edge, hchange_edge, fref_edge : std_logic;

  signal cycle_start_edge_ten_mhz : std_logic;

  signal init : std_logic;
  signal loop_ctrl : std_logic;
  signal dds_freq_limit_en : std_logic;
  signal blr_disable : std_logic;

  signal pll_frequency, pll_freqdelay, pll_phasedelay, pll_gain, dds_freq_min, dds_freq_max : std_logic_vector(31 downto 0);

  signal switch_table_addr : std_logic_vector(3 downto 0);
  signal switch_table_data : std_logic_vector(31 downto 0);
  signal phase_table_addr  : std_logic_vector(12 downto 0);
  signal phase_table_data  : std_logic_vector(7 downto 0);

  signal result_valid                 : std_logic;
  signal result_0, result_1, result_2 : std_logic_vector (23 downto 0);
  signal acq                          : std_logic;
  signal fref_in : std_logic;
  signal cycle                        : std_logic_vector(31 downto 0);
  signal ctime                        : std_logic_vector(31 downto 0);
  signal ms_pulse_out                 : std_logic;
  signal hchange_sync : std_logic;
  signal injection_sync : std_logic;
  signal cal_start_sync : std_logic;
  
  signal cycle_init_val   : std_logic_vector(31 downto 0);
  signal cycle_val_set    : std_logic;
  signal switch_state_reg : std_logic_vector(3 downto 0);
  signal diagnostics,diagnostics1      : std_logic_vector(255 downto 0);
  signal diag_control      : std_logic_vector(31 downto 0);
  signal        diag_control_set  :  std_logic;
  signal diag_delay      : std_logic_vector(31 downto 0);
  signal  diag_delay_set  :  std_logic;
  signal  diag_trigger      : std_logic_vector(31 downto 0);
  signal  diag_status       : std_logic_vector(31 downto 0);

  signal sdram_addr        : std_logic_vector(24 downto 0);
  signal sdram_data        : std_logic_vector(63 downto 0);
  signal sdram_write       : std_logic;
  signal timing_addr       : std_logic_vector(11 downto 0);
  signal timing_data       : std_logic_vector(63 downto 0);
  signal timing_write      : std_logic;
  signal info_addr         : std_logic_vector(5 downto 0);
  signal info_data         : std_logic_vector(63 downto 0);
  signal info_write        : std_logic;
  signal diagnostics_data  : std_logic_vector(63 downto 0);
  signal diagnostics_addr  : std_logic_vector(11 downto 0);
  signal diagnostics_write : std_logic;
  signal bunchmean0_data   : std_logic_vector(63 downto 0);
  signal bunchmean0_addr   : std_logic_vector(11 downto 0);
  signal bunchmean0_write  : std_logic;
  signal bunchmean1_data   : std_logic_vector(63 downto 0);
  signal bunchmean1_addr   : std_logic_vector(11 downto 0);
  signal bunchmean1_write  : std_logic;

  signal bunchmean_acc : std_logic_vector(1 downto 0);
  signal bunchmean_acc_reg : std_logic_vector(1 downto 0);
  
  signal fref_reg : std_logic;

  signal diag_irq : std_logic;
  signal fifo_irq : std_logic;
  signal phase_table_sync : std_Logic;

  signal irq0,irq1,irq2,irq3,irq4,irq5 : std_logic_vector(7 downto 0);
  
begin  -- rtl

  t0 : timing
    port map(
      clk                     => clk,
      rst                     => rst,
      cycle_start_in          => cycle_start,
      cycle_stop_in           => cycle_stop,
      cal_start_in            => cal_start,
      cal_stop_in             => cal_stop,
      injection_in            => injection,
      hchange_in              => hchange,
      fref_in                 => fref,
      fref_test               => fref_test,
      ten_mhz_clk             => ten_mhz_clk,
      test_timing_in          => test_timing,
      test_timing_wstb        => test_timing_wstb,
      cycle_start_out         => cycle_start_edge,
      cycle_start_out_ten_mhz => cycle_start_edge_ten_mhz,
      cycle_stop_out          => cycle_stop_edge,
      cal_start_out           => cal_start_edge,
      cal_stop_out            => cal_stop_edge,
      injection_out           => injection_edge,
      hchange_out             => hchange_edge,
      fref_out                => fref_reg);


  
  ps0 : pllspu
    port map(
      clk                 => clk,
      rst                 => rst,
      ten_mhz_clk         => ten_mhz_clk,
      -- Timing Inputs (already edge detected)
      cycle_start         => cycle_start_edge,
      cycle_start_ten_mhz => cycle_start_edge_ten_mhz,
      cycle_stop          => cycle_stop_edge,
      cal_start           => cal_start_edge,
      cal_stop            => cal_stop_edge,
      injection           => injection_edge,
      hchange             => hchange_edge,
      fref                => fref,
      frev_in             => fref_reg,
      -- PLL Registers
      init                => init,
      loop_ctrl           => loop_ctrl,
      dds_freq_limit_en   => dds_freq_limit_en,
      blr_disable         => blr_disable,
      pll_frequency       => pll_frequency,
      pll_freqdelay       => pll_freqdelay,
      pll_phasedelay      => pll_phasedelay,
      pll_gain            => pll_gain,
      dds_freq_min        => dds_freq_min,
      dds_freq_max        => dds_freq_max,
      use_neighbour_pll   => control(11 downto 8),
        -- Cross connections to other pupes plls
      pll_mixed_sigma0_out  => pll_mixed_sigma0_out,  
      pll_mixed_sigma1_out  => pll_mixed_sigma1_out,  
      pll_mixed_sigma0a_in  => pll_mixed_sigma0a_in,  
      pll_mixed_sigma1a_in  => pll_mixed_sigma1a_in,  
      pll_mixed_sigma0b_in  => pll_mixed_sigma0b_in,  
      pll_mixed_sigma1b_in  => pll_mixed_sigma1b_in,   
      -- ADC Inputs
      sigma               => sigma,
      deltax              => deltax,
      deltay              => deltay,
      -- Switch Table Read Only Interface
      switch_table_addr   => switch_table_addr,
      switch_table_data   => switch_table_data,
      -- Phase Table Read Only Interface
      phase_table_addr    => phase_table_addr,
      phase_table_data    => phase_table_data,
      -- Outputs to Data Logging Module
      result_valid        => result_valid,
      result_0            => result_0,
      result_1            => result_1,
      result_2            => result_2,
      acq                 => acq,
      cycle               => cycle,
      ctime               => ctime,
      ms_pulse_out        => ms_pulse_out,
      phase_table_sync    => phase_table_sync,
      hchange_sync        => hchange_sync,
      injection_sync      => injection_sync,
      cal_start_sync      => cal_start_sync,
      phase_table_msb_out => phase_table_msb_out,
      -- State Machine PCI Control
      cycle_init_val      => cycle_init_val,
      cycle_val_set       => cycle_val_set,
      switch_state_reg    => switch_state_reg,
      -- Diagnostics Output
      diagnostics         => diagnostics);


  diagnostics1 <= diagnostics;
  --diagnostics1 <= diagnostics(255 downto 64) & debug;  -- Feed local bus debug
-- data into daignostics capture logic
  
  dl0 : data_logger
    port map (
      clk                 => clk,
      rst                 => rst,
      ten_mhz_clk         => ten_mhz_clk,
      -- Inputs from PLL
      ms_pulse            => ms_pulse_out,
      acq                 => acq,
      cycle               => cycle,
      ctime               => ctime,
      sigma_avg           => result_0(18 downto 3),
      deltax_avg          => result_1(18 downto 3),
      deltay_avg          => result_2(18 downto 3),
      avg_valid           => result_valid,
      init                => init,
      bunchmean_acc       => bunchmean_acc,
      phase_table_sync    => phase_table_sync,
      hchange_sync        => hchange_sync,
      injection_sync      => injection_sync,
      cal_start_sync      => cal_start_sync,
      -- Timing Inputs (already edge detected)
      cycle_start_edge         => cycle_start_edge,
      cycle_stop_edge          => cycle_stop_edge,
      cal_start_edge           => cal_start_edge,
      cal_stop_edge            => cal_stop_edge,
      injection_edge           => injection_edge,
      hchange_edge             => hchange_edge,
      cycle_start         => cycle_start,
      cycle_start_ten_mhz => cycle_start_edge_ten_mhz,
      cycle_stop          => cycle_stop,
      cal_start           => cal_start,
      cal_stop            => cal_stop,
      injection           => injection,
      hchange             => hchange,
      fref                => fref,
      -- Diagnostics from PLL
      diagnostics         => diagnostics1,
      -- Host Control Registers
      diag_control     => diag_control,
      diag_control_set => diag_control_set,
      diag_delay     => diag_delay,
      diag_delay_set => diag_delay_set,
      diag_trigger     => diag_trigger,
      diag_status      => diag_status,
      diag_irq         => diag_irq,
      -- Write Signals to Memory
      sdram_addr          => sdram_addr,
      sdram_data          => sdram_data,
      sdram_write         => sdram_write,
      timing_addr         => timing_addr,
      timing_data         => timing_data,
      timing_write        => timing_write,
      info_addr           => info_addr,
      info_data           => info_data,
      info_write          => info_write,
      diagnostics_data    => diagnostics_data,
      diagnostics_addr    => diagnostics_addr,
      diagnostics_write   => diagnostics_write,
      bunchmean0_data     => bunchmean0_data,
      bunchmean0_addr     => bunchmean0_addr,
      bunchmean0_write    => bunchmean0_write,
      bunchmean1_data     => bunchmean1_data,
      bunchmean1_addr     => bunchmean1_addr,
      bunchmean1_write    => bunchmean1_write);

  hi0: host_interface
    generic map(
      index => index)
    port map(
      -- Host Abstract Interface
      clk               => clk,
      rst               => rst,
      system_in         => system_in,
      system_out        => system_out,
      -- Connection to Timing, PLL and Data Logger
      control           => control,
      status            => status,
      test_timing       => test_timing,
      test_timing_wstb  => test_timing_wstb,
      cycle             => cycle,
      cycle_init_val    => cycle_init_val,
      cycle_val_set     => cycle_val_set,
      ctime             => ctime,
      time_tbladdr      => time_tbladdr,
      pll_frequency     => pll_frequency,
      pll_freqdelay       => pll_freqdelay,
      pll_phasedelay      => pll_phasedelay,
      pll_gain            => pll_gain,
      dds_freq_min        => dds_freq_min,
      dds_freq_max        => dds_freq_max,
      diag_control     => diag_control,
      diag_control_set => diag_control_set,
      diag_delay     => diag_delay,
      diag_delay_set => diag_delay_set,
      diag_trigger     => diag_trigger,
      diag_status      => diag_status,
      sdram_addr          => sdram_addr,
      sdram_data          => sdram_data,
      sdram_write         => sdram_write,
      fifo_irq            => fifo_irq,
      timing_addr         => timing_addr,
      timing_data         => timing_data,
      timing_write        => timing_write,
      info_addr           => info_addr,
      info_data           => info_data,
      info_write          => info_write,
      phase_table_addr => phase_table_addr,
      phase_table_data => phase_table_data,
      switch_table_addr => switch_table_addr,
      switch_table_data => switch_table_data,
      diagnostics_data    => diagnostics_data,
      diagnostics_addr    => diagnostics_addr,
      diagnostics_write   => diagnostics_write,
      bunchmean0_data     => bunchmean0_data,
      bunchmean0_addr     => bunchmean0_addr,
      bunchmean0_write    => bunchmean0_write,
      bunchmean1_data     => bunchmean1_data,
      bunchmean1_addr     => bunchmean1_addr,
      bunchmean1_write    => bunchmean1_write);
  

  -- Connect Up miscellaneous registers
  init <= control(0);
  loop_ctrl <= control(1);
  dds_freq_limit_en <= control(2);
  blr_disable <= control(3);
  status(7 downto 4) <= switch_state_reg;
  status(3 downto 0) <= control(3 downto 0);
  bunchmean_acc <= phase_table_data(7 downto 6) and not bunchmean_acc_reg;
  status(11 downto 8) <= control(11 downto 8);
  
  process (clk,rst)
  begin  -- process
    if rst = '1' then               -- asynchronous reset
      time_tbladdr <= (others => '0');
      bunchmean_acc_reg <= "00";
    elsif clk'event and clk = '1' then  -- rising clock edge
      if timing_write = '1' then
        time_tbladdr <= EXT(timing_addr,32);
      end if;
      bunchmean_acc_reg <= phase_table_data(7 downto 6);      
    end if;
  end process;

  irq0(0) <= cycle_start_edge;
  irq0(1) <= cycle_stop_edge;
  irq0(2) <= '1' when switch_table_addr = "1110" else '0';  -- Error State
  irq0(3) <= diag_irq;
  irq0(4) <= fifo_irq;
  process (clk)
  begin  -- process
    if clk'event and clk = '1' then  -- rising clock edge
      irq0(7 downto 5) <= switch_table_data(7 downto 5);  -- User defined interrupts
      irq1 <= irq0;
      irq2 <= irq1;
      irq3 <= irq2;
      irq4 <= irq3;
      irq5 <= irq4;
      -- Hold IRQ pulse for 6 clock cycles.
      irq <= irq0 or irq1 or irq2 or irq3 or irq4 or irq5;
    end if;
  end process;
  
  
end rtl;
