/*******************************************************************************
 *	TmsLib.h	TMS Client library functions
 *			T.Barnaby,	BEAM Ltd,	2007-03-14
 *				updated by	D.Korchagin,	CERN AB-BI-SW,	2007-08-31
 *******************************************************************************
 */

#ifndef TmsLib_H
#define TmsLib_H		1

#include <TmsD.h>
#include <TmsC.h>

namespace Tms {

const unsigned int	tmsNumPickups		= 40;		///< The default number of pick ups
const unsigned int	tmsPhaseTableSize	= 512;		///< The size of the Phase Table

/// The timing signal bits
enum TimingSig		{
			TimingSigClock		= 0x01,	///< 10MHz System Clock
			TimingSigCycleStart	= 0x02,	///< CYCLE_START event
			TimingSigCycleStop	= 0x04,	///< CYCLE_STOP event
			TimingSigCalStart	= 0x08,	///< CAL_START event
			TimingSigCalStop	= 0x10,	///< CAL_STOP event
			TimingSigInjection	= 0x20,	///< INJECTION event
			TimingSigHChange	= 0x40,	///< HCHANGE event
			TimingSigFRef		= 0x80	///< FREF signal
			};

/// The Diagnostics Capture Clock settings
enum CaptureClock	{
			ClkAdcDiv_1	= 0x00,		///< ADC Clock
			ClkAdcDiv_2	= 0x01,		///< ADC Clock divided by 2
			ClkAdcDiv_5	= 0x02,		///< ADC Clock divided by 5
			ClkAdcDiv_10	= 0x03,		///< ADC Clock divided by 10
			ClkAdcDiv_20	= 0x04,		///< ADC Clock divided by 20
			ClkAdcDiv_50	= 0x05,		///< ADC Clock divided by 50
			ClkAdcDiv_100	= 0x06,		///< ADC Clock divided by 100
			ClkAdcDiv_200	= 0x07,		///< ADC Clock divided by 200
			ClkAdcDiv_500	= 0x08,		///< ADC Clock divided by 500
			ClkAdcDiv_1000	= 0x09,		///< ADC Clock divided by 1000
			ClkAdcDiv_2000	= 0x0A,		///< ADC Clock divided by 2000
			ClkAdcDiv_5000	= 0x0B,		///< ADC Clock divided by 5000
			ClkAdcDiv_10000	= 0x0C,		///< ADC Clock divided by 10000
			ClkAdcDiv_20000	= 0x0D,		///< ADC Clock divided by 20000
			ClkAdcDiv_50000	= 0x0E,		///< ADC Clock divided by 50000
			ClkAdcDiv_100000= 0x0F,		///< ADC Clock divided by 100000
			ClkMs		= 0x10,		///< Millisecond Clock
			ClkFref		= 0x11		///< FREF
};

/// The Tms State entry
#ifndef __Lynx__
union TmsState {
	struct {
		unsigned int	aquireData:1;
		unsigned int	pllReference1:1;
		unsigned int	pllReference2:1;
		unsigned int	pllFeedbackSelect:1;
		unsigned int	pllLO1FromAddress:1;
		unsigned int	pllLO2FromAddress:1;
		unsigned int	spare0:2;
		unsigned int	cycleStop:4;
		unsigned int	calStop:4;
		unsigned int	calStart:4;
		unsigned int	injection:4;
		unsigned int	hchange:4;
		unsigned int	delay:4;
	};
	unsigned int		value;
};
#else
struct TmsState {
	unsigned int	delay:4;
	unsigned int	hchange:4;
	unsigned int	injection:4;
	unsigned int	calStart:4;
	unsigned int	calStop:4;
	unsigned int	cycleStop:4;
	unsigned int	spare0:2;
	unsigned int	pllLO2FromAddress:1;
	unsigned int	pllLO1FromAddress:1;
	unsigned int	pllFeedbackSelect:1;
	unsigned int	pllReference2:1;
	unsigned int	pllReference1:1;
	unsigned int	aquireData:1;
	unsigned int&	value() { return (unsigned int &)*this; }
};
#endif

/// The Tms Phase Table Entry
#ifndef __Lynx__
union TmsPhase {
	struct {
		unsigned int	lo1:1;
		unsigned int	blr:1;
		unsigned int	gate:1;
		unsigned int	lo2:1;
		unsigned int	spare:2;
		unsigned int	meanFilter1:1;
		unsigned int	meanFilter2:1;
	};
	unsigned char		value;
};
#else
struct TmsPhase {
	unsigned int	meanFilter2:1;
	unsigned int	meanFilter1:1;
	unsigned int	spare:2;
	unsigned int	lo2:1;
	unsigned int	gate:1;
	unsigned int	blr:1;
	unsigned int	lo1:1;
	unsigned char&	value() { return (unsigned char &)*this; }
};
#endif

/// Internal CycleParameter management class
class CycleParamDb {
public:
				CycleParamDb(BString baseDir = ".");

	BError			getCycleTypes(BList<BString>& typeList);			///< Get the list of CycleParameter types in the directory
	BError			getFileNames(BList<BString>& fileList);				///< Get a list of all of the CycleParameter file names
	
	BError			getCycleParams(BString fileName, Tms::CycleParam& param);	///< Get the CycleParameter from the given file name
	BError			setCycleParams(Tms::CycleParam param);				///< Set the CycleParameters. Writes to the appropriate file name
	BError			deleteCycleParams(BString cycleType, UInt32 puChannel);		///< Deletes a CycleParameter definition file.

	BError			readCycleParams(BString fileName, Tms::CycleParam& param);	///< Reads a set of CycleParameters from a file.
	BError			writeCycleParams(BString fileName, Tms::CycleParam param);	///< Writes a set of CycleParameters to a file.
private:
	BString			obaseDir;
};

}
#endif
