/*******************************************************************************
 *	TapeSigGen.cc	TapeDigitiser signal generator utility
 *			T.Barnaby,	BEAM Ltd,	2006-09-12
 *******************************************************************************
 */

#include <stdio.h>
#include <syslog.h>
#include <unistd.h>

#include <SigGen.h>
#include <Tests.h>


void usage(){
	fprintf(stderr, "tmssiggen: Version: %s\n", VERSION);
	fprintf(stderr, "  Usage: Generates a Tms System test signal on 8 channels of the signal generator card or\n");
	fprintf(stderr, "    to a file.\n");
	fprintf(stderr, "  -t <type>     - Specifies the signal type (sine, beam1 etc)\n");
	fprintf(stderr, "  -T            - List all test types supported\n");
	fprintf(stderr, "  -l <n>        - Signal level (1.0 full)\n");
	fprintf(stderr, "  -s <freq>     - Sample rate (Default 150000000)\n");
	fprintf(stderr, "  -f <freq>     - FREF frequency (Default 437KHz)\n");
	fprintf(stderr, "  -p <period>   - TMS period to output, defaults to complete TMS cycle\n");
	fprintf(stderr, "  File options\n");
	fprintf(stderr, "  -o <fileName> - Outputs data to an ASCII file rather than signal generator card\n");
	fprintf(stderr, "  -b <fileName> - Outputs data to a TMS TestData binary file rather than signal generator card\n");
	fprintf(stderr, "  -c <num>      - Channel number to output, default is all\n");
	fprintf(stderr, "  -i 1          - Set integer output mode (14 bits)\n");
}

int main(int argc, char** argv){
	BError			err;
	int			a;
	BString			type = "";
	int			listTypes = 0;
	Test*			test = 0;
	TestParams		params;
	BString			s;
	BList<Test*>&		tests = testsInit();
	BIter			i;
		
	params.setValue("sampleRate", "150000000");
	params.setValue("amplitude", "1.0");
	params.setValue("fref", "437000");
	params.setValue("cycleStartPhase", "45");
	
	for(a = 1; a < argc; a++){
		if(argv[a][0] == '-'){
			switch(argv[a][1]){
			case 't':	type = argv[++a];				break;
			case 'T':	listTypes = 1;					break;
			case 's':	params.setValue("sampleRate",  argv[++a]);	break;
			case 'l':	params.setValue("amplitude",  argv[++a]);	break;
			case 'f':	params.setValue("fref", argv[++a]);		break;
			case 'o':	params.setValue("fileName", argv[++a]);
					params.setValue("fileType", "txt");
											break;
			case 'b':	params.setValue("fileName", argv[++a]);
					params.setValue("fileType", "psd");
											break;
			case 'c':	params.setValue("channel", argv[++a]);		break;
			case 'i':	params.setValue("integer", argv[++a]);		break;
			case 'p':	params.setValue("period", argv[++a]);		break;
			default:	usage();					return 1;
			}
		}
		else {
		}
	}
	
	if(listTypes){
		for(tests.start(i); !tests.isEnd(i); tests.next(i)){
			printf("%s: %s\n", tests[i]->name().retStr(), tests[i]->info().retStr());
		}
		return 0;
	}

	if(type == ""){
		usage();
		return 1;
	}
	
	for(tests.start(i); !tests.isEnd(i); tests.next(i)){
		if(type.toLower() == tests[i]->name().toLower()){
			test = tests[i];
			break;
		}
	}
	
	if(test == 0){
		fprintf(stderr, "Signal type: %s not implemented\n", type.retStr());
		return 1;
	}

	printf("Generating signal: %s\n", test->info().retStr());	
	if(err = test->init(params)){
		fprintf(stderr, "Test Init Error: %s\n", err.getString().retStr());
		return 1;
	}
	
	if(err = test->run()){
		fprintf(stderr, "Test Run Error: %s\n", err.getString().retStr());
		return 1;
	}

	return 0;
}
