--
--
-- Wrapper for HPC Common Interface

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_misc.all;


library work;
use work.common_interface.all;
use work.adp_pupe_pkg.all;

library UNISIM;
use UNISIM.vcomponents.all;



entity wrapper is
  port(
    lclk       : in    std_logic;
    clk200_p   : in    std_logic;
    clk200_n   : in    std_logic;
    lreseto_l  : in    std_logic;
    lwrite     : in    std_logic;
    lads_l     : in    std_logic;
    lblast_l   : in    std_logic;
    lbterm_l   : out   std_logic;
    lad        : inout std_logic_vector(63 downto 0);
    lreadyi_l  : out   std_logic;
    lbe_l      : in    std_logic_vector(7 downto 0);
    fholda     : in    std_logic;
    finti_l    : out   std_logic;
    DDR0_RAS_N : out   std_logic;
    DDR0_CAS_N : out   std_logic;
    DDR0_WE_N  : out   std_logic;
    DDR0_ODT   : out   std_logic;
    DDR0_CKE   : out   std_logic;
    DDR0_CS_N  : out   std_logic;
    DDR0_DQ    : inout std_logic_vector(31 downto 0);
    DDR0_DQS   : inout std_logic_vector(3 downto 0);
    DDR0_DQS_N : inout std_logic_vector(3 downto 0);
    DDR0_DM    : out   std_logic_vector(3 downto 0);
    DDR0_CK    : out   std_logic_vector(1 downto 0);
    DDR0_CK_N  : out   std_logic_vector(1 downto 0);
    DDR0_BA    : out   std_logic_vector(2 downto 0);
    DDR0_A     : out   std_logic_vector(12 downto 0);
    DDR1_RAS_N : out   std_logic;
    DDR1_CAS_N : out   std_logic;
    DDR1_WE_N  : out   std_logic;
    DDR1_ODT   : out   std_logic;
    DDR1_CKE   : out   std_logic;
    DDR1_CS_N  : out   std_logic;
    DDR1_DQ    : inout std_logic_vector(31 downto 0);
    DDR1_DQS   : inout std_logic_vector(3 downto 0);
    DDR1_DQS_N : inout std_logic_vector(3 downto 0);
    DDR1_DM    : out   std_logic_vector(3 downto 0);
    DDR1_CK    : out   std_logic_vector(1 downto 0);
    DDR1_CK_N  : out   std_logic_vector(1 downto 0);
    DDR1_BA    : out   std_logic_vector(2 downto 0);
    DDR1_A     : out   std_logic_vector(12 downto 0);
    DDR2_RAS_N : out   std_logic;
    DDR2_CAS_N : out   std_logic;
    DDR2_WE_N  : out   std_logic;
    DDR2_ODT   : out   std_logic;
    DDR2_CKE   : out   std_logic;
    DDR2_CS_N  : out   std_logic;
    DDR2_DQ    : inout std_logic_vector(31 downto 0);
    DDR2_DQS   : inout std_logic_vector(3 downto 0);
    DDR2_DQS_N : inout std_logic_vector(3 downto 0);
    DDR2_DM    : out   std_logic_vector(3 downto 0);
    DDR2_CK    : out   std_logic_vector(1 downto 0);
    DDR2_CK_N  : out   std_logic_vector(1 downto 0);
    DDR2_BA    : out   std_logic_vector(2 downto 0);
    DDR2_A     : out   std_logic_vector(12 downto 0);
    DDR3_RAS_N : out   std_logic;
    DDR3_CAS_N : out   std_logic;
    DDR3_WE_N  : out   std_logic;
    DDR3_ODT   : out   std_logic;
    DDR3_CKE   : out   std_logic;
    DDR3_CS_N  : out   std_logic;
    DDR3_DQ    : inout std_logic_vector(31 downto 0);
    DDR3_DQS   : inout std_logic_vector(3 downto 0);
    DDR3_DQS_N : inout std_logic_vector(3 downto 0);
    DDR3_DM    : out   std_logic_vector(3 downto 0);
    DDR3_CK    : out   std_logic_vector(1 downto 0);
    DDR3_CK_N  : out   std_logic_vector(1 downto 0);
    DDR3_BA    : out   std_logic_vector(2 downto 0);
    DDR3_A     : out   std_logic_vector(12 downto 0);
    -- MGTs
--    PCIE_RX_P  : in    std_logic_vector(7 downto 0);
--    PCIE_RX_N  : in    std_logic_vector(7 downto 0);
--    PCIE_TX_P  : out   std_logic_vector(7 downto 0);
--    PCIE_TX_N  : out   std_logic_vector(7 downto 0);
--    REAR_RX_P  : in    std_logic_vector(3 downto 0);
--    REAR_RX_N  : in    std_logic_vector(3 downto 0);
--    REAR_TX_P  : out   std_logic_vector(3 downto 0);
--    REAR_TX_N  : out   std_logic_vector(3 downto 0);
--    FRONT_RX_P : in    std_logic_vector(3 downto 0);
--    FRONT_RX_N : in    std_logic_vector(3 downto 0);
--    FRONT_TX_P : out   std_logic_vector(3 downto 0);
--    FRONT_TX_N : out   std_logic_vector(3 downto 0);


    -------------------------------------
    --Front panel debug I/O
    -------------------------------------
    debugio_dirn	:out	std_logic_vector(2 downto 0);
    debugio     	:inout	std_logic_vector(2 downto 0);
    -------------------------------------



    -------------------------------------
    --Timing bus I/O
    -------------------------------------
    tbus_bplane	:inout std_logic_vector(10 downto 1);
    tbus_header	:inout std_logic_vector(10 downto 1);
    --always inputs
    tbus_master	:in std_logic_vector(10 downto 1);
    -------------------------------------




    -------------------------------------
    -- Misc
    -------------------------------------
    red_led		:out	std_logic_vector( 1 downto 0);
    yellow_led		:out	std_logic_vector( 1 downto 0);
    -------------------------------------

    -------------------------------------
    --FPGA clocks
    -------------------------------------
    --Copy of external Synth 10Mhz refck via buffer
    fpga_10mhz_refckin_p			:in		std_logic;
    fpga_10mhz_refckin_n			:in		std_logic;

    -------------------------------------
    --Synthesiser Control
    -------------------------------------
    synth_csb	:out	std_logic;
    synth_func	:out	std_logic;
    synth_sclk	:out	std_logic;
    synth_sdio	:inout	std_logic;
    synth_sdo	:in	std_logic;
    synth_status	:in	std_logic;
    vcxo_enable	:out	std_logic;
    -------------------------------------

    -------------------------------------
    -- Sample clock from synthesiser chain
    --
    -------------------------------------
    sampck_p			:in		std_logic;
    sampck_n			:in		std_logic;
    -------------------------------------


    -------------------------------------
    -- ADC Channels
    -- Physical ordering determined by UCF
    -------------------------------------
    adc_data0		:in 	std_logic_vector(13 downto 0);
    adc_data1		:in 	std_logic_vector(13 downto 0);
    adc_data2		:in 	std_logic_vector(13 downto 0);
    adc_data3		:in 	std_logic_vector(13 downto 0);
    adc_data4		:in 	std_logic_vector(13 downto 0);
    adc_data5		:in 	std_logic_vector(13 downto 0);
    adc_data6		:in 	std_logic_vector(13 downto 0);
    adc_data7		:in 	std_logic_vector(13 downto 0);
    adc_data8		:in 	std_logic_vector(13 downto 0);
                                        -- data paths from from ADCs,
                                        --ADC_CHANCNT deep, ADC_WIDTH wide
    adc_ovf		:in		std_logic_vector(ADC_CHANCNT-1 downto 0);	-- overrange from ADCs
    adca_oe_l		:out	std_logic;						-- parallel out enable to ADC chan A
    adcb_oe_l		:out	std_logic;						-- parallel out enable to ADC chan B
    adcc_oe_l		:out	std_logic;						  -- parallel out enable to ADC chan C
    -------------------------------------
    --PHY signals
    phy_reset_l :out		std_logic;
    phy_mdck	:out		std_logic;
    phy_mdio  :inout		std_logic
    );
end wrapper;

architecture mixed of wrapper is

  component user_top_level
    port (
      clk        : in  std_logic;
      rst        : in  std_logic;
      system_in  : in  std_logic_vector(system_width-1 downto 0);
      system_out : out std_logic_vector(system_width-1 downto 0);
      -- Timing Inputs
      tbus_bplane	:inout std_logic_vector(10 downto 1);
      tbus_header	:inout std_logic_vector(10 downto 1);  
      tbus_master	:in std_logic_vector(10 downto 1);
      ten_mhz_clk       : in std_logic;
      -- IO Output
      phase_table_msb_out : out std_logic_vector(2 downto 0);
      -- ADC Inputs
      adc_in_0_0 : in std_logic_vector(13 downto 0);
      adc_in_0_1 : in std_logic_vector(13 downto 0);
      adc_in_0_2 : in std_logic_vector(13 downto 0);
      adc_in_1_0 : in std_logic_vector(13 downto 0);
      adc_in_1_1 : in std_logic_vector(13 downto 0);
      adc_in_1_2 : in std_logic_vector(13 downto 0);
      adc_in_2_0 : in std_logic_vector(13 downto 0);
      adc_in_2_1 : in std_logic_vector(13 downto 0);
      adc_in_2_2 : in std_logic_vector(13 downto 0);
      adc_enb    : out std_logic_vector(2 downto 0);
      syn_rst    : out std_logic;
      user_leds  : out std_logic_vector(1 downto 0);
      debug      : in  std_logic_vector(63 downto 0)
      );
  end component;



-- XST black box declaration
--  attribute box_type                              : string;
--  attribute box_type of user_top_level : component is "black_box";

-- Synplicity black box declaration
--  attribute syn_black_box                              : boolean;
--  attribute syn_black_box of user_top_level : component is true;


  component commonif
    port(
      lclk       : in    std_logic;
      clk200_p   : in    std_logic;
      clk200_n   : in    std_logic;
      adcclk     : in  std_logic;

      lreseto_l  : in    std_logic;
      lwrite     : in    std_logic;
      lads_l     : in    std_logic;
      lblast_l   : in    std_logic;
      lbterm_o   : out   std_logic;
      lbterm_oe  : out   std_logic;
      lad_i      : in    std_logic_vector(63 downto 0);
      lad_o      : out   std_logic_vector(63 downto 0);
      lad_oe     : out   std_logic;
      lreadyi_o  : out   std_logic;
      lreadyi_oe : out   std_logic;
      lbe_l      : in    std_logic_vector(7 downto 0);
      fholda     : in    std_logic;
      finti_l    : out   std_logic;
      -- DDRII Memory Banks
      DDR0_RAS_N : out   std_logic;
      DDR0_CAS_N : out   std_logic;
      DDR0_WE_N  : out   std_logic;
      DDR0_ODT   : out   std_logic;
      DDR0_CKE   : out   std_logic;
      DDR0_CS_N  : out   std_logic;
      DDR0_DQ    : inout std_logic_vector(31 downto 0);
      DDR0_DQS   : inout std_logic_vector(3 downto 0);
      DDR0_DQS_N : inout std_logic_vector(3 downto 0);
      DDR0_DM    : out   std_logic_vector(3 downto 0);
      DDR0_CK    : out   std_logic_vector(1 downto 0);
      DDR0_CK_N  : out   std_logic_vector(1 downto 0);
      DDR0_BA    : out   std_logic_vector(2 downto 0);
      DDR0_A     : out   std_logic_vector(12 downto 0);
      DDR1_RAS_N : out   std_logic;
      DDR1_CAS_N : out   std_logic;
      DDR1_WE_N  : out   std_logic;
      DDR1_ODT   : out   std_logic;
      DDR1_CKE   : out   std_logic;
      DDR1_CS_N  : out   std_logic;
      DDR1_DQ    : inout std_logic_vector(31 downto 0);
      DDR1_DQS   : inout std_logic_vector(3 downto 0);
      DDR1_DQS_N : inout std_logic_vector(3 downto 0);
      DDR1_DM    : out   std_logic_vector(3 downto 0);
      DDR1_CK    : out   std_logic_vector(1 downto 0);
      DDR1_CK_N  : out   std_logic_vector(1 downto 0);
      DDR1_BA    : out   std_logic_vector(2 downto 0);
      DDR1_A     : out   std_logic_vector(12 downto 0);
      DDR2_RAS_N : out   std_logic;
      DDR2_CAS_N : out   std_logic;
      DDR2_WE_N  : out   std_logic;
      DDR2_ODT   : out   std_logic;
      DDR2_CKE   : out   std_logic;
      DDR2_CS_N  : out   std_logic;
      DDR2_DQ    : inout std_logic_vector(31 downto 0);
      DDR2_DQS   : inout std_logic_vector(3 downto 0);
      DDR2_DQS_N : inout std_logic_vector(3 downto 0);
      DDR2_DM    : out   std_logic_vector(3 downto 0);
      DDR2_CK    : out   std_logic_vector(1 downto 0);
      DDR2_CK_N  : out   std_logic_vector(1 downto 0);
      DDR2_BA    : out   std_logic_vector(2 downto 0);
      DDR2_A     : out   std_logic_vector(12 downto 0);
      DDR3_RAS_N : out   std_logic;
      DDR3_CAS_N : out   std_logic;
      DDR3_WE_N  : out   std_logic;
      DDR3_ODT   : out   std_logic;
      DDR3_CKE   : out   std_logic;
      DDR3_CS_N  : out   std_logic;
      DDR3_DQ    : inout std_logic_vector(31 downto 0);
      DDR3_DQS   : inout std_logic_vector(3 downto 0);
      DDR3_DQS_N : inout std_logic_vector(3 downto 0);
      DDR3_DM    : out   std_logic_vector(3 downto 0);
      DDR3_CK    : out   std_logic_vector(1 downto 0);
      DDR3_CK_N  : out   std_logic_vector(1 downto 0);
      DDR3_BA    : out   std_logic_vector(2 downto 0);
      DDR3_A     : out   std_logic_vector(12 downto 0);
      debug      : out   std_logic_vector(63 downto 0);
      -- Interface to Synth
      synth_cntrl : out std_logic_vector(15 downto 0);
      synth_strb  : out std_logic_vector(3 downto 0);
      synth_status : in std_logic_vector(31 downto 0);
      synth_rst  : out std_logic;
      -- Interface to User Top Level
      clk        : out   std_logic;
      rst        : out   std_logic;
      system_in  : out   std_logic_vector(system_width-1 downto 0);
      system_out : in    std_logic_vector(system_width-1 downto 0));    
  end component;

  component sampck_synth is
    generic
      (
	END_CNT	:positive := 2
        );
    port
      (
        reset		:in		std_logic;
        clock		:in		std_logic;
        wr_data		:in		std_logic_vector( 7 downto 0);
        rd_data		:out		std_logic_vector( 7 downto 0);
        rd_status	:out	std_logic;

        address		:in		std_logic_vector( 7 downto 0);
        loc_func 		:in		std_logic;
        rd_strobe		:in		std_logic;
        wr_strobe		:in		std_logic;
        busy		:out		std_logic;
        busy_flag		:out		std_logic;
        init_busy		:out		std_logic;


        synth_status	:in	std_logic;		--ad9510 monitor output, fpga input
        synth_sclk	:out	std_logic;		--ad9510 serial clock,   fpga output
        synth_sdio	:out	std_logic;		--ad9510 data in, 		   fpga output
        synth_sdo	:in	std_logic;				--ad 9510 data out,		   fpga input
        synth_csb	:out	std_logic;			--ad9510 cs, active low, fpga output

        synth_func	:out	std_logic			--ad9510 multipurpose input
	--debug	:out std_logic_vector( 31 downto 0)
        );
  end component sampck_synth;

  
  signal system_in, system_out : std_logic_vector(system_width-1 downto 0);

  signal clk, rst               : std_logic;
  attribute clock_signal        : string;
  attribute clock_signal of clk : signal is "yes";
  attribute clock_signal of rst : signal is "yes";


  component IOBUF is
    port (
      I  : in    std_logic;
      IO : inout std_logic;
      O  : out   std_logic;
      T  : in    std_logic
      );
  end component;


  signal lreadyi_o  : std_logic;
  signal lreadyi_oe : std_logic;
  signal lbterm_o   : std_logic;
  signal lbterm_oe  : std_logic;
  signal lad_o      : std_logic_vector(63 downto 0);
  signal lad_i      : std_logic_vector(63 downto 0);
  signal lad_oe     : std_logic;

  -------------------------------------------------------------------
  --
  --  NULL_PAIR core component declaration
  --
  -------------------------------------------------------------------
  component NULL_PAIR
    port (
      GREFCLK_IN : in  std_logic;
      RX1N_IN    : in  std_logic_vector(1 downto 0);
      RX1P_IN    : in  std_logic_vector(1 downto 0);
      TX1N_OUT   : out std_logic_vector(1 downto 0);
      TX1P_OUT   : out std_logic_vector(1 downto 0));   
  end component;

  attribute box_type              : string;
  attribute box_type of NULL_PAIR : component is "user_black_box";

  -- Timing and ADC I/0
  -- Temporarily Defined as signals
  -- To be defined as Ports when actual I/O pins defined

  -- Timing INputs
  signal cycle_start   : std_logic;
  signal cycle_stop    : std_logic;
  signal cal_start     : std_logic;
  signal cal_stop      : std_logic;
  signal injection     : std_logic;
  signal hchange       : std_logic;
  signal fref          : std_logic;
  signal ten_mhz_clk   : std_logic;
  -- Timing Outputs
  signal cycle_start_o : std_logic;
  signal cycle_stop_o  : std_logic;
  signal cal_start_o   : std_logic;
  signal cal_stop_o    : std_logic;
  signal injection_o   : std_logic;
  signal hchange_o     : std_logic;
  signal fref_o        : std_logic;
  signal ten_mhz_clk_o : std_logic;
  -- ADC Inputs
  signal adc_in_0_0    : std_logic_vector(13 downto 0);
  signal adc_in_0_1    : std_logic_vector(13 downto 0);
  signal adc_in_0_2    : std_logic_vector(13 downto 0);
  signal adc_in_1_0    : std_logic_vector(13 downto 0);
  signal adc_in_1_1    : std_logic_vector(13 downto 0);
  signal adc_in_1_2    : std_logic_vector(13 downto 0);
  signal adc_in_2_0    : std_logic_vector(13 downto 0);
  signal adc_in_2_1    : std_logic_vector(13 downto 0);
  signal adc_in_2_2    : std_logic_vector(13 downto 0);

  signal adc_clk : std_logic;
  signal samp_clk_sig : std_logic;
  signal adc_enb : std_logic_vector(2 downto 0);
  signal syn_rst : std_logic;
  signal synth_rst_sig : std_logic;

  signal adc_samp_data : 	SLV_ARRAY_ADCCHAN;

  signal clk_counter125 : std_logic_vector(31 downto 0);
  signal clk_led125 : std_logic := '0';
  signal clk_counter10 : std_logic_vector(31 downto 0);
  signal clk_led10 : std_logic := '0';


  signal auto_rst : std_logic;
  signal lclk_i : std_logic;

  signal rst_count : std_logic_vector(3 downto 0);
  signal auto_counter : std_logic_vector(3 downto 0);
  signal auto_count : std_logic_vector(11 downto 0);
  signal cc_reg1,cc_reg2 : std_logic_vector(7 downto 0);
  
  signal synth_cntrl : std_logic_vector(15 downto 0);
  signal synth_strb : std_logic_vector(3 downto 0);
  signal synth_status_bus : std_logic_vector(31 downto 0);

  signal synth_strb_sig	:std_logic_vector( 3 downto 0);
  signal synth_strb_pl_sig  :std_logic_vector( 3 downto 0);
  signal	synth_cntrl_sig	:std_logic_vector( 23 downto 0);

  signal	synth_busy_sig	:std_logic;
  signal	synth_busy_flag_sig	:std_logic;
  signal	synth_init_busy_sig	:std_logic;
  signal	synth_status_sig	:std_logic;
  signal	synth_rd_strobe_sig	:std_logic;
  signal	synth_wr_strobe_sig	:std_logic;

  signal synth_init_sig : std_Logic;
  signal synth_func_sig : std_Logic;

  signal phase_table_msb_out : std_logic_vector(2 downto 0);

  signal user_leds : std_logic_vector(1 downto 0);

  signal debug : std_logic_vector(63 downto 0);

  component phy_cntrl is
    port
      (
        --Host controls
        reset		:in		std_logic;
        clock		:in		std_logic;
        ckdiv_cnt	:in		std_logic_vector(15 downto 0);

        rd_flag	:in		std_logic;
        wr_flag	:in		std_logic;

        busy		:out		std_logic;

        --	Params for I2C transfer
        phy_addr		:in		std_logic_vector(4 downto 0);
        reg_addr		:in		std_logic_vector(4 downto 0);

        wrdata_in		:in		std_logic_vector(15 downto 0);
        rddata_out	:out		std_logic_vector(15 downto 0);

        --PHY signals
        phy_reset_l :out		std_logic;
        i2c_clock	:out		std_logic;
        i2c_data  :inout		std_logic
	);
  end component phy_cntrl;




  
begin
  
  user0 : user_top_level
    port map (
      clk        => clk,
      rst        => rst,
      system_in  => system_in,
      system_out => system_out,
      tbus_bplane     => tbus_bplane,
      tbus_header     => tbus_header,
      tbus_master     => tbus_master,    
      ten_mhz_clk =>  ten_mhz_clk,
      phase_table_msb_out =>      phase_table_msb_out,
      adc_in_0_0 =>  adc_in_0_0,  
      adc_in_0_1 =>  adc_in_0_1,  
      adc_in_0_2 =>  adc_in_0_2,  
      adc_in_1_0 =>  adc_in_1_0,  
      adc_in_1_1 =>  adc_in_1_1,  
      adc_in_1_2 =>  adc_in_1_2,  
      adc_in_2_0 =>  adc_in_2_0,  
      adc_in_2_1 =>  adc_in_2_1,  
      adc_in_2_2 => adc_in_2_2,
      adc_enb => adc_enb,
      syn_rst => open,
      user_leds => user_leds,
      debug => debug

      );

  commonif0 : commonif
    port map (
      lclk       => lclk,
      clk200_p   => clk200_p,
      clk200_n   => clk200_n,
      adcclk => adc_clk,
      lreseto_l  => lreseto_l,
      lwrite     => lwrite,
      lads_l     => lads_l,
      lblast_l   => lblast_l,
      lbterm_o   => lbterm_o,
      lbterm_oe  => lbterm_oe,
      lad_i      => lad_i,
      lad_o      => lad_o,
      lad_oe     => lad_oe,
      lreadyi_o  => lreadyi_o,
      lreadyi_oe => lreadyi_oe,
      lbe_l      => lbe_l,
      fholda     => fholda,
      finti_l    => finti_l,
      DDR0_RAS_N => DDR0_RAS_N,
      DDR0_CAS_N => DDR0_CAS_N,
      DDR0_WE_N  => DDR0_WE_N,
      DDR0_ODT   => DDR0_ODT,
      DDR0_CKE   => DDR0_CKE,
      DDR0_CS_N  => DDR0_CS_N,
      DDR0_DQ    => DDR0_DQ,
      DDR0_DQS   => DDR0_DQS,
      DDR0_DQS_N => DDR0_DQS_N,
      DDR0_DM    => DDR0_DM,
      DDR0_CK    => DDR0_CK,
      DDR0_CK_N  => DDR0_CK_N,
      DDR0_BA    => DDR0_BA,
      DDR0_A     => DDR0_A,
      DDR1_RAS_N => DDR1_RAS_N,
      DDR1_CAS_N => DDR1_CAS_N,
      DDR1_WE_N  => DDR1_WE_N,
      DDR1_ODT   => DDR1_ODT,
      DDR1_CKE   => DDR1_CKE,
      DDR1_CS_N  => DDR1_CS_N,
      DDR1_DQ    => DDR1_DQ,
      DDR1_DQS   => DDR1_DQS,
      DDR1_DQS_N => DDR1_DQS_N,
      DDR1_DM    => DDR1_DM,
      DDR1_CK    => DDR1_CK,
      DDR1_CK_N  => DDR1_CK_N,
      DDR1_BA    => DDR1_BA,
      DDR1_A     => DDR1_A,
      DDR2_RAS_N => DDR2_RAS_N,
      DDR2_CAS_N => DDR2_CAS_N,
      DDR2_WE_N  => DDR2_WE_N,
      DDR2_ODT   => DDR2_ODT,
      DDR2_CKE   => DDR2_CKE,
      DDR2_CS_N  => DDR2_CS_N,
      DDR2_DQ    => DDR2_DQ,
      DDR2_DQS   => DDR2_DQS,
      DDR2_DQS_N => DDR2_DQS_N,
      DDR2_DM    => DDR2_DM,
      DDR2_CK    => DDR2_CK,
      DDR2_CK_N  => DDR2_CK_N,
      DDR2_BA    => DDR2_BA,
      DDR2_A     => DDR2_A,
      DDR3_RAS_N => DDR3_RAS_N,
      DDR3_CAS_N => DDR3_CAS_N,
      DDR3_WE_N  => DDR3_WE_N,
      DDR3_ODT   => DDR3_ODT,
      DDR3_CKE   => DDR3_CKE,
      DDR3_CS_N  => DDR3_CS_N,
      DDR3_DQ    => DDR3_DQ,
      DDR3_DQS   => DDR3_DQS,
      DDR3_DQS_N => DDR3_DQS_N,
      DDR3_DM    => DDR3_DM,
      DDR3_CK    => DDR3_CK,
      DDR3_CK_N  => DDR3_CK_N,
      DDR3_BA    => DDR3_BA,
      DDR3_A     => DDR3_A,
      debug => debug,
      synth_cntrl => synth_cntrl,
      synth_strb  => synth_strb,
      synth_status => synth_status_bus,
      synth_rst  => syn_rst,
      clk        => clk,
      rst        => rst,
      system_in  => system_in,
      system_out => system_out);


  -- BTERM should only be driven when the fpga is addressed otherwise
  -- float, because the control logic on the XRC might also drive it.
  --
  update_bterm : process(lbterm_o, lbterm_oe)
  begin
    if lbterm_oe = '1' then
      lbterm_l <= not lbterm_o;
    else
      lbterm_l <= 'Z';
    end if;
  end process update_bterm;

  --
  -- LREADYI# should only be driven when the fpga is addressed, otherwise
  -- float because the control logic on the XRC might also drive it.
  --
  update_ready : process(lreadyi_o, lreadyi_oe)
  begin
    if lreadyi_oe = '1' then
      lreadyi_l <= not lreadyi_o;
    else
      lreadyi_l <= 'Z';
    end if;
  end process update_ready;

  --
  -- Drive the local data bus on a read.
  --
  data_bus : process(lad_oe, lad_o)
  begin
    if lad_oe = '1' then
      lad <= lad_o;
    else
      lad <= (others => 'Z');
    end if;
  end process data_bus;

  lad_i <= lad;



--==============================================================================================
--External clock input buffer
--==============================================================================================
  ibuf_ext_clk:IBUFGDS
    port map(
      O => samp_clk_sig,
      I => sampck_p,
      IB =>sampck_n
      );
  
  adcclk_bufg : BUFG
    port map (
      I => samp_clk_sig,
      O => adc_clk);

  -- Register ADC inputs in IOBs on falling edge of ADC CLK
  
  reg_adc: process (adc_clk)
  begin  -- process reg_adc
    if adc_clk'event and adc_clk = '0' then  -- falling clock edge
      adc_samp_data(0) <= adc_data0;
      adc_samp_data(1) <= adc_data1;
      adc_samp_data(2) <= adc_data2;
      adc_samp_data(3) <= adc_data3;
      adc_samp_data(4) <= adc_data4;
      adc_samp_data(5) <= adc_data5;
      adc_samp_data(6) <= adc_data6;
      adc_samp_data(7) <= adc_data7;
      adc_samp_data(8) <= adc_data8;
    end if;
  end process reg_adc;
  
--    reg_adc: process (samp_clk_sig)
--  begin  -- process reg_adc
--    if samp_clk_sig'event and samp_clk_sig = '0' then  -- falling clock edge
--      adc_samp_data(0) <= adc_data0;
--      adc_samp_data(1) <= adc_data1;
--      adc_samp_data(2) <= adc_data2;
--      adc_samp_data(3) <= adc_data3;
--      adc_samp_data(4) <= adc_data4;
--      adc_samp_data(5) <= adc_data5;
--      adc_samp_data(6) <= adc_data6;
--      adc_samp_data(7) <= adc_data7;
--      adc_samp_data(8) <= adc_data8;
--    end if;
--  end process reg_adc;

  -- Register ADC signals on rising edge of ADC CLK
  reg_inputs: process (clk, rst)
  begin  -- process dummy_inputs
    if rst = '1' then                   
      adc_in_0_0 <= (others => '0');
      adc_in_0_1 <= (others => '0');
      adc_in_0_2 <= (others => '0');
      adc_in_1_0 <= (others => '0');
      adc_in_1_1 <= (others => '0');
      adc_in_1_2 <= (others => '0');
      adc_in_2_0 <= (others => '0');
      adc_in_2_1 <= (others => '0');
      adc_in_2_2 <= (others => '0');
    elsif clk'event and clk = '1' then
      adc_in_0_0 <=  adc_samp_data(0);
      adc_in_0_1 <=  adc_samp_data(1);  
      adc_in_0_2 <=  adc_samp_data(2);
      adc_in_1_0 <=  adc_samp_data(3);
      adc_in_1_1 <=  adc_samp_data(4);
      adc_in_1_2 <=  adc_samp_data(5);
      adc_in_2_0 <=  adc_samp_data(6);
      adc_in_2_1 <=  adc_samp_data(7);
      adc_in_2_2 <=  adc_samp_data(8);  
    end if;
  end process reg_inputs;

--==============================================================================================
--External clock input buffer
--==============================================================================================
  ibuf_ext_clk1:IBUFGDS
    port map(
      O => ten_mhz_clk,
      I =>fpga_10mhz_refckin_p,
      IB =>fpga_10mhz_refckin_n
      );
------------------------------------------------------------------------------------------------


  synth_strb_sig <= synth_strb;
  synth_cntrl_sig(15 downto 0) <= synth_cntrl;
  synth_status_bus <= EXT(synth_status_sig&synth_init_busy_sig&synth_busy_flag_sig&synth_busy_sig& synth_cntrl_sig,32);
  
--==============================================================================================
--Synthesiser control interface
-- generates a pulse following a rising edge on the appropriate bit
--decode occurs for 1 clock cycle when local address ( and wr direction) already latched
-- init signal forces the default configuration of the device
--==============================================================================================
  process(rst, lclk_i)
  begin
    if rst = '1' then
      synth_rd_strobe_sig	<='0';
      synth_wr_strobe_sig <='0';
      synth_init_sig <='0';

      synth_strb_pl_sig <=(others =>'0');
      synth_func_sig <='1';
    elsif rising_edge(lclk_i) then
      synth_strb_pl_sig <=synth_strb_sig;

      --Hard reset of the synth ( default behaviour of the func pin)
      if (synth_strb_pl_sig(3)='0' and synth_strb_sig(3)= '1') then
        synth_func_sig <='0';
        synth_init_sig <='0';
        synth_rd_strobe_sig	<='0';
        synth_wr_strobe_sig <='0';
      else
        synth_func_sig <='1';
        --initialise the synth settings to the defaults
        if (synth_strb_pl_sig(2)='0' and synth_strb_sig(2)= '1') then
          synth_rd_strobe_sig	<='0';
          synth_wr_strobe_sig <='0';
          synth_init_sig <='1';
        else
          -- read or write cycle
          synth_init_sig <='0';

          --read dominates
          if (synth_strb_pl_sig(0)='0' and synth_strb_sig(0)= '1') then
            synth_rd_strobe_sig	<='1';
            synth_wr_strobe_sig <='0';
          else
            synth_rd_strobe_sig	<='0';
            if (synth_strb_pl_sig(1)='0' and synth_strb_sig(1)= '1') then
              synth_wr_strobe_sig	<='1';
            else
              synth_wr_strobe_sig <='0';
            end if; -- pl1 wr
          end if; --pl0 rd

        end if;	-- pl2 init signal
      end if;	-- pl3 func signal
    end if;	--clock edge
  end process;

  

  synth_rst_sig <=syn_rst or rst or auto_rst;

  cksynth:sampck_synth
    generic map
    (
      END_CNT	=>2
      )
    port map
    (
      reset		=>  synth_rst_sig,
      clock		=> system_out(0),  -- lclk

      -- Local Bus Connection for Synth, not implemented
      wr_data		=> synth_cntrl_sig(7 downto 0),
      address		=> synth_cntrl_sig(15 downto 8),
      rd_data		=> synth_cntrl_sig(23 downto 16),
      rd_status	=> synth_status_sig,
      busy		 => synth_busy_sig,
      busy_flag		 => synth_busy_flag_sig,
      init_busy		 => synth_init_busy_sig,
--		loc_func     => synth_func_sig,
      loc_func     => '1',
      rd_strobe		=>synth_rd_strobe_sig,
      wr_strobe	  =>synth_wr_strobe_sig,

      synth_status	=>synth_status,
      synth_sclk	=>synth_sclk,
      synth_sdio	=> synth_sdio,
      synth_sdo	 => synth_sdo,
      synth_csb	=>synth_csb,
      synth_func	=>synth_func
      --debug	=>open
      );

  adca_oe_l <= not adc_enb(0);
  adcb_oe_l <= not adc_enb(1);
  adcc_oe_l <= not adc_enb(2);

  vcxo_enable <= '1';

  --yellow_led(0) <= clk_led10;
  --red_led(0) <= not adc_enb(2);
  yellow_led(0) <= not user_leds(0);
  red_led(0) <= not user_leds(1);
  yellow_led(1) <= clk_led125; 
  red_led(1) <= not synth_init_busy_sig;


  count_clk10: process (ten_mhz_clk)
  begin  -- process count_clk125
    if ten_mhz_clk'event and ten_mhz_clk = '1' then  -- rising clock edge
      if OR_reduce(clk_counter10) = '0' then
        clk_counter10 <= CONV_STD_LOGIC_VECTOR(10000000,32);
        clk_led10 <= not clk_led10;
      else
        clk_counter10 <= clk_counter10-1;
      end if;
    end if;
  end process count_clk10;

  count_clk125: process (adc_clk)
  begin  -- process count_clk125
    if adc_clk'event and adc_clk = '1' then  -- rising clock edge
      if OR_reduce(clk_counter125) = '0' then
        clk_counter125 <= CONV_STD_LOGIC_VECTOR(125000000,32);
        clk_led125 <= not clk_led125;
      else
        clk_counter125 <= clk_counter125-1;
      end if;
    end if;
  end process count_clk125;


--  clk_dbgio: process (adc_clk)
--  begin  -- process clk_dbgio
--    if adc_clk'event and adc_clk = '1' then  -- rising clock edge
--      debugio(0) <= not debugio(0);
--      debugio_dirn(0) <= '1';
--    end if;
--  end process clk_dbgio;
  

--  clk_dbgio1: process (ten_mhz_clk)
--  begin  -- process clk_dbgio
--    if ten_mhz_clk'event and ten_mhz_clk = '1' then  -- rising clock edge
--      debugio(1) <= not debugio(1);
--      debugio_dirn(1) <= '1';
--    end if;
--  end process clk_dbgio1;

--  clk_dbgio2: process (lclk_i)
--  begin  -- process clk_dbgio
--    if lclk_i'event and lclk_i = '1' then  -- rising clock edge
--      debugio(2) <= not debugio(2);
--      debugio_dirn(2) <= '1';
--    end if;
--  end process clk_dbgio2;

  debugio <= phase_table_msb_out;
  debugio_dirn <= "111";
  

  lclk_i <= system_out(0);

  auto_rst_pll: process (lclk_i, rst)
  begin  -- process auto_rst_pll
    if rst = '1' then                   -- asynchronous reset (active low)
      rst_count <= (others => '1');
      auto_rst <= '1';
      auto_counter <= (others => '0');
      auto_count <= (others => '0');
    elsif lclk_i'event and lclk_i = '1' then  -- rising clock edge
      if auto_counter = "0000" then
        cc_reg1 <= clk_counter125(7 downto 0);
        cc_reg2 <= cc_reg1;
        if cc_reg2 = cc_reg1 and synth_init_busy_sig = '0' then
          auto_count <= auto_count+1;
        else
          auto_count <= (others => '0');
        end if;
      end if;
      
      if OR_reduce(rst_count) = '1' then
        rst_count <= rst_count-1;
      elsif AND_reduce(auto_count) = '1' then
        rst_count <= (others => '1');
      end if;
      auto_rst <= OR_reduce(rst_count);
      auto_counter <= auto_counter+1;
    end if;
  end process auto_rst_pll;
  

  phycntrl: phy_cntrl
    port map
    (
      --Host controls
      reset		=> synth_rst_sig,
      clock		=> lclk_i,
      ckdiv_cnt =>X"0020",
      rd_flag	  => '0',
      wr_flag	  => '0',      
      busy	    => open,
      
      --	Params for I2C transfer
      phy_addr		=> "00000",
      reg_addr		=> "00000",

      wrdata_in		=> X"0000",
      rddata_out	=> open,

      --PHY signals
      phy_reset_l => phy_reset_l,
      i2c_clock	=>phy_mdck,
      i2c_data  =>phy_mdio
      );

    
end mixed;
