/*******************************************************************************
 *	BString.hxx		BString Handling
 *				T.Barnaby,	BEAM Ltd, 29/10/91
 *******************************************************************************
 */
#ifndef BSTRING_H
#define BSTRING_H	1

#include	<BRefData.h>
#include	<BList.h>
#include	<iostream>

class BString {
public:
			BString();
			BString(const BString& string);
			BString(const char* str);
			BString(char ch);
			BString(int v);
			BString(unsigned int v);
			BString(long v);
			BString(unsigned long long);
			BString(double v);
// QT support
#ifdef QSTRING_H
			BString(const QString& str){
				Init(str);
			}
#endif
	virtual		~BString();

	static		BString convert(char ch);			///< Converts char to string
	static		BString	convert(int value);			///< Converts int to string
	static		BString	convert(unsigned int value);		///< Converts uint to string
	static		BString	convert(long value);			///< Converts long to string
	static		BString	convert(double value);			///< Converts double to string
	static		BString	convert(unsigned long long value);	///< Converts u long long to string
	static		BString	convertHex(int value);			///< Converts int to string as hex value
	static		BString	convertHex(unsigned int value);		///< Converts uint to string as hex value

	BString		copy();						///< Return an independant copy
	
	virtual void	strChanged();
	
	int		len() const;						///< Length of string
	const char*	retStr() const;						///< Ptr to char* representation
	char*		retStrDup() const;					///< Ptr to newly malloc'd char*
	int		retInt() const;						///< Return string as a int
	unsigned int	retUInt() const;					///< Return string as an unsigned int
	double		retDouble() const;					///< Return string as a double
	int		compare(const BString& string) const;			///< Compare strings
	int		compareWild(const BString& string) const;		///< Compare string to string with wildcards 
	int		compareWildExpression(const BString& string) const;	///< Compare string to space deliminated patterns
	BString		add(const BString& str) const;				///< Add two strings
	BString&	truncate(int len);					///< Truncate to length len
	BString&	pad(int len);						///< Pad to length len
	
	BString&	toUpper();						///< Convert to uppercase
	BString&	toLower();						///< Convert to lowercase
	void		removeNL();						///< Remove if present NL from last char 

	BString		subString(int start, int len) const;			///< Returns substring
	int		del(int start, int len);				///< Delete substring
	int		insert(int start, BString str);				///< Insert substring
	void		printf(const char* fmt, ...) ;				///< Formated print into the string
	int		find(char ch) const;					///< Find ch in string searching forwards
	int		findReverse(char ch) const;				///< Find ch in string searching backwards

	BList<BString>	getTokenList(BString separators);			///< Break string into tokens 
	BString		removeSeparators(BString separators);			///< Remove any char from sepatators from string
	BString		pullToken(BString terminators);				///< Pull token from start of string
	BString		pullSeparators(BString separators);			///< Pull separators from start of string
	BString		pullWord();						///< Pull a word out of the head of the string
	BString		pullLine();						///< Pull a line out of the head of the string

	BString		field(int field) const;
	char**		fields();

	/* Operator functions */
	BString&	operator=(const BString& string);
	char&		operator[](int pos);
	
	int		operator==(const BString& s) const{
				return (compare(s) == 0);
			}
	int		operator==(const char* s) const{
				return (compare(s) == 0);
			}
	int		operator>(const BString& s) const{
				return (compare(s) > 0);
			}
	int		operator>(const char* s) const{
				return (compare(s) > 0);
			}
	int		operator<(const BString& s) const{
				return (compare(s) < 0);
			}
	int		operator<(const char* s) const{
				return (compare(s) < 0);
			}
	int		operator>=(const BString& s) const{
				return (compare(s) >= 0);
			}
	int		operator<=(const BString& s) const{
				return (compare(s) <= 0);
			}
	int		operator!=(const BString& s) const{
				return (compare(s) != 0);
			}
	int		operator!=(const char* s) const{
				return (compare(s) != 0);
			}
	BString 	operator+(const BString& s) const {
				return add(s);
			}
	BString 	operator+(const char* s) const {
				return add(s);
			}
	BString 	operator+=(const BString& s){
				*this = add(s);
				return *this;
			}
	BString 	operator+=(const char* s){
				*this = add(s);
				return *this;
			}
			
	/* Special operators */
	BString 	operator+(char ch) const {
				return add(convert(ch));
			}
	BString 	operator+(int i) const {
				return add(convert(i));
			}
	BString 	operator+(unsigned int i) const {
				return add(convert(i));
			}
	BString 	operator+(unsigned long long i) const {
				return add(convert(i));
			}
#ifdef ZAP
	friend BString	operator+(const char* a1, BString& a2);
#endif

			operator const char* () const {
				return retStr();
			}
protected:
	BRefData*	ostr;	

private:
	void		Init(const char* str);
	int		inString(int pos) const;
	int		isSpace(char ch) const;
};

/* OStream Functions */
std::ostream& operator<<(std::ostream& o, BString& s);
std::istream& operator>>(std::istream& i, BString& s);

#ifdef BLIST_H
typedef BList<BString>	BStringList;
#endif


#endif
