-------------------------------------------------------------------------------
-- Copyright (c) 2005 Xilinx, Inc.
-- This design is confidential and proprietary of Xilinx, All Rights Reserved.
-------------------------------------------------------------------------------
--   ____  ____
--  /   /\/   /
-- /___/  \  / Vendor: Xilinx
-- \   \   \/ Version: 1.6
--  \   \ Application : MIG
--  /   / Filename: mem_interface_small_ddr2_controller_0.vhd
-- /___/   /\ Date Last Modified:  Wed Jun 1 2005
-- \   \  /  \Date Created: Mon May 2 2005
--  \___\/\___\
--
-- Device: Virtex-4
-- Design Name: DDR2_V4
-- Description     : This module is the main control logic of the memory interface.
--                         All commands are issued from here acoording to the burst,
--                         CAS Latency and the user commands.
-------------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.numeric_std.all;
use work.mem_interface_small_parameters_0.all;

library UNISIM;
use UNISIM.VCOMPONENTS.ALL;

entity mem_interface_small_ddr2_controller_0 is

port(
                   clk0                   : in std_logic; -- controller input
                   refresh_clk            : in std_logic; -- controller input
                   rst                    : in std_logic; -- controller input
                   af_addr                : in std_logic_vector(35 downto 0); --FIFO  signal
                     af_empty               : in std_logic;                     --FIFO  signal
                   phy_Dly_Slct_Done      : in std_logic;  --Input signal for the Dummy Reads
                   COMP_DONE              : in std_logic;
                       ctrl_dummy_wr_sel      : out std_logic;
                   burst_length           : out std_logic_vector(2 downto 0);
                   ctrl_Dummyread_Start   : out std_logic;
                   ctrl_af_RdEn           : out std_logic;--FIFO read enable signals
                   ctrl_Wdf_RdEn          : out std_logic;-- FIFO read enable signals
                   ctrl_Dqs_Rst           : out std_logic;  -- Rst and Enable signals for DQS logic
                   ctrl_Dqs_En            : out std_logic;  -- Rst and Enable signals for DQS logic
                   ctrl_WrEn              : out std_logic;   --Read and Write Enable signals to the phy interface
                   ctrl_RdEn              : out std_logic;    --Read and Write Enable signals to the phy interface
                   ctrl_ddr2_address      : out std_logic_vector((row_address-1) downto 0);
                   ctrl_ddr2_ba           : out std_logic_vector((bank_address-1) downto 0);
                   ctrl_ddr2_ras_L        : out std_logic;
                   ctrl_ddr2_cas_L        : out std_logic;
                   ctrl_ddr2_we_L         : out std_logic;
                   ctrl_ddr2_cs_L         : out std_logic_vector((cs_width-1) downto 0);
                   ctrl_ddr2_cke          : out std_logic_vector((cke_width-1) downto 0);
                   ctrl_ddr2_odt          : out std_logic_vector((odt_width-1) downto 0);
                         dummy_write_flag               : out std_logic
         );
end entity ;

architecture arc_controller of mem_interface_small_ddr2_controller_0 is

-- INTERNAL SIGNALS--

    signal   init_count            : std_logic_vector(3 downto 0);
    signal   init_count_cp         : std_logic_vector(3 downto 0);
    signal   init_memory           : std_logic;
    signal   count_200_cycle       : std_logic_vector(7 downto 0);
    signal   ref_flag              : std_logic;
    signal   ref_flag_266          : std_logic;
    signal   ref_flag_266_r        : std_logic;
    signal   auto_ref              : std_logic;

    signal   next_state            : std_logic_vector(4 downto 0);
    signal   state                 : std_logic_vector(4 downto 0);
    signal   state_r2              : std_logic_vector(4 downto 0);
    signal   state_r3              : std_logic_vector(4 downto 0);

    signal   init_next_state       : std_logic_vector(4 downto 0);
    signal   init_state            : std_logic_vector(4 downto 0);
    signal   init_state_r2         : std_logic_vector(4 downto 0);
    signal   init_state_r3         : std_logic_vector(4 downto 0);

    signal  row_addr_r             : std_logic_vector((row_address -1) downto 0);
    signal  ddr2_address_init_r    : std_logic_vector((row_address -1) downto 0);
    signal  ddr2_address_r1        : std_logic_vector((row_address -1) downto 0);
    signal  ddr2_address_r2        : std_logic_vector((row_address -1) downto 0);
    signal  ddr2_ba_r1             : std_logic_vector((bank_address -1) downto 0);
    signal  ddr2_ba_r2             : std_logic_vector((bank_address -1) downto 0);

     -- counters for ddr controller
    signal   mrd_count             : std_logic;
    signal   rp_count              : std_logic_vector(2 downto 0);
    signal   rfc_count             : std_logic_vector(5 downto 0);
    signal   rcd_count             : std_logic_vector(2 downto 0);
    signal   ras_count             : std_logic_vector(3 downto 0);
    signal   wr_to_rd_count        : std_logic_vector(3 downto 0);
    signal   rd_to_wr_count        : std_logic_vector(3 downto 0);
    signal   rtp_count             : std_logic_vector(3 downto 0);
    signal   wtp_count             : std_logic_vector(3 downto 0);

    signal   refi_count            : std_logic_vector(7 downto 0);
    signal   cas_count             : std_logic_vector(2 downto 0);
    signal   cas_check_count       : std_logic_vector(3 downto 0);
    signal   wrburst_cnt           : std_logic_vector(2 downto 0);
    signal   read_burst_cnt        : std_logic_vector(2 downto 0);
    signal   ctrl_WrEn_cnt         : std_logic_vector(2 downto 0);
    signal   rdburst_cnt           : std_logic_vector(3 downto 0);
    signal   af_addr_r             : std_logic_vector(35 downto 0);
    signal   af_addr_r1            : std_logic_vector(35 downto 0);


    signal   wdf_rden_r            : std_logic;
    signal   wdf_rden_r2           : std_logic;
    signal   wdf_rden_r3           : std_logic;
    signal   wdf_rden_r4           : std_logic;
    signal   ctrl_Wdf_RdEn_int     : std_logic;

    signal   af_rden               : std_logic;
    signal   ddr2_ras_r2           : std_logic;
    signal   ddr2_cas_r2           : std_logic;
    signal   ddr2_we_r2            : std_logic;
    signal   ddr2_ras_r            : std_logic;
    signal   ddr2_cas_r            : std_logic;
    signal   ddr2_we_r             : std_logic;
    signal   ddr2_ras_r3           : std_logic;
    signal   ddr2_cas_r3           : std_logic;
    signal   ddr2_we_r3            : std_logic;

    signal   idle_cnt              : std_logic_vector(3 downto 0);

    signal   ctrl_Dummyread_Start_r1 : std_logic;
    signal   ctrl_Dummyread_Start_r2 : std_logic;
    signal   ctrl_Dummyread_Start_r3 : std_logic;
    signal   ctrl_Dummyread_Start_r4 : std_logic;

    signal   conflict_resolved_r     : std_logic;

    signal       dqs_reset            : std_logic;
    signal   dqs_reset_r1         : std_logic;
    signal   dqs_reset_r2         : std_logic;
    signal   dqs_reset_r3         : std_logic;
    signal   dqs_reset_r4         : std_logic;
    signal   dqs_reset_r5         : std_logic;
    signal   dqs_reset_r6         : std_logic;


    signal   dqs_en               : std_logic;
    signal   dqs_en_r1            : std_logic;
    signal   dqs_en_r2            : std_logic;
    signal   dqs_en_r3            : std_logic;
    signal   dqs_en_r4            : std_logic;
    signal   dqs_en_r5            : std_logic;
    signal   dqs_en_r6            : std_logic;

    signal   ctrl_wdf_read_en     : std_logic;
    signal   ctrl_wdf_read_en_r1  : std_logic;
    signal   ctrl_wdf_read_en_r2  : std_logic;
    signal   ctrl_wdf_read_en_r3  : std_logic;
    signal   ctrl_wdf_read_en_r4  : std_logic;
    signal   ctrl_wdf_read_en_r5  : std_logic;
    signal   ctrl_wdf_read_en_r6  : std_logic;

    signal   ddr2_cs_r1           : std_logic_vector((cs_width-1) downto 0);
    signal   ddr2_cs_r            : std_logic_vector((cs_width-1) downto 0);
    signal   ddr2_cke_r           : std_logic_vector((cke_width-1) downto 0);
    signal   chip_cnt             : std_logic_vector(1 downto 0);
    signal   auto_cnt             : std_logic_vector(2 downto 0);

--  Precharge fix for deep memory
    signal   pre_cnt             : std_logic_vector(2 downto 0);

-- FIFO read enable signals
-- Rst and Enable signals for DQS logic
-- Read and Write Enable signals to the phy interface

    signal  dummy_read_en         : std_logic;
    signal  ctrl_init_done        : std_logic;

    signal   count_200cycle_done_r : std_logic;

    signal   init_done             : std_logic;

    signal   burst_cnt             : std_logic_vector(2 downto 0);


    signal   ctrl_write_en         : std_logic;
    signal   ctrl_write_en_r1      : std_logic;
    signal   ctrl_write_en_r2      : std_logic;
    signal   ctrl_write_en_r3      : std_logic;
    signal   ctrl_write_en_r4      : std_logic;
    signal   ctrl_write_en_r5      : std_logic;
    signal   ctrl_write_en_r6      : std_logic;


    signal   ctrl_read_en          : std_logic;
    signal   ctrl_read_en_r        : std_logic;
    signal   ctrl_read_en_r1       : std_logic;
    signal   ctrl_read_en_r2       : std_logic;
    signal   ctrl_read_en_r3       : std_logic;
    signal   ctrl_read_en_r4       : std_logic;

    signal   odt_cnt               : std_logic_vector(3 downto 0);
    signal   odt_en_cnt            : std_logic_vector(3 downto 0);
    signal   odt_en                : std_logic_vector((odt_width-1) downto 0);

    signal   conflict_detect       : std_logic;
    signal   conflict_detect_r     : std_logic;

    signal   load_mode_reg         : std_logic_vector((row_address-1) downto 0);
    signal   ext_mode_reg          : std_logic_vector((row_address-1) downto 0);


    signal   CAS_LATENCY_VALUE     : std_logic_vector(2 downto 0);
    signal   BURST_LENGTH_VALUE    : std_logic_vector(2 downto 0);
    signal   ADDITIVE_LATENCY_VALUE: std_logic_vector(2 downto 0);
    signal   ODT_ENABLE            : std_logic;
    signal   REGISTERED_VALUE       : std_logic;
    signal   ECC_VALUE              : std_logic;

    signal   WR                    : std_logic;
    signal   RD                    : std_logic;
    signal   LMR                   : std_logic;
    signal   PRE                   : std_logic;
    signal   REF                   : std_logic;
    signal   ACT                   : std_logic;

    signal   WR_r                  : std_logic;
    signal   RD_r                  : std_logic;
    signal   LMR_r                 : std_logic;
    signal   PRE_r                 : std_logic;
    signal   REF_r                 : std_logic;
    signal   ACT_r                 : std_logic;
    signal   af_empty_r            : std_logic;
    signal   LMR_PRE_REF_ACT_cmd_r : std_logic;
    signal   cke_200us_cnt  :  std_logic_vector(4 downto 0);
    signal   done_200us   : std_logic;
    signal   dummy_write_state_r  : std_logic;
    signal   dummy_write_state_r1 : std_logic;

    signal  dummy_write_state   : std_logic;
    signal  ctrl_dummy_write    : std_logic;

    signal dummy_write_flag_r      : std_logic;

    signal   command_address       : std_logic_vector(2 downto 0);
    signal   zeroes                : std_logic_vector(3 downto 0);
    signal   ctrl_odt              : std_logic_vector((odt_width-1) downto 0);
    signal   cs_width0             : std_logic_vector(1 downto 0);
    signal   cs_width1             : std_logic_vector(2 downto 0);

    signal   s1_h, s2_h, s3_h ,s5_h   : std_logic_vector(3 downto 0);
    signal s4_h                       :   std_logic_vector(2 downto 0);
    signal COMP_DONE_r                :   std_logic;

    signal   ctrl_Dqs_Rst_r1   : std_logic;
    signal   ctrl_Dqs_En_r1    : std_logic;
    signal   ctrl_WrEn_r1      : std_logic;
    signal   ctrl_RdEn_r1      : std_logic;

    signal   ddr2_cs_r_out     : std_logic_vector((cs_width-1) downto 0);
    signal   ddr2_cs_r_odt     : std_logic_vector((cs_width-1) downto 0);

    signal   count6            : std_logic_vector(5 downto 0);

    


attribute equivalent_register_removal : string;
attribute equivalent_register_removal of ddr2_cs_r_odt : signal is "no";


constant INIT_IDLE                   : std_logic_vector(4 downto 0) := "00000";--  5'h00//0_0000
constant INIT_LOAD_MODE          : std_logic_vector(4 downto 0) := "00001";--  5'h01//0_0001
constant INIT_MODE_REGISTER_WAIT     : std_logic_vector(4 downto 0) := "00010";--  5'h02//0_0010
constant INIT_PRECHARGE              : std_logic_vector(4 downto 0) := "00011";--  5'h03//0_0011
constant INIT_PRECHARGE_WAIT         : std_logic_vector(4 downto 0) := "00100";--  5'h04//0_0100
constant INIT_AUTO_REFRESH           : std_logic_vector(4 downto 0) := "00101";--  5'h05//0_0101
constant INIT_AUTO_REFRESH_WAIT      : std_logic_vector(4 downto 0) := "00110";--  5'h06//0_0110
constant INIT_COUNT_200              : std_logic_vector(4 downto 0) := "00111";--  5'h07//0_0111
constant INIT_COUNT_200_WAIT         : std_logic_vector(4 downto 0) := "01000";--  5'h08//0_1000
constant INIT_DUMMY_READ_CYCLES      : std_logic_vector(4 downto 0) := "01001";--  5'h09//0_1001
constant INIT_DUMMY_ACTIVE           : std_logic_vector(4 downto 0) := "01010";--  5'h0A//0_1010
constant INIT_DUMMY_ACTIVE_WAIT      : std_logic_vector(4 downto 0) := "01011";--  5'h0B//0_1011
constant INIT_DUMMY_READ             : std_logic_vector(4 downto 0) := "01100";--  5'h0C//0_1100
constant INIT_DUMMY_READ_WAIT        : std_logic_vector(4 downto 0) := "01101";--  5'h0D//0_1101
constant INIT_DUMMY_FIRST_READ       : std_logic_vector(4 downto 0) := "01110";--  5'h0E//0_1110
constant INIT_DEEP_MEMORY_ST         : std_logic_vector(4 downto 0) := "01111";--  5'h0F//0_1111
constant INIT_ACTIVE                     : std_logic_vector(4 downto 0) := "10000";--  5'h10//1_0000
constant INIT_ACTIVE_WAIT        : std_logic_vector(4 downto 0) := "10001";--  5'h11//1_0001
constant INIT_PATTERN_WRITE          : std_logic_vector(4 downto 0) := "10010";--  5'h12//1_0010
constant INIT_PATTERN_WRITE_READ     : std_logic_vector(4 downto 0) := "10011";--  5'h13//1_0011
constant INIT_PATTERN_READ           : std_logic_vector(4 downto 0) := "10100";--  5'h14//1_0100
constant INIT_PATTERN_READ_WAIT      : std_logic_vector(4 downto 0) := "10101";--  5'h15//1_0101

constant IDLE                    : std_logic_vector(4 downto 0) := "00000";--  5'h00//0_0000
constant LOAD_MODE                       : std_logic_vector(4 downto 0) := "00001";--  5'h01//0_0001
constant MODE_REGISTER_WAIT              : std_logic_vector(4 downto 0) := "00010";--  5'h02//0_0010
constant PRECHARGE                       : std_logic_vector(4 downto 0) := "00011";--  5'h03//0_0011
constant PRECHARGE_WAIT                  : std_logic_vector(4 downto 0) := "00100";--  5'h04//0_0100
constant AUTO_REFRESH                    : std_logic_vector(4 downto 0) := "00101";--  5'h05//0_0101
constant AUTO_REFRESH_WAIT               : std_logic_vector(4 downto 0) := "00110";--  5'h06//0_0110
constant ACTIVE                          : std_logic_vector(4 downto 0) := "00111";--  5'h07//0_0111
constant ACTIVE_WAIT                     : std_logic_vector(4 downto 0) := "01000";--  5'h08//0_1000
constant FIRST_READ                      : std_logic_vector(4 downto 0) := "01001";--  5'h09//0_1001
constant BURST_READ              : std_logic_vector(4 downto 0) := "01010";--  5'h0A//0_1010
constant READ_WAIT                       : std_logic_vector(4 downto 0) := "01011";--  5'h0B//0_1011
constant FIRST_WRITE                     : std_logic_vector(4 downto 0) := "01100";--  5'h0C//0_1100
constant BURST_WRITE                     : std_logic_vector(4 downto 0) := "01101";--  5'h0D//0_1101
constant WRITE_WAIT                      : std_logic_vector(4 downto 0) := "01110";--  5'h0E//0_1110
constant WRITE_READ                      : std_logic_vector(4 downto 0) := "01111";--  5'h0F//0_1111
constant READ_WRITE                      : std_logic_vector(4 downto 0) := "10000";--  5'h10//1_0000

constant cntnext                         : std_logic_vector(5 downto 0) := "101000";

begin
    
    
    REGISTERED_VALUE <= '0';

    CAS_LATENCY_VALUE        <=   load_mode_reg(6 downto 4);
    BURST_LENGTH_VALUE       <=   load_mode_reg(2 downto 0);
    ADDITIVE_LATENCY_VALUE   <=   ext_mode_reg(5 downto 3);
    ODT_ENABLE               <=   ext_mode_reg(2) or ext_mode_reg(6);
    burst_length             <=   burst_cnt;
    command_address          <=   af_addr(34 downto 32);
    zeroes                   <=   (others=> '0');
    
    ECC_VALUE <= '0';

    
   -- fifo control signals

   ctrl_af_RdEn              <=   af_rden;
   conflict_detect          <= af_addr(35) and ctrl_init_done and (not af_empty); 
   cs_width1   <=    (('0' & cs_width0) + "001");

   dummy_write_flag         <= dummy_write_flag_r;

process(clk0)
  begin
  if (clk0='1' and clk0'event) then
     if (rst='1') then
       ctrl_dummy_write  <= '1';
     elsif(init_state = INIT_PATTERN_READ) then
       ctrl_dummy_write  <= '0';
     end if;
end if;
end process;

process(clk0)
  begin
  if (clk0='1' and clk0'event) then
     if (rst='1') then
       COMP_DONE_r        <= '0';
     else
       COMP_DONE_r        <= COMP_DONE;
     end if;
end if;
end process;

ctrl_Wdf_RdEn     <= ctrl_Wdf_RdEn_int  when (ctrl_dummy_write = '0') else '0';

-- commands

  process(command_address,ctrl_init_done,af_empty)
  begin
     WR <= '0';
     RD <= '0';
     LMR <= '0';
     PRE <= '0';
     REF <= '0';
     ACT <= '0';
     if ( ctrl_init_done='1' and af_empty = '0') then

        case  command_address is
          when "000"      =>   LMR<='1';
          when "001"      =>   REF<='1';
          when "010"      =>   PRE<='1';
          when "011"      =>   ACT<='1';
          when "100"      =>   WR<= '1';
          when "101"      =>   RD<= '1';
          when others     =>   null;
        end case;
     end if;
  end process;

-- register address outputs
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
     if (rst='1') then
             WR_r <= '0';
             RD_r <= '0';
             LMR_r <='0';
             PRE_r <='0';
             REF_r <='0';
             ACT_r <='0';
             af_empty_r <= '0';
             LMR_PRE_REF_ACT_cmd_r <= '0';
     else
             WR_r <= WR;
             RD_r <= RD;
             LMR_r <= LMR;
             PRE_r <= PRE;
             REF_r <= REF;
             ACT_r <= ACT;
             af_empty_r <= af_empty;
             LMR_PRE_REF_ACT_cmd_r <= LMR or PRE or REF or ACT;
    end if;
 end if;
end process;

   
-- register address outputs
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
     if (rst='1') then
         af_addr_r          <= (others=>'0');
         af_addr_r1         <= (others=>'0');
         conflict_detect_r  <= '0';
     else
          af_addr_r <= af_addr;
          af_addr_r1 <= af_addr_r;
          conflict_detect_r <= conflict_detect ;
    end if;
  end if;
end process;

process(clk0)
 begin
  if (clk0='1' and clk0'event) then
      if (rst='1') then
          load_mode_reg         <= load_mode_register;
      elsif((state =LOAD_MODE) and (LMR_r='1') and (af_addr_r(((bank_address + row_address + col_ap_width)-1) downto (col_ap_width + row_address))="00")) then
           load_mode_reg <=  af_addr((row_address-1) downto 0);
      end if;
  end if;
end process;


process(clk0)
 begin
  if (clk0='1' and clk0'event) then
      if (rst='1') then
          ext_mode_reg         <= ext_load_mode_register;
       elsif((state = LOAD_MODE) and (LMR_r='1') and (af_addr_r(((bank_address + row_address + col_ap_width)-1) downto (col_ap_width + row_address)) ="01")) then
          ext_mode_reg <=  af_addr((row_address-1) downto 0);
      end if;
 end if;
end process;

-- to initialize memory
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
      if ((rst='1') or (init_state = INIT_DEEP_MEMORY_ST)) then
          init_memory <= '1';
           elsif (init_count_cp ="1111") then
          init_memory <= '0';
      else
          init_memory <= init_memory;
     end if;
 end if;
end process;


-- mrd count
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
       if (rst='1') then
           mrd_count <= '0';
       elsif (state = LOAD_MODE) then
           mrd_count <= mrd_count_value;
       elsif ( mrd_count /= '0') then
           mrd_count <= '0';
       else
           mrd_count <= '0';
       end if;
   end if;
end process;

-- rp count
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
      if (rst='1') then
          rp_count(2 downto 0) <= "000" ;
      elsif (state = PRECHARGE) then
          rp_count(2 downto 0) <= rp_count_value;
      elsif (rp_count(2 downto 0) /= "000") then
          rp_count(2 downto 0) <= rp_count(2 downto 0) - "001";
      else
          rp_count(2 downto 0) <= "000";
      end if;
  end if;
end process;

-- rfc count
 process(clk0)
   begin
    if (clk0='1' and clk0'event) then
      if (rst='1') then
          rfc_count(5 downto 0) <= (others => '0');
      elsif (state = AUTO_REFRESH) then
          rfc_count(5 downto 0) <= rfc_count_value;
      elsif (rfc_count(5 downto 0) /= "000000") then
          rfc_count(5 downto 0) <= rfc_count(5 downto 0) - "000001";
      else
          rfc_count(5 downto 0) <= "000000";
      end if;
   end if;
 end process;

--  rcd count - 20ns
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
     if (rst='1') then
       rcd_count(2 downto 0) <= "000";
     elsif (state = ACTIVE) then
       rcd_count(2 downto 0) <= rcd_count_value - ADDITIVE_LATENCY_VALUE - "001";
     elsif (rcd_count(2 downto 0) /= "000") then
       rcd_count(2 downto 0) <= rcd_count(2 downto 0) - "001";
     else
       rcd_count(2 downto 0) <=  "000";
     end if;
  end if;
end process;


-- ras count - active to precharge
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
      if (rst='1') then
          ras_count(3 downto 0) <= "0000";
      elsif (state = ACTIVE) then
          ras_count(3 downto 0) <= ras_count_value;
      elsif (ras_count(3 downto 1) = "000") then
           if (ras_count(0) /= '0') then
               ras_count(0) <= '0';
           end if;
      else
           ras_count(3 downto 0) <= ras_count(3 downto 0) - "0001";
       end if;
  end if;
end process;

-- AL+BL/2+TRTP-2
-- rtp count - read to precharge
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
      if (rst='1') then
          rtp_count(3 downto 0) <= "0000";
      elsif ((state = FIRST_READ) or (state = BURST_READ)) then
          rtp_count(3 downto 0) <= (('0' & trtp_count_value) + ('0' & burst_cnt) + ('0' & ADDITIVE_LATENCY_VALUE) -"0011") ;
      elsif (rtp_count(3 downto 1) = "000") then
              if (rtp_count(0) /= '0') then
                  rtp_count(0) <= '0';
              end if;
      else
           rtp_count(3 downto 0) <= rtp_count(3 downto 0) - "0001";
      end if;
  end if;
end process;


-- WL+BL/2+TWR
-- wtp count - write to precharge
process(clk0)
 begin
    if (clk0='1' and clk0'event) then
        if (rst='1') then
            wtp_count(3 downto 0) <= "0000";
        elsif ((state = FIRST_WRITE) or (state = BURST_WRITE)) then
            wtp_count(3 downto 0) <= (('0' & twr_count_value) + ('0' & burst_cnt) + ('0' & CAS_LATENCY_VALUE) + ('0' & ADDITIVE_LATENCY_VALUE) -"0011")  ;
        elsif (wtp_count(3 downto 1) = "000") then
            if (wtp_count(0) /= '0') then
                  wtp_count(0) <= '0';
              end if;
        else
            wtp_count(3 downto 0) <= wtp_count(3 downto 0) - "0001";
        end if;
   end if;
end process;

 -- write to read counter
 -- write to read includes : write latency + burst time + tWTR

process(clk0)
 begin
  if (clk0='1' and clk0'event) then
   if (rst='1') then
       wr_to_rd_count(3 downto 0) <= "0000";
   elsif ((state = FIRST_WRITE) or (state = BURST_WRITE)) then
       wr_to_rd_count(3 downto 0) <= (('0' & twtr_count_value) + ('0' & burst_cnt) + ('0' & ADDITIVE_LATENCY_VALUE) + ('0' & CAS_LATENCY_VALUE) - "0010");
   elsif (wr_to_rd_count(3 downto 0) /= "0000") then
       wr_to_rd_count(3 downto 0) <= wr_to_rd_count(3 downto 0) - "0001";
   else
       wr_to_rd_count(3 downto 0) <= "0000";
   end if;
 end if;
end process;


-- read to write counter
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
      if (rst='1') then
          rd_to_wr_count(3 downto 0) <= "0000";
      elsif ((state = FIRST_READ) or (state = BURST_READ)) then
          rd_to_wr_count(3 downto 0) <= ( ('0' & ADDITIVE_LATENCY_VALUE) + ("000" & REGISTERED_VALUE) + ('0' & burst_cnt) );
      elsif (rd_to_wr_count(3 downto 0) /= "0000") then
          rd_to_wr_count(3 downto 0) <= rd_to_wr_count(3 downto 0) - "0001";
      else
          rd_to_wr_count(3 downto 0) <= "0000";
      end if;
  end if;
end process;

 -- auto refresh interval counter in refresh_clk domain
process(refresh_clk)
 begin
   if (refresh_clk='1' and refresh_clk'event) then
       if (rst='1') then
           refi_count <= "00000000";
       elsif (refi_count = max_ref_cnt) then
           refi_count <= "00000000";
       else
           refi_count <= refi_count + "00000001";
       end if;
   end if;
 end process;

process(refi_count, done_200us)
 begin
     if ((refi_count = max_ref_cnt) and (done_200us = '1')) then
         ref_flag <= '1';
     else
         ref_flag <= '0';
     end if;
end process;


-- 200us counter for cke
process(refresh_clk)
 begin
   if (refresh_clk='1' and refresh_clk'event) then
        if (rst='1') then
            cke_200us_cnt <= "11011";
         --   cke_200us_cnt <= "00001";

        elsif (refi_count((max_ref_width-1) downto 0) =  max_ref_cnt) then
            cke_200us_cnt  <=  cke_200us_cnt - "00001";
        else
            cke_200us_cnt  <= cke_200us_cnt;
        end if;
   end if;
 end process;

-- refresh detect in 266 MHz clock
process(clk0)
 begin
   if (clk0='1' and clk0'event) then
   if (rst='1') then
      ref_flag_266   <= '0';
      ref_flag_266_r <= '0';
      done_200us <= '0';
   else
      ref_flag_266   <= ref_flag;
      ref_flag_266_r <= ref_flag_266;
      if (done_200us = '0') then
         if (cke_200us_cnt = "00000")    then
                                         done_200us <=  '1' ;
                        end if;
      end if;
   end if;
   end if;
end process;


-- refresh flag detect
-- auto_ref high indicates auto_refresh requirement
-- auto_ref is held high until auto refresh command is issued.
process(clk0)
 begin
   if (clk0='1' and clk0'event) then
    if (rst='1') then
       auto_ref <= '0';
    elsif (ref_flag_266 ='1' and ref_flag_266_r ='0') then
       auto_ref <= '1';
    elsif (state = AUTO_REFRESH) then
       auto_ref <= '0';
    else
       auto_ref <= auto_ref;
    end if;
   end if;
 end process;

-- 200 clocks counter - count value : C8
-- required for initialization
process(clk0)
 begin
   if (clk0='1' and clk0'event) then
     if (rst='1') then
         count_200_cycle(7 downto 0) <= "00000000";
     elsif (init_state = INIT_COUNT_200) then
         count_200_cycle(7 downto 0) <=  X"C8" ;
     elsif (count_200_cycle(7 downto 0) /= "00000000") then
         count_200_cycle(7 downto 0)<= count_200_cycle(7 downto 0) - "00000001";
     else
         count_200_cycle(7 downto 0) <= "00000000";
     end if;
  end if;
 end process;

process(clk0)
 begin
   if (clk0='1' and clk0'event) then
       if (rst='1') then
           count_200cycle_done_r<= '0';
       elsif ((init_memory='1') and (count_200_cycle = "00000000")) then
           count_200cycle_done_r<= '1';
       else
           count_200cycle_done_r<= '0';
       end if;
   end if;
end process;


process(clk0)
 begin
   if (clk0='1' and clk0'event) then
       if (rst='1') then
           init_done <= '0';
       elsif ((init_count_cp = "1101") and (count_200cycle_done_r ='1') and (Phy_Mode = '0'))then
           init_done <= '1';
       elsif ((Phy_Mode = '1') and (COMP_DONE_r ='1') and (count6 = "010100")) then --  Precharge fix after pattern read
           init_done <= '1';
       else
           init_done <= init_done;
       end if;
   end if;
   end process;

    ctrl_init_done      <= init_done;


process( BURST_LENGTH_VALUE)
begin
  if (BURST_LENGTH_VALUE = "010") then
       burst_cnt <="010";
  elsif (BURST_LENGTH_VALUE = "011") then
       burst_cnt <="100";
  else
       burst_cnt <="000";
  end if;
end process;

process(clk0)
 begin
   if (clk0='1' and clk0'event) then
       if ((rst='1') or (init_state = INIT_DEEP_MEMORY_ST))then
            init_count(3 downto 0) <= "0000";
       elsif (init_memory='1') then
             if (init_state = INIT_LOAD_MODE or init_state = INIT_PRECHARGE or init_state = INIT_AUTO_REFRESH or
                 init_state = INIT_DUMMY_READ_CYCLES or init_state = INIT_COUNT_200 or init_state = INIT_DEEP_MEMORY_ST
                 or init_state = INIT_PATTERN_WRITE) then
                 init_count(3 downto 0) <= init_count(3 downto 0) + "0001";
             elsif(init_count = "1111") then
                 init_count(3 downto 0) <= "0000";
               else
                 init_count(3 downto 0) <= init_count(3 downto 0);
             end if;
       end if;
   end if;
 end process;


process(clk0)
 begin
    if (clk0='1' and clk0'event) then
              if ((rst='1') or  (init_state = INIT_DEEP_MEMORY_ST)) then
                   init_count_cp(3 downto 0) <= "0000";
              elsif (init_memory='1') then
                 if (init_state = INIT_LOAD_MODE or init_state = INIT_PRECHARGE or init_state = INIT_AUTO_REFRESH or
                     init_state = INIT_DUMMY_READ_CYCLES or init_state = INIT_COUNT_200 or init_state = INIT_DEEP_MEMORY_ST
                     or init_state = INIT_PATTERN_WRITE) then
                     init_count_cp(3 downto 0) <= init_count_cp(3 downto 0) + "0001";
                 elsif(init_count_cp = "1111") then
                     init_count_cp(3 downto 0) <= "0000";
                 else
                     init_count_cp(3 downto 0) <= init_count_cp(3 downto 0);
                end if;
              end if;
    end if;
end process;


process(clk0)
 begin
    if (clk0='1' and clk0'event) then
        if (rst='1') then
            chip_cnt <= "00";
        elsif ( init_state = INIT_DEEP_MEMORY_ST) then
            chip_cnt <= chip_cnt + "01";
        else
            chip_cnt <= chip_cnt;
        end if;
    end if;
end process;

 process(clk0)
 begin
    if (clk0='1' and clk0'event) then
       if (rst='1' or state = PRECHARGE) then
           auto_cnt <= "000";
      elsif ( state = AUTO_REFRESH and init_memory = '0') then
          auto_cnt <= auto_cnt + "001";
      else
         auto_cnt <= auto_cnt;
      end if;
    end if;
end process;

---  Precharge fix for deep memory
 process(clk0)
 begin
    if (clk0='1' and clk0'event) then
       if (rst='1' or state = auto_refresh) then
           pre_cnt <= "000";
      elsif ( state = PRECHARGE and init_memory = '0' and auto_ref = '1') then
          pre_cnt <= pre_cnt + "001";
      else
         pre_cnt <= pre_cnt;
      end if;
    end if;
end process;

-- write burst count
process(clk0)
 begin
    if (clk0='1' and clk0'event) then
        if (rst='1') then
            wrburst_cnt(2 downto 0) <= "000";
        elsif (state = FIRST_WRITE or state = BURST_WRITE or init_state = INIT_PATTERN_WRITE) then
            wrburst_cnt(2 downto 0) <= burst_cnt(2 downto 0);
        elsif (wrburst_cnt(2 downto 0) /= "000") then
            wrburst_cnt(2 downto 0) <= wrburst_cnt(2 downto 0) - "001";
        else
            wrburst_cnt(2 downto 0) <= "000";
        end if;
    end if;
end process;

-- read burst count for state machine
process(clk0)
 begin
    if (clk0='1' and clk0'event) then
        if (rst='1') then
           read_burst_cnt(2 downto 0) <= "000";
        elsif (state = FIRST_READ or state = BURST_READ) then
           read_burst_cnt(2 downto 0) <= burst_cnt(2 downto 0);
        elsif (read_burst_cnt(2 downto 0) /= "000") then
           read_burst_cnt(2 downto 0) <= read_burst_cnt(2 downto 0) - "001";
        else
           read_burst_cnt(2 downto 0) <= "000";
        end if;
    end if;
end process;

-- count to generate write enable to the data path
process(clk0)
 begin
    if (clk0='1' and clk0'event) then
        if (rst='1') then
            ctrl_WrEn_cnt(2 downto 0) <= "000";
        elsif (wdf_rden_r='1' or dummy_write_state_r='1') then
            ctrl_WrEn_cnt(2 downto 0) <= burst_cnt(2 downto 0);
        elsif (ctrl_WrEn_cnt(2 downto 0) /= "000") then
            ctrl_WrEn_cnt(2 downto 0) <= ctrl_WrEn_cnt(2 downto 0)- "001";
        else
            ctrl_WrEn_cnt(2 downto 0) <= "000";
        end if;
    end if;
 end process;

-- write enable to data path
process(ctrl_WrEn_cnt)
 begin
       if (ctrl_WrEn_cnt(2 downto 0) /= "000") then
            ctrl_write_en <= '1';
        else
            ctrl_write_en <= '0';
        end if;
end process;

-- write enable to data path
s1_h<=(('0' & ADDITIVE_LATENCY_VALUE) + ('0' & CAS_LATENCY_VALUE) + ("000" & REGISTERED_VALUE) + ("000" & ECC_VALUE)); -- internal signal to calculate the value of expression used in CASE
process(clk0)
 begin

 if (clk0='1' and clk0'event) then
   if (rst='1') then
      ctrl_WrEn_r1       <= '0';
      ctrl_write_en_r1<= '0';
      ctrl_write_en_r2<= '0';
      ctrl_write_en_r3<= '0';
      ctrl_write_en_r4<= '0';
      ctrl_write_en_r5<= '0';
      ctrl_write_en_r6<= '0';
   else
      ctrl_write_en_r1<= ctrl_write_en;
      ctrl_write_en_r2<= ctrl_write_en_r1;
      ctrl_write_en_r3<= ctrl_write_en_r2;
      ctrl_write_en_r4<= ctrl_write_en_r3;
      ctrl_write_en_r5<= ctrl_write_en_r4;
      ctrl_write_en_r6<= ctrl_write_en_r5;

      case  s1_h is
                  when  "0011" => ctrl_WrEn_r1 <= ctrl_write_en;
             when  "0100" => ctrl_WrEn_r1 <= ctrl_write_en_r1;
             when  "0101" => ctrl_WrEn_r1 <= ctrl_write_en_r2;
             when  "0110" => ctrl_WrEn_r1 <= ctrl_write_en_r3;
             when  "0111" => ctrl_WrEn_r1 <= ctrl_write_en_r4;
             when  "1000" => ctrl_WrEn_r1 <= ctrl_write_en_r5;
             when  "1001" => ctrl_WrEn_r1 <= ctrl_write_en_r6;
             when  others => ctrl_WrEn_r1 <= '0';
      end case ; -- case(ADDITIVE_LATENCY_VALUE + CAS_LATENCY_VALUE )

   end if;
end if;
end process;

-- DQS enable to data path

process(state, init_state)
 begin
   if (state = FIRST_WRITE or init_state = INIT_PATTERN_WRITE) then
       dqs_reset <= '1';
   else
       dqs_reset <= '0';
   end if;
end process;

s2_h<=(('0' & ADDITIVE_LATENCY_VALUE) + ('0' & CAS_LATENCY_VALUE)+("000" & REGISTERED_VALUE)+("000" & ECC_VALUE));

process(clk0)
 begin

 if (clk0='1' and clk0'event) then
     if (rst='1') then
      ctrl_Dqs_Rst_r1 <= '0';
      dqs_reset_r1<=  '0';
      dqs_reset_r2<=  '0';
      dqs_reset_r3<=  '0';
      dqs_reset_r4<=  '0';
      dqs_reset_r5<=  '0';
      dqs_reset_r6<=  '0';

    else
      dqs_reset_r1 <= dqs_reset;
      dqs_reset_r2 <= dqs_reset_r1;
      dqs_reset_r3 <= dqs_reset_r2;
      dqs_reset_r4 <= dqs_reset_r3;
      dqs_reset_r5 <= dqs_reset_r4;
      dqs_reset_r6 <= dqs_reset_r5;

      case s2_h is

        when "0011" => ctrl_Dqs_Rst_r1 <= dqs_reset;
        when "0100" => ctrl_Dqs_Rst_r1 <= dqs_reset_r1;
        when "0101" => ctrl_Dqs_Rst_r1 <= dqs_reset_r2;
      when "0110" => ctrl_Dqs_Rst_r1 <= dqs_reset_r3;
        when "0111" => ctrl_Dqs_Rst_r1 <= dqs_reset_r4;
        when "1000" => ctrl_Dqs_Rst_r1 <= dqs_reset_r5;
        when "1001" => ctrl_Dqs_Rst_r1 <= dqs_reset_r6;
        when others => ctrl_Dqs_Rst_r1 <= '0';
      end case;   -- case(ADDITIVE_LATENCY_VALUE + CAS_LATENCY_VALUE )
   end  if;
end if;
end process;

process(state , init_state, wrburst_cnt)
  begin
    if ((state = FIRST_WRITE) or (state = BURST_WRITE) or (init_state = INIT_PATTERN_WRITE) or (wrburst_cnt /= "000")) then
     dqs_en <= '1';
    else
     dqs_en <= '0';
    end if;
end process;

 s3_h<=(('0' & ADDITIVE_LATENCY_VALUE)+('0' & CAS_LATENCY_VALUE)+("000" & REGISTERED_VALUE)+("000" & ECC_VALUE));
process(clk0)
 begin

 if (clk0='1' and clk0'event) then
   if (rst='1') then
      ctrl_Dqs_En_r1 <= '0';
      dqs_en_r1   <= '0';
      dqs_en_r2   <= '0';
      dqs_en_r3   <= '0';
      dqs_en_r4   <= '0';
      dqs_en_r5   <= '0';
      dqs_en_r6   <= '0';

  else
      dqs_en_r1 <= dqs_en;
      dqs_en_r2 <= dqs_en_r1;
      dqs_en_r3 <= dqs_en_r2;
      dqs_en_r4 <= dqs_en_r3;
      dqs_en_r5 <= dqs_en_r4;
      dqs_en_r6 <= dqs_en_r5;

      case s3_h is

        when "0011" => ctrl_Dqs_En_r1 <= dqs_en;
        when "0100" => ctrl_Dqs_En_r1 <= dqs_en_r1;
        when "0101" => ctrl_Dqs_En_r1 <= dqs_en_r2;
        when "0110" => ctrl_Dqs_En_r1 <= dqs_en_r3;
        when "0111" =>ctrl_Dqs_En_r1 <= dqs_en_r4;
        when "1000" => ctrl_Dqs_En_r1 <= dqs_en_r5;
        when "1001" => ctrl_Dqs_En_r1 <= dqs_en_r6;
        when others => ctrl_Dqs_En_r1 <= '0';
      end case ;
   end if;
end if;
end process;


-- cas count
process(clk0)
 begin
 if (clk0='1' and clk0'event) then
     if (rst='1') then
         cas_count(2 downto 0) <= "000";
     elsif (init_state = INIT_DUMMY_FIRST_READ) then
         cas_count(2 downto 0) <= CAS_LATENCY_VALUE + ("00" & REGISTERED_VALUE);
     elsif (cas_count(2 downto 0) /= "000") then
         cas_count(2 downto 0) <= cas_count(2 downto 0) - "001";
     else
         cas_count(2 downto 0)<= "000";
     end if;
 end if;
end process;

 -- dummy_read enable
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
      if (rst='1') then
          dummy_read_en <= '0';
      elsif (init_state = INIT_DUMMY_READ) then
          dummy_read_en <= '1';
      elsif (phy_Dly_Slct_Done = '1') then
          dummy_read_en <= '0';
      else
          dummy_read_en <= dummy_read_en;
      end if;
  end if;
end process;


-- ctrl_Dummyread_Start signal generation to the data path module
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
      if (rst='1') then
          ctrl_Dummyread_Start_r1 <= '0';
      elsif ((dummy_read_en = '1') and (cas_count = "000")) then
          ctrl_Dummyread_Start_r1 <= '1';
      elsif (phy_Dly_Slct_Done = '1') then
          ctrl_Dummyread_Start_r1 <= '0';
      else
          ctrl_Dummyread_Start_r1 <= ctrl_Dummyread_Start_r1;
      end if;
  end if;
end process;

-- register ctrl_Dummyread_Start signal
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
      if (rst='1') then
          ctrl_Dummyread_Start_r2 <= '0';
          ctrl_Dummyread_Start_r3 <= '0';
          ctrl_Dummyread_Start_r4 <= '0';
          ctrl_Dummyread_Start    <= '0';
      else
          ctrl_Dummyread_Start_r2 <=  ctrl_Dummyread_Start_r1;
          ctrl_Dummyread_Start_r3 <=  ctrl_Dummyread_Start_r2;
          ctrl_Dummyread_Start_r4 <=  ctrl_Dummyread_Start_r3;
          ctrl_Dummyread_Start    <=  ctrl_Dummyread_Start_r4;
      end if;
   end if;
end process;

-- read_wait/write_wait to idle count
-- the state machine waits for 15 clock cycles in the write wait state for any wr/rd commands
-- to be issued. If no commands are issued in 15 clock cycles, the statemachine issues
-- enters the idle state and stays in the idle state until an auto refresh is required.

process(clk0)
 begin
  if (clk0='1' and clk0'event) then
   if (rst='1') then
       idle_cnt(3 downto 0) <= "0000";
   elsif (state = FIRST_READ or state = FIRST_WRITE or state = BURST_READ or  state = BURST_WRITE) then
       idle_cnt(3 downto 0) <= "1111" ;
   elsif (idle_cnt(3 downto 0) /= "0000") then
       idle_cnt(3 downto 0)<= idle_cnt(3 downto 0) - "0001";
   else
       idle_cnt(3 downto 0)<= "0000";
   end if;
 end if;
end process;


process(clk0)
 begin
  if (clk0='1' and clk0'event) then
     if (rst='1') then
          cas_check_count(3 downto 0) <= "0000";
     elsif ((state_r2 = FIRST_READ) or (init_state_r2 = INIT_PATTERN_READ))   then
          cas_check_count(3 downto 0) <= (('0' & CAS_LATENCY_VALUE) - "0001");
     elsif (cas_check_count(3 downto 0) /= "0000") then
          cas_check_count(3 downto 0) <= cas_check_count(3 downto 0) - "0001";
     else
          cas_check_count(3 downto 0) <= "0000";
     end if;
 end if;
end process;

process(clk0)
begin
 if (clk0='1' and clk0'event) then
     if (rst='1') then
       rdburst_cnt(3 downto 0) <= "0000";
     elsif (cas_check_count = "0010") then
          rdburst_cnt(3 downto 0) <= ( '0' & burst_cnt(2 downto 0));
     elsif(state_r3 = BURST_READ) then
        if(burst_cnt(2) ='1') then
            rdburst_cnt(3 downto 0) <= ((burst_cnt(2 downto 0) & '0' )  -("0111" - ('0' & CAS_LATENCY_VALUE)));
        else
           rdburst_cnt(3 downto 0)<= ((burst_cnt(2 downto 0) & '0' ) -("0101" - ('0' & CAS_LATENCY_VALUE)));
        end if;
     elsif (rdburst_cnt(3 downto 0) /= "0000")  then
          rdburst_cnt(3 downto 0)<= rdburst_cnt(3 downto 0) - "0001";
     else
         rdburst_cnt(3 downto 0) <= "0000";
     end if;
end if;
end process;
-- read enable to data path
process(rdburst_cnt)
begin
   if (rdburst_cnt = "0000") then
      ctrl_read_en <= '0';
   else
      ctrl_read_en <= '1';
   end if;
end process;

process(clk0)
 begin
  if (clk0='1' and clk0'event) then
   if (rst='1') then
      ctrl_read_en_r  <= '0';
      ctrl_read_en_r1 <= '0';
      ctrl_read_en_r2 <= '0';
      ctrl_read_en_r3 <= '0';
      ctrl_read_en_r4 <= '0';

   else
      ctrl_read_en_r  <= ctrl_read_en;
      ctrl_read_en_r1 <= ctrl_read_en_r;
      ctrl_read_en_r2 <= ctrl_read_en_r1;
      ctrl_read_en_r3 <= ctrl_read_en_r2;
      ctrl_read_en_r4 <= ctrl_read_en_r3;

   end if;
end if;
end process;

s4_h<=(ADDITIVE_LATENCY_VALUE + ("00" & ECC_VALUE) + ("00" & REGISTERED_VALUE));

process(clk0)
 begin

  if (clk0='1' and clk0'event) then
   if (rst='1') then
      ctrl_RdEn_r1  <= '0';
   else
     case s4_h is
      when  "000" => ctrl_RdEn_r1  <= ctrl_read_en;
      when  "001" => ctrl_RdEn_r1  <= ctrl_read_en_r;
      when  "010" => ctrl_RdEn_r1  <= ctrl_read_en_r1;
      when  "011" => ctrl_RdEn_r1  <= ctrl_read_en_r2;
      when  "100" => ctrl_RdEn_r1  <= ctrl_read_en_r3;
      when  others => ctrl_RdEn_r1 <= '0';
     end case ; -- case(ADDITIVE_LATENCY_VALUE)
   end if; -- else: !if(rst)
end if;
end process;

--**********************************************************************************************************************************************************//
--  Code Changed
--**********************************************************************************************************************************************************//
-- write address FIFO read enable signals


   af_rden  <= '1' when ((state = FIRST_WRITE) or (state = FIRST_READ) or (state = BURST_WRITE) or (state = BURST_READ) or
       ((state = MODE_REGISTER_WAIT) and (LMR_r='1') and (mrd_count='0')) or ((state = PRECHARGE )and (PRE_r='1')) or
      ((state = AUTO_REFRESH) and (REF_r='1')) or ((state = ACTIVE ) and  (ACT_r='1'))) else '0';


-- write data fifo read enable
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
   if (rst = '1') then
       wdf_rden_r  <= '0';
   elsif ((state = FIRST_WRITE) or (state =BURST_WRITE) or (init_state = INIT_PATTERN_WRITE))   then -- place holder for BURST_WRITE
       wdf_rden_r  <= '1';
   else
       wdf_rden_r  <= '0';
   end if;
  end if;
 end process;

process(clk0)
 begin
  if (clk0='1' and clk0'event) then
   if (rst ='1') then
       wdf_rden_r2 <= '0';
       wdf_rden_r3 <= '0';
       wdf_rden_r4 <= '0';
   else
       wdf_rden_r2 <= wdf_rden_r;
       wdf_rden_r3 <= wdf_rden_r2;
       wdf_rden_r4 <= wdf_rden_r3;
   end if; --else: !if(rst)
  end if;
 end process;

-- Read enable to the data fifo

process(burst_cnt, wdf_rden_r, wdf_rden_r2, wdf_rden_r3, wdf_rden_r4)
  begin
   if (burst_cnt = "010") then
       ctrl_wdf_read_en<= (wdf_rden_r or wdf_rden_r2) ;
   elsif (burst_cnt = "100") then
       ctrl_wdf_read_en<= (wdf_rden_r or wdf_rden_r2 or wdf_rden_r3 or wdf_rden_r4) ;
   else
       ctrl_wdf_read_en<= '0';
   end if;
end process;
s5_h<=(('0' & ADDITIVE_LATENCY_VALUE)+('0' & CAS_LATENCY_VALUE)+("000" & REGISTERED_VALUE));
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
   if (rst='1') then
       ctrl_Wdf_RdEn_int <= '0';
       ctrl_wdf_read_en_r1<= '0';
       ctrl_wdf_read_en_r2<= '0';
       ctrl_wdf_read_en_r3<= '0';
       ctrl_wdf_read_en_r4<= '0';
       ctrl_wdf_read_en_r5<= '0';
       ctrl_wdf_read_en_r6<= '0';
  else
       ctrl_wdf_read_en_r1<= ctrl_wdf_read_en;
       ctrl_wdf_read_en_r2<= ctrl_wdf_read_en_r1;
       ctrl_wdf_read_en_r3<= ctrl_wdf_read_en_r2;
       ctrl_wdf_read_en_r4<= ctrl_wdf_read_en_r3;
       ctrl_wdf_read_en_r5<= ctrl_wdf_read_en_r4;
       ctrl_wdf_read_en_r6<= ctrl_wdf_read_en_r5;
 case s5_h is
       when   "0011" => ctrl_Wdf_RdEn_int  <= ctrl_wdf_read_en;
       when   "0100" => ctrl_Wdf_RdEn_int  <= ctrl_wdf_read_en_r1;
       when   "0101" => ctrl_Wdf_RdEn_int  <= ctrl_wdf_read_en_r2;
       when   "0110" => ctrl_Wdf_RdEn_int  <= ctrl_wdf_read_en_r3;
       when   "0111" => ctrl_Wdf_RdEn_int  <= ctrl_wdf_read_en_r4;
       when   "1000" => ctrl_Wdf_RdEn_int  <= ctrl_wdf_read_en_r5;
       when   "1001" => ctrl_Wdf_RdEn_int  <= ctrl_wdf_read_en_r6;
       when   others => ctrl_Wdf_RdEn_int  <= '0';
 end case ;
 end if;-- else: !if(rst)
 end if;
end process;

process(clk0)
 begin
  if (clk0='1' and clk0'event) then
      if (rst='1') then
        count6 <= "000000";
      else
         case init_state is

            when INIT_PRECHARGE_WAIT | INIT_MODE_REGISTER_WAIT | INIT_AUTO_REFRESH_WAIT |
            INIT_ACTIVE_WAIT | INIT_PATTERN_WRITE_READ | INIT_PATTERN_READ_WAIT |
            INIT_DUMMY_READ_WAIT | INIT_DUMMY_ACTIVE_WAIT =>
                count6 <= count6 + "000001";

            when others =>  count6 <= "000000";
         end case;
      end if;
  end if;
end process;

process(clk0)
 begin
  if (clk0='1' and clk0'event) then
      if (rst='1') then
        init_state <= INIT_IDLE;
      else
        init_state <= init_next_state;
      end if;
  end if;
end process;

process(clk0)
 begin
  if (clk0='1' and clk0'event) then
      if (rst='1') then
        state <= IDLE;
      else
        state <= next_state;
      end if;
  end if;
end process;

process(clk0)
 begin
  if (clk0='1' and clk0'event) then
      if (rst='1') then
        dummy_write_flag_r <= '0';
      elsif(phy_Dly_Slct_Done = '1' and  COMP_DONE_r = '0') then
        dummy_write_flag_r <= '1';
      elsif(COMP_DONE_r = '1') then
        dummy_write_flag_r <= '0';
      end if;
  end if;
end process;

dummy_write_state <= '1' when (init_state = INIT_PATTERN_WRITE) else '0';

process(clk0)
 begin
  if (clk0='1' and clk0'event) then
      if (rst='1') then
       dummy_write_state_r <= '0';
       dummy_write_state_r1 <= '0';
      else
       dummy_write_state_r <= dummy_write_state;
       dummy_write_state_r1 <= dummy_write_state_r;
      end if;
  end if;
end process;

cs_width0 <= "00"  when  cs_width = 1 else
               "01"  when  cs_width = 2 else
               "10"  when  cs_width = 3 else
               "11" when  cs_width = 4 ;


-- init state machine
process(RD, RD_r, WR, WR_r, LMR_r, PRE_r, ACT_r, REF_r,
          auto_ref, chip_cnt,auto_cnt, conflict_detect, conflict_detect_r,
          conflict_resolved_r, count_200cycle_done_r, idle_cnt,
          init_count, init_memory, mrd_count, LMR_PRE_REF_ACT_cmd_r,
          phy_Dly_Slct_Done, ras_count, rcd_count, rd_to_wr_count,
          read_burst_cnt, rfc_count, rp_count, rtp_count,
          init_state, wr_to_rd_count, wrburst_cnt, wtp_count, done_200us,
          cs_width0, dummy_write_flag_r, COMP_DONE_r, count6)
begin
   init_next_state <= init_state;
      case init_state is
      when INIT_IDLE =>
        if (init_memory='1' and  done_200us = '1') then
          case init_count is -- synthesis parallel_case full_case
            when "0000" =>
                init_next_state <= INIT_COUNT_200;

            when "0001" =>
                if (count_200cycle_done_r = '1') then
                  init_next_state <= INIT_PRECHARGE;
                else
                  init_next_state <= INIT_IDLE;
                end if;

            when "0010" =>
                  init_next_state <= INIT_LOAD_MODE; -- emr(2)

            when "0011" =>
                  init_next_state <= INIT_LOAD_MODE; -- emr(3)

            when "0100" =>
                  init_next_state <= INIT_LOAD_MODE; -- emr

            when "0101" =>
                  init_next_state <= INIT_LOAD_MODE; --lmr

            when "0110" =>
                  init_next_state <= INIT_PRECHARGE;

            when "0111" =>
                  init_next_state <= INIT_AUTO_REFRESH;

            when "1000" =>
                  init_next_state <= INIT_AUTO_REFRESH;

            when "1001" =>
                  init_next_state <= INIT_LOAD_MODE;

            when "1010" =>                                   -- EMR OCD DEFAULT
                  init_next_state <= INIT_LOAD_MODE;

            when "1011" =>               -- EMR OCD EXIT
                  init_next_state <= INIT_LOAD_MODE;

            when "1100" =>
                  init_next_state <= INIT_COUNT_200;

            when "1101" =>
                if (chip_cnt < cs_width0)  then
             -- if (chip_cnt < (cs_width- 1))  then
                  init_next_state <= INIT_DEEP_MEMORY_ST;
                    elsif (Phy_Mode ='1'  and count_200cycle_done_r = '1')  then
                      init_next_state <= INIT_DUMMY_READ_CYCLES;
                    else
                      init_next_state <= INIT_IDLE;
                  end if;

            when "1110" =>
                if (phy_Dly_Slct_Done = '1') then
                   init_next_state <= INIT_PRECHARGE;
                else
                   init_next_state <= INIT_IDLE;
                end if;


            when "1111" =>
                if (COMP_DONE_r = '1') then
                   init_next_state <= INIT_IDLE;
                end if;

            when others =>    init_next_state <= INIT_IDLE;

          end case;  -- case(init_count)


        end if; -- case: INIT_IDLE

      when INIT_DEEP_MEMORY_ST    =>
           init_next_state <= INIT_IDLE;

      when INIT_COUNT_200   =>
           init_next_state <= INIT_COUNT_200_WAIT;

      when INIT_COUNT_200_WAIT =>
           if (count_200cycle_done_r ='1') then
              init_next_state <= INIT_IDLE;
           else
              init_next_state <= INIT_COUNT_200_WAIT;
           end if;

      when INIT_DUMMY_READ_CYCLES =>
           init_next_state <= INIT_DUMMY_ACTIVE ;

      when INIT_DUMMY_ACTIVE       =>
           init_next_state <= INIT_DUMMY_ACTIVE_WAIT;

      when INIT_DUMMY_ACTIVE_WAIT =>
           if (count6 = cntnext) then
              init_next_state <= INIT_DUMMY_FIRST_READ;
           else
              init_next_state <= INIT_DUMMY_ACTIVE_WAIT;
           end if;

      when INIT_DUMMY_FIRST_READ =>
           init_next_state <= INIT_DUMMY_READ_WAIT;

      when INIT_DUMMY_READ =>
           init_next_state <= INIT_DUMMY_READ_WAIT;

      when INIT_DUMMY_READ_WAIT =>
           if (phy_Dly_Slct_Done ='1') then
             if(count6 = cntnext) then
                init_next_state <= INIT_PATTERN_WRITE;
             else
                init_next_state <= INIT_DUMMY_READ_WAIT;
             end if;
           else
             init_next_state <= INIT_DUMMY_READ;
           end if;

     -- pattern write
      when INIT_PRECHARGE   =>
           init_next_state <= INIT_PRECHARGE_WAIT;

      when INIT_PRECHARGE_WAIT   =>
           if (count6 = cntnext) then
              init_next_state <= INIT_IDLE;
             else
              init_next_state <= INIT_PRECHARGE_WAIT;
           end if;

      when INIT_LOAD_MODE    =>
           init_next_state <= INIT_MODE_REGISTER_WAIT;

      when INIT_MODE_REGISTER_WAIT =>
           if (count6 = cntnext) then
              init_next_state <= INIT_IDLE;
           else
              init_next_state <= INIT_MODE_REGISTER_WAIT;
           end if;

      when INIT_AUTO_REFRESH   =>
           init_next_state <= INIT_AUTO_REFRESH_WAIT;

      when INIT_AUTO_REFRESH_WAIT  =>
           if (count6 = cntnext) then
              init_next_state <= INIT_IDLE;
           else
              init_next_state <= INIT_AUTO_REFRESH_WAIT;
           end if;

      when INIT_ACTIVE     =>
           init_next_state <= INIT_ACTIVE_WAIT;

      when INIT_ACTIVE_WAIT    =>         -- first active or when a new row is opened
           if (count6 = cntnext) then
              init_next_state <= INIT_IDLE;
             else
                init_next_state <= INIT_ACTIVE_WAIT;
           end if;

      when INIT_PATTERN_WRITE   =>
           init_next_state <= INIT_PATTERN_WRITE_READ;

      when INIT_PATTERN_WRITE_READ  =>
           if (count6 = cntnext) then
              init_next_state <= INIT_PATTERN_READ;
           else
              init_next_state <= INIT_PATTERN_WRITE_READ;
           end if;

      when INIT_PATTERN_READ =>
           init_next_state <= INIT_PATTERN_READ_WAIT;

      when INIT_PATTERN_READ_WAIT =>
           if (COMP_DONE_r = '1') then
              init_next_state <= INIT_PRECHARGE; --  Precharge fix after pattern read
           else
              init_next_state <= INIT_PATTERN_READ_WAIT;
           end if;

      when others =>  init_next_state <= INIT_IDLE;

      end case;
end  process;

-- main control state machine
process(RD, RD_r, WR, WR_r, LMR_r, PRE_r, ACT_r, REF_r,
          auto_ref, chip_cnt,auto_cnt, conflict_detect, conflict_detect_r,
          conflict_resolved_r, count_200cycle_done_r, idle_cnt,
          init_count, init_memory, mrd_count, LMR_PRE_REF_ACT_cmd_r,
          phy_Dly_Slct_Done, ras_count, rcd_count, rd_to_wr_count,
          read_burst_cnt, rfc_count, rp_count, rtp_count, pre_cnt, ---  Precharge fix for deep memory
          state, wr_to_rd_count, wrburst_cnt, wtp_count, done_200us,
          cs_width1, dummy_write_flag_r, COMP_DONE_r, init_done, af_empty_r)
begin
   next_state <= state;
      case state is
      when IDLE =>
             if ((conflict_detect_r='1' or LMR_PRE_REF_ACT_cmd_r='1' or auto_ref='1') and ras_count = "0000" and init_done = '1')  then
              next_state <= PRECHARGE;
             elsif ((WR_r='1'  or RD_r='1'  ) and (ras_count = "0000")) then
              next_state <= ACTIVE;
             end if;

      when PRECHARGE =>
           next_state <= PRECHARGE_WAIT;

---  Precharge fix for deep memory
     when  PRECHARGE_WAIT  =>
          if (rp_count ="000") then
              if (auto_ref='1' or REF_r='1') then
                  if ( (pre_cnt < cs_width1) and init_memory = '0')then
                     next_state <= PRECHARGE;
                  else
                     next_state <= AUTO_REFRESH;
                  end if;
              elsif (LMR_r='1') then
                  next_state <= LOAD_MODE;
              elsif (conflict_detect_r='1' or ACT_r='1') then
                  next_state <= ACTIVE;
              else
                   next_state <= IDLE;
              end if;
         else
              next_state <= PRECHARGE_WAIT;
        end if;

      when LOAD_MODE=>
           next_state <= MODE_REGISTER_WAIT;

      when MODE_REGISTER_WAIT =>
           if (mrd_count = '0') then
              next_state <= IDLE;
           else
              next_state <= MODE_REGISTER_WAIT;
           end if;

      when AUTO_REFRESH =>
           next_state <= AUTO_REFRESH_WAIT;

      when AUTO_REFRESH_WAIT =>
           if ( (auto_cnt < cs_width1) and rfc_count = "000001" and init_memory = '0')then
               next_state <= AUTO_REFRESH;
           elsif ((rfc_count = "000001") and (conflict_detect_r ='1')) then
               next_state <= ACTIVE;
           elsif (rfc_count = "000001") then
               next_state <= IDLE;
           else
               next_state <= AUTO_REFRESH_WAIT;
           end if;

      when ACTIVE =>
           next_state <= ACTIVE_WAIT;

      when ACTIVE_WAIT =>
           if (rcd_count = "000") then    -- first active or when a new row is opened
               if (WR ='1') then
               next_state <= FIRST_WRITE;
             elsif (RD ='1') then
               next_state <= FIRST_READ;
             else
             next_state <= IDLE;
             end if;
             else
               next_state <= ACTIVE_WAIT;
           end if;

      when FIRST_WRITE =>
           next_state <= WRITE_WAIT;

      when BURST_WRITE  =>
           next_state <= WRITE_WAIT;

      when WRITE_WAIT =>
           if (((conflict_detect = '1') and (conflict_resolved_r = '0')) or (auto_ref = '1')) then
               if ((wtp_count = "0000") and (ras_count = "0000")) then
               next_state <= PRECHARGE;
             else
               next_state <= WRITE_WAIT;
             end if;
           elsif (RD = '1' ) then
             next_state <= WRITE_READ;
           elsif ((WR = '1') and (wrburst_cnt = "010")) then
             next_state <= BURST_WRITE;
           elsif ((WR = '1') and (wrburst_cnt = "000")) then --added to improve the efficiency
	     next_state <= FIRST_WRITE;
           elsif (idle_cnt = "0000") then
             next_state <= PRECHARGE;
           else
             next_state <= WRITE_WAIT;
           end if;

      when WRITE_READ =>
           if (wr_to_rd_count = "0000") then
             next_state <= FIRST_READ;
           else
             next_state <= WRITE_READ;
           end if;

      when FIRST_READ =>
           next_state <= READ_WAIT;

      when BURST_READ =>
           next_state <= READ_WAIT;

      when READ_WAIT =>
           if (((conflict_detect='1') and  (conflict_resolved_r='0')) or (auto_ref='1')) then
              if(rtp_count = "0000" and  ras_count = "0000") then
                next_state <= PRECHARGE;
              else
                next_state <= READ_WAIT;
              end if;
           elsif (WR='1')then
              next_state <= READ_WRITE;
           elsif ((RD='1') and (read_burst_cnt <="010")) then
              if(af_empty_r = '1') then
                next_state <= FIRST_READ;
              else
                next_state <= BURST_READ;
              end if;
           elsif (idle_cnt = "0000") then
             next_state <= PRECHARGE;
           else
             next_state <= READ_WAIT;
           end if;

      when READ_WRITE  =>
           if (rd_to_wr_count = "0000") then
             next_state <= FIRST_WRITE;
           else
             next_state <= READ_WRITE;
           end if;

      when others =>  next_state <= IDLE;

      end case;
end  process;

-- register command outputs
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
    if (rst ='1') then
      state_r2 <= idle ;
      state_r3 <= idle ;
      init_state_r2 <= "00000";
      init_state_r3 <= "00000";
    else
      state_r2 <= state;
      state_r3 <= state_r2;
      init_state_r2 <= init_state;
      init_state_r3 <= init_state_r2;
    end if;
  end if;
end process;

-- commands to the memory
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
   if (rst ='1') then
      ddr2_ras_r <= '1';
   elsif (state = LOAD_MODE or state = PRECHARGE or state = ACTIVE or state = AUTO_REFRESH
            or init_state = INIT_LOAD_MODE or init_state = INIT_PRECHARGE or init_state = INIT_ACTIVE or init_state = INIT_AUTO_REFRESH
                or init_state = INIT_DUMMY_ACTIVE ) then
      ddr2_ras_r <= '0';
   else
      ddr2_ras_r <= '1';
  end if;
 end if;
 end process;

-- commands to the memory
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
    if (rst='1') then
       ddr2_cas_r <= '1';
    elsif (state = LOAD_MODE or state = FIRST_WRITE or state = BURST_WRITE or
          state = FIRST_READ or state = BURST_READ or init_state = INIT_DUMMY_FIRST_READ or
            init_state = INIT_LOAD_MODE or init_state = INIT_AUTO_REFRESH or
          state = AUTO_REFRESH or init_state= INIT_DUMMY_READ or init_state = INIT_PATTERN_READ or init_state = INIT_PATTERN_WRITE) then
       ddr2_cas_r <= '0';
    else
       ddr2_cas_r <= '1';
    end if;
  end if;
end process;

-- commands to the memory
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
    if (rst ='1') then
      ddr2_we_r <= '1';
    elsif (state = LOAD_MODE or state = PRECHARGE or state = FIRST_WRITE or state = BURST_WRITE or init_state = INIT_PATTERN_WRITE
            or init_state = INIT_LOAD_MODE or init_state = INIT_PRECHARGE)  then
      ddr2_we_r <= '0';
    else
      ddr2_we_r <= '1';
   end if;
 end if;
end process;

-- register commands to the memory
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
   if (rst='1') then
      ddr2_ras_r2 <= '1';
      ddr2_cas_r2 <= '1';
      ddr2_we_r2 <= '1';
   else
      ddr2_ras_r2  <= ddr2_ras_r;
      ddr2_cas_r2  <= ddr2_cas_r;
      ddr2_we_r2   <= ddr2_we_r;
   end if;
 end if;
end process;

-- register commands to the memory
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
--   if (rst='1')  then
--        ddr2_ras_r3 <= '1';
--        ddr2_cas_r3 <= '1';
--        ddr2_we_r3  <= '1';
--   else
        ddr2_ras_r3  <= ddr2_ras_r2;
        ddr2_cas_r3  <= ddr2_cas_r2;
        ddr2_we_r3   <= ddr2_we_r2;
--  end if;  -- else: !if(rst)
end if;
end process;

process(clk0)
 begin
  if (clk0='1' and clk0'event) then
  if (rst='1') then
      row_addr_r((row_address-1) downto 0) <= (others => '0');
  else
      row_addr_r((row_address-1) downto 0) <= af_addr(((row_address + col_ap_width)-1) downto col_ap_width);
  end if;
  end if;
end process;

-- chip enable generation logic
process(clk0)
 begin
  if (clk0='1' and clk0'event) then
      if (rst = '1')  then
          ddr2_cs_r((cs_width-1) downto 0) <=  (others => '0');
      else
          if (af_addr_r((chip_address + bank_address +row_address + (col_ap_width-1)) downto (bank_address + row_address + col_ap_width)) =
              cs_h0((chip_address - 1) downto 0)) then
                       ddr2_cs_r((cs_width-1) downto  0) <=  cs_hE((cs_width-1) downto  0);
          elsif (af_addr_r((chip_address + bank_address + row_address + (col_ap_width-1)) downto (bank_address +row_address + col_ap_width)) =
                 cs_h1((chip_address - 1) downto 0))   then
                     ddr2_cs_r((cs_width-1) downto 0) <=  cs_hD((cs_width-1) downto  0);
          elsif (af_addr_r((chip_address + bank_address +row_address + (col_ap_width-1)) downto (bank_address + row_address + col_ap_width)) =
                  cs_h2((chip_address - 1) downto 0)) then
                     ddr2_cs_r((cs_width-1) downto 0) <= cs_hB((cs_width-1) downto  0);
         elsif (af_addr_r((chip_address + bank_address +row_address + (col_ap_width-1)) downto (bank_address + row_address + col_ap_width)) =
                    cs_h3((chip_address - 1) downto 0)) then
                         ddr2_cs_r((cs_width-1) downto 0) <=  cs_h7((cs_width-1) downto  0);
         else
                     ddr2_cs_r((cs_width-1) downto 0) <=  cs_hF((cs_width-1) downto  0);
         end if;
      end if;
  end if;
end process;

-- address during init
process(clk0)
 begin
    if (clk0='1' and clk0'event) then
        if (rst = '1') then
            ddr2_address_init_r <= (others =>'0');
        else
            if (init_state_r2 = INIT_PRECHARGE) then
               ddr2_address_init_r <= ( 10 => '1' , others => '0'); --    *A10 = 1 for precharge all *
            elsif (init_state_r2 = INIT_LOAD_MODE and init_count_cp ="0101") then
                 ddr2_address_init_r <= ext_mode_reg;                                           -- A0 == 0 for DLL enable
            elsif (init_state_r2 = INIT_LOAD_MODE and init_count_cp ="0110") then
               ddr2_address_init_r <= add_const1((row_address-1) downto 0) or  load_mode_reg ;
            elsif (init_state_r2 = INIT_LOAD_MODE and init_count_cp ="1010") then
               ddr2_address_init_r <= load_mode_reg;    --  Write recovery = 4; cas_latency = 4; burst length = 4
            elsif (init_state_r2 = INIT_LOAD_MODE and init_count_cp ="1011") then
               ddr2_address_init_r <= (add_const2((row_address-1) downto 0) or ext_mode_reg); -- OCD DEFAULT
            elsif (init_state_r2 = INIT_LOAD_MODE and init_count_cp = "1100") then
               ddr2_address_init_r <= (add_const3((row_address-1) downto 0) or ext_mode_reg);  -- OCD EXIT
            elsif(init_state_r2 = INIT_DUMMY_ACTIVE) then
               ddr2_address_init_r <= row_addr_r;
              else
               ddr2_address_init_r <= (others =>'0');
            end if;
        end if;
    end if;
end process;

process(clk0)
 begin
    if (clk0='1' and clk0'event) then
        if (rst = '1') then
            ddr2_address_r1 <= (others => '0');
        elsif ((state_r2 = ACTIVE )) then
            ddr2_address_r1 <= row_addr_r;
        elsif (state_r2 = FIRST_WRITE or state_r2 = BURST_WRITE or
                state_r2 = FIRST_READ or state_r2 = BURST_READ) then
               ddr2_address_r1 <=  af_addr_r1((row_address-1) downto 0) and add_const4((row_address-1) downto 0);
--           ddr2_address_r1 <= (zeroes(row_address - col_ap_width -1 downto 0 ) &  af_addr_r1((col_ap_width -1) downto 0));
        elsif (state_r2 = PRECHARGE) then
            if(PRE_r = '1') then
            ddr2_address_r1 <= af_addr_r1((row_address-1) downto 0);
            else
            ddr2_address_r1 <= (10 => '1', others => '0');
            end if;
        elsif (state_r2 = LOAD_MODE) then
               ddr2_address_r1 <= af_addr_r1((row_address-1) downto 0);
        else
            ddr2_address_r1 <= (others => '0');
        end if;
    end if;
end process;

process(clk0)
 begin
    if (clk0='1' and clk0'event) then
        if (rst = '1') then
            ddr2_address_r2 <= (others => '0');
        elsif (init_memory = '1') then
            ddr2_address_r2 <= ddr2_address_init_r;
        else
            ddr2_address_r2 <= ddr2_address_r1;
        end if;
    end if;
 end process;

process(clk0)
 begin
    if (clk0='1' and clk0'event) then
        if (rst='1') then
            ddr2_ba_r1((bank_address-1) downto 0) <= ( others => '0');
        elsif ((init_memory = '1') and (init_state_r2 = INIT_LOAD_MODE) ) then
           if (init_count_cp = X"3") then
               ddr2_ba_r1((bank_address-1) downto 0) <=   ( 1 => '1', others => '0');  -- emr2
           elsif (init_count_cp = X"4") then
               ddr2_ba_r1((bank_address-1) downto 0) <= ( 0 | 1 => '1', others => '0'); -- emr3
           elsif (init_count_cp = X"5" or init_count_cp = X"B" or init_count_cp = X"C") then
               ddr2_ba_r1((bank_address-1) downto 0) <= ( 0  => '1', others => '0');  -- emr
           else
               ddr2_ba_r1((bank_address-1) downto 0) <= (others => '0');
           end if;
        elsif ((state_r2 = ACTIVE) or (init_state_r2 = INIT_DUMMY_ACTIVE ) or (state_r2 = LOAD_MODE) or ((state_r2 = PRECHARGE) and (PRE_r='1'))
                or (init_state_r2 = INIT_ACTIVE) or (init_state_r2 = INIT_LOAD_MODE) or (init_state_r2 = INIT_PRECHARGE)) then
              ddr2_ba_r1((bank_address-1) downto 0)  <= af_addr_r(((bank_address + row_address + col_ap_width)-1) downto (col_ap_width + row_address));
       else
         ddr2_ba_r1((bank_address-1) downto 0)  <= ddr2_ba_r1((bank_address-1) downto 0) ;
       end if;
   end if;
end process;

process(clk0)
begin
   if (clk0='1' and clk0'event) then
        if (rst='1') then
             ddr2_ba_r2((bank_address-1) downto 0) <= ( others => '0');
          else
             ddr2_ba_r2((bank_address-1) downto 0) <= ddr2_ba_r1;
          end if;
   end if;
end process;

process(clk0)
 begin
   if (clk0='1' and clk0'event) then
        if (rst = '1') then
          ddr2_cs_r1((cs_width-1) downto 0) <= ( others => '1');
        elsif (init_memory = '1' ) then
           if (chip_cnt = "00") then
                  ddr2_cs_r1((cs_width-1) downto 0) <=  cs_hE((cs_width-1) downto 0);
           elsif (chip_cnt = "01") then

               ddr2_cs_r1((cs_width-1) downto 0) <=   cs_hD((cs_width-1) downto 0);
           elsif (chip_cnt = "10") then

               ddr2_cs_r1((cs_width-1) downto 0) <=  cs_hB((cs_width-1) downto 0);
           elsif (chip_cnt = "11") then

               ddr2_cs_r1((cs_width-1) downto 0) <=  cs_h7((cs_width-1) downto 0);
           else

               ddr2_cs_r1((cs_width-1) downto 0) <= cs_hF((cs_width-1) downto 0);
             end if;
---  Precharge fix for deep memory
        elsif (state_r2 = PRECHARGE ) then
         if (pre_cnt = "001") then
                            ddr2_cs_r1((cs_width-1) downto 0) <=  cs_hE((cs_width-1) downto 0);
         elsif (pre_cnt = "010") then
                      ddr2_cs_r1((cs_width-1) downto 0) <=   cs_hD((cs_width-1) downto 0);
         elsif (pre_cnt = "011") then
                        ddr2_cs_r1((cs_width-1) downto 0) <=  cs_hB((cs_width-1) downto 0);
         elsif (pre_cnt = "100") then
                       ddr2_cs_r1((cs_width-1) downto 0) <=  cs_h7((cs_width-1) downto 0);
         elsif (pre_cnt = "000") then
                ddr2_cs_r1((cs_width-1) downto 0) <= ddr2_cs_r1((cs_width-1) downto 0); --cs_hF
         else
                ddr2_cs_r1((cs_width-1) downto 0) <= cs_hF((cs_width-1) downto 0);
         end if;
        elsif (state_r2 = AUTO_REFRESH) then
           if (auto_cnt = "001") then
                            ddr2_cs_r1((cs_width-1) downto 0) <=  cs_hE((cs_width-1) downto 0);
           elsif (auto_cnt = "010") then
                      ddr2_cs_r1((cs_width-1) downto 0) <=   cs_hD((cs_width-1) downto 0);
           elsif (auto_cnt = "011") then
                        ddr2_cs_r1((cs_width-1) downto 0) <=  cs_hB((cs_width-1) downto 0);
           elsif (auto_cnt = "100") then
                       ddr2_cs_r1((cs_width-1) downto 0) <=  cs_h7((cs_width-1) downto 0);
           else
                ddr2_cs_r1((cs_width-1) downto 0) <= cs_hF((cs_width-1) downto 0);
             end if;
        elsif ((state_r2 = ACTIVE ) or (init_state_r2 = INIT_DUMMY_ACTIVE  ) or (state_r2 = LOAD_MODE) or (state_r2 = PRECHARGE_WAIT )
               or (init_state_r2 = INIT_ACTIVE ) or (init_state_r2 = INIT_LOAD_MODE) or (init_state_r2 = INIT_PRECHARGE_WAIT ))  then
           ddr2_cs_r1((cs_width-1) downto 0) <= ddr2_cs_r((cs_width-1) downto 0);
        else
             ddr2_cs_r1((cs_width-1) downto 0)<= ddr2_cs_r1((cs_width-1) downto 0);
        end if;
   end if;
end process ;

process(clk0)
 begin
    if (clk0 = '1' and clk0'event) then
        if (rst = '1') then
                ddr2_cs_r_out <= (others => '1');
                ddr2_cs_r_odt <= (others => '1');
        else
                ddr2_cs_r_out <= ddr2_cs_r1;
                ddr2_cs_r_odt <= ddr2_cs_r1;
        end if;
    end if;
end process;

process(clk0)
 begin
    if (clk0='1' and clk0'event) then
        if (rst='1') then
            conflict_resolved_r <= '0';
        else
           if ((state = PRECHARGE_WAIT)  and (conflict_detect_r = '1')) then
                conflict_resolved_r  <= '1';
           elsif(af_rden='1') then
                conflict_resolved_r  <= '0';
           end if;
        end if;
    end if;
 end process;

process(clk0)
 begin
  if (clk0='1' and clk0'event) then
    if (rst ='1') then
          ddr2_cke_r  <=cs_h0((cke_width-1) downto 0) ;
    elsif (done_200us ='1') then
         ddr2_cke_r <= cs_hF((cke_width-1) downto 0);
    end if;
  end if;

 end process;

-- odt

process(clk0)
 begin
    if (clk0='1' and clk0'event) then
        if (rst ='1') then
            odt_en_cnt <= "0000";
        elsif((state = FIRST_WRITE) and (ODT_ENABLE = '1')) then
            odt_en_cnt <= (('0' & ADDITIVE_LATENCY_VALUE) + ('0' & CAS_LATENCY_VALUE) - "0010");
        elsif((state = FIRST_READ) and (ODT_ENABLE = '1')) then
            odt_en_cnt <= (('0' & ADDITIVE_LATENCY_VALUE) + ('0' & CAS_LATENCY_VALUE)- "0001");
        elsif(odt_en_cnt /= "0000") then
            odt_en_cnt <= odt_en_cnt - "0001";
        else
            odt_en_cnt <= "0000";
        end if;
    end if;
 end process;

process(clk0)
 begin
    if (clk0='1' and clk0'event) then
        if (rst = '1') then
            odt_cnt <= "0000";
        elsif((state = FIRST_WRITE) or (state = BURST_WRITE)) then
            odt_cnt <= (('0' & ADDITIVE_LATENCY_VALUE) + ( '0'& CAS_LATENCY_VALUE) + ('0' & burst_cnt)+ "0010");    -- AS modified, added + "0010"
        elsif((state = FIRST_READ) or (state = BURST_READ)) then
            odt_cnt <= (('0' & ADDITIVE_LATENCY_VALUE) + ('0' & CAS_LATENCY_VALUE) + ('0' & burst_cnt) + "0010");   -- AS modified was + "0001"
        elsif(odt_cnt /= "0000") then
            odt_cnt <= odt_cnt - "0001";
        else
            odt_cnt <= "0000";
        end if;
    end if;
 end process;

-- odt_en logic is made combinational to add a flop to the ctrl_odt logic

process(rst, odt_en_cnt, odt_cnt)
 begin
        if (rst = '1') then
            odt_en  <= (others => '0');
--        elsif(odt_en_cnt = "0001") then                                       -- AS commented
--            odt_en((cs_width-1) downto 0)  <= cs_hF((cs_width-1) downto 0);
        elsif(odt_cnt > "0000") then                   -- AS modified was (odt_cnt > "0010" and odt_en_cnt <= "0001")
            odt_en((cs_width-1) downto 0)  <= cs_hF((cs_width-1) downto 0);
--        elsif (odt_cnt = "0010") then                                         -- AS commented
  --          odt_en <= (others => '0');
        else
            odt_en <= (others => '0');
        end if;
 end process;

-- added for deep designs




process(clk0)
 begin
    if (clk0='1' and clk0'event) then
        if (rst = '1') then
            ctrl_odt  <= (others => '0');
        else
          case cs_width is
            when 1 =>
               if (ddr2_cs_r_odt((cs_width-1) downto 0) =  cs_h0((cs_width - 1) downto 0)) then
                  ctrl_odt <= (odt_en((cs_width-1) downto 0) and cs_h1((cs_width-1) downto 0));
               else
                  ctrl_odt <= (others => '0');
               end if;

            when 2 =>
              if (ddr2_cs_r_odt((cs_width-1) downto 0)  =  cs_h2((cs_width - 1) downto 0)) then
                   ctrl_odt <= (odt_en((cs_width-1) downto 0) and cs_hA((cs_width-1) downto 0));
               elsif (ddr2_cs_r_odt((cs_width-1) downto 0)  =  cs_h1((cs_width - 1) downto 0)) then
                  ctrl_odt <= (odt_en((cs_width-1) downto 0) and  cs_h1((cs_width-1) downto 0));
               else
                   ctrl_odt <= (others => '0');
               end if;

            when 3 =>
             if (ddr2_cs_r_odt((cs_width-1) downto 0)  =  cs_h6((cs_width - 1) downto 0)) then
                   ctrl_odt <= (odt_en((cs_width-1) downto 0) and cs_D100((cs_width-1) downto 0));
             elsif(ddr2_cs_r_odt((cs_width-1) downto 0) =  cs_h5((cs_width - 1) downto 0))  then
                   ctrl_odt <= (odt_en((cs_width-1) downto 0) and cs_D100((cs_width-1) downto 0));
             elsif (ddr2_cs_r_odt((cs_width-1) downto 0)  = cs_h3((cs_width - 1) downto 0)) then
                   ctrl_odt <= (odt_en((cs_width-1) downto 0) and cs_hA((cs_width-1) downto 0));
             else
                   ctrl_odt <= (others => '0');
             end if;

            when 4 =>
             if (ddr2_cs_r_odt((cs_width-1) downto 0) =  cs_hE((cs_width - 1) downto 0)) then
                   ctrl_odt <= (odt_en((cs_width-1) downto 0) and cs_D1000((cs_width-1) downto 0));
             elsif (ddr2_cs_r_odt((cs_width-1) downto 0)  = cs_hD((cs_width - 1) downto 0)) then
                  ctrl_odt <= (odt_en((cs_width-1) downto 0) and cs_D1000((cs_width-1) downto 0));
             elsif (ddr2_cs_r_odt((cs_width-1) downto 0) = cs_hB((cs_width - 1) downto 0)) then
                   ctrl_odt <= (odt_en((cs_width-1) downto 0) and cs_D1000((cs_width-1) downto 0));
             elsif (ddr2_cs_r_odt((cs_width-1) downto 0)  =    cs_h7((cs_width - 1) downto 0)) then
                   ctrl_odt <= (odt_en((cs_width-1) downto 0) and cs_D100((cs_width-1) downto 0));
             else
                   ctrl_odt <= (others => '0');
             end if;

            when others =>  ctrl_odt <= ctrl_odt;

            end case;
        end if;
    end if;
end process;



  ctrl_ddr2_address((row_address-1) downto 0)  <= ddr2_address_r2((row_address-1) downto 0);
  ctrl_ddr2_ba ((bank_address-1) downto 0)     <= ddr2_ba_r2((bank_address-1) downto 0);
  ctrl_ddr2_ras_L <= ddr2_ras_r3;
  ctrl_ddr2_cas_L <= ddr2_cas_r3;
  ctrl_ddr2_we_L  <= ddr2_we_r3;
  ctrl_ddr2_odt  <= ctrl_odt;
  ctrl_ddr2_cs_L <= ddr2_cs_r_out;

  ctrl_Dqs_Rst  <= ctrl_Dqs_Rst_r1;
  ctrl_Dqs_En   <= ctrl_Dqs_En_r1;
  ctrl_WrEn     <= ctrl_WrEn_r1;
  ctrl_RdEn     <= ctrl_RdEn_r1;
  ctrl_dummy_wr_sel <= ctrl_Wdf_RdEn_int  when (ctrl_dummy_write = '1') else '0';


ctrl_ddr2_cke  <= ddr2_cke_r;

end arc_controller ;


