/*******************************************************************************
 *	TapeStateGen.cc	TMS System State Table Generation utility
 *			T.Barnaby,	BEAM Ltd,	2007-03-14
 *******************************************************************************
 */

#include <stdio.h>
#include <syslog.h>
#include <unistd.h>
#include <getopt.h>
#include <SigGen.h>
#include <Gen.h>

void usage(){
	fprintf(stderr, "tmsStateGen: Version: %s\n", VERSION);
	fprintf(stderr, "Usage: tmsstategen [options] <filename>\n");
	fprintf(stderr, "Generates a Tms State Table for a particular BEAM type to a file.\n");
	fprintf(stderr, "  -t <type>     - Specifies the signal type (beam1 etc)\n");
	fprintf(stderr, "  -T            - List all signal types supported\n");
	fprintf(stderr, "  -p            - Produce Phase Table ASCII dump file\n");
	fprintf(stderr, "  -fd <n>       - Set pllInitialFrequencyDelay to n ms\n");
	fprintf(stderr, "  -phaseDelay <n> - Add this to all PhaseDelay parameters\n");
	fprintf(stderr, "  -phaseDelayAll <n> - Set all PhaseDelay parameters to this value\n");
	fprintf(stderr, "  -fref <n>     - Set FREF to n Hz\n");
	fprintf(stderr, "  -name <n>     - Set name n\n");
	fprintf(stderr, "  -info <n>     - Prefix to standard info\n");
}

static struct option options[] = {
		{ "?",			0, NULL, 0 },
		{ "h",			0, NULL, 0 },
		{ "help",		0, NULL, 0 },

		{ "T",			0, NULL, 0 },
		{ "t",			1, NULL, 0 },
		{ "p",			0, NULL, 0 },

		{ "fd",			1, NULL, 0 },
		{ "phaseDelay",		1, NULL, 0 },
		{ "phaseDelayAll",	1, NULL, 0 },
		{ "fref",		1, NULL, 0 },
		{ "name",		1, NULL, 0 },
		{ "info",		1, NULL, 0 },
		{ 0,0,0,0 }
};

int main(int argc, char** argv){
	BError			err;
	Gen			gen;
	PNameValueList		params;
	BString			s;
	BIter			i;
	int			c;
	int			optIndex = 0;
	BString			fileName = "";
	BString			cycleType = "";
	int			listTypes = 0;
	double			fref = 437000;
	double			sampleRate;
	int			pllInitialFrequencyDelay = 0;
	BString			name;
	BList<BString>		sl;
	
	while((c = getopt_long_only(argc, argv, "", options, &optIndex)) == 0){
		s = options[optIndex].name;
		if(s == "help" || s == "h" || s == "?"){
			usage();
			return 1;
		}
		else if(s == "T"){
			listTypes = 1;
		}
		else if(s == "t"){
			cycleType = optarg;
		}
		else if(s == "p"){
			params.setValue("phaseTableFile", "1");
		}
		else if(s == "fd"){
			pllInitialFrequencyDelay = atoi(optarg);
		}
		else if(s == "phaseDelay"){
			params.setValue("phaseDelay", atoi(optarg));
		}
		else if(s == "phaseDelayAll"){
			params.setValue("phaseDelayAll", atoi(optarg));
		}
		else if(s == "fref"){
			fref = atoi(optarg);
		}
		else if(s == "name"){
			name = optarg;
		}
		else if(s == "info"){
			params.setValue("info", optarg);
		}
	}
	
	if(!listTypes && (optind == argc)){
		usage();
		return 1;
	}

	sampleRate = 125000000;

	if(name == "")
		name = cycleType;

	params.setValue("sampleRate", sampleRate);
	params.setValue("fref", fref);
	params.setValue("fileName", argv[optind++]);
	params.setValue("useMsbFref", 1);
	params.setValue("pllInitialFrequencyDelay", pllInitialFrequencyDelay);
	params.setValue("name", name);

#ifdef ZAP	
	printf("SampleRate: %f\n", sampleRate);
	printf("FRef: %f\n", fref);
	printf("PllInitialFrequencyDelay: %d\n", pllInitialFrequencyDelay);
#endif

	if(listTypes){
		sl = gen.getTypes();
		for(sl.start(i); !sl.isEnd(i); sl.next(i)){
			printf("%s\n", sl[i].retStr());
		}
		return 0;
	}

	if(cycleType == ""){
		usage();
		return 1;
	}
	
	printf("Generating TMS CycleParams for Cycle Type: %s\n", cycleType.retStr());	
	if(err = gen.init(params)){
		fprintf(stderr, "Gen Init Error: %s\n", err.getString().retStr());
		return 1;
	}
	
	if(err = gen.generate(cycleType)){
		fprintf(stderr, "Gen Run Error: %s\n", err.getString().retStr());
		return 1;
	}

	return 0;
}
